/*!
* \file       vd_diaglog_snapshotdata.cpp
*
* \brief      DiagLog Report Record snapshot data
*
* \details    Used to handle the snapshot data related to each ITC.
*
* \component  Diaglog
*
* \ingroup    DiagFramework
*
* \copyright  (c) 2018 Robert Bosch GmbH
*
* The reproduction, distribution and utilization of this file as
* well as the communication of its contents to others without express
* authorization is prohibited. Offenders will be held liable for the
* payment of damages. All rights reserved in the event of the grant
* of a patent, utility model or design.
*/

// first include diaglog settings
#include <common/framework/vd_diaglog_settings.h>


#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_snapshotdata
   #include "vd_diaglog_snapshotdata.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record
   #include <common/framework/vd_diaglog_report_record.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_memory_master
   #include <common/framework/vd_diaglog_memory_master.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_snapshotdata.cpp.trc.h"
#endif

#define DIAGLOG_SNAPSHOT_REPORT_SEM   "DIAGLOG_SNAPSHOT_REPORT_SEM"
OSAL_tSemHandle vdl_tclSnapShotData::m_hReportSem = OSAL_C_INVALID_HANDLE;

#ifndef VARIANT_S_FTR_ENABLE_TRANSACTION_SAFETY
#define DIAGLOG_SNAPSHOT_FILE OSAL_C_STRING_DEVICE_FFS2"/persistent/diagnosis/diaglogsnapshotdata.log"
#else
#define DIAGLOG_SNAPSHOT_FILE "/var/opt/bosch/persistent/diagnosis/diaglogsnapshotdata.log"
#endif

#define SNAPSHOT_ALL_RECORDS     0xFF
#define SNAPSHOT_MAX_DATA_SIZE   10000

/*static*/ tU32 vdl_tclSnapshotRecord::m_HandleCounter = 0;

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::~vdl_tclSnapShotData
//
// DESCRIPTION: destrcutor
//
// PARAMETER:  none
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclSnapShotData::vdl_tclSnapShotData():
vdl_tclStorageUserInterface(DIAGLOG_SNAPSHOT_FILE)
{
   _BP_TRY_BEGIN
   {
      vInit();
   }

   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclSnapShotData::vdl_tclSnapShotData => FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::~vdl_tclSnapShotData
//
// DESCRIPTION: destrcutor
//
// PARAMETER:  none
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclSnapShotData::~vdl_tclSnapShotData()
{
   _BP_TRY_BEGIN
   {
      vClose();
   }

   _BP_CATCH_ALL
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclSnapShotData::~vdl_tclSnapShotData => FAILED"));
      NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
// PRIVATE
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::vInit
//
// DESCRIPTION: Init snapshot data
//
// PARAMETER:  none
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::vInit() 
{
   ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::bOnInit => Create Semaphore m_hReportSem"));
   // Create semaphore to avoid parallel processing
   tS32 s32Result = OSAL_s32SemaphoreCreate(DIAGLOG_SNAPSHOT_REPORT_SEM, &m_hReportSem, 1); 
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      ETG_TRACE_ERRMEM(("!!! Diaglog: vdl_tclSnapShotData::bOnInit => OSAL_s32SemaphoreCreate DIAGLOG_SNAPSHOT_REPORTMEMSEM failed: Result=%x, Error=%x",s32Result, static_cast<tUInt>(u32ErrorCode)));
      NORMAL_M_ASSERT_ALWAYS();
      m_hReportSem = OSAL_C_INVALID_HANDLE;
   }

   if (false == loadDataBaseIfNotLoaded())
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclSnapShotData::vOnInit => Database not loaded"));
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::vClose
//
// DESCRIPTION: Exit snapshot data
//
// PARAMETER:  none
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::vClose() 
{
   ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::vClose => Delete Semaphore m_hReportSem"));

   tS32 s32Result = OSAL_s32SemaphoreDelete(DIAGLOG_SNAPSHOT_REPORT_SEM);
   if(OSAL_OK != s32Result)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: vdl_tclSnapShotData::vClose => OSAL_s32SemaphoreDelete failed: Result=%x, Error=%x",
                               s32Result,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
   m_hReportSem = OSAL_C_INVALID_HANDLE;

   // finally save Database NOW
   saveNowIfModified();
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::requestSnapShotData
//
// DESCRIPTION: go through the list and forward request to get the SnapShot Data
//              from the components/diagnosis
//
// PARAMETER:  none
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::requestSnapShotData()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::requestSnapShotData"));  
   tSnapshotRecordList tempRecordList;
   tSnapshotRecordListIt iter;

   LockSnapShotRecordSemaphore();
   {
      iter = m_SnapShotRecordList.begin();
      while(iter != m_SnapShotRecordList.end())
      {
         // only new elements
         if(iter->m_state == SNAPSHOT_ADDED)
         {
            // copy to temp buffer (do not send diretly in Semaphore!!!)
            tempRecordList.push_back(*iter);
            // set to next state
            iter->m_state = SNAPSHOT_WAIT_FOR_RESPONSE;
         }
         ++iter; // next element
      }
   }
   FreeSnapShotRecordSemaphore();

   // now go through the temp list and send it out
   iter = tempRecordList.begin();
   while(iter != tempRecordList.end())
   {
      // now we left the common part
      (void) requestSnapShotDataBySnapshotRecord(*iter);
      ++iter; // next element
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::requestSnapShotData"));  
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord
//
// DESCRIPTION: request the Data for the requested snapshot record
//              this is the dummy function and should be overwritten by project File
//
// PARAMETER:  snapshotRecord: DID to get from e.g. Diagnostic
//
// RETURNVALUE: void
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord(vdl_tclSnapshotRecord snapshotRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord"));  
   bool bReturn = false;
   tU16 DID = snapshotRecord.m_DID;
   tU32 handle = snapshotRecord.m_Handle;
   ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord => DID:%04x handle:%08x",DID,handle));  
   //lint -e764 // switch statement does not have a case
   switch(DID)
   {
      case 0x4711:
      {
         // dummy
         // set snap shot to 0x12345678 km

         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord => DUMMY Data:0x12345678km"));  
         std::vector<tU8> data;
         data.push_back(0x12);
         data.push_back(0x34);
         data.push_back(0x56);
         data.push_back(0x78);
         copyDataToSnapShot(handle,data);
         break;
      }

      case 0x0815:
      {
         // dummy
         // set snap shot to 0x33445566 

         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord => DUMMY Data:0x33445566"));  
         std::vector<tU8> data;
         data.push_back(0x33);
         data.push_back(0x44);
         data.push_back(0x55);
         data.push_back(0x66);
         copyDataToSnapShot(handle,data);
         break;
      }

      case 0x4444:
      {
         // AgeingCounter
         tU16  ITC  = snapshotRecord.m_ITC; 
         tU8   ID   = snapshotRecord.m_memoryID; 
         tU8 ageingCtr = 0;

         if(vdDiagLog_tclApp::m_poInstance != NULL)
         {
            vdl_tclReportRecord* pRecord = (vdDiagLog_tclApp::m_poInstance)->m_oMemoryMaster.pGetReportRecord(ID, ITC);

            if(pRecord != NULL)
            {
               ageingCtr = pRecord->u8GetAgeingCounter();
            }
            else
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
         }

         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord => AgeingCounter Data:%d",ageingCtr));  
         std::vector<tU8> data;
         data.push_back(ageingCtr);
         copyDataToSnapShot(handle,data);
         break;
      }

      default:
      {
         break;
      }
   }// switch(DID)
   //lint +e764
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::requestSnapShotDataBySnapshotRecord"));  
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::findSnapshotRecordByHandle
//
// DESCRIPTION: search for the record by given handle
//              HINT: must be called with LockSnapShotRecordSemaphore!!!
//
// PARAMETER:  ID: unique record handle
//
// RETURNVALUE: pointer to SnapshotRecord
//
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclSnapshotRecord* vdl_tclSnapShotData::findSnapshotRecordByHandle(tU32 handle)
{
   // search in List
   for (tSnapshotRecordListIt iter = m_SnapShotRecordList.begin(); iter != m_SnapShotRecordList.end(); ++iter)
   {
      if(iter->m_Handle == handle)
      {
         // found
         return &(*iter);
      }
   }
   return NULL;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::removeSnapshotRecordByHandle
//
// DESCRIPTION: search for the record by given handle and delete it
//              HINT: must be called with LockSnapShotRecordSemaphore!!!
//
// PARAMETER:  ID: unique record handle
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::removeSnapshotRecordByHandle(tU32 handle)
{
   // search in List
   for (tSnapshotRecordListIt iter = m_SnapShotRecordList.begin(); iter != m_SnapShotRecordList.end(); ++iter)
   {
      if(iter->m_Handle == handle)
      {
         // found
         (void) m_SnapShotRecordList.erase(iter);
         break;
      }
   }
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::addToRecordMap
//
// DESCRIPTION: add a new List element to existing List for SnapShot Data
//
// PARAMETER:  recordMap      : referenze to record Map
//             snapshotRecord : Data to add
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::addToRecordMap(tSSD_RecordMap& recordMap,vdl_tclSnapshotRecord snapshotRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::addToRecordMap")); 
   tU8 recordId = snapshotRecord.m_RecordId;

   // first find if RecordId still exist
   tSSD_RecordMapIt iter = recordMap.find(recordId);
   if(iter != recordMap.end()) 
   {
      ETG_TRACE_USR4_THR(( "--- vdl_tclSnapShotData::addToRecordMap -> record still in list")); 
      // record still in list
      tSSD_DIDMap& didMap = iter->second; // referenz to second element, only for better understanding
      addToDidMap(didMap, snapshotRecord);
   }
   else 
   {
      // not found
      create_RecordMapMember( recordMap, snapshotRecord);
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::addToRecordMap")); 
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::addToDidMap
//
// DESCRIPTION: add a new List element to existing List for SnapShot Data
//
// PARAMETER:  didMap         : referenze to DID Map
//             snapshotRecord : Data to add
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::addToDidMap(tSSD_DIDMap& didMap,vdl_tclSnapshotRecord snapshotRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::addToDidMap")); 
   tU16 didId = snapshotRecord.m_DID;
   // first find if DID still exist
   tSSD_DIDMapIt iter = didMap.find(didId);
   if(iter != didMap.end()) 
   {
      ETG_TRACE_USR4_THR(( "--- vdl_tclSnapShotData::addToDidMap -> DID still in list")); 
      // DID still in list
      tSSD_Data& data = iter->second; // referenz to second element, only for better understanding

      // this is the lowest elemet so write here the Data
      data = snapshotRecord.m_data; // overwrite
   }
   else 
   {
      // not found
      create_DIDMapMember( didMap, snapshotRecord);
   }
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::addToDidMap")); 
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::create_mainMapMember
//
// DESCRIPTION: create a new List element for SnapShot Data
//
// PARAMETER:  ITCMap         : reference to SnapShotMap
//             snapshotRecord : Data to add
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::create_mainMapMember(tSSD_Map& mainMap, vdl_tclSnapshotRecord snapshotRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::create_mainMapMember")); 

   // first create new recordMap
   tSSD_RecordMap recordMap;
   create_RecordMapMember(recordMap,snapshotRecord);

   // now insert the new pair of identifer and recordMap to mainMap
   mainMap.insert( std::make_pair(snapshotRecord.m_Identifier, recordMap));

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::create_mainMapMember")); 
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::create_RecordMap
//
// DESCRIPTION: create a new List element for SnapShot Data
//
// PARAMETER:  RecordMap      : referenze to Record Map
//             snapshotRecord : Data to add
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::create_RecordMapMember(tSSD_RecordMap& recordMap, vdl_tclSnapshotRecord snapshotRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::create_RecordMapMember")); 

   // first create new didMap
   tSSD_DIDMap didMap;
   create_DIDMapMember(didMap,snapshotRecord);

   // now insert the new pair of reordId and didMap to recordMap
   recordMap.insert( std::make_pair(snapshotRecord.m_RecordId, didMap));

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::create_RecordMapMember")); 
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::create_ITCMap
//
// DESCRIPTION: create a new List element for SnapShot Data
//
// PARAMETER:  DIDMap         : referenze to DID Map
//             snapshotRecord : Data to add
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::create_DIDMapMember(tSSD_DIDMap& didMap, vdl_tclSnapshotRecord snapshotRecord)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::create_DIDMapMember"));  

   // now insert the new pair of DID and data to didMap
   didMap.insert( std::make_pair(snapshotRecord.m_DID,snapshotRecord.m_data));

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::create_DIDMapMember"));  
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclSnapShotData::LockSnapShotRecordSemaphore()
//
// DESCRIPTION: lock the Memory Semaphore
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::LockSnapShotRecordSemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphoreWait(vdl_tclSnapShotData::m_hReportSem, OSAL_C_U32_INFINITE);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: LockSnapShotRecordSemaphore() => OSAL_s32SemaphoreWait failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }

}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclMemoryMaster::FreeMemorySemaphore()
//
// DESCRIPTION: free the Memory Semaphore
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::FreeSnapShotRecordSemaphore() const
{
   tS32 s32Ret = OSAL_s32SemaphorePost(vdl_tclSnapShotData::m_hReportSem);
   if(OSAL_OK != s32Ret)
   {
      tU32 u32ErrorCode= OSAL_u32ErrorCode();
      et_vErrmemStringNormal( TR_COMP_DIAGNOSIS, "Diaglog: FreeSnapShotRecordSemaphore() => OSAL_s32SemaphorePost failed: Result=%x, Error=%x",
                               s32Ret,
                               u32ErrorCode);
      NORMAL_M_ASSERT_ALWAYS();
   }
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: void vdl_tclMemoryMaster::FreeMemorySemaphore()
//
// DESCRIPTION: free the Memory Semaphore
//
// PARAMETER:  none
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::copyDidMapToSnapShotData(tSSD_DIDMap& didMap, vdl_tsSnapShotDataStruct& snapShotData) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::copyDidMapToSnapShotData")); 

   ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::copyDidMapToSnapShotData => didMap Size:%d",didMap.size())); 
   // over all DID's of this Record
   for(tSSD_DIDMapIt didMap_iter = didMap.begin(); didMap_iter != didMap.end(); ++didMap_iter) 
   {
      tU16        did         = didMap_iter->first;   // get did
      tSSD_Data   dataVector  = didMap_iter->second;  // get data vector
      
      ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::copyDidMapToSnapShotData => found DID:%04x",did)); 
      // write the DID to the data (first 2 Bytes)
      snapShotData.data.push_back((tU8)(did >> 8));     // write DID high byte
      snapShotData.data.push_back((tU8)(did & 0xFF));   // write DID low byte

      ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::copyDidMapToSnapShotData => dataVector Size:%d",dataVector.size())); 
      // over all data of this DID
      for(tSSD_DataCIt data_iter = dataVector.begin(); data_iter != dataVector.end(); ++data_iter) 
      {
         tU8 data = *data_iter;  // data 
         snapShotData.data.push_back(data); // write data stream 
      }// for(tSSD_Data data_iter = dataVector.begin(); data_iter != dataVector.end(); ++data_iter) 
   }// for(tSSD_DIDMapCIt didMap_iter = didMap.begin(); didMap_iter != didMap.end(); ++didMap_iter) 
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::copyDidMapToSnapShotData")); 
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::addSnapShotData
//
// DESCRIPTION: add a new Data to SnapShot List
//              -> DTC
//                 -> RecordID
//                    -> DID ID
//                       -> Data
//
// PARAMETER:  snapshotRecord : Data to add
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::addSnapShotData(vdl_tclSnapshotRecord snapshotRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::addSnapShotData"));  
   tU32 identifier = snapshotRecord.m_Identifier;

   ETG_TRACE_USR4_THR(( "--- vdl_tclSnapShotData::addSnapShotData => Memory:%d DTC:%06x RecNO:%02x, DID:%04x - handle:%08x",snapshotRecord.m_memoryID,snapshotRecord.m_DTC,snapshotRecord.m_RecordId,snapshotRecord.m_DID,snapshotRecord.m_Handle));   

   // first find if ITC still exist
   tSSD_MapIt iter = m_SnapShotDataList.find(identifier);
   if(iter != m_SnapShotDataList.end()) 
   {
      // DTC still in list
      tSSD_RecordMap& recordMap = iter->second; // referenz to second element, only for better understanding
      addToRecordMap(recordMap, snapshotRecord);
   }
   else 
   {
      // not found
      create_mainMapMember( m_SnapShotDataList, snapshotRecord);
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::addSnapShotData"));  
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
// PUBLIC
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::getSnapShotDataByDTC
//
// DESCRIPTION: get SnapShotData from Memory, requested by Diagnosis
//
// PARAMETER: MemoryID           : Memory (Customer/BOSCH)
//            DTC                : DTC to read
//            recordNumber       : Record ID to read (0xff = all)
//            SnapShotDataList   : referenz to List, to fill by this function
//
// RETURNVALUE: TRUE = Data valid
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSnapShotData::getSnapShotDataByDTC(tU8 MemoryId, tU32 DTC,  tU8 requestedRecordNumber, tSnapShotDataList& SnapShotDataList, bool isDTCSet)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::getSnapShotDataByDTC")); 
   bool bReturn = true;

   tU32 identifier = calcIdentifier(MemoryId,DTC);
   ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => identifier:%08x",identifier)); 

   LockSnapShotRecordSemaphore();
   {
      // copy the Data to given Buffer (UDS Format) to be save if removed meanwhile
      tSSD_MapCIt itcMap_iter = m_SnapShotDataList.find(identifier);
      if(itcMap_iter != m_SnapShotDataList.end()) 
      {
         tSSD_RecordMap recordMap = itcMap_iter->second; // get Record Map
         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => MemoryId:%d DTC:%06x",MemoryId,DTC)); 

		if(isDTCSet == true)
		{
         switch(requestedRecordNumber)
         {
            case SNAPSHOT_ALL_RECORDS:
            {
               ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => SNAPSHOT_ALL_RECORDS")); 

               // over all records
               for(tSSD_RecordMapCIt recordMap_iter = recordMap.begin(); recordMap_iter != recordMap.end(); ++recordMap_iter) 
               {
                  vdl_tsSnapShotDataStruct oSnapShotData;
                  tU8         recordNumber = recordMap_iter->first;        // get recordNumber from Element
                  tSSD_DIDMap didMap       = recordMap_iter->second;       // get didMap from Element

                  ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => RecordNumber:%d found",requestedRecordNumber)); 
                  oSnapShotData.RecordNumber = recordNumber;               // write the record Number

                  // copy the data from the Map to the result Buffer
                  copyDidMapToSnapShotData(didMap,oSnapShotData);          // write Data
                  oSnapShotData.NumberOfDid =(tU8)didMap.size();               // write number of included DIDs

                  // wirte data to LIST
                  SnapShotDataList.push_back(oSnapShotData);
               }// for(tSSD_RecordMapCIt recordMap_iter = recordMap.begin(); recordMap_iter != recordMap.end(); ++recordMap_iter) 
               break;
            }// case SNAPSHOT_ALL_RECORDS:

            default: // single record
            {
               // search for RecordNumber in Map
               tSSD_RecordMapCIt recordMap_iter = recordMap.find(requestedRecordNumber);
               if(recordMap_iter != recordMap.end()) 
               {
                  // found
                  ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => RecordNumber:%d found",requestedRecordNumber)); 

                  vdl_tsSnapShotDataStruct oSnapShotData;
                  tU8         recordNumber = recordMap_iter->first;        // get recordNumber from Element
                  tSSD_DIDMap didMap       = recordMap_iter->second;       // get didMap from Element

                  oSnapShotData.RecordNumber = recordNumber;               // write the record Number

                  // copy the data from the Map to the result Buffer
                  copyDidMapToSnapShotData(didMap,oSnapShotData);          // write Data
                  oSnapShotData.NumberOfDid =(tU8)didMap.size();               // write number of included DIDs

                  // wirte data to LIST
                  SnapShotDataList.push_back(oSnapShotData);
               }
               else
               {
                  ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => RecordNumber:%d NOT FOUND!!!",requestedRecordNumber)); 
               }
               break;
			}// default: // single record
            }// default: // single record
         }// switch(requestedRecordNumber)
      }// if(itcMap_iter != m_SnapShotDataList.end()) 
      else
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotDataByDTC => DTC:%06x NOT FOUND !!!",DTC)); 
      }
   }
   FreeSnapShotRecordSemaphore();

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::getSnapShotDataByDTC"));  
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::updateSnapShotDataByRecord
//
// DESCRIPTION: this call is done from the record to indicate,
//              that we have to get SnapShot Data
//
// PARAMETER:  pRecord : Pointer to record. 
//             HINT never copy the pointer!!! it could be destroyed!!!
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::updateSnapShotDataByRecord(vdl_tclReportRecord const* pRecord, vdl_teDiaglogSnapShotTrigger trigger)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::updateSnapShotDataByRecord"));  
   if(pRecord == NULL)
   {
      // invalid pointer
      ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::updateSnapShotDataByRecord => pRecord == NULL!!!"));
      return;
   }

   // copy the Data to local buffer, never store the pointer!!!
   tU32 DTC       = pRecord->u32DTC();
   tU16 ITC       = pRecord->u16ITC();
   tU8  memoryId  = pRecord->u8GetMemoryId();
   tU8 const*  recordPtr = pRecord->pGetSnapShotData();   
   if(recordPtr == NULL)
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::updateSnapShotDataByRecord => NO snapshot data found in configuration !!!"));
      // invalid pointer
      return;
   }

   LockSnapShotRecordSemaphore();
   {
      for(int i=0; i < DIAGLOG_MAX_SNAPSHOT_RECORDS_SUPPORTED; i++)
      {
         // over each element in snapshot list
         tU8 const record = recordPtr[i]; 
         if(record != 0)
         {
            // we found a configured record
            tU8 nrOfDids = getNrOfDIDs(record);
            if(nrOfDids > 0)
            {
               for(tU8 didPos=0; didPos < nrOfDids; didPos++)
               {
                  bool valid = false; // need this for differ between Extended and Snapshot Data
                  tU16 DID = getDID(record,didPos);
                  if(DID != 0)
                  {
                     tU32 type = getDIDType(record,didPos);
                     ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::updateSnapShotDataByRecord => Type:%x Trigger:%x",type,trigger));
                     if(  (type == DIAGLOG_FREEZE_FRAME_DATA_TYPE_EXTENDEDDATA)
                        &&(trigger == DIAGLOG_SSD_TRIGGER_BY_READ))
                     {
                        valid = true;
                     }
                     else if(  (type == DIAGLOG_FREEZE_FRAME_DATA_TYPE_SNAPSHOTDATA)
                             &&(trigger == DIAGLOG_SSD_TRIGGER_BY_FAIL))
                     {
                        valid = true;
                     }

                     if(valid == true)
                     {
                        ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::updateSnapShotDataByRecord => create Record:%02d DID:%04x",record,DID));
                        // create new record
                        vdl_tclSnapshotRecord snapshotRecord(DTC,ITC,memoryId,record,DID);
                        m_SnapShotRecordList.push_back(snapshotRecord);
                     }
                  }
                  else
                  {
                     // 0 marks end of List
                     break;
                  }
               }// for(tU8 didPos=0; didPos < DIAGLOG_MAX_EXTENDED_RECORDS_SUPPORTED; didPos++)
            }// if(nrOfDids > 0)
            else
            {
               ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::updateSnapShotDataByRecord => undefined record"));
            }// else // if(nrOfDids > 0)
         }// if(record != 0)
         else
         {
            // 0 marks end of List
            break;
         }
      }// for(int i=0;i<MAX_SNAPSHOT_RECORDS_SUPPORTED; i++)
   }
   FreeSnapShotRecordSemaphore();

   // forward DataRequest
   requestSnapShotData();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::updateSnapShotDataByRecord"));  
}


///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::copyDataToSnapShot
//
// DESCRIPTION: copy received data to memory. 
//              This is called if response to SnapShot Data (handle) is received
//
// PARAMETER:  handle   : unique ID
//             data     : snapshot data (tU8 vector)
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::copyDataToSnapShot(tU32 handle, tSSD_Data data)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::copyDataToSnapShot"));  
   // we must protect it by semaphore, otherwise it could be overwritten meanwhile (asynchrone)
   LockSnapShotRecordSemaphore();
   {
      vdl_tclSnapshotRecord* record = findSnapshotRecordByHandle(handle);
      if(record != NULL)
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::copyDataToSnapShot => Receive Data for handle:%08x",handle));  
         // found the record
         record->m_data = data;
         record->m_state = SNAPSHOT_DATA_RECEIVED;
         ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::copyDataToSnapShot => Memory:%d DTC:%06x RecNO:%02x, DID:%04x - handle:%08x",record->m_memoryID,record->m_DTC,record->m_RecordId,record->m_DID,record->m_Handle));   
         addSnapShotData(*record);
         // the handle is now used, delete it
         removeSnapshotRecordByHandle(handle);
         // do nothing elese after remove !!!
      }
   }
   FreeSnapShotRecordSemaphore();
   saveDatabase();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::copyDataToSnapShot"));  
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::copyDataToSnapShot
//
// DESCRIPTION: copy received data to memory. 
//              This is called if response to SnapShot Data (handle) is received
//
// PARAMETER:  handle   : unique ID
//             data     : snapshot data (tU8 vector)
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::traceSnapShotMap(void)
{
   ETG_TRACE_FATAL(( "*************************************************************************"));  
   ETG_TRACE_FATAL(( "************* vdl_tclSnapShotData::traceSnapShotMap => start ************"));  
   ETG_TRACE_FATAL(( "*************************************************************************"));  
   // we must protect it by semaphore, otherwise it could be overwritten meanwhile (asynchrone)
   LockSnapShotRecordSemaphore();
   {
      ETG_TRACE_FATAL(( "*************************************************************************"));  
      ETG_TRACE_FATAL(( "*************              m_SnapShotDataList               *************"));  
      ETG_TRACE_FATAL(( "*************************************************************************"));  
      ETG_TRACE_FATAL(( "                                                                         "));  
      for(tSSD_MapCIt itcMap_iter = m_SnapShotDataList.begin(); itcMap_iter != m_SnapShotDataList.end(); ++itcMap_iter) 
      {
         tU32 identifier = itcMap_iter->first;           // identifer
         tSSD_RecordMap recordMap = itcMap_iter->second; // get Record Map
         tU32 DTC = getDtcFromIdentifier(identifier);
         tU8 MemoryId = getMemoryIdFromIdentifier(identifier);

         ETG_TRACE_FATAL(( "-> ID:%02x DTC:%06x                                                      ",MemoryId,DTC)); 
         for(tSSD_RecordMapCIt recordMap_iter = recordMap.begin(); recordMap_iter != recordMap.end(); ++recordMap_iter) 
         {
            tU8         recordNumber = recordMap_iter->first;        // get recordNumber from Element
            tSSD_DIDMap didMap       = recordMap_iter->second;       // get didMap from Element
            ETG_TRACE_FATAL(( "   -> Record:%02x                                                          ",recordNumber)); 
            for(tSSD_DIDMapCIt didMap_iter = didMap.begin(); didMap_iter != didMap.end(); ++didMap_iter) 
            {
               tU16        did         = didMap_iter->first;   // get did
               tSSD_Data   dataVector  = didMap_iter->second;  // get data vector
               
               ETG_TRACE_FATAL(( "      -> DID:%04x                                                        ",did)); 
               tU16 size = (tU16)dataVector.size();
               tU8 aData[size];
               tU8 i=0;
               // over all data of this DID
               for(tSSD_DataCIt data_iter = dataVector.begin(); data_iter != dataVector.end(); ++data_iter) 
               {
                  aData[i++]=*data_iter;
               }
               ETG_TRACE_FATAL(( "         -> DATA:%*x                                                      ",ETG_LIST_LEN(size), ETG_LIST_PTR_T8(aData)));  //lint !e772 // Trace only
            }// for(tSSD_DIDMapCIt didMap_iter = didMap.begin(); didMap_iter != didMap.end(); ++didMap_iter) 
         }
      }
      ETG_TRACE_FATAL(( "*************************************************************************"));  
      ETG_TRACE_FATAL(( "*************              m_SnapShotRecordList             *************"));  
      ETG_TRACE_FATAL(( "*************************************************************************"));  
      ETG_TRACE_FATAL(( "                                                                         "));  
      for (tSnapshotRecordListIt iter = m_SnapShotRecordList.begin(); iter != m_SnapShotRecordList.end(); ++iter)
      {
         tU32 handle = iter->m_Handle;
         tU8  MemId  = iter->m_memoryID;
         tU32 DTC    = iter->m_DTC;
         tU16 ITC    = iter->m_ITC;
         tU8  RecNo  = iter->m_RecordId;
         tU16 DID    = iter->m_DID;
         ETG_TRACE_FATAL(("Handle:%08x MemID:%d DTC:%06x ITC:%04x RecNO:%02x DID:%04x             ",handle,MemId,DTC,ITC,RecNo,DID)); 
      }

   }
   FreeSnapShotRecordSemaphore();
   ETG_TRACE_FATAL(( "*************************************************************************"));  
   ETG_TRACE_FATAL(( "************* vdl_tclSnapShotData::traceSnapShotMap => end   ************"));  
   ETG_TRACE_FATAL(( "*************************************************************************"));  
}
///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::getSnapShotIdList
//
// DESCRIPTION: get SnapShotID List from Memory, requested by Diagnosis
//
// PARAMETER:   MemoryId         : only DTC belonging to this memory
//              SnapShotIdList   : reference to SnapShot LIst
//
// RETURNVALUE: true if successfull
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSnapShotData::getSnapShotIdList(tU8 MemoryId, tSnapShotIdList& SnapShotIdList)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::getSnapShotIdList"));  
   bool bReturn = true;

   ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotIdList => MemoryID:%d",MemoryId));  
   vdl_tsSnapShotIdStruct oSnapShotIdElement;

   for(tSSD_MapCIt itc_iter = m_SnapShotDataList.begin(); itc_iter != m_SnapShotDataList.end(); ++itc_iter) 
   {
      tU32              identifier  = itc_iter->first;   // get identifier
      tSSD_RecordMap    recordMap   = itc_iter->second;  // get record Map

      // only use if DTC belong to correct memory
      if(isIdentifierOfMemoryID(MemoryId,identifier))
      {
         for(tSSD_RecordMapCIt record_iter = recordMap.begin(); record_iter != recordMap.end(); ++record_iter) 
         {

            tU8 recordNumber = record_iter->first;   // get recordNumber
            oSnapShotIdElement.DTC           = getDtcFromIdentifier(identifier);
            oSnapShotIdElement.RecordNumber  = recordNumber;
            ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::getSnapShotIdList => add MemoryID:%d DTC:%06x RecNo:%02x",MemoryId,oSnapShotIdElement.DTC,oSnapShotIdElement.RecordNumber));  
            SnapShotIdList.push_back(oSnapShotIdElement);
         }   
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::getSnapShotIdList"));  
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::clearSnapShotByDTC
//
// DESCRIPTION: clear the SnapShot from the List
//
// PARAMETER: MemoryID           : Memory (Customer/BOSCH)
//            DTC                : DTC to read
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
void vdl_tclSnapShotData::clearSnapShotByDTC(tU8 MemoryId, tU32 DTC)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::clearSnapShotByDTC"));  
   tU32 identifier = calcIdentifier(MemoryId,DTC);
   // we must protect it by semaphore, otherwise it could be overwritten meanwhile (asynchrone)
   LockSnapShotRecordSemaphore();
   {
      // first find if ITC still exist
      tSSD_MapIt iter = m_SnapShotDataList.find(identifier);
      if(iter != m_SnapShotDataList.end()) 
      {
         // found
         ETG_TRACE_USR1_THR(( "--- vdl_tclSnapShotData::clearSnapShotByDTC => remove MemoryId:%d DTC:%06x",MemoryId,DTC));  
         m_SnapShotDataList.erase (iter);
      }
   }
   FreeSnapShotRecordSemaphore();
   saveDatabase();
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::clearSnapShotByDTC"));  
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::vSave
//
// DESCRIPTION: SAVE data
//
// PARAMETER:  tStream - array to fill with class member data
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSnapShotData::vSave(tStream& roData)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::vSave"));  
   
   tU32 size=0;

   tU8 aData[SNAPSHOT_MAX_DATA_SIZE+100];
   LockSnapShotRecordSemaphore();
   {
      // write noOfIdentifier
      tU16  noOfIdentifier  = (tU16)m_SnapShotDataList.size();    // get noOfIdentifier
      aData[size++] = static_cast<tU8>((noOfIdentifier >> 8)  & 0xFF);
      aData[size++] = static_cast<tU8>(noOfIdentifier & 0xFF);

      // first collect Data to an Byte-Array
      for(tSSD_MapCIt itcMap_iter = m_SnapShotDataList.begin(); itcMap_iter != m_SnapShotDataList.end(); ++itcMap_iter) 
      {
         tU16 dtcSize=0;
         tU32 identifier = itcMap_iter->first;           // identifer
         tSSD_RecordMap recordMap = itcMap_iter->second; // get Record Map

         // write identifier
         aData[size++] = static_cast<tU8>((identifier >> 24) & 0xFF);
         aData[size++] = static_cast<tU8>((identifier >> 16) & 0xFF);
         aData[size++] = static_cast<tU8>((identifier >> 8)  & 0xFF);
         aData[size++] = static_cast<tU8>(identifier & 0xFF);
         ETG_TRACE_USR2_THR(( "-> Identifier:%08x                                                      ",identifier));

         // write noOfRecords
         tU8  noOfRecords  = (tU8)recordMap.size();    // get No of Records
         aData[size++] = noOfRecords;

         for(tSSD_RecordMapCIt recordMap_iter = recordMap.begin(); recordMap_iter != recordMap.end(); ++recordMap_iter) 
         {
            tU8         recordNumber = recordMap_iter->first;        // get recordNumber from Element
            tSSD_DIDMap didMap       = recordMap_iter->second;       // get didMap from Element

            // write recordNumber
            aData[size++] = recordNumber;
            ETG_TRACE_USR2_THR(( "   -> Record:%02x                                                          ",recordNumber)); 

            // write noOFDids
            tU8  noOFDids  = (tU8)didMap.size();    // get No of Records
            aData[size++] = noOFDids;

            for(tSSD_DIDMapCIt didMap_iter = didMap.begin(); didMap_iter != didMap.end(); ++didMap_iter) 
            {
               tU16        did         = didMap_iter->first;   // get did
               tSSD_Data   dataVector  = didMap_iter->second;  // get data vector
             
               // write DID
               aData[size++] = static_cast<tU8>((did >> 8) & 0xFF);
               aData[size++] = static_cast<tU8>(did & 0xFF);
               ETG_TRACE_USR2_THR(( "      -> DID:%04x                                                        ",did)); 

               // write noOFData
               tU16  noOFData = (tU16)dataVector.size();    // get Size of Vector
               aData[size++] = static_cast<tU8>((noOFData >> 8) & 0xFF);      
               aData[size++] = static_cast<tU8>(noOFData & 0xFF);
               ETG_TRACE_USR2_THR(( "        -> noOFData:%d                                                      ",noOFData)); 

               tU8 aVecData[noOFData];
               tU8 vecIter = 0;
               // over all data of this DID
               for(tSSD_DataCIt data_iter = dataVector.begin(); data_iter != dataVector.end(); ++data_iter) 
               {
                  // write data
                  aData[size++]=*data_iter; 
                  aVecData[vecIter++]=*data_iter;
               }
               ETG_TRACE_USR2_THR(( "         -> DATA:%*x                                                      ",ETG_LIST_LEN(noOFData), ETG_LIST_PTR_T8(aVecData))); //lint !e772 // Trace only

               // this is a komplex structure, so we add a validator here, this must be end of each Element
               tU16 validator = 0xAFFE;
               aData[size++] = static_cast<tU8>((validator >> 8) & 0xFF);      
               aData[size++] = static_cast<tU8>(validator & 0xFF);

            }// for(tSSD_DIDMapCIt didMap_iter = didMap.begin(); didMap_iter != didMap.end(); ++didMap_iter) 
         }
         if((size+dtcSize)>SNAPSHOT_MAX_DATA_SIZE)//lint !e845 // The right argument to operator '+' is certain to be 0  // dtcSize could be 0 in first round
         {
            ETG_TRACE_FATAL_THR(( "!!! vdl_tclSnapShotData::vSave => to much Data %d > %d",size,SNAPSHOT_MAX_DATA_SIZE)); 
            break;
         }
         size += dtcSize; //lint !e845 
      }
   }
   FreeSnapShotRecordSemaphore();
   ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::vSave => Write %d Bytes",size)); 

   for(tU32 i=0;i<size;i++)
   {
      roData.push_back(static_cast<tS8>(aData[i]));
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::vSave"));  
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::bLoad
//
// DESCRIPTION: LOAD data
//
// PARAMETER:  tStream - array to fill with class member data
//
// RETURNVALUE: bool true=succesfull / false=failed
//
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSnapShotData::bLoad(tStream& roData)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSnapShotData::bLoad")); 
   bool bReturn            = true;
   tStreamCIterator iData  = roData.begin();
   tS32 size               = (tS32)roData.size();
   tS32 s32BytesRead       = 0;
   tU8 u8Value;

   ETG_TRACE_USR2_THR(( "--- vdl_tclSnapShotData::bLoad => Size:%d",size)); 
   if(size < 2) // sizeof noOfIdentifier
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => data to short [noOfIdentifier], recreate a new one")); 
      return false;
   }
  
   // read noOfIdentifier
   tU16 noOfIdentifier;
   u8Value = static_cast<tU8>(*iData++);
   noOfIdentifier = static_cast<tU16>(u8Value << 8);
   u8Value = static_cast<tU8>(*iData++);
   noOfIdentifier = static_cast<tU16>(noOfIdentifier | u8Value);
   s32BytesRead+= 2;

   if((size-s32BytesRead) < 5) // sizeof identifier + noOfRecords
   {
      ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => data to short [identifier + noOfRecords], recreate a new one")); 
      return false;
   }

   for (int i=0; i<noOfIdentifier;i++)
   {
      // read Identifier
      tU32 identifier;
      u8Value = static_cast<tU8>(*iData++);
      identifier = (static_cast<tU32>(u8Value) << 24);
      u8Value = static_cast<tU8>(*iData++);
      identifier |= (static_cast<tU32>(u8Value) << 16);
      u8Value = static_cast<tU8>(*iData++);
      identifier |= (static_cast<tU32>(u8Value) << 8);
      u8Value = static_cast<tU8>(*iData++);
      identifier |= static_cast<tU32>(u8Value);
      s32BytesRead+= 4;
      ETG_TRACE_USR2_THR(( "-> Identifier:%08x                                                      ",identifier));

      // read Number of Records
      tU8 noOfRecords = static_cast<tU8>(*iData++);
      s32BytesRead+= 1;

      if((size-s32BytesRead) < 2) // sizeof RecordNumber + noOfDids
      {
         ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => data to short [RecordNumber + noOfDids], recreate a new one")); 
         return false;
      }

      for (int j=0; j<noOfRecords;j++)
      {
         // read RecordNumber
         tU8 recordNumber = static_cast<tU8>(*iData++);
         s32BytesRead+= 1;
         ETG_TRACE_USR2_THR(( "   -> Record:%02x                                                          ",recordNumber)); 

         // read noOfIdentifier
         tU8 noOfDids = static_cast<tU8>(*iData++);
         s32BytesRead+= 1;

         if((size-s32BytesRead) < 4) // sizeof did + noOfData
         {
            ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => data to short [did + noOfData], recreate a new one")); 
            return false;
         }

         for (int k=0; k<noOfDids;k++)
         {
            // read did
            tU16 did;
            u8Value = static_cast<tU8>(*iData++);
            did = static_cast<tU16>(u8Value << 8);
            u8Value = static_cast<tU8>(*iData++);
            did = static_cast<tU16>(did | u8Value);
            s32BytesRead+= 2;
            ETG_TRACE_USR2_THR(( "      -> DID:%04x                                                        ",did)); 

            // read noOfData
            tU16 noOfData;
            u8Value = static_cast<tU8>(*iData++);
            noOfData = static_cast<tU16>(u8Value << 8);
            u8Value = static_cast<tU8>(*iData++);
            noOfData = static_cast<tU16>(noOfData | u8Value);
            s32BytesRead+= 2;
            ETG_TRACE_USR2_THR(( "        -> NoOFData:%d                                                      ",noOfData)); 

            std::vector<tU8> data;
            for (int l=0; l<noOfData;l++)
            {
               // read data
               data.push_back((tU8)(*iData++));
               s32BytesRead++;
            }// for (int l=0; l<noOfData;l++)

            if((size-s32BytesRead) < 0) 
            {
               // this we can check not before
               ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => data to short [data], recreate a new one")); 
               return false;
            }

            if((size-s32BytesRead) < 2) 
            {
               ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => data to short [validator], recreate a new one")); 
               return false;
            }
            // read validator
            tU16 validator;
            u8Value = static_cast<tU8>(*iData++);
            validator = static_cast<tU16>(u8Value << 8);
            u8Value = static_cast<tU8>(*iData++);
            validator = static_cast<tU16>(validator | u8Value);
            s32BytesRead+= 2;
            if(validator == 0xAFFE)
            {
               // if we found the validator data can be used independed if the following are invalid
               vdl_tclSnapshotRecord record(identifier,recordNumber,did,data);
               addSnapShotData(record);
            }
            else
            {
               ETG_TRACE_ERR_THR(( "!!! vdl_tclSnapShotData::bLoad => wrong validator")); 
               return false;
            }
         }// for (int k=0; k<noOfDids;k++)
      }// for (int j=0; j<noOFRecords;j++)
   }// for (int i=0; i<noOfIdentifier;i++)



   ETG_TRACE_USR3_THR(( "<-- vdl_tclSnapShotData::bLoad")); 
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
//
// FUNCTION: vdl_tclSnapShotData::vGetSizeOfLoadBuffer
//
// DESCRIPTION: GetSize of expected File
//
// PARAMETER:  Max : referenz to Maxvalue
//             Min : referenz to Minvalue
//
// RETURNVALUE: none
//
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSnapShotData::vGetSizeOfLoadBuffer(tS32& Max, tS32& Min)
{
   // we have no fix size so read what you get
   Max=SNAPSHOT_MAX_DATA_SIZE; 
   Min=0; 
}
