//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_report_record.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Report Record base class
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 27.10.05  | ESA2 Barber                  | Initial version
// 15.10.12  | BSOT Plischke                | new Diaglog 
//
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_ageing_single
   #include "vd_diaglog_ageing_single.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_trace
   #include <common/framework/vd_diaglog_trace.h>  
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record_const_data
   #include <common/framework/vd_diaglog_report_record_const_data.h>
#endif

/* --Includes. */
/*
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_ageing_single.cpp.trc.h"   
#endif

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclSingleAgeing::vdl_tclSingleAgeing
// 
// DESCRIPTION: constructor
//
// PARAMETER:  none
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclSingleAgeing::vdl_tclSingleAgeing(vdl_tclReportMemory* const pMemoryObject):
   m_pMemoryObject(pMemoryObject)
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::vdl_tclSingleAgeing"));

   if(NULL == pMemoryObject)
   {
      // should never happen
      NORMAL_M_ASSERT_ALWAYS();
   }//if(NULL == pMemoryObject)
   else
   {
      tReportArray oReports = m_pMemoryObject->m_oReports;

      for(tReportArrayCIt itReport =  oReports.begin();
          itReport != oReports.end();
          ++itReport)
      {
         vdl_tclReportRecord* pReportRecord = (*itReport); // get report record
         if(NULL != pReportRecord)
         {
            vdl_teDiaglogAgeingCycle configuredAgeingTrigger = pReportRecord->getAgeingTrigger();
            if(configuredAgeingTrigger == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT) //lint !e774 !e506
            {
               pReportRecord->m_u8AgeingCounter = 255;
            }
            else
            {
               pReportRecord->m_u8AgeingCounter = 0;
            }
         }// if(NULL != pReportRecord)
      }// for(tReportArrayCIt itReport =  oReports.begin();...
   }// else // if(NULL == pMemoryObject)

	ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::vdl_tclSingleAgeing"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::~vdl_tclReportRecord
// 
// DESCRIPTION: default destructor
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclSingleAgeing::~vdl_tclSingleAgeing()
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::~vdl_tclSingleAgeing"));
	ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::~vdl_tclSingleAgeing"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclSingleAgeing::processSystemEvent()
// 
// DESCRIPTION: call for Power Event change
//
// PARAMETER:  systemEvent : Power Event
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSingleAgeing::processSystemEvent(vdl_teDiaglogSystemEvent systemEvent)
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::processSystemEvent"));

   switch(systemEvent)
   {
      case DIAGLOG_EVENT_IGNITION_CYCLE_START:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_IGNITION_CYCLE_START"));
         vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_IGNITION_CYCLE_START);
         break;
      }
      case DIAGLOG_EVENT_IGNITION_CYCLE_END:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_IGNITION_CYCLE_END"));
         vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_IGNITION_CYCLE_END);
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_START:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_OPERATION_CYCLE_START"));
         vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_OPERATION_CYCLE_START);
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_END:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_OPERATION_CYCLE_END"));
         vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_OPERATION_CYCLE_END);
         break;
      }
      case DIAGLOG_EVENT_ACC_CYCLE_START:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_ACC_CYCLE_START:"));
         vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_ACC_CYCLE_START);
         break;
      }
      case DIAGLOG_EVENT_ACC_CYCLE_END:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_ACC_CYCLE_END"));
         vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_ACC_CYCLE_END);
         break;
      }
      case DIAGLOG_EVENT_CRANK_CYCLE_START:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_CRANK_CYCLE_START"));
         // unused
         break;
      }
      case DIAGLOG_EVENT_CRANK_CYCLE_END:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::processSystemEvent => DIAGLOG_EVENT_CRANK_CYCLE_END"));
         // unused
         break;
      }
      default:
      {
         // this one should support all, if used or not, just not to forget one
         ETG_TRACE_FATAL_THR(( "!!! vdl_tclSingleAgeing::processSystemEvent => systemEvent:%d not supported",systemEvent));
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }// switch(systemEvent)


	ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::processSystemEvent"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclSingleAgeing::vOnAgeingEvent
// 
// DESCRIPTION: external triggered AgeingEvent
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSingleAgeing::vOnAgeingEvent(const tU8 u8AgeingCounter) 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::vOnAgeingEvent"));  

   vCheckForAgeing(DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT, u8AgeingCounter);

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::vOnAgeingEvent"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclSingleAgeing::onClear
// 
// DESCRIPTION: Clear received
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSingleAgeing::onClearAll() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::onClearAll"));  

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::onClearAll"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclSingleAgeing::onClearSingle
// 
// DESCRIPTION: Clear received
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSingleAgeing::onClearSingle(vdl_tclReportRecord* pRecord) 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::onClearAll"));  

   if(pRecord != NULL)
   {
      vdl_teDiaglogAgeingCycle configuredAgeingTrigger = pRecord->getAgeingTrigger();

      if(configuredAgeingTrigger == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT) //lint !e774 !e506
      {
         pRecord->m_u8AgeingCounter = 255;
      }
      else
      {
         pRecord->m_u8AgeingCounter = 0;
      }
   }// if(pRecord != NULL)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::onClearAll"));
}



///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclSingleAgeing::onFailed
// 
// DESCRIPTION: Failed received
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSingleAgeing::onFailed(vdl_tclReportRecord* pRecord)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::onFailed"));  

   if(pRecord != NULL)
   {
      vdl_teDiaglogAgeingCycle configuredAgeingTrigger = pRecord->getAgeingTrigger();

      if(configuredAgeingTrigger == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT) //lint !e774 !e506
      {
         tU16 AgingCounter = tU16(vdl_tclAgeing::u8GetExternalAgeingCounter() + 40);
         if(AgingCounter >= 255)
         {
            pRecord->m_u8AgeingCounter = (tU8) (AgingCounter - 255);
         }
         else
         {
            pRecord->m_u8AgeingCounter = (tU8) (AgingCounter);
         }
         ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::onFailed => DTC:%06x AgeingCounter:%d",pRecord->u32DTC(),pRecord->m_u8AgeingCounter));   
      }
      else
      {
         pRecord->m_u8AgeingCounter = 0;
      }
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::onFailed"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclSingleAgeing::vDoAgeing
// 
// DESCRIPTION: called to increment ageing counter
//
// PARAMETER:  none
//
// RETURNVALUE: true:   ageing finished
//              false:  ageing still running
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSingleAgeing::doAgeing(vdl_tclReportRecord* const pReportRecord) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::doAgeing"));
   bool bReturn = false;

   if(NULL != pReportRecord)
   {
      tU8 u8MaxAge = pReportRecord->u8GetMaxAge();
      // increment ageing counter but not above max age
      if(pReportRecord->m_u8AgeingCounter < u8MaxAge)
      {
         pReportRecord->m_u8AgeingCounter++;
      }
      ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::doAgeing => MemoryID:%d DTC:%x m_u8AgeingCounter=%d MaxAge=%d",pReportRecord->u8GetMemoryId(),pReportRecord->u32DTC(), pReportRecord->m_u8AgeingCounter,u8MaxAge));
      
      if((pReportRecord->m_u8AgeingCounter) >= u8MaxAge)
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::doAgeing => CLEAR DTC:%x",pReportRecord->u32DTC()));
         // max Age reached, relaern now
         bReturn = true;
      }// if((pReportRecord->m_u8AgeingCounter) >= u8MaxAge)
   }// if(NULL != pReportRecord)
   else
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclSingleAgeing::doAgeing => pReportRecord = NULL"));
      NORMAL_M_ASSERT_ALWAYS();
   }


   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::vDoAgeing"));
   return bReturn;
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclSingleAgeing::doAgeingByExternalAgeingCounter
// 
// DESCRIPTION: called to increment ageing counter
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSingleAgeing::doAgeingByExternalAgeingCounter(const vdl_tclReportRecord* const pReportRecord, const tU8 u8MessageCounter) const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::doAgeingByExternalAgeingCounter"));
   bool bReturn = false;

   if(NULL != pReportRecord)
   {
      if(pReportRecord->m_u8AgeingCounter == u8MessageCounter)
      {
         ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::doAgeingByExternalAgeingCounter => CLEAR DTC:%x",pReportRecord->u32DTC()));
         // max Age reached, relaern now
         bReturn = true;
      }// if(pReportRecord->m_u8AgeingCounter == u8MessageCounter)
   }// if(NULL != pReportRecord)
   else
   {
      ETG_TRACE_ERRMEM(( "!!! vdl_tclSingleAgeing::doAgeingByExternalAgeingCounter => pReportRecord = NULL"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::doAgeingByExternalAgeingCounter"));
   return bReturn;
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclSingleAgeing::bLoad
// 
// DESCRIPTION: load raw vdl_tclSingleAgeing RAM data into class members
//
// PARAMETER:  vdl_tclDiagLogObject::tStreamCIterator - array of log file data
//
// RETURNVALUE: bool true=succesfull / false=failed
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclSingleAgeing::bLoad(tStreamCIterator& /*iPos*/)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::bLoad"));

   // nothing to do here, handeld by Record

   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::bLoad"));

   return true;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclSingleAgeing::vSave
// 
// DESCRIPTION: save raw vdl_tclSingleAgeing RAM data
//
// PARAMETER:  tStream - array to fill with class member data
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclSingleAgeing::vSave(tStream& /*roData*/)const
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::vSave"));

   // nothing to do here, handeld by Record
     
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::vSave"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclSingleAgeing::vCheckForAgeing
// 
// DESCRIPTION: check if we have to age
// the following is NOT valid for DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT
// a)
// m_bAgeingAllowed = false & m_u8MaxAge = e.g.40
// -> age DTC, but do not clear if Max reached
//
// b)
// m_bAgeingAllowed = true & m_u8MaxAge = e.g.40
// -> age DTC and clear if Max reached
//
// c)
// m_bAgeingAllowed = false & m_u8MaxAge = 0
// -> do not age and do not clear
//
// d)
// m_bAgeingAllowed = true & m_u8MaxAge = 0
// -> only valid for one Cycle, clear after this cycle
//
// PARAMETER:  const vdl_teDiaglogAgeingCycle ageingEvent: Event Trigger
//             const tU8 u8MessageCounter                : AgegingCounter, only valid for External Event
//
// RETURNVALUE: tVoid
//
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclSingleAgeing::vCheckForAgeing(const vdl_teDiaglogAgeingCycle ageingEvent, const tU8 u8MessageCounter)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclSingleAgeing::vCheckForAgeing"));
   if( m_pMemoryObject == NULL)
   {
      NORMAL_M_ASSERT_ALWAYS();
      return;
   }
   tReportArray oReports = m_pMemoryObject->m_oReports;
   // over all recoreds
   tReportArrayIt iPos = oReports.begin();
   while(iPos!=(oReports.end()))
   {
      vdl_tclReportRecord* pReportRecord = (*iPos); // get report record
      if(NULL != pReportRecord)
      {
         // is forbidden by configuration?
         if(  (false == pReportRecord->bIsAgeingAllowedByConf())
            &&(pReportRecord->u8GetMaxAge() == 0))
         {
            // this object is forbidden to age
            ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::vCheckForAgeing => DTC:%x; bIsAgeingAllowedByConf = FALSE && MaxAge = 0 => IGNORE",pReportRecord->u32DTC()));
            ++iPos; // next iterartor
            continue;
         }// if(false == pReportRecord->bIsAgeingAllowedByConf())

         // check for configuration
         vdl_teDiaglogAgeingCycle configuredAgeingTrigger = pReportRecord->getAgeingTrigger();

         // check if trigger is defined
         if(configuredAgeingTrigger == DIAGLOG_AGEING_CYCLE_UNUSED)
         {
            ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::vCheckForAgeing => DTC:%x; do not have an ageing trigger defined => IGNORE",pReportRecord->u32DTC()));
            ++iPos; // next iterartor
            continue;
         }// if(false == pReportRecord->bIsAgeingAllowedByConf())

         // check if the Ageing Event match
         if(configuredAgeingTrigger != ageingEvent)
         {
            // this object has not to age by this event
            ETG_TRACE_USR4_THR(( "--- vdl_tclSingleAgeing::vCheckForAgeing => DTC:%x; wrong Event %x<>%x => IGNORE",pReportRecord->u32DTC(), configuredAgeingTrigger,ageingEvent));
            ++iPos; // next iterartor
            continue;
         }// if(false == pReportRecord->bIsAgeingAllowedByConf())

         // is forbidden in case of Blockmode?
         if(AGEING_BLOCKMODE == DIAGLOG_AGEING_BLOCKING_BY_BLOCKINGMODE_DIAG)//lint !e774 !e506
         {
            // get current Blocking Mode and check if Diagnosis ia actuall Blocking
            tU32 u32BlockingModeActive = (  (vdDiagLog_tclApp::m_poInstance)->u32GetSystemBlockingMode()   // get current system Blockmode
                                          & pReportRecord->u32GetDTCBlockingMode()                         // get Record BlockMode condiation
                                          & DIAGLOG_BLOCKING_MODE_DIAGNOSIS);                              // only check for DIAGLOG_BLOCKING_MODE_DIAGNOSIS

            if(u32BlockingModeActive)
            {
               // this one has not to check
               ETG_TRACE_USR1_THR(( "--- vdl_tclSingleAgeing::vCheckForAgeing => DTC:%x AGEING_BLOCKMODE = DIAGLOG_AGEING_BLOCKING_BY_BLOCKINGMODE_DIAG",pReportRecord->u32DTC()));
               ++iPos; // next iterartor
               continue;
            }
         }// if(AGEING_BLOCKMODE == DIAGLOG_AGEING_BLOCKING_BY_BLOCKINGMODE_DIAG)

         ETG_TRACE_USR2_THR(( "--- vdl_tclSingleAgeing::vCheckForAgeing => DTC:%x  Status=%x",pReportRecord->u32DTC(), pReportRecord->u32GetStatus()));

         // check if statusbits allow ageing
         if(  (pReportRecord->bIsDTCEnabled() == true)                                          // is calibrated
            &&(pReportRecord->bGetIsHistory() == true)                                          // is currently confirmed
            &&(pReportRecord->bGetIsActive() == false))                                         // is currently NOT active
         {
            if(ageingEvent == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT)
            {
               if(true == pReportRecord->bIsAgeingAllowedByConf())
               {
                  // forbidden, will be ignored!!!
                  NORMAL_M_ASSERT_ALWAYS();
               }
               if(doAgeingByExternalAgeingCounter(pReportRecord, u8MessageCounter)== true)
               {
                  // true means ageing finished
                  if(m_pMemoryObject!=NULL)
                  {
                     m_pMemoryObject->vAgeingCompleted(pReportRecord);
                     if(pReportRecord->isDynamicalDTC())
                     {
                        // we have to delete it
                        iPos = m_pMemoryObject->deleteRecordByIter(iPos);
                        continue; // delete returns iPos to next report
                     }// if(pReportRecord->isDynamicalDTC()
                     else
                     {
                        // we have only to clear it
                        pReportRecord->vClear();
                     }// else // if(pReportRecord->isDynamicalDTC()
                  }// if(m_pMemoryObject!=NULL)
               }// if(doAgeingByExternalAgeingCounter(pReportRecord)== true)
            }// if(ageingEvent == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT)
            else
            {
               if(pReportRecord->u8GetAgeingCounter()!= pReportRecord->u8GetMaxAge())   // max age still reached (only for not deleteable DTC'S)
               {
                  if(doAgeing(pReportRecord)== true)
                  {
                     // true means ageing finished
                     if(m_pMemoryObject!=NULL)
                     {
                        m_pMemoryObject->vAgeingCompleted(pReportRecord);

                        if(pReportRecord->bIsAgeingAllowedByConf())
                        {
                           if(pReportRecord->isDynamicalDTC())
                           {
                              // we have to delete it
                              iPos = m_pMemoryObject->deleteRecordByIter(iPos);
                              continue; // delete returns iPos to next report
                           }// if(pReportRecord->isDynamicalDTC()
                           else
                           {
                              // we have only to clear it
                              pReportRecord->vClear();
                           }// else // if(pReportRecord->isDynamicalDTC()
                        }// if(pReportRecord->bIsAgeingAllowedByConf())
                        else
                        {
                           // do not clear this DTC
                        }// else // if(pReportRecord->bIsAgeingAllowedByConf())
                     }// if(m_pMemoryObject!=NULL)
                  }// if(doAgeing(pReportRecord)== true)
               }// if(pReportRecord->u8GetAgeingCounter()!= pReportRecord->bIsAgeingAllowedByConf())
               else
               {
                  // do nothing
               }
            }// else if(ageingEvent == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT)
         }// if(  (bIsDTCEnabled() == true)...
      }// if(NULL != pReportRecord)
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclSingleAgeing::vCheckForAgeing => NULL == pReportRecord"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      ++iPos; // next iterartor
   }// for(tReportArrayCIt iPos =  oReports.begin();...
   ETG_TRACE_USR3_THR(( "<-- vdl_tclSingleAgeing::vCheckForAgeing"));
}
