//////////////////////////////////////////////////////////////////////////
// FILE:         vd_diaglog_report_record.cpp
// PROJECT:      common
// SW-COMPONENT: DiagLog
//----------------------------------------------------------------------
//
// DESCRIPTION: DiagLog Report Record base class
//              
//----------------------------------------------------------------------
// COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      | Author                       | Modification
// 27.10.05  | ESA2 Barber                  | Initial version
// 15.10.12  | BSOT Plischke                | new Diaglog 
//
//////////////////////////////////////////////////////////////////////////
// first include diaglog settings
#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_main
   #include <common/framework/vd_diaglog_main.h>
#endif

#include <common/framework/vd_diaglog_settings.h>

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_ageing_global
   #include "vd_diaglog_ageing_global.h"
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_trace
   #include <common/framework/vd_diaglog_trace.h>  
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_system_variables
   #include <common/framework/vd_diaglog_system_variables.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_memory
   #include <common/framework/vd_diaglog_report_memory.h>
#endif

#ifndef VD_DIAGLOG_INCLUDEGUARD_vd_diaglog_report_record_const_data
   #include <common/framework/vd_diaglog_report_record_const_data.h>
#endif

/*
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGLOG_INFO
#include "trcGenProj/Header/vd_diaglog_ageing_global.cpp.trc.h"   
#endif

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclGlobalAgeing::vdl_tclGlobalAgeing
// 
// DESCRIPTION: constructor
//
// PARAMETER:  none
//
// RETURNVALUE: void
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclGlobalAgeing::vdl_tclGlobalAgeing(vdl_tclReportMemory* const pMemoryObject):
   m_s8AgeingCounter(-1),
   m_pMemoryObject(pMemoryObject)
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::vdl_tclGlobalAgeing"));

   vdl_tclAgeing::m_ps8GlobalAgeingCounter = &m_s8AgeingCounter;

	ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::vdl_tclGlobalAgeing"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclReportRecord::~vdl_tclReportRecord
// 
// DESCRIPTION: default destructor
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
vdl_tclGlobalAgeing::~vdl_tclGlobalAgeing()
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::~vdl_tclGlobalAgeing"));

   vdl_tclAgeing::m_ps8GlobalAgeingCounter = NULL;

	ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::~vdl_tclGlobalAgeing"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclGlobalAgeing::processSystemEvent()
// 
// DESCRIPTION: call for Power Event change
//
// PARAMETER:  systemEvent : Power Event
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::processSystemEvent(vdl_teDiaglogSystemEvent systemEvent)
{
	ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::processSystemEvent"));

   switch(systemEvent)
   {
      case DIAGLOG_EVENT_IGNITION_CYCLE_START:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclGlobalAgeing::processSystemEvent => DIAGLOG_EVENT_IGNITION_CYCLE_START"));
         if(AGEING_CYCLE_GLOBAL == DIAGLOG_AGEING_CYCLE_ON_IGNITION_CYCLE_START) //lint !e774 !e506
         {
            vCheckForAgeing();
         }
         break;
      }
      case DIAGLOG_EVENT_IGNITION_CYCLE_END:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclGlobalAgeing::processSystemEvent => DIAGLOG_EVENT_IGNITION_CYCLE_END"));
         if(AGEING_CYCLE_GLOBAL == DIAGLOG_AGEING_CYCLE_ON_IGNITION_CYCLE_END) //lint !e774 !e506
         {
            vCheckForAgeing();
         }
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_START:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclGlobalAgeing::processSystemEvent => DIAGLOG_EVENT_OPERATION_CYCLE_START"));
         if(AGEING_CYCLE_GLOBAL == DIAGLOG_AGEING_CYCLE_ON_OPERATION_CYCLE_START) //lint !e774 !e506
         {
            vCheckForAgeing();
         }
         break;
      }
      case DIAGLOG_EVENT_OPERATION_CYCLE_END:
      {
         ETG_TRACE_USR2_THR(( "--- vdl_tclGlobalAgeing::processSystemEvent => DIAGLOG_EVENT_OPERATION_CYCLE_END"));
         if(AGEING_CYCLE_GLOBAL == DIAGLOG_AGEING_CYCLE_ON_OPERATION_CYCLE_END) //lint !e774 !e506
         {
            vCheckForAgeing();
         }
         break;
      }
      case DIAGLOG_EVENT_ACC_CYCLE_START:
      case DIAGLOG_EVENT_ACC_CYCLE_END:
      case DIAGLOG_EVENT_CRANK_CYCLE_START:
      case DIAGLOG_EVENT_CRANK_CYCLE_END:
      {
         // unused
         break;
      }
      default:
      {
         // this one should support all, if used or not, just not to forget one
         ETG_TRACE_FATAL_THR(( "!!! vdl_tclGlobalAgeing::processSystemEvent => systemEvent:%d not supported",systemEvent));
         NORMAL_M_ASSERT_ALWAYS();
         break;
      }
   }// switch(systemEvent)


	ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::processSystemEvent"));
}


///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclGlobalAgeing::vOnAgeingEvent
// 
// DESCRIPTION: external triggered AgeingEvent
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::vOnAgeingEvent(const tU8 /*u8AgeingCounter*/) 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::vOnAgeingEvent"));  

   if(AGEING_CYCLE_GLOBAL == DIAGLOG_AGEING_CYCLE_ON_EXTERNAL_EVENT) //lint !e774 !e506
   {
      vCheckForAgeing();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::vOnAgeingEvent"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclGlobalAgeing::onClearAll
// 
// DESCRIPTION: Clear received
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::onClearAll() 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::onClear"));  

   m_s8AgeingCounter = -1;

   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::onClear"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclGlobalAgeing::onClear
// 
// DESCRIPTION: Clear received
//
// PARAMETER:  pRecord: not used for global ageing
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::onClearSingle(vdl_tclReportRecord* /*pRecord*/)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::onClearSingle"));  

   bool isAnyITCFailed = false;

   if(m_pMemoryObject != NULL)
   {
      isAnyITCFailed = m_pMemoryObject->isAnyItcActiveOrHistory();
      if(isAnyITCFailed == false)
      {
         ETG_TRACE_COMP_THR(( "--- vdl_tclGlobalAgeing::onClearSingle => clear aging counter"));
         // reset ageing counter
         m_s8AgeingCounter = -1;
      }
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::v"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclGlobalAgeing::onFailed
// 
// DESCRIPTION: Failed received
//
// PARAMETER:  tU32 u32AgeingCounter
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::onFailed(vdl_tclReportRecord* /*pRecord*/)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::onFailed"));  
   m_s8AgeingCounter = 0;
   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::onFailed"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION:tVoid vdl_tclGlobalAgeing::vDoAgeing
// 
// DESCRIPTION: called to increment ageing counter
//
// PARAMETER:  none
//
// RETURNVALUE: none
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::vDoAgeing()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::vDoAgeing"));

   tU8 u8MaxAge = vdl_tclGlobalSystemVariables::u8GetMaxAge();
   // increment ageing counter but not above max age
   if(  (m_s8AgeingCounter < u8MaxAge)
      &&(m_s8AgeingCounter >= 0))
   {
      ++m_s8AgeingCounter;
   }
   
   if(m_s8AgeingCounter >= u8MaxAge)
   {
      // Reset Ageing Counter
      m_s8AgeingCounter = -1;
      // max Age reached, relaern now
      if(m_pMemoryObject != NULL)
      {
         m_pMemoryObject->vClearAll();
         // write Info to EM-Trace
         ETG_TRACE_ERRMEM(("DIAGLOG => ClearByAgeing DTC Memory Id:%d",m_pMemoryObject->u8GetMemoryId()));
      }
      else
      {
         ETG_TRACE_ERRMEM(( "!!! vdl_tclGlobalAgeing::vDoAgeing => NULL == m_pMemoryObject"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }// if(m_s8AgeingCounter >= u8MaxAge)
   else
   {
      if(m_pMemoryObject != NULL)
      {
         ETG_TRACE_COMP_THR(( "--- vdl_tclGlobalAgeing::vDoAgeing => Memory:%d m_u8GlobalAgeingCounter=%d MaxAge=%d",m_pMemoryObject->u8GetMemoryId(), m_s8AgeingCounter,u8MaxAge));
      }
   }// else // if(m_s8AgeingCounter >= u8MaxAge)

   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::vDoAgeing"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: bool vdl_tclGlobalAgeing::bLoad
// 
// DESCRIPTION: load raw vdl_tclGlobalAgeing RAM data into class members
//
// PARAMETER:  vdl_tclDiagLogObject::tStreamCIterator - array of log file data
//
// RETURNVALUE: bool true=succesfull / false=failed
// 
///////////////////////////////////////////////////////////////////////////////////
//
bool vdl_tclGlobalAgeing::bLoad(tStreamCIterator& iPos)
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::bLoad"));

   m_s8AgeingCounter = static_cast<tS8>(*iPos++);

   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::bLoad"));
   return true;
}
///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: tVoid vdl_tclGlobalAgeing::vSave
// 
// DESCRIPTION: save raw vdl_tclGlobalAgeing RAM data
//
// PARAMETER:  tStream - array to fill with class member data
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
//
tVoid vdl_tclGlobalAgeing::vSave(tStream& roData)const 
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::vSave"));


   roData.push_back(static_cast<tS8>(m_s8AgeingCounter));
   ETG_TRACE_USR1_THR(( "--- vdl_tclGlobalAgeing::vSave: m_u8GlobalAgeingCounter = %02d", m_s8AgeingCounter));

     
   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::vSave"));
}

///////////////////////////////////////////////////////////////////////////////////
// 
// FUNCTION: vdl_tclGlobalAgeing::vCheckForAgeing
// 
// DESCRIPTION: check if we have to age
//
// PARAMETER:  none
//
// RETURNVALUE: tVoid
// 
///////////////////////////////////////////////////////////////////////////////////
tVoid vdl_tclGlobalAgeing::vCheckForAgeing()
{
   ETG_TRACE_USR3_THR(( "--> vdl_tclGlobalAgeing::vCheckForAgeing"));
   
   if(m_pMemoryObject != NULL)
   {
      bool bIncrementCycleCounter = true;
      tU16 ITC = m_pMemoryObject->getFirstActiveITC();
      tU8 memoryId = m_pMemoryObject->u8GetMemoryId();
      if(ITC != 0)
      {
         // we found one active, stop ageing and reset ageing counter
         ETG_TRACE_COMP_THR(( "--- vdl_tclGlobalAgeing::vCheckForAgeing => MemoryId:%d forbidden because of ITC:%04x is active",memoryId,ITC));
         bIncrementCycleCounter = false;
         m_s8AgeingCounter = 0;
      }
      if (bIncrementCycleCounter)
      {
         // do aeging now
         vDoAgeing();
      }
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR3_THR(( "<-- vdl_tclGlobalAgeing::vCheckForAgeing"));
}

