/************************************************************************
* FILE:         dia_tclUDS_BD_GetMacAddress.cpp
* PROJECT:      DIAG Core
* SW-COMPONENT: Diagnostic application
*----------------------------------------------------------------------
*
* DESCRIPTION: Classes for UDS service Get Display Type
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 23.09.10  | TMS Plischke       | initial version
*
*************************************************************************/

#include "dia_tclUDS_BD_GetMacAddress.h"

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#define DATA_START            (1/*sizeof(SID)*/ + mDIDLen)
#define DATA_LENGTH           (17U)    //string length "AA:BB:CC:DD:EE:FF"
#define DIA_MAC_ADDRESS_LEN   (6U)

//-------------------------------------------------------------------------------

dia_tclUDS_BD_GetMacAddress::dia_tclUDS_BD_GetMacAddress(tCString name, tU8 sid, tU16 did):
      dia_ServiceHandlerUDS(name,sid,(tU16) did)
{
   dia_tclFnctTrace trc("dia_tclUDS_BD_GetMacAddress::dia_tclUDS_BD_GetMacAddress");
}

//-------------------------------------------------------------------------------

dia_tclUDS_BD_GetMacAddress::~dia_tclUDS_BD_GetMacAddress()
{
   _BP_TRY_BEGIN
   {
     (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("dia_tclUDS_BD_GetMacAddress::~dia_tclUDS_BD_GetMacAddress - Exception caught!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_tclUDS_BD_GetMacAddress::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_tclUDS_BD_GetMacAddress::vProcessRequest");

   tBool errorDetected = TRUE;

   dia_IBluetooth* pIBluetooth = 0;
   tDiaResult sysAdapterResult = querySysAdapterInterface<dia_IBluetooth>(&pIBluetooth);

   if ((DIA_SUCCESS==sysAdapterResult) && pIBluetooth)
   {
      (void) setSysAdapterListener<dia_IBluetoothListener>(this);
      if ( pIBluetooth->vGetBluetoothMacAddress() != DIA_SUCCESS )
      {
         DIA_TR_ERR("!!! dia_tclUDS_BD_GetMacAddress::vProcessRequest => ERROR: vGetBluetoothMacAddress != DIA_SUCCESS");
      }
      else
      {
         errorDetected = FALSE;
         DIA_TR_INF("dia_tclUDS_BD_GetMacAddress::vProcessRequest => Send Successfully to Server");
      }
   }
   else
   {
      DIA_TR_INF("!!! dia_tclUDS_BD_GetMacAddress::vProcessRequest ERROR: sysAdapterResult=0x%X pIBluetooth=0x%p", sysAdapterResult, pIBluetooth);
   }

   if ( errorDetected )
   {
      DIA_TR_INF("dia_tclUDS_BD_GetMacAddress::vProcessRequest => ERROR DETECTED");
      (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);
      oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      vResReadyAndQuit();
   }
}

//-------------------------------------------------------------------------------

void
dia_tclUDS_BD_GetMacAddress::vOnReadMacAddressResult(const tU8* const pu8Result)
{
   dia_tclFnctTrace trc("dia_tclUDS_BD_GetMacAddress::vOnReadMacAddressResult");

   (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength((tU16)(DATA_START + DATA_LENGTH));
   const tU8 u8ConvertTable[16] =
   {
      '0', '1', '2', '3',  '4', '5', '6', '7',
      '8', '9', 'A', 'B',  'C', 'D', 'E', 'F'
   };

   for(tU8 i=0; i < DIA_MAC_ADDRESS_LEN; ++i)
   {
      tU16 dataStart = (tU16)(DATA_START + (3 * i));
      oDiagMsgBuffer().vSetDataU8(dataStart, u8ConvertTable[(tU8)((pu8Result[i] >> 4) & 0x0F)]);
      oDiagMsgBuffer().vSetDataU8((tU16)(dataStart + 1), u8ConvertTable[(tU8)((pu8Result[i]) & 0x0F)]);

      if (i < (DIA_MAC_ADDRESS_LEN-1))
      {
         oDiagMsgBuffer().vSetDataU8((tU16)(dataStart + 2), ':');
      }
   }

   vResReadyAndQuit();
   DIA_TR_INF("dia_tclUDS_BD_GetMacAddress::vOnReadMacAddressResult MAC %02X:%02X:%02X:%02X:%02X:%02X",
               pu8Result[0], pu8Result[1], pu8Result[2], pu8Result[3], pu8Result[4], pu8Result[5]);
}

//-------------------------------------------------------------------------------

void
dia_tclUDS_BD_GetMacAddress::vOnReadMacAddressError (const tU32 errCode )
{
   dia_tclFnctTrace trc("dia_tclUDS_BD_GetMacAddress::vOnReadMacAddressError");

   (void) unsetSysAdapterListener<dia_IBluetoothListener>(this);

   DIA_TR_INF( "dia_tclUDS_BD_GetMacAddress::vOnReadMacAddressError=> errCode=%08X",errCode);
   oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   vResReadyAndQuit();
}
