/*!
 * \file       dia_SrvHandlerMicrophonePowerSupplyInfo.cpp
 *
 * \brief      Service Handler for Read of Microphone Connection Status
 *
 * \details    Connection Status is current value and power supply status
 *             Microphone status could be delivered by RVC (A-IVI project) and AUDIO (other G3G projects).
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

// TTFis: DIA_REQ UDS 04 22 10 20 (MIC_1)
// TTFis: DIA_REQ UDS 04 22 10 21 (MIC_2)


#include "project/services/production/dia_SrvHandlerMicrophonePowerSupplyInfo.h"

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif


//-----------------------------------------------------------------------------

dia_SrvHandlerMicrophonePowerSupplyInfo::dia_SrvHandlerMicrophonePowerSupplyInfo ( tCString name, tU8 sid, tU16 did, dia_eMicrophoneNumber micNo )
   : dia_ServiceHandlerUDS(name,sid,(tU16) did),
     mMicNo(micNo),
     mCurrentMeasurementImpossible(FALSE)
{
#ifndef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandlerMicrophonePowerSupplyInfo::dia_SrvHandlerMicrophonePowerSupplyInfo(name,sid,did,micNo)");
#endif
}

//-----------------------------------------------------------------------------

dia_SrvHandlerMicrophonePowerSupplyInfo::~dia_SrvHandlerMicrophonePowerSupplyInfo( void )
{
    _BP_TRY_BEGIN
    {
       (void) unsetSysAdapterListener<dia_IMicrophoneListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandlerMicrophonePowerSupplyInfo::~dia_SrvHandlerMicrophonePowerSupplyInfo !!!");
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_SrvHandlerMicrophonePowerSupplyInfo::vProcessRequest ( const std::vector<void*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandlerMicrophonePowerSupplyInfo::vProcessRequest");
   tBool errorDetected = TRUE;

   dia_IMicrophone* pInterface = OSAL_NULL;
   if ( querySysAdapterInterface<dia_IMicrophone>(&pInterface) == DIA_SUCCESS )
   {
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vProcessRequest - querySysAdapterInterface OK.");
      if ( pInterface )
      {
         DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vProcessRequest - pInterface OK.");
         (void) setSysAdapterListener<dia_IMicrophoneListener>(this);

         if(oDiagMsgBuffer().u16GetDataLength() == DATA_LENGTH)
         {
            DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vProcessRequest - MsgLength OK -> Send here!");
         }

         if ( pInterface->getMicrophoneConnectionInfo(mMicNo) == DIA_SUCCESS )
         {
            DIA_TR_INF("pInterface->getMicrophoneConnectionInfo SUCCESS.");
            errorDetected = FALSE;
         }
         else
         {
            DIA_TR_INF("pInterface->getMicrophoneConnectionInfo FAILED.");
         }
      }
   }
   else
   {
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vProcessRequest - querySysAdapterInterface FAILED.");
   }

   if ( errorDetected )
   {
      (void) unsetSysAdapterListener<dia_IMicrophoneListener>(this);
      DIA_TR_ERR(" ---  SEND TO MICROPHONE SERVER FAILED!!!!");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

//-----------------------------------------------------------------------------

//! DEPRECATED! PLEASE USE vOnMicrophoneConnectionInfo INSTEAD.
//! This method was kept because of backward compatibility.

void
dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo(const std::vector<dia_tMicrophoneStatus>& Mic_info)
{
   dia_tclFnctTrace trc("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo");

   (void) unsetSysAdapterListener<dia_IMicrophoneListener>(this);

   dia_eMicrophoneStatus eConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_UNKNOWN;
   tU16 current = 0;

   std::vector<dia_tMicrophoneStatus>::const_iterator it=Mic_info.begin();
   DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo DIAG REQ => mMicNo=%d",mMicNo);

   for(;it!=Mic_info.end();++it)
   {
      dia_tMicrophoneStatus Mic=(*it);

      if(Mic.Mic_num==mMicNo)
      {
         DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo => MIC_NUM=%d",Mic.Mic_num);
         DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo => Current=%d",Mic.u16CurrentValue);
         DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo => Connection=%d",Mic.ConnectionStatus);
         current=Mic.u16CurrentValue;
         eConnectionStatus=Mic.ConnectionStatus;
         break;
      }
   }

   if(( current <= CURRENT_LIMIT )&& ( eConnectionStatus < DIA_EN_MICROPHONE_CONNECTION_COUNT ))
   {
      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);

      current = (tU16) (current*10); // as defined in CDD-File

      switch(eConnectionStatus)
      {
         case DIA_EN_MICROPHONE_CONNECTION_OPEN:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x01));
            DIA_TR_INF("vOnMic1PowerSupplyInfo_Get => DIA_EN_MICROPHONE_CONNECTION_OPEN");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_SHORT:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x02));
            DIA_TR_INF("vOnMic1PowerSupplyInfo_Get => DIA_EN_MICROPHONE_CONNECTION_SHORT");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_OK:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x00));
            DIA_TR_INF("vOnMic1PowerSupplyInfo_Get => DIA_EN_MICROPHONE_CONNECTION_OK");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_SHORT_TO_BATTERY:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x02));
            DIA_TR_INF("vOnMic1PowerSupplyInfo_Get => DIA_EN_MICROPHONE_CONNECTION_SHORT_TO_BATTERY");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_UNKNOWN:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x80));
            DIA_TR_INF("vOnMic1PowerSupplyInfo_Get => DIA_EN_MICROPHONE_CONNECTION_UNKNOWN");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_UNDEFINED:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0xFF));
            DIA_TR_INF("vOnMic1PowerSupplyInfo_Get => DIA_EN_MICROPHONE_CONNECTION_UNDEFINED");
         break;

         default:
            DIA_TR_ERR("vOnMic1PowerSupplyInfo_Get - Microphone Connection Open Status!!!");
         break;
      }

      if (TRUE==GetCurrentMeasurementImpossible())
      {
         //current measurement is impossible
         if (DIA_SUCCESS!=dia_getProperty(DIA_PROP_CM_MIC_INVALID_CURRENT_VALUE, current))
         {
            DIA_TR_ERR("### dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo Error: invalid current cannot be got.");
            current = 0;
         }
         else
         {
            DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo current=0x%04X was overwritten.", current);
         }
      }
      else
      {
         DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo current measurement is possible");
      }

      (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 1, ((tU8) (current >> 8)));
      (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 2, ((tU8) (current)));

      vResReadyAndQuit();
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo  SUCCESSFUL");
   }
   else
   {
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophonePowersupplyInfo  FAIL");
      // Sending back the Negative answer
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

void
dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo ( tU16 micID, const dia_MicrophoneInfo& micInfo)
{
   dia_tclFnctTrace trc("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo");

   (void) unsetSysAdapterListener<dia_IMicrophoneListener>(this);

   dia_eMicrophoneConnectionStatus eConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN;
   tU16 current = 0;

   if (micID==mMicNo)
   {
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo => micID=%d",             micID);
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo => Current=%d",           micInfo.mCurrent);
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo => ConnectionStatus=%d",  micInfo.mConnectionStatus);
      current = micInfo.mCurrent;
      eConnectionStatus = micInfo.mConnectionStatus;
   }
   else
   {
      DIA_TR_ERR("### actual mic ID 0x%02X, expected mic ID 0x%02X. WRONG MICROPHONE ID. FAILED.", micID, mMicNo);
   }

   if ((micID==mMicNo) && (current <= CURRENT_LIMIT) && (eConnectionStatus < DIA_EN_MICROPHONE_CONNECTION_STATUS_COUNT))
   {
      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);

      
      current = (tU16) (current*10); // as defined in CDD-File

      switch(eConnectionStatus)
      {
         case DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x80));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_STATUS_OK:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x00));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_OK");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_STATUS_OPEN_LOAD:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x01));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_OPEN_LOAD");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x02));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_ACROSS:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x02));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_ACROSS");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_BATTERY:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x02));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_BATTERY");
         break;

         case DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_GROUND:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0x02));
            DIA_TR_INF("vOnMicrophoneConnectionInfo => DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_GROUND");
         break;

         default:
            (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 0, ((tU8) 0xFF));
            DIA_TR_ERR("vOnMicrophoneConnectionInfo - Microphone Connection Open Status!!!");
         break;
      }

      if (TRUE==GetCurrentMeasurementImpossible())
      {
         //current measurement is impossible
         if (DIA_SUCCESS!=dia_getProperty(DIA_PROP_CM_MIC_INVALID_CURRENT_VALUE, current))
         {
            DIA_TR_ERR("### dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo Error: invalid current cannot be got.");
            current = 0;
         }
         else
         {
            DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo current=0x%04X was overwritten.", current);
         }
      }
      else
      {
         DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo current measurement is possible");
      }

      (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 1, ((tU8) (current >> 8)));
      (void) oDiagMsgBuffer().vSetDataU8(DATA_START + 2, ((tU8) (current)));

      vResReadyAndQuit();
      DIA_TR_INF("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo  SUCCESSFUL");
   }
   else
   {
      DIA_TR_ERR("dia_SrvHandlerMicrophonePowerSupplyInfo::vOnMicrophoneConnectionInfo  FAIL");
      // Sending back the Negative answer
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}
