/**
 * \file      dia_RoutineCtrlSWUPUpdateMapData.cpp
 *
 * \date      29.03.2017
 *
 * \autor     kaa1hi
 *
 * \brief     Routine to trigger full map update for Navi
 *
 * \details    Reply for request result sends ("IN PROGRESS"):
 *             1/ previous status of Map Update
 *             2/ send status request to HMI App Navigation
 *
 *             Start:       31 01 11 11
 *             Req Result:  31 03 11 11
 *
 * \copyright Robert Bosch GmbH 2017
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CTRL_SWUP_UPDATE_MAP_DATA__
#include <project/services/production/dia_RoutineCtrlSWUPUpdateMapData.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_MANAGER__
#include <common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h>
#endif

namespace dia
{

static tU8 sResultMapping[DIA_EN_RTCTRL_STATUS_COUNT] = {
      0xFF, // DIA_EN_RTCTRL_STATUS_UNKNOWN
      0xFF, // DIA_EN_RTCTRL_STATUS_IDLE
      0x03, // DIA_EN_RTCTRL_STATUS_IN_PROGRESS
      0x00, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK
      0x01, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK
      0x02, // DIA_EN_RTCTRL_STATUS_ABORTED
      0xFF  // DIA_EN_RTCTRL_STATUS_TIMED_OUT
};

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlSWUPUpdateMapData::RoutineCtrlSWUPUpdateMapData(void)
   :  dia_Routine("dia_RoutineCtrlSWUPUpdateMapData", DIA_C_U16_ID_RTCTRL_SWUP_UPDATE_MAP_DATA, DIA_EN_RTCTRL_ID_SWUP_UPDATE_MAP_DATA, DIA_EN_RTCTRL_TYPE_LONG_TERM),
      mProgress(0)
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlSWUPUpdateMapData::RoutineCtrlSWUPUpdateMapData");
}

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlSWUPUpdateMapData::~RoutineCtrlSWUPUpdateMapData ( void )
{
   _BP_TRY_BEGIN
   {
      (void) RoutineCtrlSWUPUpdateMapData::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("dia::RoutineCtrlSWUPUpdateMapData::~RoutineCtrlSWUPUpdateMapData - Exception caught!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}


//-----------------------------------------------------------------------------

tDiaResult
RoutineCtrlSWUPUpdateMapData::tearDown ( void )
{
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
RoutineCtrlSWUPUpdateMapData::vOnServiceTimeout ( void )
{
   dia_tclFnctTrace oTrace("RoutineCtrlSWUPUpdateMapData::vOnServiceTimeout()");
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlSWUPUpdateMapData::start( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("RoutineCtrlSWUPUpdateMapData::start()");
   tDiaResult retCode = DIA_FAILED;
   dia_IHMINavigationDataUpdate* pHMINavigationDataUpdate = NULL;

   // prepare processing of the routine
   vInitialize();

   mProgress = DEFAULT_PROGRESS;

   tDiaResult queryResult = querySysAdapterInterface<dia_IHMINavigationDataUpdate>(&pHMINavigationDataUpdate);

   if (DIA_SUCCESS==queryResult)
   {
      if (pHMINavigationDataUpdate)
      {
         (void) setSysAdapterListener<dia_IHMINavigationDataUpdateListener>(this);

         if (pHMINavigationDataUpdate->performFullUpdateViaUSB() == DIA_SUCCESS)
         {
            DIA_TR_INF("RoutineCtrlSWUPUpdateMapData::start performFullUpdateViaUSB() success.");
            retCode = DIA_SUCCESS;
         }
         else
         {
            DIA_TR_ERR("RoutineCtrlSWUPUpdateMapData::start performFullUpdateViaUSB() with ERRORS!");
            (void) unsetSysAdapterListener<dia_IHMINavigationDataUpdateListener>(this);
         }
      }
      else
      {
         DIA_TR_ERR("RoutineCtrlSWUPUpdateMapData::start pHMINavigationDataUpdate is NULL.");
      }
   }
   else
   {
      DIA_TR_ERR("querySysAdapterInterface<dia_IHMINavigationDataUpdate> ERROR! queryResult=0x%08X", queryResult);
   }

   DIA_TR_INF("RoutineCtrlSWUPUpdateMapData::start retCode = 0x%08X", retCode);

   return retCode;
}

//------------------------------------------------------------------------------
//! Return value: If false, negative response. If true, positive response;

tDiaResult
RoutineCtrlSWUPUpdateMapData::requestResult( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("RoutineCtrlSWUPUpdateMapData::requestResult()");
   tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

   results.clear();
   if ( !(mResults.empty()) )
   {
      /* this code is executed in case of reply for start request only */
      DIA_TR_INF("RoutineCtrlSWUPUpdateMapData::requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); iter++ )
      {
         results.push_back(*iter);
      }

      // clear buffer, such reply is necessary once
      mResults.clear();

      // keep the result until new calculation is started
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_INF("requestResult (%d)", mStatus);
      switch ( mStatus )
      {
         case DIA_EN_RTCTRL_STATUS_IN_PROGRESS:
         {
            DIA_TR_INF("RoutineCtrlSWUPUpdateMapData::requestResult DIA_EN_RTCTRL_STATUS_IN_PROGRESS.");

            //send what we got last time
            results.push_back(sResultMapping[mStatus]);
            results.push_back(mProgress);
            mIsResultReady = TRUE;
            retCode = DIA_SUCCESS;

            //send status request
            dia_IHMINavigationDataUpdate* pHMINavigationDataUpdate = NULL;
            tDiaResult queryResult = querySysAdapterInterface<dia_IHMINavigationDataUpdate>(&pHMINavigationDataUpdate);
            if (DIA_SUCCESS==queryResult)
            {
               if (pHMINavigationDataUpdate)
               {
                  (void) setSysAdapterListener<dia_IHMINavigationDataUpdateListener>(this);

                  if (pHMINavigationDataUpdate->getNavDataUpdateStatus() == DIA_SUCCESS)
                  {
                     DIA_TR_INF("RoutineCtrlSWUPUpdateMapData::requestResult getNavDataUpdateStatus() success.");
                     retCode = DIA_SUCCESS;
                  }
                  else
                  {
                     DIA_TR_ERR("RoutineCtrlSWUPUpdateMapData::requestResult performFullUpdateViaUSB() with ERRORS!");
                     (void) unsetSysAdapterListener<dia_IHMINavigationDataUpdateListener>(this);
                  }
               }
               else
               {
                  DIA_TR_ERR("RoutineCtrlSWUPUpdateMapData::requestResult pHMINavigationDataUpdate is NULL.");
               }
            }
            else
            {
               DIA_TR_ERR("querySysAdapterInterface<dia_IHMINavigationDataUpdate> ERROR! queryResult=0x%08X", queryResult);
            }
         }
         break;

         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK:
         {
            DIA_TR_INF("RoutineCtrlSWUPUpdateMapData::requestResult DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK.");
            results.push_back(sResultMapping[mStatus]);
            results.push_back(MAX_PROGRESS);
            mIsResultReady = TRUE;
            retCode = DIA_SUCCESS;
         }
         break;

         case DIA_EN_RTCTRL_STATUS_ABORTED:
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK:
         {
            if (DIA_EN_RTCTRL_STATUS_ABORTED==mStatus)            DIA_TR_INF("requestResult: DIA_EN_RTCTRL_STATUS_ABORTED, progress %d",  mProgress);
            if (DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK==mStatus)  DIA_TR_INF("requestResult: DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK, progress %d", mProgress);
            results.push_back(sResultMapping[mStatus]);
            results.push_back(mProgress);
            mIsResultReady = TRUE;
            retCode = DIA_SUCCESS;
         }
         break;

         default:
            DIA_TR_ERR("RoutineCtrlSWUPUpdateMapData::requestResult mStatus = %d.", mStatus);
         break;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

void
RoutineCtrlSWUPUpdateMapData::vOnPerformFullUpdateViaUSB( dia_eFullUpdateViaUSBConfirmation ack )
{
   dia_tclFnctTrace oTrace("RoutineCtrlSWUPUpdateMapData::vOnPerformFullUpdateViaUSB()");

   (void) unsetSysAdapterListener<dia_IHMINavigationDataUpdateListener>(this);

   switch (ack)
   {
      case DIA_EN_FULL_UPDATE_CONFIRMATION_ACCEPTED:
         DIA_TR_INF("DIA_EN_FULL_UPDATE_CONFIRMATION_ACCEPTED ==> IN_PROGRESS");
         eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
      break;

      case DIA_EN_FULL_UPDATE_CONFIRMATION_DECLINED_CANNOT_PROCESS:
         DIA_TR_INF("DIA_EN_FULL_UPDATE_CONFIRMATION_DECLINED_CANNOT_PROCESS ==> ABORTED");
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
      break;

      case DIA_EN_FULL_UPDATE_CONFIRMATION_DECLINED_REQUEST_ACTIVE:
         DIA_TR_INF("DIA_EN_FULL_UPDATE_CONFIRMATION_DECLINED_REQUEST_ACTIVE ==> IN_PROGRESS");
         eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
      break;

      default:
         DIA_TR_INF("Unknown ack (%d) ==> ABORTED", ack);
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
      break;
   }

   //1 bytes output
   mResults.push_back(sResultMapping[mStatus]);

   mIsResultReady = TRUE;
   dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
}

//------------------------------------------------------------------------------

void
RoutineCtrlSWUPUpdateMapData::vOnGetNavDataUpdateStatus( tU8 progress, dia_eMapUpdateStatus updateStatus )
{
   dia_tclFnctTrace oTrace("RoutineCtrlSWUPUpdateMapData::vOnGetNavDataUpdateStatus(tU8)");

   (void) unsetSysAdapterListener<dia_IHMINavigationDataUpdateListener>(this);

   DIA_TR_INF("progress is %d", progress);
   DIA_TR_INF("mStatus is %d. sResultMapping[mStatus] is %d.", mStatus, sResultMapping[mStatus]);

   switch(updateStatus)
   {
      case DIA_EN_MAP_UPDATE_STATUS_COMPLETED_OK:
      {
         if (progress<=MAX_PROGRESS)
         {
            DIA_TR_INF("DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK. Progress value is in range (0,100).");
            eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);
         }
         else
         {
            DIA_TR_ERR("progress set to 0, because it was greater than MAX_PROGRESS:%d,  progress:%d.", MAX_PROGRESS, progress);
            progress = DEFAULT_PROGRESS;
            eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
         }
      }
      break;

      case DIA_EN_MAP_UPDATE_STATUS_COMPLETED_NOK:
      {
         DIA_TR_INF("vOnGetNavDataUpdateStatus: DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK.");
         eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
         if (progress<=MAX_PROGRESS)
         {
            /* do nothing, take the value of progress */
         }
         else
         {
            progress = MAX_PROGRESS;
         }
      }
      break;

      case DIA_EN_MAP_UPDATE_STATUS_ABORTED:
      {
         DIA_TR_INF("vOnGetNavDataUpdateStatus: DIA_EN_RTCTRL_STATUS_ABORTED.");
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);

         if (progress<=MAX_PROGRESS)
         {
            /* do nothing, take the value of progress */
         }
         else
         {
            progress = DEFAULT_PROGRESS;
         }
      }
      break;

      case DIA_EN_MAP_UPDATE_STATUS_RUNNING:
      {
         if (progress<=MAX_PROGRESS)
         {
            DIA_TR_INF("vOnGetNavDataUpdateStatus: DIA_EN_MAP_UPDATE_STATUS_RUNNING. Progress value is in range (0,100).");
            eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
         }
         else
         {
            progress = 0;
            DIA_TR_ERR("vOnGetNavDataUpdateStatus: progress set to 0, because it was greater than %d.", MAX_PROGRESS);
            eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
         }
      }
      break;

      case DIA_EN_MAP_UPDATE_STATUS_STOPPED:
      {
         DIA_TR_INF("vOnGetNavDataUpdateStatus: DIA_EN_MAP_UPDATE_STATUS_STOPPED.");
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
         progress = 0;
      }
      break;

      case DIA_EN_MAP_UPDATE_STATUS_NOT_RUN:
      {
         DIA_TR_INF("vOnGetNavDataUpdateStatus: DIA_EN_MAP_UPDATE_STATUS_NOT_RUN.");
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
         progress = 0;
      }
      break;

      default:
      {
         DIA_TR_INF("vOnGetNavDataUpdateStatus: DIA_EN_MAP_UPDATE_STATUS_NOT_RUN.");
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
         progress = 0;
      }
      break;
   }

   mProgress = progress;

   mIsResultReady = TRUE;
   dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
}

}
