/*!
 * \file       dia_SrvHandler_WriteBootDigestClearITC.cpp
 *
 * \brief      Service Handler for Digest Writing
 *
 * \details    Write digest to property. If writing was successful, then positive response shall be sent.
 *             Otherwise negative response shall be sent. Clear three corresponding DTC ((did<<8) | 0x51).
 *             Clear DTC shall be performed always (even if writing to property failed).
 *             Set last result of CheckProgrammingDependency to COMPLETED_NOK (see DIA_PROP_AIVI_RESULT_CHECK_PROG_DEP) -- AIVI-57691
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */


#include "project/services/customer/dia_SrvHandler_WriteBootDigestClearITC.h"

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY_METHODS_PROJECT__
#include "project/framework/factory/dia_factoryMethodsProject.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CTRL_CHECK_PROGRAMMING_DEPENDENCIES__
#include "dia_RoutineCtrlCheckProgrammingDependencies.h"
#endif

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"


const tU8 DIA_AIVI_DTC_PROGRAMMING_FAILURE = 0x51;

const tU8  DIA_DIGEST_LENGTH = 20;
const tU8  DIA_MSG_DATA_OFFSET = 4;     // Offset from oDiagMsgBuffer-start to 1st data-byte (e.g. MemorySize-Byte)
const tU16 DIA_DATA_START = 3;
const tU8  DIA_NUMBER_OF_NOTIFICATION = 1;

//-----------------------------------------------------------------------------
dia_SrvHandler_WriteBootDigestClearITC::dia_SrvHandler_WriteBootDigestClearITC(tCString name, tU8 sid, tU16 did, tU32 propID)
   : dia_SrvHandlerGenericIdentWrite(name, sid, did),
     mPropID(propID),
     mReceivePassedCnt(0),
     mReceiveFailedCnt(0)
{
   dia_tclFnctTrace trc("dia_SrvHandler_WriteBootDigestClearITC::dia_SrvHandler_WriteBootDigestClearITC()");
}

//-----------------------------------------------------------------------------
dia_SrvHandler_WriteBootDigestClearITC::~dia_SrvHandler_WriteBootDigestClearITC( void )
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandler_WriteBootDigestClearITC::~dia_SrvHandler_WriteBootDigestClearITC !!!");
       NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest( const std::vector<void*>& vecArgs )
{
   dia_tclFnctTrace oTrace("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest()");

   //write digest to property
   {
      if ( 0==mPropID )
      {
         DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC: FAILED mPropID=0x%08X.", mPropID);
         vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
         return;
      }

      tU16 payloadSize = oDiagMsgBuffer().u16GetDataLength()-DIA_MSG_DATA_OFFSET;
      if (DIA_DIGEST_LENGTH!=payloadSize)
      {
         // Sending back the Negative answer
         DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC: FAILED payloadSize=%d, exp=%d", payloadSize, DIA_DIGEST_LENGTH);
         vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
         return;
      }

      tU8 const* pU8 = oDiagMsgBuffer().u8GetBuffer();
      tU8 u8propData[DIA_DIGEST_LENGTH] = {0};

      // copy digest
      (void) ::memcpy(u8propData, &pU8[DIA_MSG_DATA_OFFSET], DIA_DIGEST_LENGTH);

      tDiaResult setResult = dia_setProperty(mPropID, u8propData, DIA_DIGEST_LENGTH);
      if ( DIA_SUCCESS != setResult )
      {
         DIA_TR_ERR("dia_SrvHandler_WriteBootDigestClearITC: set ERROR CODE 0x%08X.", setResult);
         vSendNegativeResponse( (DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT==setResult)? DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT: DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
         return;
      }
   }

   //clear ITC event if property write was unsuccessful.
   {
      dia_IErrorLog* pInterface = 0;
      tBool errorDetected = TRUE;

      tDiaResult res = querySysAdapterInterface<dia_IErrorLog>(&pInterface);
      if ((DIA_SUCCESS==res) && pInterface)
      {
         (void) setSysAdapterListener<dia_IErrorLogListener>(this);

         tU32 groupOfDtcProgramming = ((getDID()<<8) | DIA_AIVI_DTC_PROGRAMMING_FAILURE);
         DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest groupOfDtcProgramming = 0x%08X", groupOfDtcProgramming);

         tDiaResult resProgramming = pInterface->clearDTC(DIAGLOG_MEMORY_REPROGRAMMING, groupOfDtcProgramming);

         if (DIA_SUCCESS==resProgramming)
         {
            DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest --- clearDTC success");
            errorDetected = FALSE;
            mReceivePassedCnt = 0;
            mReceiveFailedCnt = 0;
         }
         else
         {
            DIA_TR_ERR("### dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest ---  SEND TO DIAGLOG FAILED!!!!");
            DIA_TR_ERR("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest resProgramming = 0x%08X", resProgramming);
         }
      }
      else
      {
         if (DIA_SUCCESS!=res)   DIA_TR_ERR("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest res=0x%08X", res);
         if (NULL==pInterface)   DIA_TR_ERR("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest pInterface is NULL");
      }

      if ( errorDetected )
      {
         DIA_TR_ERR("### dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest Error detected!");
         (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
         vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      }
      else
      {
         DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest great success");
      }
   }

   //set last result of CheckProgrammingDependency to COMPLETED_NOK.
   {
      tDiaResult wrCode = dia_setProperty(DIA_PROP_AIVI_RESULT_CHECK_PROG_DEP, (tU8)PROG_DEPENDENCY_RESULT_TYPE_COMPLETED_NOK);
      if (DIA_SUCCESS!=wrCode)
      {
         DIA_TR_ERR("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest Set Prop failed. wrCode=0x%08X.", wrCode);
      }
      else
      {
         DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC::vProcessRequest Set Prop passed.");
      }
   }
}

void
dia_SrvHandler_WriteBootDigestClearITC::vOnClearResult()
{
   dia_tclFnctTrace trc("dia_SrvHandler_WriteBootDigestClearITC::vOnClearResult");

   mReceivePassedCnt++;
   DIA_TR_INF("::vOnClearResult mReceiveFailedCnt=%d, mReceivePassedCnt=%d.", mReceiveFailedCnt, mReceivePassedCnt);

   if (DIA_NUMBER_OF_NOTIFICATION==mReceivePassedCnt)
   {
      unsetSysAdapterListener<dia_IErrorLogListener>(this);

      DIA_TR_INF("dia_SrvHandler_WriteBootDigestClearITC::vOnClearResult Sending back the positive answer");
      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(DIA_DATA_START);
      vResReadyAndQuit();
   }
   else if (DIA_NUMBER_OF_NOTIFICATION==(mReceivePassedCnt+mReceiveFailedCnt))
   {
      unsetSysAdapterListener<dia_IErrorLogListener>(this);

      DIA_TR_ERR("## ::vOnClearResult At least one error.");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

void
dia_SrvHandler_WriteBootDigestClearITC::vOnClearError(tU32 u32Error)
{
   dia_tclFnctTrace trc("dia_SrvHandler_WriteBootDigestClearITC::vOnClearError(tU32)");

   DIA_TR_ERR( "### vOnClearError => ERROR: 0x%02X",u32Error);

   mReceiveFailedCnt++;
   DIA_TR_INF("::vOnClearError mReceiveFailedCnt=%d, mReceivePassedCnt=%d.", mReceiveFailedCnt, mReceivePassedCnt);

   if (DIA_NUMBER_OF_NOTIFICATION==(mReceivePassedCnt+mReceiveFailedCnt))
   {
      unsetSysAdapterListener<dia_IErrorLogListener>(this);

      DIA_TR_ERR("## ::vOnClearResult At least one error.");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

void
dia_SrvHandler_WriteBootDigestClearITC::vOnTimeout()
{
   dia_tclFnctTrace trc("dia_SrvHandler_WriteBootDigestClearITC::vOnTimeout()");

   unsetSysAdapterListener<dia_IErrorLogListener>(this);
}
