/*!
 * \file       dia_SrvHandler_WifiMonitoring.cpp
 *
 * \brief      Service Handler for Wifi Monitoring
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */


#ifndef __INCLUDED_DIA_COMMON_CORE__
#include "common/depricated/dia_common_core.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_SERVICE_HANDLER_WIFI_MONITORING__
#include "dia_SrvHandler_WifiMonitoring.h"
#endif


#define DATA_START 3
#define DATA_LENGTH 53

#define ADDRESS_START 0
#define ADDRESS_SIZE 15

#define CONFIGURATION_START 15

#define CONFIGURATION2_START 16
#define STRENGTH_MASK 0x7F
#define STRENGTH_STA 0x00
#define STRENGTH_AP 0x7F
#define BAND_BIT 7

#define CHANNEL_START 17
#define CHANNEL_LENGTH 3

#define SSID_START 20
#define SSID_LENGTH 33


//-----------------------------------------------------------------------------

dia_SrvHandler_WifiMonitoring::dia_SrvHandler_WifiMonitoring(tCString name, tU8 sid, tU16 did)
   : dia_ServiceHandlerUDS(name, sid, did)
{
   dia_tclFnctTrace trc("dia_SrvHandler_WifiMonitoring::dia_SrvHandler_WifiMonitoring");
}

//-----------------------------------------------------------------------------

dia_SrvHandler_WifiMonitoring::~dia_SrvHandler_WifiMonitoring()
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandler_WifiMonitoring::~dia_SrvHandler_WifiMonitoring !!!");
       NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_WifiMonitoring::vProcessRequest( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace oTrace("dia_SrvHandler_WifiMonitoring::vProcessRequest");

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);

   for (unsigned i = 0; i < DATA_LENGTH; i++)
   {
      (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+i), 0);
   }

   tBool errorDetected = TRUE;

   dia_IWifiSetups* pInterface = 0;
   if ((querySysAdapterInterface<dia_IWifiSetups>(&pInterface) == DIA_SUCCESS) && pInterface)
   {
      (void) setSysAdapterListener<dia_IWifiSetupsListener>(this);
      if (pInterface->GetActiveSetups() == DIA_SUCCESS)
      {
         errorDetected = FALSE;
      }
   }

   if ( errorDetected )
   {
      DIA_TR_ERR("### dia_SrvHandler_WifiMonitoring::vProcessRequest Error detected!");
      (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_WifiMonitoring::OnActiveSetups ( const std::vector<dia_stWifiSetup>& setupList )
{
   dia_tclFnctTrace oTrace("dia_SrvHandler_WifiMonitoring::OnActiveSetups");

   tBool readyToRespond = TRUE;
   unsigned i;
   bool bAP = false;
   bool bSTA = false;
   tU32 u32ConnectedDevices = 0;
   tU32 u32Channel = 0;
   dia_enWifiFrequency eFrequency = DIA_ENUM_WIFI_FREQ_UNKNOWN;

   (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);

   std::string strAddress = "255.255.255.255";
   for (unsigned i = 0; i < ADDRESS_SIZE; i++)
   {
      (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+ADDRESS_START+i), strAddress[i]);
   }

   std::vector<dia_stWifiSetup>::const_iterator it = setupList.begin();
   for ( ; it != setupList.end(); it++)
   {
      switch (it->mMode)
      {
         case DIA_ENUM_WIFI_MODE_AP:
         {
            DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups DIA_ENUM_WIFI_MODE_AP.");

            if (it->mPoweredOn)
            {
               DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups DIA_ENUM_WIFI_MODE_AP mPoweredOn true.");
               if (it->mSSID.size()>0)  DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups it->mSSID (len=%d): %s.", (it->mSSID.size()-1), dia::utils::bin2hexstr(it->mSSID).c_str());

               std::string ssid( it->mSSID.begin(), it->mSSID.end() );
               setSSID(ssid);

               bAP = true;
               u32Channel = it->mChannel;
               eFrequency = it->mFrequency;
               (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+CONFIGURATION2_START), STRENGTH_AP);
            }
            else
            {
               DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups DIA_ENUM_WIFI_MODE_AP mPoweredOn false.");
            }

            u32ConnectedDevices = it->mConnectedDevices;

            break;
         }

         case DIA_ENUM_WIFI_MODE_STA:
         {
            if (it->mPoweredOn)
            {
               DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups DIA_ENUM_WIFI_MODE_STA mPoweredOn true.");
               if (it->mSSID.size()>0)  DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups mSSID: %s.", dia::utils::bin2hexstr(it->mSSID).c_str());

               bSTA = true;
               u32Channel = it->mChannel;
               eFrequency = it->mFrequency;
               (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+CONFIGURATION2_START), STRENGTH_STA);

               tBool errorDetected = TRUE;
               dia_IWifiConnman* pInterface = 0;
               if ((querySysAdapterInterface<dia_IWifiConnman>(&pInterface) == DIA_SUCCESS) && pInterface)
               {
                  DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups: interface got.");
                  (void) setSysAdapterListener<dia_IWifiConnmanListener>(this);
                  if (pInterface->GetWifiStations() == DIA_SUCCESS)
                  {
                     DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups GetWifiStations() success.");
                     errorDetected = FALSE;
                     readyToRespond = FALSE;
                  }
               }

               if ( errorDetected )
               {
                  DIA_TR_ERR("### dia_SrvHandler_WifiMonitoring::OnActiveSetups: Failed to get WIFI station information!");
                  (void) unsetSysAdapterListener<dia_IWifiConnmanListener>(this);
               }
            }
            else
            {
               DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups DIA_ENUM_WIFI_MODE_STA mPoweredOn false.");
            }
            break;
         }

         default:
         {
            DIA_TR_ERR("Unexpected WIFI Mode = %d", it->mMode);
            break;
         }
      }
   }

   tU8 bConfiguration = 0; //Wifi if OFF
   if (bSTA)
   {
      DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups Wifi is in client mode.");
      bConfiguration = 1; //Wifi is in client mode
   }
   if (bAP)
   {
      DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnActiveSetups Wifi is in hotspot mode without internet sharing.");
      bConfiguration = 2; //Wifi is in hotspot mode without internet sharing
      //TODO:
      //3: Wifi is in hotspot mode with internet sharing
      //4: Wifi is in hotspot mode due to Carplay Wireless

      bConfiguration |= ((u32ConnectedDevices & 0x07) << 4);
   }

   (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+CONFIGURATION_START), bConfiguration);

   std::string strChannel;
   strChannel.resize(CHANNEL_LENGTH + 1);
   ::snprintf(&strChannel[0], strChannel.size(), "%03u", u32Channel);
   for (i = 0; i < CHANNEL_LENGTH; i++)
   {
      (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+CHANNEL_START+i), strChannel[i]);
   }

   tU8 u8Config2 = oDiagMsgBuffer().u8GetData((tU16) (DATA_START+CONFIGURATION2_START));
   u8Config2 |= (eFrequency == DIA_ENUM_WIFI_FREQ_5GHZ)? (1 << BAND_BIT): 0;
   (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+CONFIGURATION2_START), u8Config2);

   if (readyToRespond)
   {
      vResReadyAndQuit();
   }
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_WifiMonitoring::OnWifiStations ( const std::vector<dia_stWifiStation>& stationList )
{
   dia_tclFnctTrace oTrace("dia_SrvHandler_WifiMonitoring::OnWifiStations");

   (void) unsetSysAdapterListener<dia_IWifiConnmanListener>(this);

   DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnWifiStations stationList.size()=%d.", stationList.size());

   std::vector<dia_stWifiStation>::const_iterator it = stationList.begin();
   for ( ; it != stationList.end(); it++)
   {
      DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnWifiStations: Addr='%s', State=%d, Strength=%d",
            it->mAddress.c_str(), it->mState, it->mStrength);

      if (     (it->mState == DIA_ENUM_WIFI_STATE_ONLINE) //Connected State of a Wifi link
            || (it->mState == DIA_ENUM_WIFI_STATE_READY) //Connected State of an Ethernet link
            )
      {
         if (it->mState == DIA_ENUM_WIFI_STATE_ONLINE)  DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnWifiStations: DIA_ENUM_WIFI_STATE_ONLINE");
         if (it->mState == DIA_ENUM_WIFI_STATE_READY)   DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnWifiStations: DIA_ENUM_WIFI_STATE_READY");

         tU8 u8Config2 = oDiagMsgBuffer().u8GetData((tU16) (DATA_START+CONFIGURATION2_START));
         u8Config2 |= (it->mStrength & STRENGTH_MASK);
         (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+CONFIGURATION2_START), u8Config2);

         //IP address
         if (it->mAddress.size())
         {
            DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnWifiStations: Setting address to %s.", it->mAddress.c_str());
            unsigned i;
            for (i = 0; (i < it->mAddress.size()) && (i < ADDRESS_SIZE); i++)
            {
               (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+ADDRESS_START+i), it->mAddress[i]);
            }
            if (i < ADDRESS_SIZE)
            {
               (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+ADDRESS_START+i), '\x00');
               i++;
            }
            for ( ; i < ADDRESS_SIZE; i++)
            {
               (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+ADDRESS_START+i), ' ');
            }
         }

         //SSID
         if ((it->mName.size())>0)
         {
            DIA_TR_INF("dia_SrvHandler_WifiMonitoring::OnWifiStations: Length of SSID is %d.", it->mName.size());
            setSSID(it->mName);
         }

         break;
      }
   }

   vResReadyAndQuit();
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_WifiMonitoring::vOnTimeout()
{
   dia_tclFnctTrace trc("dia_SrvHandler_WifiMonitoring::vOnTimeout()");

   (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
   (void) unsetSysAdapterListener<dia_IWifiConnmanListener>(this);

   vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_WifiMonitoring::setSSID(const std::string& ssid)
{
   dia_tclFnctTrace trc("dia_SrvHandler_WifiMonitoring::setSSID(const std::string& ssid)");

   DIA_TR_INF("dia_SrvHandler_WifiMonitoring::setSSID Setting name of SSID to %s.", ssid.c_str());

   unsigned i;
   for (i = 0; (i < ssid.size()) && (i < SSID_LENGTH); i++)
   {
      (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+SSID_START+i), ssid[i]);
   }

   if (i < SSID_LENGTH)
   {
      (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+SSID_START+i), '\x00');
      i++;
   }

   for ( ; i < SSID_LENGTH; i++)
   {
      (void) oDiagMsgBuffer().vSetDataU8((tU16) (DATA_START+SSID_START+i), ' ');
   }
}


