/*****************************************************************************************
*
* \file    dia_SrvHandler_ReadBTDevice.cpp
*
* \brief   Service Handler for Read BT Device
*
* \author  nce5kor	
*
* \date    02.12.2015
*
* (c) 2014 Robert Bosch Car Multimedia GmbH
* --------------------------------------------------------
* Date 	     	  | Author	          | Modification
* 7.7.17          | mky6kor           | Updated Data length to 156 bytes to allow 6 devices.
****************************************************************************************/

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_PROTOCOLS_UDS__
#include "common/framework/protocols/uds/dia_common_uds.h"
#endif

#ifndef __DIA_SRVHANDLER_READBTDEVICE_H__
#include "project/services/customer/dia_SrvHandler_ReadBTDevice.h"
#endif

#define DATA_START ((tU8) 3)
#define DATA_LENGTH ((tU8) 156)
#define DEFAULT_RESPONSE_BYTE ((tU8)0xFF)
#define BT_NAME_LENGTH ((tU8) 20)
#define BT_MAC_ADDR_LENGTH ((tU8) 6)
#define ASCII_TO_CHAR_ALPHABET_SMALL 87
#define ASCII_TO_CHAR_ALPHABET_BIG 55
#define ASCII_TO_CHAR_NUMBER 48
#define FOUR_BITS 4


/******************************************************************************
* FUNCTION:    dia_SrvHandler_ReadBTDevice
*
* DESCRIPTION: Constructor
*
******************************************************************************/

dia_SrvHandler_ReadBTDevice::dia_SrvHandler_ReadBTDevice ( void )
   : dia_ServiceHandlerUDS("dia_SrvHandler_ReadBTDevice", DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, DIA_C_U16_DID_AIVI_BT_CONNECTED_DEVICE) //,
{
   dia_tclFnctTrace oTrace("dia_SrvHandler_ReadBTDevice::dia_SrvHandler_ReadBTDevice()");
}

/******************************************************************************
* FUNCTION:    ~dia_SrvHandler_ReadBTDevice
*
* DESCRIPTION: Destructor
*
******************************************************************************/
dia_SrvHandler_ReadBTDevice::~dia_SrvHandler_ReadBTDevice ( void )
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IBluetoothMostListener>(this);
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandler_ReadBTDevice::~dia_SrvHandler_ReadBTDevice !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

/******************************************************************************
* FUNCTION:    vOnTimeout
*
* DESCRIPTION: Timeout handling function
*
* PARAMETER:   void
*
* RETURNVALUE: void
*
******************************************************************************/

void
dia_SrvHandler_ReadBTDevice::vOnTimeout ( void )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadBTDevice::vOnTimeout");
   (void) unsetSysAdapterListener<dia_IBluetoothMostListener>(this);
   vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
}

/******************************************************************************
* FUNCTION:    vProcessRequest
*
* DESCRIPTION: This function receives request from tester.
			   Data is analyzed and prepared for diagnosis-internal communication
*
* PARAMETER:   const std::vector<void*> - list of input params
*
* RETURNVALUE: void
*
******************************************************************************/

void
dia_SrvHandler_ReadBTDevice::vProcessRequest ( const std::vector<void*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadBTDevice::vProcessRequest");

   tDiaResult errCode = DIA_FAILED;

   dia_IBluetoothMost* pBT = 0;
   if ( (querySysAdapterInterface<dia_IBluetoothMost>(&pBT) == DIA_SUCCESS) && pBT )
   {
	   DIA_TR_INF("dia_SrvHandler_ReadBTDevice::vProcessRequest - querySysAdapterInterface & pBT OK");
      (void) setSysAdapterListener<dia_IBluetoothMostListener>(this);
      errCode = pBT->getBTMostDeviceList();
   }
   else
   {
	   DIA_TR_INF("dia_SrvHandler_ReadBTDevice::vProcessRequest - querySysAdapterInterface & pBT NOT OK");
   }

   if ( errCode != DIA_SUCCESS )
   {
	   DIA_TR_INF("dia_SrvHandler_ReadBTDevice::vProcessRequest - Send to BT Failed");
      (void) unsetSysAdapterListener<dia_IBluetoothMostListener>(this);
	  vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

/******************************************************************************************
* FUNCTION:    vOnBTConnectedDeviceList
*
* DESCRIPTION: Response Handling Function to deliver info about the connected BT devices.
*
* PARAMETER: 
* tU8 - number of paired devices
* const dia_tBTExtendedDeviceInfo* - list of paired devices
*
* RETURNVALUE: void
*
******************************************************************************************/
void
dia_SrvHandler_ReadBTDevice::vOnBTConnectedDeviceList( tU8 numOfPairedDev , const dia_tBTExtendedDeviceInfo* devInfo)
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadBTDevice::vOnBTConnectedDeviceList");
   DIA_TR_INF("Total Number of BT Devices Found = %d", numOfPairedDev);

   (void) unsetSysAdapterListener<dia_IBluetoothMostListener>(this);

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START + DATA_LENGTH);

   for(tU8 btcount = OSAL_NULL; btcount < numOfPairedDev; btcount++)
   {
	   if (devInfo[btcount].btDevAddr && devInfo[btcount].btDevName )
	   {
	     DIA_TR_INF("dia_SrvHandler_ReadBTDevice - BT Device Number = %d", btcount);
	     DIA_TR_INF("dia_SrvHandler_ReadBTDevice - Device Name = \"%s\"", devInfo[btcount].btDevName);
	     DIA_TR_INF("dia_SrvHandler_ReadBTDevice - Device Address = \"%s\"", devInfo[btcount].btDevAddr);

		 tU8 start_index = (tU8)(btcount*(BT_NAME_LENGTH + BT_MAC_ADDR_LENGTH));
		 for(tU8 namelen = OSAL_NULL; namelen < BT_NAME_LENGTH; namelen++)
		 {
			 if(namelen < strlen(devInfo[btcount].btDevName))
			 {
				 (void) oDiagMsgBuffer().vSetDataU8(DATA_START + start_index + namelen,devInfo[btcount].btDevName[namelen]);					 			
			 }
			 else
			 {
				 (void) oDiagMsgBuffer().vSetDataU8(DATA_START + start_index + namelen,0x00);					
			 }
		 }				
		 for(tU8 addrlen = OSAL_NULL; addrlen < BT_MAC_ADDR_LENGTH; addrlen++)
		 {
			 tU8 byte1 = devInfo[btcount].btDevAddr[addrlen * 2];
			 tU8 byte2 = devInfo[btcount].btDevAddr[(addrlen * 2) + 1];

			 byte1 = (byte1 < 'A') ? (byte1 - ASCII_TO_CHAR_NUMBER) : ((byte1 < 'a')?(byte1 - ASCII_TO_CHAR_ALPHABET_BIG):(byte1 - ASCII_TO_CHAR_ALPHABET_SMALL));
			 byte2 = (byte2 < 'A') ? (byte2 - ASCII_TO_CHAR_NUMBER) : ((byte2 < 'a')?(byte2 - ASCII_TO_CHAR_ALPHABET_BIG):(byte2 - ASCII_TO_CHAR_ALPHABET_SMALL));

			 tU8 addr_byte = (tU8)((byte1 << FOUR_BITS) | byte2);
			 
			(void) oDiagMsgBuffer().vSetDataU8(DATA_START + start_index + BT_NAME_LENGTH + addrlen, addr_byte);
		 }
	   }
   }

   if(numOfPairedDev < MAX_BT_DEVICE_COUNT)
   {
	   DIA_TR_INF("Total Number of BT Devices %d is Less than MAX %d ", numOfPairedDev, MAX_BT_DEVICE_COUNT );
	   for(tU8 i = (tU8)(numOfPairedDev *(BT_NAME_LENGTH + BT_MAC_ADDR_LENGTH)); i < DATA_LENGTH ; i++)
	   {
		   (void) (oDiagMsgBuffer().vSetDataU8( DATA_START+i , DEFAULT_RESPONSE_BYTE ));
	   }
   }

   vResReadyAndQuit();
}


