/*
 * dia_SrvHandler_ECUTemperatureRead.cpp
 *
 *  Created on: 06.09.2016
 *      Author: dvd5kor
 *
 * TTFis:\> DIA_REQ UDS 03 22 10 07
 *
 * -----------------------------------------------------------------------------------
 * Date 			| Author			 | Modification
 * 06.09.2016  		| dvd5kor	  		 | AIVI-26189  
 * ------------------------------------------------------------------------------------
 */

#include "dia_SrvHandler_ECUTemperatureRead.h"

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define DATA_START 3
#define DATA_LENGTH 1
#define TEMP_LOW_LIMIT          0x0000
#define TEMP_HIGH_LIMIT         0x0500 // 1280 deci-degC = 128 degC

//-----------------------------------------------------------------------------

dia_SrvHandler_ECUTemperatureRead::dia_SrvHandler_ECUTemperatureRead ( void )
    : dia_ServiceHandlerUDS("dia_SrvHandler_ECUTemperatureRead",DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,(tU16) DIA_C_U16_DID_AIVI_ECU_TEMP /*DID*/),
		mTemperatureID(DIA_EN_HEATCTRL_TEMPERATURE_ID_UNKNOWN)
{
   dia_tclFnctTrace trc("dia_SrvHandler_ECUTemperatureRead::dia_SrvHandler_ECUTemperatureRead(tVoid)");
}

//-----------------------------------------------------------------------------
dia_SrvHandler_ECUTemperatureRead::dia_SrvHandler_ECUTemperatureRead ( tCString name, tU8 sid, tU16 did, tU8 temperatureID )
   : dia_ServiceHandlerUDS(name,sid,(tU16) did),
     mTemperatureID(temperatureID)
{
	dia_tclFnctTrace trc("dia_SrvHandler_ECUTemperatureRead::dia_SrvHandler_ECUTemperatureRead(tVoid)");
}
//-----------------------------------------------------------------------------

dia_SrvHandler_ECUTemperatureRead::~dia_SrvHandler_ECUTemperatureRead ( void )
{
    _BP_TRY_BEGIN
    {
       (tVoid) unsetSysAdapterListener<dia_IHeatControlListener>(this);
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_SrvHandler_ECUTemperatureRead::~dia_SrvHandler_ECUTemperatureRead !!!");
        DIA_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_ECUTemperatureRead::vProcessRequest ( const std::vector<tVoid*>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ECUTemperatureRead::vProcessRequest");
   tDiaResult requestErrorState = DIA_FAILED;

   dia_IHeatControl* pInterface = 0;
   if ( querySysAdapterInterface<dia_IHeatControl>(&pInterface) == DIA_SUCCESS )
   {
      DIA_TR_INF("dia_SrvHandler_ECUTemperatureRead::vProcessRequest - querySysAdapterInterface OK");
      if ( pInterface )
      {
         DIA_TR_INF("dia_SrvHandler_ECUTemperatureRead::vProcessRequest - pInterface OK");
         (tVoid) setSysAdapterListener<dia_IHeatControlListener>(this);
		 
		 dia_eHeatCtrlTemperatureClass temperatureClass = getTemperatureClass ();
         if ( temperatureClass != DIA_EN_HEATCTRL_TEMPERATURE_CLASS_ALL)
         {
            requestErrorState = pInterface->getTemperatures(temperatureClass);
         }
         else
         {
            DIA_TR_INF("dia_SrvHandler_ECUTemperatureRead::vProcessRequest unsupported Interface.");
            requestErrorState = DIA_E_NOT_SUPPORTED;
         }
      }
   } 
   else 
   {
      DIA_TR_INF("dia_SrvHandler_ECUTemperatureRead::vProcessRequest - querySysAdapterInterface NOK !");
   }

   if ( requestErrorState != DIA_SUCCESS )
   {
      if ( pInterface )
      {
         (void) unsetSysAdapterListener<dia_IHeatControlListener>(this);
      }
      DIA_TR_ERR(" ---  SEND TO HEAT CONTROL SERVER FAILED!!!!");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }
}

/******************************************************************************
* FUNCTION:    vOnTemperatureInfo
*
* DESCRIPTION: Response handling Function
*
* PARAMETER:   list - tempList
*
* RETURNVALUE: VOID
*
******************************************************************************/

void
dia_SrvHandler_ECUTemperatureRead::vOnTemperatureInfo ( std::list<dia_tHeatCtrlTemperature>& tempList )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ECUTemperatureRead:vOnTemperatureInfo");

   (tVoid) unsetSysAdapterListener<dia_IHeatControlListener>(this);

   if ( tempList.empty() )
   {
      oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      DIA_TR_INF("dia_SrvHandler_ECUTemperatureRead::vOnTemperatureInfo list<dia_tHeatCtrlTemperature> is empty!");

      vResReadyAndQuit();
   }
   else
   {
		tS16 temperature = 0;
   
		std::list<dia_tHeatCtrlTemperature>::iterator iter = tempList.begin();
		for ( ; iter != tempList.end(); iter++ )
		{
			if((*iter).mID == mTemperatureID)
			{
				temperature = (tS16)(*iter).mTemperature;
			}
		}

		// assemble response
		DIA_TR_INF("dia_SrvHandler_ECUTemperatureRead::vOnTemperatureInfo - Assemble positive response with temp: %d", temperature);
   
		oDiagMsgBuffer().vSetPosResp();
		oDiagMsgBuffer().vSetDataLength(DATA_START+DATA_LENGTH);
		tU8 u8DataPos = DATA_START;

		if ((temperature < TEMP_LOW_LIMIT) || (temperature > TEMP_HIGH_LIMIT))
		{
			DIA_TR_ERR("dia_SrvHandler_ECUTemperatureRead::vOnTemperatureInfo - the value of temperature is out of limit !!!!");
			(void) oDiagMsgBuffer().vSetDataU8(u8DataPos, 0xFF);
		}
		else
		{
			(void) oDiagMsgBuffer().vSetDataU8(u8DataPos++, (tU8)(temperature/10) );
		}
		vResReadyAndQuit();
	}
}

//------------------------------------------------------------------------------

void
dia_SrvHandler_ECUTemperatureRead::vOnTimeout ( void )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ECUTemperatureRead::vOnTimeout");
   (tVoid) unsetSysAdapterListener<dia_IHeatControlListener>(this);
   oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   vResReadyAndQuit();
}

//-------------------------------------------------------------------------------
//-----------------------------------------------------------------------------
dia_eHeatCtrlTemperatureClass
dia_SrvHandler_ECUTemperatureRead::getTemperatureClass ( void ) const
{
   dia_eHeatCtrlTemperatureClass tempClass = DIA_EN_HEATCTRL_TEMPERATURE_CLASS_ALL;

   switch(mTemperatureID)
   {
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_V850:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_PANEL:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_CDDRIVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_DVDDRIVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_GPS:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_GYRO:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_EMMC:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_AMP:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_IMX:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_HEATSINK:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_DISPLAY:
      {
         tempClass = DIA_EN_HEATCTRL_TEMPERATURE_CLASS_REAL;
         break;
     }

   case DIA_EN_HEATCTRL_TEMPERATURE_ID_AMPLIFIER:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_CPU:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_EXPBOARD:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_BOARD:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_DRIVE:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_SYSTEM:
   case DIA_EN_HEATCTRL_TEMPERATURE_ID_MAIN:
      {
         tempClass = DIA_EN_HEATCTRL_TEMPERATURE_CLASS_ESTIMATED;
         break;
     }

   default:
      {
         break;
     }
   }

   return tempClass;
}


