 /*
 * \file        dia_SrvHandler_ConfigurationPartNumber.cpp
 *
 * \brief       {insert brief description here}
 *
 * \details    1. Read 10 bytes ECU Part Number (see property DIA_PROP_AIVI_ECU_PART_NUMBER).
 *                1a. if nok, NRC 0x22 (Scope1 and Scope 2.0 Only) or go to next point (Scope2.1)
 *                1b. if ok, go to next point.
 *             2. Trigger test for CRC check of the virtual buffers with variable (see dia_GenericConfigLongItemDidWrite) *
 *             3. Trigger test dia_TestConfigHash_AIVI (read config data, calculate SHA-1,
 *                compare to reference SHA-1 DIA_PROP_AIVI_CONFIG_HASH_VALUE, both Configuration Part Number
 *                DIA_PROP_AIVI_CONFIG_PARTS_NUMBER and DIA_PROP_AIVI_DEF_CONFIG_PARTS_NUMBER, set ITC according to comparison result of PNs and FP,
 *                write calculated fingerprint to registry DIA_PROP_CUSTOMER_CONFIG_FINGERPRINT in
 *                ascii format AABBCCDD-...-EEFFAABB if both FPs match AND PNs do not match,
 *                otherwise write zeros {0000000-000000-...-00000000})
 * Scope2.0 <= 4. Send response to the tester (read property DIA_PROP_CUSTOMER_CONFIG_FINGERPRINT)
 *                4a. if both fingerprints don't match OR Configuration Part Number for both Customer and Customer Default are identical
 *                    (property DIA_PROP_CUSTOMER_CONFIG_FINGERPRINT is zeros, see also properties DIA_PROP_AIVI_CONFIG_PARTS_NUMBER and DIA_PROP_AIVI_DEF_CONFIG_PARTS_NUMBER),
 *                    - send back ten bytes 0x30 ('0') and
 *                    - write ten bytes 0x30 ('0') to registry (property DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER)
 *                4b. otherwise
 *                    - send back the Configuration Part Number (see property DIA_PROP_AIVI_CONFIG_PARTS_NUMBER)
 *                    - write the Configuration Part Number to registry (see property DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER)
 * Scope 2.1   4. Send response to the tester (read property DIA_PROP_AIVI_CONFIG_PARTS_NUMBER)
 *                4a. send DIA_PROP_AIVI_CONFIG_PARTS_NUMBER if reading is successful.
 *                4b. otherwise send ten bytes 0x30 ('0').
 *
 * \author      kaa1hi
 * \date        May 06, 2015
 *
 * \copyright   Robert Bosch Car Multimedia 2015
 */


#include "project/services/customer/dia_SrvHandler_ConfigurationPartNumber.h"

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_TEST_CONTROLLER__
#include "common/framework/test/dia_TestController.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_TEST__
#include "common/framework/test/dia_common_test.h"
#endif

#ifndef __INCLUDED_DIA_TEST_SERVICE__
#include "common/framework/test/dia_TestService.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_TEST_CONFIG_HASH_H_
#include "common/framework/test/dia_TestConfigHash.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#define DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE         10
#define DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER_SIZE  ((tU16)11)  /* 10 ascii + 1 termination */

const tU8 ASCII_ZERO = 0x30;

//-----------------------------------------------------------------------------
dia_SrvHandler_ConfigurationPartNumber::dia_SrvHandler_ConfigurationPartNumber(bool Scope2_1Defined)
   : dia_ServiceHandlerUDS("dia_SrvHandler_ConfigurationPartNumber", DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, DIA_C_U16_DID_AIVI_CONFIG_PARTS_NUMBER),
     dia_ConfigNotifier("dia_SrvHandler_ConfigurationPartNumber", DIA_PROP_CUSTOMER_CONFIG_FINGERPRINT),
     mValidComparison(false),
     mScope21Defined(Scope2_1Defined)
{
   dia_tclFnctTrace trc("dia_SrvHandler_ConfigurationPartNumber::dia_SrvHandler_ConfigurationPartNumber()");
}

//-----------------------------------------------------------------------------
dia_SrvHandler_ConfigurationPartNumber::~dia_SrvHandler_ConfigurationPartNumber( void )
{
}

//-------------------------------------------------------------------------------

void
dia_SrvHandler_ConfigurationPartNumber::vProcessRequest ( const std::vector<tArgsType>& vecArgs )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest");

   if (mScope21Defined) DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest Scope2.1");
   else                 DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest Scope1 and Scope2.0");

   tU8 ECUPartNumber[DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE + 1 /*termination*/] = { 0 };

   //read ECU Part Number from KDS
   tU32 size;
   bool ECUReadFailed = true;

   if ( DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE==(size = dia_getPropertySize(DIA_PROP_AIVI_ECU_PART_NUMBER)) )
   {
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest DIA_PROP_AIVI_ECU_PART_NUMBER size OK");
      tDiaResult getPropRes;

      if ( (getPropRes = dia_getProperty(DIA_PROP_AIVI_ECU_PART_NUMBER, ECUPartNumber, DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE)) == DIA_SUCCESS )
      {
         DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest DIA_PROP_AIVI_ECU_PART_NUMBER OK");

         DIA_TR_INF("ECUPartNumber 0x%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
               ECUPartNumber[ 0+0], ECUPartNumber[ 0+1], ECUPartNumber[ 0+2], ECUPartNumber[ 0+3],
               ECUPartNumber[ 4+0], ECUPartNumber[ 4+1], ECUPartNumber[ 4+2], ECUPartNumber[ 4+3],
               ECUPartNumber[ 8+0], ECUPartNumber[ 8+1] );

         ECUReadFailed = false;
      }
      else
      {
         DIA_TR_ERR("### READ DIA_PROP_AIVI_ECU_PART_NUMBER FAILED! getPropRes=0x%08X ###", getPropRes);
      }
   }

   if (ECUReadFailed)
   {
      DIA_TR_ERR("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest dia_getPropertySize FAILED. size=%d", size);

      if (!mScope21Defined)
      {
         // Sending back the Negative answer
         vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
         return;
      }
   }

   // Test if the virtual buffers of ECNR/Audio Parameters with variable length need to be flushed, see dia_GenericConfigLongItemDidWrite
   dia_TestController* pTestCtrl = getInstanceOfTestController();
   if (pTestCtrl)
   {
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest DIA_EN_TESTCONDITION_GENERIC_CONFIG_ITEM_VAR_LENGTH_WRITE");

      std::list<dia_Test*> testRepVarLengthParamSets;
      pTestCtrl->queryTests(DIA_EN_TESTCONDITION_GENERIC_CONFIG_ITEM_VAR_LENGTH_WRITE, testRepVarLengthParamSets);

      std::list<dia_Test*>::iterator it;
      for (it = testRepVarLengthParamSets.begin(); it != testRepVarLengthParamSets.end(); it++)
      {
         dia_Test* pParamSetTest = *it;
         if (pParamSetTest && (pParamSetTest->getTestType() == DIA_EN_TEST_TYPE_SERVICE))
         {
            dia_TestService* pParamSetService = static_cast<dia_TestService*>(pParamSetTest);

            DIA_TR_INF("Executing Flush on (Name = %s)", pParamSetService->getTestName());

            pParamSetService->setEmbeddedMode(TRUE);
            const std::vector<tArgsType> nullArgs;
            // Args to the ServiceHandler are only assigned when the ServiceHandler is added to the Engine!
            // An empty arg list indicates the trigger to flush all chunks of a variable length param set
            pParamSetService->vProcessRequest(nullArgs);
         }
      }
   }

   // calculate configuration hash, see dia_TestConfigHash_AIVI
   bool testConfigHashFailed = true;
   if ( pTestCtrl )
   {
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest DIA_EN_TESTCONDITION_CODINGVALUE_CHANGE");

      std::list<dia_Test*> cmdRep;
      pTestCtrl->queryTests(DIA_EN_TESTCONDITION_CODINGVALUE_CHANGE, cmdRep);

      if ( !cmdRep.empty() )
      {
         DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest cmdRep.size()=%d", cmdRep.size());
         tU32 expectedUID = dia_getHashCodeFromString("dia_TestConfigHash");

         std::list<dia_Test*>::iterator it;
         for (it = cmdRep.begin(); it != cmdRep.end(); ++it)
         {
            if ((*it) && ((*it)->getTestType() == DIA_EN_TEST_TYPE_SERVICE) )
            {
               if ((*it)->getTestUID()==expectedUID)
               {
                  DIA_TR_INF("Executing Test (Type = %d, Name = %s)", (*it)->getTestType(), (*it)->getTestName());

                  dia_TestService* pTestService = static_cast<dia_TestService*>((*it));
                  pTestService->setEmbeddedMode(TRUE);

                  SetValidComparison(false);
                  pTestService->vProcessRequest(vecArgs);
                  testConfigHashFailed = false;

                  break;
               }
               else
               {
                  DIA_TR_INF("UID: act 0x%08X != exp 0x%08X. Skip (\"%s\")", (*it)->getTestUID(), expectedUID, (*it)->getTestName());
               }
            }
            else
            {
               DIA_TR_ERR("### dia_SrvHandler_ConfigurationPartNumber::vProcessRequest pCurrentTest null or not service ###");
            }
         }
      }
      else
      {
         DIA_TR_ERR("### dia_SrvHandler_ConfigurationPartNumber::vProcessRequest cmdRep is empty ###");
      }
   }
   else
   {
      DIA_TR_ERR("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest pTestCtrl is NULL");
   }

   if (testConfigHashFailed)
   {
      DIA_TR_ERR("### dia_SrvHandler_ConfigurationPartNumber::vProcessRequest Embedded Test Not Executed.");

      if (!mScope21Defined)
      {
         // Sending back the Negative answer
         vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
         return;
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   //read both Configuration Part Number: DIA_PROP_AIVI_CONFIG_PARTS_NUMBER
   /////////////////////////////////////////////////////////////////////////////
   tU8 configurationPartNumberNonDefCD[DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE + 1 /*termination*/] = {0};
   tDiaResult retCodeNonDefCD = dia_getProperty(DIA_PROP_AIVI_CONFIG_PARTS_NUMBER, configurationPartNumberNonDefCD, DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE);

   if (DIA_SUCCESS!=retCodeNonDefCD)
   {
      DIA_TR_ERR("UNABLE TO READ ITEM: DIA_PROP_AIVI_CONFIG_PARTS_NUMBER     --- Send ECU Part Number !!");

      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength((1 + 2/*DID Len*/)+DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE);
      for (tU16 i = 0; i < DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE; ++i)
      {
         (void) oDiagMsgBuffer().vSetDataU8((1 + 2/*DID Len*/) + i,  mScope21Defined ? ASCII_ZERO: ECUPartNumber[i]);
      }

      vResReadyAndQuit();
      return;
   }
   else
   {
      DIA_TR_INF("Read DIA_PROP_AIVI_CONFIG_PARTS_NUMBER was successful.");

      tU8* p = &configurationPartNumberNonDefCD[0];
      DIA_TR_INF("AIVI_CONFIG_PARTS_NUMBER     Hex: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X", p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9] );

      if (mScope21Defined)
      {
         DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest DIA_PROP_AIVI_CONFIG_PARTS_NUMBER Scope2.1");
         oDiagMsgBuffer().vSetPosResp();
         oDiagMsgBuffer().vSetDataLength((1 + 2/*DID Len*/)+DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE);
         for (tU16 i = 0; i < DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE; ++i)
         {
            (void) oDiagMsgBuffer().vSetDataU8((1 + 2/*DID Len*/) + i, configurationPartNumberNonDefCD[i]);
         }
      }
   }

   /////////////////////////////////////////////////////////////////////////////
   //read Calculated Fingerprint
   /////////////////////////////////////////////////////////////////////////////
   tU8 refZeros[DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE] = "{00000000-00000000-00000000-00000000-00000000}";
   tU8 actualCustConfigFPLongFormat[DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE] = {0};
   tU32 actualCustConfigFPLongFormatLength = DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE;
   tDiaResult retCode = dia_getProperty(DIA_PROP_CUSTOMER_CONFIG_FINGERPRINT, actualCustConfigFPLongFormat, &actualCustConfigFPLongFormatLength);
   tS32 compResultFP = (tS32)memcmp(actualCustConfigFPLongFormat, refZeros, DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE);

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength((1 + 2/*DID Len*/)+DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE);

   DIA_TR_INF("mValidComparison              : %s", ((mValidComparison) ? "true" : "false") );
   DIA_TR_INF("retCode of prop reading       : 0x%08x", retCode );
   DIA_TR_INF("memcmp for fp                 : %s", ((0==compResultFP) ? "zeros in registry (FPs don't match or PNs do match)" : "correct hash in registry") );

   if (!mValidComparison || (DIA_SUCCESS!=retCode) || (0==compResultFP) )
   {
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest Send back ECU Part Number");

      if (!mScope21Defined)
      {
         for (tU16 i = 0; i < DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE; ++i)
         {
            ECUPartNumber[i] = ASCII_ZERO;
            (void) oDiagMsgBuffer().vSetDataU8((1 + 2/*DID Len*/) + i, ECUPartNumber[i]);
         }
      }

      //write ECU Part Number to registry
      if (DIA_SUCCESS!=dia_setProperty(DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER, (tU8*)ECUPartNumber, DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER_SIZE))
      {
         DIA_TR_ERR("### ERROR: UNABLE TO WRITE ECU PART NUMBER TO REGISTRY ###");

         if (!mScope21Defined)
         {
            vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
            return;
         }
      }
   }
   else
   {
      DIA_TR_INF("FP=%s", actualCustConfigFPLongFormat);
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest fingerprints match");
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::vProcessRequest Send back Configuration Part Number");

      if (!mScope21Defined)
      {
         for (tU16 i = 0; i < DIA_PROP_AIVI_ECU_PART_NUMBER_SIZE; ++i)
         {
            (void) oDiagMsgBuffer().vSetDataU8((1 + 2/*DID Len*/) + i, configurationPartNumberNonDefCD[i]);
         }
      }

      //write the Configuration Part Number to registry
      if (DIA_SUCCESS!=dia_setProperty(DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER, (tU8*)configurationPartNumberNonDefCD, DIA_PROP_CUSTOMER_CONFIG_PART_NUMBER_SIZE))
      {
         DIA_TR_ERR("### ERROR: UNABLE TO WRITE CONFIGURATION PART NUMBER TO REGISTRY ###");

         if (!mScope21Defined)
         {
            vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
            return;
         }
      }
   }

   vResReadyAndQuit();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SrvHandler_ConfigurationPartNumber::onPropertyUpdate ( tU32 propID )
{
   dia_tclFnctTrace oTrace("dia_SrvHandler_ConfigurationPartNumber::onPropertyUpdate()");

   DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::onPropertyUpdate(propID=0x%08X)", propID);

   if (DIA_PROP_CUSTOMER_CONFIG_FINGERPRINT==propID)
   {
      DIA_TR_INF("dia_SrvHandler_ConfigurationPartNumber::onPropertyUpdate mValidComparison set to true");
      SetValidComparison(true);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_SrvHandler_ConfigurationPartNumber::SetValidComparison(bool validComparison)
{
   mValidComparison = validComparison;
}
