/*!
 * \file       dia_SrvHandlerVINIdentWrite.cpp
 *
 * \brief      Service Handler for writing VIN
 *
 * \details    VIN writing is allowed without Customer Security Access if:
 *             1/ current VIN is not available, or
 *             2/ all bytes of current VIN are 0x30.
 *
 *             VIN writing is possible (with or without Customer Security Access) only if incoming VIN is different than following patterns:
 *             1/ all bytes of VIN are 0x00
 *             2/ all bytes of VIN are 0x30
 *             3/ all bytes of VIN are 0xFF
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

// TTFis: DIA_REQ UDS 03 2E F1 90 xx (17 bytes)

#ifndef __INCLUDED_SERVICE_HANDLER_VIN_IDENT_WRITE__
#include "project/services/customer/dia_SrvHandlerVINIdentWrite.h"
#endif 

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SECURITY__
#include "common/framework/security/dia_common_security.h"
#endif

#define DATA_START (1/*sizeof(SID)*/ + mDIDLen)

//-----------------------------------------------------------------------------

dia_SrvHandlerVINIdentWrite::dia_SrvHandlerVINIdentWrite ( tCString name, tU8 sid, tU16 did )
    : dia_SrvHandlerGenericIdentWrite(name,sid,did)
{
   dia_tclFnctTrace trc("dia_SrvHandlerVINIdentWrite::dia_SrvHandlerVINIdentWrite(tCString,tU8,tU16)");
}
//-----------------------------------------------------------------------------

dia_SrvHandlerVINIdentWrite::~dia_SrvHandlerVINIdentWrite ( void )
{
}

//-----------------------------------------------------------------------------

void
dia_SrvHandlerVINIdentWrite::vProcessRequest ( const std::vector<void*>& vecArgs )
{
   dia_tclFnctTrace trc("dia_SrvHandlerVINIdentWrite::vOnProcessRequest");

   tBool allowWrite = false;

   /* SrvHandler Defines according to CDD-File */
   #define MSG_DATA_OFFSET 4                        // Offset from oDiagMsgBuffer-start to 1st data-byte (e.g. MemorySize-Byte)
   /*------------------------------------------*/

   if ( !(vecArgs.size()) )
   {
      // Sending back the Negative answer
      DIA_TR_INF("dia_SrvHandlerVINIdentWrite: FAILED (Property Not Available) !!");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      return;
   }

   tU32 propID = (tU32) vecArgs[0];
   tU16 propLength = dia_getPropertySize(propID);

   // extract the data from the received message
   tU8 const* pU8 = oDiagMsgBuffer().u8GetBuffer();
   tU8 u8propData[DIA_PROP_LENGTH_MAX] = {0};
   // initialize the data memory
   (void) ::memset(u8propData,0,DIA_PROP_LENGTH_MAX);

   // copy until end of msg. (max ms length == 240B) or '\0' (0x0)
   tU16 u16EleCount = 0;
   for ( ; u16EleCount < (oDiagMsgBuffer().u16GetDataLength()-MSG_DATA_OFFSET); u16EleCount++)
   {
      if ( u16EleCount < DIA_PROP_LENGTH_MAX )
      {
         u8propData[u16EleCount] = pU8[u16EleCount+MSG_DATA_OFFSET];
         DIA_TR_INF("dia_SrvHandlerVINIdentWrite::u8propData[%d] 0x%02x", u16EleCount, u8propData[u16EleCount]);
      }
   }
   
   tDiaResult retCode = checkLength(u16EleCount, propLength, propLength);
   if ( retCode != DIA_SUCCESS )
   {
      // Sending back the Negative answer
      DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: FAILED (SIZE CHECK). Provided size: %d != Actual size: %d", u16EleCount, propLength);
      vSendNegativeResponse(DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT);
      return;
   }

   //check for currently stored value of VIN
   {
      std::vector<tU8> dataVec;

      dataVec.reserve(propLength);
      dataVec.resize(propLength);

      DIA_TR_INF("dia_SrvHandlerVINIdentWrite::vOnProcessRequest - reading currently stored VIN value");
      if (DIA_FAILED == dia_getProperty(propID,dataVec))
      {
       DIA_TR_INF("dia_SrvHandlerVINIdentWrite::vOnProcessRequest - VIN is uninitialised");
       allowWrite = true;
     }
     else
     {
       DIA_TR_INF("dia_SrvHandlerVINIdentWrite::vOnProcessRequest - VIN is available.");
       DIA_TR_INF("dia_SrvHandlerVINIdentWrite::vOnProcessRequest - check if VIN is default value.");
       allowWrite = true;
       for (tU16 i = 0; i < dataVec.size(); ++i)
       {
         DIA_TR_INF("dataVec[%d] = 0x%02x",i, dataVec[i]);
         if(dataVec[i] != 0x30)
         {
            allowWrite = false;
            break;
         }
       }
     }
   }

   //check if write is allowed without security unlocked.
   if(allowWrite == true)
   {
     //check if valid value is being written
     tBool validInput = false;
     if ((u8propData[0] == 0x30) || (u8propData[0] == 0xFF) || (u8propData[0] == 0x00))
     {
        validInput = false;
        for (tU16 i = 0; i < u16EleCount; ++i)
        {
           if(u8propData[i] != u8propData[0])
           {
              DIA_TR_INF("dia_SrvHandlerVINIdentWrite: input is not one of the prohibited values, allow right!!");
              validInput = true;
              break;
           }
        }
     }
     else
     {
        validInput = true;
     }

     if (validInput == true)
     {
        DIA_TR_INF("dia_SrvHandlerVINIdentWrite: (WRITE DATA TO PERSISTENT MEMORY) without checking security lock!!");
        retCode = dia_setProperty ( propID, u8propData, u16EleCount );
        if ( retCode != DIA_SUCCESS )
        {
          // Sending back the Negative answer
          DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: FAILED (WRITE DATA TO PERSISTENT MEMORY) !!");
          vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
          return;
       }
     }
     else
     {
        // Sending back the Negative answer
        DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: invalid Input (prohibited value) !!");
        vSendNegativeResponse(DIA_E_U8_UDS_OUT_OF_RANGE);
        return;
     }
   }
   else
   {
         //check if valid value is being written
         tBool validInput = false;
         if ((u8propData[0] == 0x30) || (u8propData[0] == 0xFF) || (u8propData[0] == 0x00))
         {
            validInput = false;
            for (tU16 i = 0; i < u16EleCount; ++i)
            {
               if(u8propData[i] != u8propData[0])
               {
                  DIA_TR_INF("dia_SrvHandlerVINIdentWrite: input is not one of the prohibited values, allow right!!");
                  validInput = true;
                  break;
               }
            }
         }
         else
         {
            validInput = true;
         }

         if (validInput == true)
         {
             DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: check if customer security level is UNLOCKED");
             if(DIA_SUCCESS != checkCustSecurityLevelUnlocked())
             {
               DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: FAILED (not allowed to write without unlocking device)!!");
               vSendNegativeResponse(DIA_E_U8_UDS_SECURITY_ACCESS_DENIED);
               return;
             }
             else
             {
        	  DIA_TR_INF("dia_SrvHandlerVINIdentWrite: (WRITE DATA TO PERSISTENT MEMORY) with checking security lock!!");
              retCode = dia_setProperty ( propID, u8propData, u16EleCount );
             }

            if ( retCode != DIA_SUCCESS )
            {
              // Sending back the Negative answer
              DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: FAILED (WRITE DATA TO PERSISTENT MEMORY) !!");
              vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
              return;
           }
         }
         else
         {
            // Sending back the Negative answer
            DIA_TR_ERR("dia_SrvHandlerVINIdentWrite: invalid Input (prohibited value) !!");
            vSendNegativeResponse(DIA_E_U8_UDS_OUT_OF_RANGE);
            return;
         }
   }

   // Sending back the positive answer
   DIA_TR_INF("dia_SrvHandlerVINIdentWrite: SUCCEEDED");
   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START);
   vResReadyAndQuit();
}


tDiaResult
dia_SrvHandlerVINIdentWrite::checkCustSecurityLevelUnlocked ( void ) const
{
   dia_tclFnctTrace trc("dia_SrvHandlerVINIdentWrite::checkCustSecurityLevelUnlocked");
   
   std::list<dia_SecurityLevel*> activeSecurityLevels;
   dia_SecurityManager* pSecMgr = getInstanceOfSecurityManager();
   if ( pSecMgr )
   {
      pSecMgr->getActiveLevels(activeSecurityLevels);
      std::list<dia_SecurityLevel*>::const_iterator activeLevelIter = activeSecurityLevels.begin();
      for ( ; activeLevelIter != activeSecurityLevels.end(); ++activeLevelIter )
      {
       if ( (*activeLevelIter) && ((*activeLevelIter)->getUID() == DIA_UID_SECURITY_LEVEL_PROJECT_01) )
         {
           DIA_TR_INF("dia_SrvHandlerVINIdentWrite: checkCustSecurityLevelUnlocked - DIA_UID_SECURITY_LEVEL_PROJECT_01 is unlocked/active!!");
         return DIA_SUCCESS;
       }
     }
      
     DIA_TR_ERR("!!! dia_SrvHandlerVINIdentWrite::checkCustSecurityLevelUnlocked - DIA_UID_SECURITY_LEVEL_PROJECT_01 is LOCKED!!!");
     return DIA_FAILED;
   }
   else
   {
     DIA_TR_ERR("!!! dia_SrvHandlerVINIdentWrite::checkCustSecurityLevelUnlocked => ERROR: Security manager is not available");
     return DIA_FAILED;
   }

}

