/*!
 * \file       dia_RoutineCtrlIdentifySubSystem.cpp
 *
 * \brief      Service handler that will list out all the possible subsystems connected
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Rajesh Kumar(RBEI/ECA2), Arjun Manjunath Sanu (RBEI/ECA2), Kirty Mayank (RBEI/ECA2)
 *
 * \date       30.09.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include <common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_MANAGER__
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include <common/framework/engine/dia_Session.h>
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_MANAGER__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisManager.h"
#endif

#ifndef __INCLUDED_DIA_ROUTINE_IDENTIFY_SUBSYSTEM_H_
#include "project/services/customer/dia_RoutineCtrlIdentifySubSystem.h"
#endif

static tU8 sRoutineResultMapping[DIA_EN_RTCTRL_STATUS_COUNT] = {
		0xFF, // DIA_EN_RTCTRL_STATUS_UNKNOWN
		0xFF, // DIA_EN_RTCTRL_STATUS_IDLE
		0x03, // DIA_EN_RTCTRL_STATUS_IN_PROGRESS
		0x00, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK
		0x01, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK
		0x02, // DIA_EN_RTCTRL_STATUS_ABORTED
		0xFF  // DIA_EN_RTCTRL_STATUS_TIMED_OUT
};

//------------------------------------------------------------------------------

const std::map<const dia_eIdentifyEntitiesRoutineStatus, const dia_eRoutineStatus> dia_RoutineCtrlIdentifySubSystem::_IdentifyEntity_RoutineStatus_Map = 
{
	{ DIA_EN_IDENTIFY_ENTITIES_UNKNOWN, 	DIA_EN_RTCTRL_STATUS_ABORTED			},
	{ DIA_EN_IDENTIFY_ENTITIES_RUNNING, 	DIA_EN_RTCTRL_STATUS_IN_PROGRESS		},
	{ DIA_EN_IDENTIFY_ENTITIES_NOK, 		DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK	},
	{ DIA_EN_IDENTIFY_ENTITIES_OK, 			DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK	},
	{ DIA_EN_IDENTIFY_ENTITIES_ABORTED, 	DIA_EN_RTCTRL_STATUS_ABORTED			},
	{ DIA_EN_IDENTIFY_ENTITIES_TIMED_OUT, 	DIA_EN_RTCTRL_STATUS_TIMED_OUT			}
};

//------------------------------------------------------------------------------

dia_RoutineCtrlIdentifySubSystem::dia_RoutineCtrlIdentifySubSystem( void )
: dia_Routine("dia_RoutineCtrlIdentifySubSystem", DIA_C_U16_DID_CENTER_CMC_19_IDENTIFY_SUB_SYSTEM, DIA_EN_RTCTRL_ID_PROJECT_14, DIA_EN_RTCTRL_TYPE_LONG_TERM),
  _started(false)
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::dia_RoutineCtrlIdentifySubSystem");
	(void)registerSessionChange();
}

//------------------------------------------------------------------------------

dia_RoutineCtrlIdentifySubSystem::~dia_RoutineCtrlIdentifySubSystem( void )
{
	_BP_TRY_BEGIN
    {
		DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::~dia_RoutineCtrlIdentifySubSystem - unregisterSessionChange");
		(void)unregisterSessionChange();
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_RoutineCtrlIdentifySubSystem::~dia_RoutineCtrlIdentifySubSystem !!!");
        NORMAL_M_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
	
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlIdentifySubSystem::start ( std::vector<tU8>& /*params*/, tU8 /*timerValue*/ )
{
	dia_tclFnctTrace oTrace(" dia_RoutineCtrlIdentifySubSystem::start()");

	// prepare processing of the routine
	vInitialize();

	tDiaResult retCode = DIA_FAILED;

	//! get the instance of subsystem manager
	dia_SubsystemDiagnosisManager* pSubManager = getInstanceOfSubsystemDiagnosisManager();

	if( !pSubManager )
	{
		// Sending back the negative response since Subsystem manager is not available
		DIA_TR_ERR("dia_RoutineCtrlIdentifySubSystem::start FAILED (NO ACCESS TO SUBSYSTEM MANAGER) !!");
		eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
		mIsResultReady = TRUE;
	}
	else
	{	
		retCode = pSubManager->identyfyAllSubsystems();	
	}

	if(DIA_E_SUBSYSTEM_IDENTIFY_ENTITIES_RUNNING == retCode)
	{
		eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
		// set the flag to be used incase request result called first
		_started = true;
		retCode = DIA_SUCCESS;
		DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemStartUpdate:DIA_EN_RTCTRL_STATUS_IN_PROGRESS");
	}
	else
	{
		eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
		DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemStartUpdate:DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK");
	}
	
	//negative answer doesn't need any bytes in output buffer
    if (DIA_FAILED != retCode)
    {
       // one byte expected
       mResults.push_back(sRoutineResultMapping[eGetStatus()]);

       mIsResultReady = TRUE;
    }
	
	DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::start retCode = 0x%08X", retCode);

	return retCode;
}

//--------------------------------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlIdentifySubSystem::requestResult ( void )
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::requestResult(void)");
	
	tDiaResult retCode = DIA_SUCCESS;
	
	if(_started == false)
	{
		return DIA_E_SEQUENCE_ERROR;
	}
	
	// get the instance of subsystem manager
	dia_SubsystemDiagnosisManager* pSubManager = getInstanceOfSubsystemDiagnosisManager();
	if( pSubManager )
	{
		//! Set the listener to get informed about the identified Subsystem later
		if (setSysAdapterListener<dia_ISubsystemDiagnosisListener>(this) == DIA_SUCCESS)
		{
			(void)pSubManager->identyfyAllSubsystemsRequestResult();
			DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem ---> DIA_E_ROUTINE_RESULT_PENDING");
			retCode = DIA_E_ROUTINE_RESULT_PENDING;	
		}
		else
		{
			DIA_TR_ERR("dia_RoutineCtrlIdentifySubSystem --> querySysAdapterInterface returned NULL PTR");
		}
	}
	else
	{
		DIA_TR_ERR("dia_RoutineCtrlIdentifySubSystem --> FAILED (NO ACCESS TO SUBSYSTEM MANAGER) !!");
	}
	
	return retCode;
}

//--------------------------------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlIdentifySubSystem::requestResult ( std::vector<tU8>& results )
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::requestResult(...)");

	tDiaResult retCode = DIA_E_SEQUENCE_ERROR;
	results.clear();
	
	if ( !(mResults.empty()) )
	{
		/* this code is executed in case of reply for start request only */
		DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::requestResult --- 1");

		std::vector<tU8>::iterator iter = mResults.begin();
		for ( ; iter != mResults.end(); iter++ )
		{
			results.push_back(*iter);
		}
		// clear buffer, such reply is necessary once
		mResults.clear();
		// prepare processing of the routine
		vInitialize();
		// keep the result until new calculation is started
		retCode = DIA_SUCCESS;
	}
	
	DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::requestResult %s", (DIA_SUCCESS == retCode? "DIA_SUCCESS": "DIA_E_SEQUENCE_ERROR"));

	return retCode;
}

//------------------------------------------------------------------------------------

void dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate (const std::list<dia_tSubsystemInfo>& nodeList, dia_eIdentifyEntitiesRoutineStatus status)
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate");
	
	// prepare processing of the routine
	vInitialize();
	
	(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisListener>(this);
	
	auto it = _IdentifyEntity_RoutineStatus_Map.find(status);
	
	//! Populate the response results with status and subsytems node list info
	if(it != _IdentifyEntity_RoutineStatus_Map.end())
	{
		eSetStatus(it->second);
		mResults.push_back(sRoutineResultMapping[eGetStatus()]);
	}
	
	switch ( status )
	{
		case DIA_EN_IDENTIFY_ENTITIES_ABORTED:
		{
			_started = false;
			DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate -- Subsystems Identification Aborted!");
		}
		break;
		
		case DIA_EN_IDENTIFY_ENTITIES_NOK:
		{
			_started = false;
			DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate -- Subsystems Identification Completed, NoK!");
		}
		break;
		
		case DIA_EN_IDENTIFY_ENTITIES_OK:
		{
			_started = false;
			DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate -- Subsystems Identification Complete, Ok!");
			//vExtractAndFillNodeData(nodeList);
		}
		break;
		
		case DIA_EN_IDENTIFY_ENTITIES_RUNNING:
		{
			DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate -- Subsystems Identification in Progress!");
			
			/*if(!nodeList.empty())
			{
				DIA_TR_INF(" -- Node List not empty! -- ");
				vExtractAndFillNodeData(nodeList);
			}*/
		}
		break;
		
		default:
		{
			_started = false;
			DIA_TR_INF("dia_RoutineCtrlIdentifySubSystem::vOnIdentifySubsystemsRequestUpdate -- Subsystems Identification in Unknown state!");
		}
		break;
	}
	
	vExtractAndFillNodeData(nodeList);

	mIsResultReady = TRUE;
    dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
}

//-----------------------------------------------------------------------

void 
dia_RoutineCtrlIdentifySubSystem::vExtractAndFillNodeData(const std::list<dia_tSubsystemInfo>& nodeList)
{
	dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::vExtractAndFillNodeData(...)");
	
	tU32 size = nodeList.size();

	std::list<dia_tSubsystemInfo>::const_iterator  iter = nodeList.begin();

	for ( ; iter != nodeList.end(); ++iter )
	{
		dia_tSubsystemInfo const &subsystem = *iter;
		DIA_TR_INF("subsystemSupplierName is 0x%04X",subsystem.subsystemEntityName);

		for(int i = DIA_C_SUBSYSTEM_ENTITY_NAME_LEN; i > 0 ; i--)
		{
			mResults.push_back(static_cast<tU8>(((subsystem.subsystemEntityName) >> (8 * (i-1)))));
		}

		DIA_TR_INF("subsystemSupplierName is 0x%02X",subsystem.subsystemSupplierName);
		mResults.push_back(subsystem.subsystemSupplierName);

		DIA_TR_INF("subsystemHWVariant is 0x%02X",iter->subsystemHWVariant);
		mResults.push_back(subsystem.subsystemHWVariant);

		DIA_TR_INF("subsystemSWVariant is 0x%04X",iter->subsystemSWVariant);
		for(int i = DIA_C_SUBSYSTEM_ENTITY_SW_VAR_LEN; i > 0 ; i--)
		{
			mResults.push_back(static_cast<tU8>(((subsystem.subsystemSWVariant) >> (8 * (i-1)))));
		}

		DIA_TR_INF("subsystemIPAddress is 0x%08X",iter->subsystemIPAddress);
		for(int i = DIA_C_SUBSYSTEM_ENTITY_IP_ADDR_LEN; i > 0 ; i--)
		{
			mResults.push_back(static_cast<tU8>(((subsystem.subsystemIPAddress) >> (8 * (i-1)))));
		}

		DIA_TR_INF("subsystemEntityID is 0x%X%X%X%X%X%X", subsystem.subsystemEntityID[0],
													  subsystem.subsystemEntityID[1],
													  subsystem.subsystemEntityID[2],
													  subsystem.subsystemEntityID[3],
													  subsystem.subsystemEntityID[4],
													  subsystem.subsystemEntityID[5]);
													  
		for(int i = 0; i< DIA_C_SUBSYSTEM_ENTITY_ID_LEN; ++i)
		{
			mResults.push_back(subsystem.subsystemEntityID[i]);
		}
	}

	for(int i = size; i < DIA_C_SUBSYSTEM_IDENTIFY_ENTITIES_RESP_MAX_NUM_SUBSYSTEMS; i++)
	{
		for(int j = 0; j<DIA_C_SUBSYSTEM_IDENTIFY_ENTITIES_RESP_SIZE_PER_SUBSYTEM; j++)
		{
			mResults.push_back(DIA_C_SUBSYSTEM_IDENTIFY_ENTITIES_RESP_DEFAULT_DATA);
		}
	}
}

//-----------------------------------------------------------------------

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlIdentifySubSystem::registerSessionChange ( void )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::registerSessionChange");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->addListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! dia_RoutineCtrlIdentifySubSystem::registerSessionChange => ERROR: Unable to register for Session changes");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

void
dia_RoutineCtrlIdentifySubSystem::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::vOnSessionChanged()");

	// reset the flag just in case
	_started = false;
   
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlIdentifySubSystem::unregisterSessionChange ( void )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlIdentifySubSystem::unregisterSessionChange");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->removeListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! dia_RoutineCtrlIdentifySubSystem::unregisterSessionChange => ERROR: Unable to deregister for Session changes");
   }

   return retVal;
}

