/*!
 * \file       dia_RoutineCtrlCheckReprogrammingDtc.cpp
 *
 * \brief      Trigger corresponding test for a specific DTC, update DTC status and get it back.
 *
 * \details    This routine is implemented for all DTCs concerning logical blocks from 00 to XX
 *             (Y logical blocks x 3 failure types = 3Y DTCs).
 *             Test of logical blocks which belongs to Audio component are triggered by DiagLog process:
 *             Step 1. Register for active DTC.
 *             Step 2. Clear DTC (group: reprogramming DTC, only specific DTC).
 *             ..... wait for reply from DiagLog
 *             Step 4. Callback vOnDtcStatusChanged
 *
 * \component  Diagnosis
 *
 * \ingroup    Routine Control for the customer
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_UDS_RTCTRL__
#include "common/framework/protocols/uds/rtctrl/dia_common_uds_rtctrl.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_EXT_DIAGLOG_IF__
#include "project/framework/application/ext_diaglog_if.h"      //label for ITC
#endif

#ifndef __INCLUDED_DIA_INTERFACE_ERRORLOG__
#include "common/interfaces/dia_IErrorLog.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#include "dia_RoutineCtrlCheckReprogrammingDtc.h"

static const tU8  RSRRoutineInfo = 0x10;     /* defined by the customer*/
static const tU32 PARAMETER_LENGTH = 3;      /* 2 bytes of logical block + 1 byte DTC Failure Type */

static const tU8  ROUTINE_STATUS_SUCCESS = 0x00;
static const tU8  ROUTINE_STATUS_FAILURE = 0x01;

static tU8 sResultMapping[DIA_EN_RTCTRL_STATUS_COUNT] = {
      0xFF, // DIA_EN_RTCTRL_STATUS_UNKNOWN
      0xFF, // DIA_EN_RTCTRL_STATUS_IDLE
      0xFF, // DIA_EN_RTCTRL_STATUS_IN_PROGRESS
      0x00, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK
      0x01, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK
      0xFF, // DIA_EN_RTCTRL_STATUS_ABORTED
      0xFF  // DIA_EN_RTCTRL_STATUS_TIMED_OUT
};

enum dia_ITC_value
{
   DIA_ITC_PASSED = 0,     /* =0 */
   DIA_ITC_FAILED          /* =1 */
};

#define DTC_STATUS_CHANGE_TIMER  2000
#define DTC_STATUS_CHANGE_MAX    18

/*static*/ dia_IErrorLogNotification* dia_RoutineCtrlCheckReprogrammingDtc::pErrorLogNotification = NULL;

//-----------------------------------------------------------------------------

dia_RoutineCtrlCheckReprogrammingDtc::dia_RoutineCtrlCheckReprogrammingDtc ( tCString name, tU16 udsID,
      const std::map<dia_e_logical_block,std::pair <tU32,tU32> >& mapPairDataDigest )
   : dia_Routine(name, udsID,
         DIA_EN_RTCTRL_TYPE_SHORT_TERM),
         mAreHashesMatching(false),
         mLastDTCNumber(0),
         mLastDTCStatus(0xFF),
         mMapPairDataDigests(mapPairDataDigest),
         mTimerCount(0)
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::dia_RoutineCtrlCheckReprogrammingDtc()");
#if 0
   mTimer.s32Create();
#endif
}

//-----------------------------------------------------------------------------

dia_RoutineCtrlCheckReprogrammingDtc::~dia_RoutineCtrlCheckReprogrammingDtc ( void )
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
#if 0
      dia_IErrorLog* pInterface = 0;
      if ((querySysAdapterInterface<dia_IErrorLog>(&pInterface) == DIA_SUCCESS) && pInterface)
      {
         (void)pInterface->unregisterForActiveDtc();
      }
#endif

#if 0
      mTimer.s32Delete();
      mTimer.removeTimerListener(this);
#endif
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: dia_RoutineCtrlCheckReprogrammingDtc::~dia_RoutineCtrlCheckReprogrammingDtc !!!");
       NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlCheckReprogrammingDtc::clearDTC(tU32 dtc)
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::clearDTC()");

   DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::clearDTC DTC=0x%06X.", dtc);

   tBool errorDetected = TRUE;
   dia_IErrorLog* pInterface = 0;
   if ((querySysAdapterInterface<dia_IErrorLog>(&pInterface) == DIA_SUCCESS) && pInterface)
   {
      DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::clearDTC interface got.");
      (void) setSysAdapterListener<dia_IErrorLogListener>(this);

      if (pInterface->clearDTC(DIAGLOG_MEMORY_REPROGRAMMING, dtc) == DIA_SUCCESS)
      {
         DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::clearDTC clearDTC passed.");
         errorDetected = FALSE;
      }
      else
      {
         DIA_TR_ERR("### clearDTC FAILED!!!! ###");
      }
   }

   if ( errorDetected )
   {
      DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::clearDTC unset system adapter listener.");
      (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);

      eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
      mIsResultReady = TRUE;
   }

   DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::clearDTC returned %s.", (errorDetected ? "DIA_FAILED": "DIA_SUCCESS"));

   return (errorDetected ? DIA_FAILED: DIA_SUCCESS);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlCheckReprogrammingDtc::setITC(tU16 itc, bool IsDtcPassed)
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::setITC()");
   tU8 message[4] = {0};
   tDiaResult retVal = DIA_FAILED;

   DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::setITC - result %s", ( IsDtcPassed? "DIA_ITC_PASSED": "DIA_ITC_FAILED"));

   // currently only 2 Bytes ITC supported
   message[0] = 3; // the first byte of the message is length of data container
   message[1] = (tU8)(itc >> 8);
   message[2] = (tU8)(itc & 0xFF);
   message[3] = IsDtcPassed ? DIA_ITC_PASSED: DIA_ITC_FAILED;

   dia_IErrorLog* pErrorLog = 0;
   tDiaResult retValQuery;
   if ((DIA_SUCCESS==(retValQuery=querySysAdapterInterface<dia_IErrorLog>(&pErrorLog))) && pErrorLog)
   {
      (void) setSysAdapterListener<dia_IErrorLogListener>(this);
      if ( pErrorLog->u32TestDTCStart(&message[0]) != DIA_SUCCESS )
      {
         DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::setITC UNABLE TO SET ITC label=0x%04X", itc);
      }
      else
      {
         DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::setITC itc=0x%04X u32TestDTCStart successful", itc);
         retVal = DIA_SUCCESS;
      }
   }
   else
   {
      DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::setITC - UNABLE TO SEND TEST RESULT!!!");
      DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::setITC - ERROR pErrorLog=0x%p", pErrorLog);
      DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::setITC - ERROR retValQuery=0x%08X", retValQuery);
   }

   if (DIA_SUCCESS!=retVal)
   {
      DIA_TR_ERR("### dia_RoutineCtrlCheckReprogrammingDtc::setITC unset system adapter listener.");
      (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
      eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
      mIsResultReady = TRUE;
   }

   return retVal;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlCheckReprogrammingDtc::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::start()");
   tDiaResult retCode = DIA_FAILED;
   tU32 logBlock = 0;
   tU32 dataID = 0;
   tU32 digestID = 0;

   DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::start - params.size()=%d", params.size());

   mAreHashesMatching = false;

   for (tU8 i=0; i<params.size(); i++)
   {
      DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::start - params[%d]=0x%02X", i, params[i]);
   }

   if (PARAMETER_LENGTH!=params.size())
   {
      DIA_TR_ERR("### params.size()=%d is different than expected %d bytes. ###", params.size(), PARAMETER_LENGTH);
      return DIA_FAILED;   //NRC 0x22
   }

   if ((FTB_PROGRAMMING_FAILURE!=params[2]) && (FTB_INTERNAL_FAILURE!=params[2]) && (FTB_ALGORITHM_FAILURE!=params[2]))
   {
      DIA_TR_ERR("### params[2]=0x%02X is different than expected 0x%02X or 0x%02X or 0x%02X. ###", params[2], FTB_PROGRAMMING_FAILURE, FTB_INTERNAL_FAILURE, FTB_ALGORITHM_FAILURE);
      return DIA_E_OUT_OF_RANGE;   //NRC 0x31
   }

   logBlock = (params[0]<<8) + params[1];
   enum dia_e_logical_block blockID = (dia_e_logical_block)((logBlock) & 0xFFFF);

   std::map<dia_e_logical_block,std::pair <tU32,tU32> > ::const_iterator it = mMapPairDataDigests.find(blockID);
   if ( it != mMapPairDataDigests.end() )
   {
      dataID   = it->second.first;
      digestID = it->second.second;
   }
   else
   {
      DIA_TR_ERR("### blockID = 0x%08X not found in the repo. mMapPairDataDigests.size()=%d ###", blockID, mMapPairDataDigests.size());

      std::map<dia_e_logical_block,std::pair <tU32,tU32> > ::const_iterator i = mMapPairDataDigests.begin();
      for (; i!=mMapPairDataDigests.end(); ++i)
      {
         DIA_TR_INF("mMapPairDataDigests: BlockID=0x%04X dataID=0x%08X digestID=0x%08X", i->first, i->second.first, i->second.second);
      }

      return DIA_E_OUT_OF_RANGE;      //NRC 0x31
   }

   // prepare processing of the routine
   vInitialize();

   tU32 dtc = (params[0]<<16) + (params[1]<<8) + params[2];

   mLastDTCNumber = dtc;
   mLastDTCStatus = 0xFF;

   if (!pErrorLogNotification)
   {
      if ((querySysAdapterInterface<dia_IErrorLogNotification>(&pErrorLogNotification) == DIA_SUCCESS) && pErrorLogNotification)
      {
         (void) setSysAdapterListener<dia_IErrorLogNotificationListener>(this);

         retCode = pErrorLogNotification->registerDtcStatusChange(DIAGLOG_MEMORY_REPROGRAMMING, dtc, 0);
         if (retCode != DIA_SUCCESS)
         {
            DIA_TR_ERR("### registerDtcStatusChange failed. ###");
         }
      }
      else
      {
         DIA_TR_ERR("### dia_IErrorLogNotification failed. ###");
      }
   }
   else
   {
      (void) setSysAdapterListener<dia_IErrorLogNotificationListener>(this);
      retCode = DIA_SUCCESS;
   }

   if (retCode != DIA_SUCCESS)
   {
      return DIA_FAILED;      //NRC 0x22
   }

#if 0
   dia_IErrorLogNotification* pInterface = 0;
   if ((querySysAdapterInterface<dia_IErrorLogNotification>(&pInterface) == DIA_SUCCESS) && pInterface)
   {
      (void) setSysAdapterListener<dia_IErrorLogNotificationListener>(this);

      if (DIA_SUCCESS == pInterface->registerDtcStatusChange(DIAGLOG_MEMORY_REPROGRAMMING, /*dtc*/0xfdffff, 0))
      {
         mTimerCount = 0;
         if (DIA_SUCCESS==clearDTC(dtc))
         {
            retCode = DIA_SUCCESS;
            DIA_TR_INF("cleared DTC 0x%06X successfully", dtc);
         }
         else
         {
            DIA_TR_ERR("### Clear DTC 0x%06X failed. ###", dtc);
         }
      }
      else
      {
         DIA_TR_ERR("### registerDtcStatusChange failed. ###");
      }
   }
   else
   {
      DIA_TR_ERR("### dia_IErrorLogNotification failed. ###");
   }
#endif

   retCode = clearDTC(dtc);
   if (retCode == DIA_SUCCESS)
   {
      DIA_TR_INF("cleared DTC 0x%06X successfully", dtc);
   }
   else
   {
      DIA_TR_ERR("### Clear DTC 0x%06X failed. ###", dtc);
      return DIA_FAILED;      //NRC 0x22
   }

   //check digest and trigger ITC
   if (FTB_PROGRAMMING_FAILURE==params[2])
   {
      DIA_TR_INF("--- FTB_PROGRAMMING_FAILURE ---");
      tU16 itc = ITC_AIVI_CAL_PROGRAMMING_FAILURE_START |  (blockID & 0x00FF);

      do
      {
         std::vector<tU8> data;
         std::vector<tU8> calcHash;
         std::vector<tU8> digest;

         DIA_TR_INF("itc = 0x%04X", itc);

         data.clear();
         if ((dia_getProperty(dataID,data) != DIA_SUCCESS) || (!data.size()) )
         {
            DIA_TR_ERR("##### FAILED TO GET DATA TO VECTOR: dataID = 0x%08X #####", dataID);
            continue;
         }

         if (false==dia_nsUtilities::generateHash(data, calcHash))
         {
            DIA_TR_ERR("### dia_nsUtilities::generateHash FAILED dataID=0x%08X.", dataID);
            continue;
         }

         if ((dia_getProperty(digestID,digest) != DIA_SUCCESS) || (!digest.size()) )
         {
            DIA_TR_ERR("##### FAILED TO GET DATA TO VECTOR: digestID = 0x%08X #####", digestID);
            continue;
         }

         if ((DIA_CONFIG_HASH_BINARY_SIZE!=digest.size()) || (DIA_CONFIG_HASH_BINARY_SIZE!=calcHash.size()))
         {
            DIA_TR_ERR("##### digest.size()=%d  calcHash.size()=%d #####", digest.size(), calcHash.size());
            continue;
         }

         if (calcHash==digest)
         {
            DIA_TR_INF("SHA-1 of data of dataID 0x%08X and digestID 0x%08X do match", dataID, digestID);

            mAreHashesMatching = true;
         }
         else
         {
            DIA_TR_ERR("##### SHA-1 of data of dataID 0x%08X and digestID 0x%08X don't match #####", dataID, digestID);

            if (DIA_CONFIG_HASH_BINARY_SIZE==digest.size())
            {
               tU8* p = &digest[0];
               DIA_TR_ERR("digest hex: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
                     p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15], p[16], p[17], p[18], p[19]);
            }

            if (DIA_CONFIG_HASH_BINARY_SIZE==calcHash.size())
            {
               tU8* p = &calcHash[0];
               DIA_TR_ERR("calcHash hex: %02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X",
                     p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9], p[10], p[11], p[12], p[13], p[14], p[15], p[16], p[17], p[18], p[19]);
            }
         }
      } while(0);

      if (DIA_SUCCESS==setITC(itc, mAreHashesMatching))
      {
         eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
         retCode = DIA_SUCCESS;

         //waiting for callback from DiagLog (mIsResultReady is set to FALSE now)
      }
      else
      {
         DIA_TR_ERR("### setITC failed for FTB_PROGRAMMING_FAILURE ###");
      }
   }
   else if (FTB_INTERNAL_FAILURE==params[2])
   {
      DIA_TR_INF("--- FTB_INTERNAL_FAILURE ---");
      std::vector<tU8> data;

      tU16 itc = ITC_AIVI_CAL_SYSTEM_INT_FAILURE_START |  (blockID & 0x00FF);

      //test by attempting to read the data block
      tDiaResult getPropRetCode =  dia_getProperty(dataID, data);
      if (DIA_SUCCESS != getPropRetCode)
      {
         if (DIA_E_NOT_AVAILABLE==getPropRetCode)  DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::start: item 0x%08X NOT_AVAILABLE in DP.", dataID);
         else                                      DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::start: item 0x%08X, retCode 0x%08X FAILED !!", dataID, getPropRetCode);
      }
      else
      {
         DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::start: dia_getProperty(0x%08X) returned DIA_SUCCESS.", dataID);
      }

      //last byte in the response
      mAreHashesMatching = ((DIA_E_NOT_AVAILABLE==getPropRetCode) || ((DIA_SUCCESS==getPropRetCode) && (data.size()>0))) ? true : false;

      DIA_TR_INF("data.size() = %d", data.size());
      DIA_TR_INF("isDtcPassed is %s.", (mAreHashesMatching) ? "TRUE" : "FALSE");

      if (DIA_SUCCESS==setITC(itc, mAreHashesMatching))
      {
         retCode = DIA_SUCCESS;
         eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);

         //waiting for callback from DiagLog (mIsResultReady is set to FALSE now)
      }
      else
      {
         DIA_TR_ERR("### setITC failed for FTB_INTERNAL_FAILURE ###");
      }
   }
   else if (FTB_ALGORITHM_FAILURE==params[2])
   {
      DIA_TR_INF("--- FTB_ALGORITHM_FAILURE ---");

      //Set DTC to PASSED for block 00.
      if (DIA_LOGICAL_BLOCK_00==logBlock)
      {
         DIA_TR_INF("--- DIA_LOGICAL_BLOCK_00 ---");

         tU16 itc = ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_START |  (blockID & 0x00FF);

         //Set always PASSED.
         mAreHashesMatching = true;
         if (DIA_SUCCESS==setITC(itc, true))
         {
            retCode = DIA_SUCCESS;
            eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);

            //waiting for callback from DiagLog (mIsResultReady is set to FALSE now)
         }
         else
         {
            DIA_TR_ERR("### setITC failed for FTB_INTERNAL_FAILURE ###");
         }
      }
      else
      {
         DIA_TR_INF("--- It is audio block (block id 0x%04X) ---", logBlock);
         //Trigger corresponding test by Audio using SendNextTestResult (registerForActiveDtc and Clear DTC)
      }
   }
   else
   {
      DIA_TR_ERR("### NEVER SHOULD BE REACHED ###");
      retCode = DIA_FAILED;      //NRC 0x22 //Coverity Fix :CID-139773)
   }

   //DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::start returned %s", (retCode==DIA_SUCCESS ? "DIA_SUCCESS": "DIA_FAILED"));
   DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::start returned 0x%08X", retCode);

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlCheckReprogrammingDtc::requestResult ( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::requestResult()");
   tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

   results.clear();
   if ( !(mResults.empty()) )
   {
      DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); iter++ )
      {
         results.push_back(*iter);
      }
      mResults.clear();
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::requestResult --- 2");
      switch ( mStatus )
      {
         case DIA_EN_RTCTRL_STATUS_IN_PROGRESS:
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK:
         {
            //DiagLog error or another failure
            if (DIA_EN_RTCTRL_STATUS_IN_PROGRESS==mStatus)        DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::requestResult DIA_EN_RTCTRL_STATUS_IN_PROGRESS");
            if (DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK==mStatus)  DIA_TR_ERR("dia_RoutineCtrlCheckReprogrammingDtc::requestResult DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK");
            mIsResultReady = TRUE;
            retCode = DIA_FAILED;      //NRC 0x22
         }
         break;

         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK:
         {
            DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::requestResult --- 3 (%d)", mStatus);
            // collect results
            mResults.push_back(RSRRoutineInfo);
            results.push_back(sResultMapping[mStatus]);
            mIsResultReady = TRUE;
            retCode = DIA_SUCCESS;
         }
         break;

         default:
         {
            DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::requestResult --- 4");
            DIA_TR_INF("dia_RoutineCtrlCheckReprogrammingDtc::requestResult mStatus = %d.", mStatus);
         }
         break;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnServiceTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnServiceTimeout()");

#if 0
   mTimer.s32SetTime(0,0);
   mTimer.removeTimerListener(this);
#endif
   (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
   (void) unsetSysAdapterListener<dia_IErrorLogNotificationListener>(this);

#if 0
   dia_IErrorLogNotification* pInterface = 0;
   if ((querySysAdapterInterface<dia_IErrorLogNotification>(&pInterface) == DIA_SUCCESS) && pInterface)
   {
      if (DIA_SUCCESS != pInterface->unregisterDtcStatusChange(DIAGLOG_MEMORY_REPROGRAMMING, mLastDTCNumber, 0))
      {
         DIA_TR_ERR("### unregisterForActiveDtc FAILED ####");
      }
   }
   else
   {
      DIA_TR_ERR("### CANNOT GET ERROR LOG NOTIFICATION INTERFACE ####");
   }
#endif

   eSetStatus(DIA_EN_RTCTRL_STATUS_TIMED_OUT);
   mIsResultReady = TRUE;
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnDiaglogWriteTestResult(void)
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnDiaglogWriteTestResult()");

#if 0
   (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);

   eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);

   //2 bytes output
   mResults.push_back(RSRRoutineInfo);
   mResults.push_back(mAreHashesMatching ? ROUTINE_STATUS_SUCCESS: ROUTINE_STATUS_FAILURE);

   mIsResultReady = TRUE;
   dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
#endif
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnDiaglogWriteTestError(const tU32 errCode )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnDiaglogWriteTestError()");

#if 0
   (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
   DIA_TR_ERR( "dia_RoutineCtrlCheckReprogrammingDtc::vProcessRequest => ERROR: 0x%08X", errCode);

   eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);

   mIsResultReady = TRUE;
   dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
#endif
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnClearResult(void)
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnClearResult()");

   (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnClearError(const tU32 errCode )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnClearError()");

   (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
   DIA_TR_ERR( "dia_RoutineCtrlCheckReprogrammingDtc::vProcessRequest => ERROR: 0x%08X", errCode);

   eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);

   mIsResultReady = TRUE;
   dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnDtcStatusChanged ( const tU32 DTCNumber, const tU8 DTCStatus )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnDtcStatusChanged()");

   DIA_TR_INF("DTC 0x%06X changed and its status is 0x%02X.", DTCNumber, DTCStatus);

   tDiaResult retVal = DIA_FAILED;

   if (mLastDTCNumber == DTCNumber)
   {
      // Wait for status transition:
      if ((mLastDTCStatus == 0x10) // from TestNotRunSinceDTCCleared
            &&                     // to
            ((DTCStatus == 0x00)   //Passed
            ||                     //or
            (DTCStatus == 0x01)))  //Failed
      {
#if 0
         mTimer.s32SetTime(0,0);
         mTimer.removeTimerListener(this);
#endif
         (void) unsetSysAdapterListener<dia_IErrorLogListener>(this);
         (void) unsetSysAdapterListener<dia_IErrorLogNotificationListener>(this);

#if 0
         dia_IErrorLogNotification* pInterface = 0;
         if ((querySysAdapterInterface<dia_IErrorLogNotification>(&pInterface) == DIA_SUCCESS) && pInterface)
         {
            if (DIA_SUCCESS == pInterface->unregisterDtcStatusChange(DIAGLOG_MEMORY_REPROGRAMMING, DTCNumber, 0))
            {
               retVal = DIA_SUCCESS;
            }
         }

         if (retVal != DIA_SUCCESS)
         {
            DIA_TR_ERR("### unregisterDtcStatusChange FAILED ####");
         }
#endif

         DIA_TR_INF( "dia_RoutineCtrlCheckReprogrammingDtc::vOnDtcStatusChanged COMPLETED AND OK");

         eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);

         //2 bytes output
         mResults.push_back(RSRRoutineInfo);
         mResults.push_back(DTCStatus);
         mIsResultReady = TRUE;
         dia_RoutineCtrlManager::getInstance()->vOnRoutineUpdate(*this);
      }
      else
      {
         mLastDTCStatus = DTCStatus;
#if 0
         DIA_TR_INF("Starting timer for DTC Status Change (mTimerCount = %d)", mTimerCount);
         mTimer.addTimerListener(this);
         mTimer.s32SetTime(DTC_STATUS_CHANGE_TIMER, 0);
#endif
      }
   }
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlCheckReprogrammingDtc::vOnTimerElapsed ( dia_TimerID id )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlCheckReprogrammingDtc::vOnTimerElapsed()");

   mTimerCount++;
   DIA_TR_INF("Timer elapsed %d. time", mTimerCount);
   if (mTimerCount < DTC_STATUS_CHANGE_MAX)
   {
      if (DIA_SUCCESS==clearDTC(mLastDTCNumber))
      {
         DIA_TR_INF("cleared DTC 0x%06X successfully", mLastDTCNumber);
      }
      else
      {
         DIA_TR_ERR("### Clear DTC 0x%06X failed. ###", mLastDTCNumber);
      }
   }
   else
   {
      DIA_TR_ERR("Unable to get DTC Status change for 0x%06X (status is 0x%02X).", mLastDTCNumber, mLastDTCStatus);
      DIA_ASSERT_ALWAYS();
   }
}
