/*
 * dia_ReadConfigItem.cpp
 *
 *  Created on: Dec 20, 2016
 *      Author: hld5kor
 *
 *TTFis:\> DIA_REQ CMD 0 F 0061 3 01
 *TTFis:\> DIA_REQ CMD 0 F 0061 1 01
 *TTFis:\> DIA_REQ CMD 0 F 0061 4 01
 *
 * -----------------------------------------------------------------------------------
 * Date          | Author          | Modification                               |
 * 06.09.16          | hld5kor            | AIVI-26969                                 |
 * ------------------------------------------------------------------------------------
 */

#ifndef __DIA_READCONFIGITEM_H___
#include "dia_ReadConfigItem.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_CONFIG__
#include "common/framework/config/dia_common_config.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define DATA_ZERO               0x00
#define DATA_ONE                    0x01
#define DATA_THREE                  0x03


//----------------------------------------------------------------------------------------

dia_ReadConfigItem::dia_ReadConfigItem( void )
     :dia_PropertyBag(DIA_C_STR_PROPBAG_CONFIGITEM),dataLength(0),
      dia_ConfigNotifier("dia_ReadConfigItem")
{
   dia_tclFnctTrace oTrace("dia_ReadConfigItem::dia_ReadConfigItem");
}

//----------------------------------------------------------------------------------------

dia_ReadConfigItem::~dia_ReadConfigItem( void )
{
}

//----------------------------------------------------------------------------------------
/**
  * \brief       Get ConfigItem Entry data and length
  *            Function retrieves the data based on Property ID.
  *
  * \param[in]   tU32     - Property ID, Pointer to the address of Data, Pointer to the Data length
  */
tDiaResult
dia_ReadConfigItem::getConfigItem ( tU32 ConfigItem, tU8** ConfItemData, tU16* ConfigItemDatalength )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::getConfigItem - Checks for data and length for the requested ConfigItem");

   tDiaResult retCode;
   tU8 *lValue;
   loadPropItem();
   tU32 pCode = getPropItem(ConfigItem);
   if( pCode != 0 )
   {
      retCode = getPropertyDetails ( pCode );
      if( retCode == DIA_SUCCESS )
      {
         if( ConfigItem == ConfigItem_Region )
         {
            lValue = &(PropDataVec[0]);
            //Reading only last 5 Bits from 1st Byte
            lValue[0] =  lValue[0] & 0x1F;
            *ConfItemData = (tU8*)(&lValue[0]);
            *ConfigItemDatalength = DATA_ONE ;
         }
         else if( ConfigItem == ConfigItem_SP_Lock_Config_Data )
         {
            lValue = &(PropDataVec[6]);
            //Reading only 4th Bit from 6th Byte
            lValue[0] = lValue[0] & 0x10;
            *ConfItemData = (tU8*)(&lValue[0]);
            *ConfigItemDatalength = DATA_THREE ;
         }
         else if( ConfigItem == ConfigItem_DeviceVariant )
         {
            lValue = &(PropDataVec[0]);
            //Reading only 1st 4 Bits from 1st Byte
            lValue[0] =  lValue[0] & 0xF0;
            *ConfItemData = (tU8*)(&lValue[0]);
            *ConfigItemDatalength = DATA_ONE ;
         }
         else if (ConfigItem == ConfigItem_AM_FM_Tuner_Feature_Control)
         {
            if (PropDataVec.size() > 43)
            {
               lValue = PropDataVec.data();
               *ConfItemData = &lValue[42];
               *ConfigItemDatalength = 2;
            }
            else
            {
               retCode = DIA_E_NOT_AVAILABLE;
            }
         }
         else if (ConfigItem == ConfigItem_DID_F18C)
         {
            retCode = DIA_E_TEMPORARY_NOT_AVAILABLE;
         }
         else
         {
            *ConfigItemDatalength = dataLength;
            *ConfItemData = (tU8 *)(&PropDataVec[0]);
         }
      }
      else
      {
         *ConfigItemDatalength = DATA_ZERO;
         *ConfItemData = NULL;
      }
   }
   else
   {
      DIA_TR_INF("dia_ReadConfigItem::getConfigItem - Invalid Config Item Entry");
      retCode = DIA_E_NOT_IMPLEMENTED;
   }

   return retCode;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::getConfigItem ( tU32 ConfigItem, dia_IConfigItemListener* pListener, void* cookie )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::getConfigItem - Starts an asynchronous read of the requested ConfigItem");

   tDiaResult retCode = DIA_E_ALREADY_STARTED;

   tU32 propID = getPropItem(ConfigItem);

   if (mConfigItemRead.find(ConfigItem) == mConfigItemRead.end())
   {
      Async act = {pListener, cookie};
      mConfigItemRead[ConfigItem] = act;

      if (ConfigItem == ConfigItem_DID_F18C)
      {
         DIA_TR_INF("Special handling for DID $F18C (ECUSerialNumber)");

         dia_IAIVISecurity* pInterface = OSAL_NULL;
         if ( (querySysAdapterInterface<dia_IAIVISecurity>(&pInterface) == DIA_SUCCESS) && pInterface )
         {
            (void) setSysAdapterListener<dia_IAIVISecurityListener>(this);

            retCode = pInterface->getEcuSerialNumber();
         }
      }
      else
      {
         retCode = dia_getProperty( propID, this );
      }
   }
   else
   {
      DIA_TR_ERR("Asynchronous read for ConfigItem=%d already started by pListener=0x%p/cookie=0x%p",
            ConfigItem, pListener, cookie);
   }

   if (retCode != DIA_SUCCESS)
   {
      getCallback(propID, retCode, NULL, 0);
   }

   return retCode;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::setConfigItem ( tU32 ConfigItem, const std::vector<tU8>& ConfItemData )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::setConfigItem - Starts a synchronous write of the requested ConfigItem");

   loadPropItem();

   tU32 propID = getPropItem(ConfigItem);
   tDiaResult retCode = dia_setProperty(propID, ConfItemData);

   return retCode;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::setConfigItem ( tU32 ConfigItem, const std::vector<tU8>& ConfItemData, dia_IConfigItemListener* pListener, void* cookie )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::setConfigItem - Starts an asynchronous write of the requested ConfigItem");

   tDiaResult retCode = DIA_FAILED;

   tU32 propID = getPropItem(ConfigItem);

   if (mConfigItemWrite.find(ConfigItem) == mConfigItemWrite.end())
   {
      Async act = {pListener, cookie};
      mConfigItemWrite[ConfigItem] = act;

      if (ConfItemData.size() == sizeof(tU8))
      {
         retCode = getInstanceOfConfigManager()->setPropertyU8(propID, ConfItemData[0], this);
      }
      else
      {
         DIA_TR_ERR("Unsupported data length (%d) in asynchronous write of ConfigItem %d", ConfItemData.size(), ConfigItem);
         for (unsigned i = 0; i < ConfItemData.size(); i++)
         {
            DIA_TR_ERR("ConfItemData[%d] = 0x%x", i, ConfItemData[i]);
         }
         retCode = DIA_E_NOT_SUPPORTED;
      }

      if (retCode != DIA_SUCCESS)
      {
         setCallbackU8(propID, retCode);
      }
   }
   else
   {
      DIA_TR_ERR("Asynchronous write for ConfigItem=%d already started by pListener=0x%p/cookie=0x%p",
            ConfigItem, pListener, cookie);

      retCode = DIA_E_ALREADY_STARTED;

      if (pListener)
      {
         pListener->vOnConfigItemSetComplete(ConfigItem, retCode, cookie);
      }
   }

   return retCode;
}

//---------------------------------------------------------------------------------------
/**
  * \brief       Get Property Details
  *            Function retrieves the data based on Property ID.
  *
  * \param[in]   tU32     - Property ID
  */
tDiaResult
dia_ReadConfigItem::getPropertyDetails ( tU32 PropID )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::getPropertyDetails - Retrieves the data based on Property ID");

   // retrieve size of property
   tU16 Proplength = dia_getPropertySize(PropID); //DIA_PROP_LENGTH_MAX;
   PropDataVec.reserve(Proplength);
   PropDataVec.resize(Proplength);
   tDiaResult retCode = dia_getProperty(PropID,PropDataVec);
   if ( retCode == DIA_SUCCESS )
   {
      DIA_TR_INF("dia_ReadConfigItem::getPropertyDetails checks for data successfully for Property");
      dataLength = (tU16) PropDataVec.size();
   }
   else
   {
      DIA_TR_INF("dia_ReadConfigItem::getPropertyDetails Get property details failed for Property");
      dataLength = 0;
   }
   return retCode;
}

//---------------------------------------------------------------------------------------
tVoid
dia_ReadConfigItem::loadPropItem(void)
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::loadPropItem");

   if (mConfigItemProp.empty())
   {
      mConfigItemProp[ConfigItem_VehicleModel]                 = DIA_PROP_AIVI_VEHICLE_INFO;
      mConfigItemProp[ConfigItem_NavUnitID]                    = DIA_PROP_CM_NAVI_UNIT;
      mConfigItemProp[ConfigItem_DeviceVariant]                = DIA_PROP_AIVI_VARIANT_CODING;
      mConfigItemProp[ConfigItem_OEMType]                      = DIA_PROP_CM_CONFIG_OEM_TYPE;
      mConfigItemProp[ConfigItem_SWVersion]                    = DIA_PROP_AIVI_ECU_SW_VERSION_NUMBER;
      mConfigItemProp[ConfigItem_Region]                       = DIA_PROP_AIVI_VEHICLE_INFO;
      mConfigItemProp[ConfigItem_VIN]                          = DIA_PROP_AIVI_VIN;
      mConfigItemProp[ConfigItem_SP_Lock_Config_Data]          = DIA_PROP_AIVI_SYSTEM_INFO_COMMON;
      mConfigItemProp[ConfigItem_SP_Lock_Restriction]          = DIA_PROP_AIVI_SPEED_LOCKOUT;
      mConfigItemProp[ConfigItem_HD_Tuner]                     = DIA_PROP_CM_HD_PARAMETER_SET;
      mConfigItemProp[ConfigItem_DRM_Tuner]                    = DIA_PROP_CM_DRM_TUNER_PARAMETER_SET;
      mConfigItemProp[ConfigItem_DAB_Tuner]                    = DIA_PROP_AIVI_DAB_SET;
      mConfigItemProp[ConfigItem_AM_Tuner]                     = DIA_PROP_AIVI_FMAM_TUNER_PARAM_SET;
      mConfigItemProp[ConfigItem_AM_FM_Tuner_Feature_Control]  = DIA_PROP_AIVI_FMAM_TUNER_PARAM_SET;
      mConfigItemProp[ConfigItem_FactoryMode]                  = DIA_PROP_AIVI_FACTORY_MODE;
      mConfigItemProp[ConfigItem_DID_F18C]                     = DIA_PROP_AIVI_ECU_SERIAL_NUMBER;
      mConfigItemProp[ConfigItem_BCM_AutoLightTiming]          = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_AUTO_LIGHT_TIM;
     mConfigItemProp[ConfigItem_BCM_AutoLightDelayTimer]       = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_AUTO_LIGHT_DLY_TIM;
     mConfigItemProp[ConfigItem_BCM_LaneChange3Flash]          = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_LANE_CHANGES_3_FLASH;
     mConfigItemProp[ConfigItem_BCM_LEDHyperdayLight]          = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_LED_HYDERDAY_LIGHT;
     mConfigItemProp[ConfigItem_BCM_AnswerBackExteriorLight]   = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_ANSWER_BACK_EXTERIOR_LIGHT;
     mConfigItemProp[ConfigItem_BCM_RoomLight]                 = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_ROOM_LIGHT;
     mConfigItemProp[ConfigItem_BCM_WelcomeLight]              = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_WELCOME_LIGHT;
     mConfigItemProp[ConfigItem_BCM_AmbientBrightness]         = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_AMBIENT_BRIGHTNESS;
     mConfigItemProp[ConfigItem_BCM_SelectiveDoorUnlock]       = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_SELECTIVE_DOOR_UNLOCK;
     mConfigItemProp[ConfigItem_BCM_AutoLock]                  = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_AUTO_LOCK;
     mConfigItemProp[ConfigItem_BCM_AutoUnlock]                = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_AUTO_UNLOCK;
     mConfigItemProp[ConfigItem_BCM_AnswerBack]                = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_ANSWER_BACK;
     mConfigItemProp[ConfigItem_BCM_WiperWithSpeed]            = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_WIPER_WITH_SPEED;
     mConfigItemProp[ConfigItem_BCM_RainSensor]                = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_RAIN_SENSOR;
     mConfigItemProp[ConfigItem_BCM_DripWipe]                  = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_DRIP_WIPE;
     mConfigItemProp[ConfigItem_BCM_ReverseLink]               = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_REVERSE_LINK;
     mConfigItemProp[ConfigItem_BCM_LockUnlockByHandsFree]     = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_LOCK_UNLOCK_BY_HANDS_FREE;
     mConfigItemProp[ConfigItem_BCM_IGN_ACCBatterySaver]       = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_IGN_ACC_BATTERY_SAVER;
     mConfigItemProp[ConfigItem_BCM_RemoteEngineStart]         = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_REMOTE_ENGINE_START;
     mConfigItemProp[ConfigItem_BCM_TouchDoorSensor]           = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_TOUCH_DOOR_SENSOR;
     mConfigItemProp[ConfigItem_BCM_Reset]                     = DIA_PROP_AIVI_VEHICLE_SETTING_BCM_CUSTOM_RESET_IT;
     mConfigItemProp[ConfigItem_ALC_AmbientBrightness]         = DIA_PROP_AIVI_VEHICLE_SETTING_ALC_AMBIENT_BRIGHTNESS;
     mConfigItemProp[ConfigItem_ALC_FrontSeatArea]             = DIA_PROP_AIVI_VEHICLE_SETTING_ALC_FRONT_SEAT_AREA;
     mConfigItemProp[ConfigItem_ALC_RearSeatArea]              = DIA_PROP_AIVI_VEHICLE_SETTING_ALC_REAR_SEAT_AREA;
     mConfigItemProp[ConfigItem_ALC_RoofAmbientLight]          = DIA_PROP_AIVI_VEHICLE_SETTING_ALC_ROOF_AMBINENT_LIGHT;
     mConfigItemProp[ConfigItem_ALC_Reset]                     = DIA_PROP_AIVI_VEHICLE_SETTING_ALC_CUSTOM_RESET_IT;
     mConfigItemProp[ConfigItem_ADP_LiftColumnUponExit]        = DIA_PROP_AIVI_VEHICLE_SETTING_ADP_LIFT_COLUMN_UPON_EXIT;
     mConfigItemProp[ConfigItem_ADP_AdjustDriverSeatUponExit]  = DIA_PROP_AIVI_VEHICLE_SETTING_ADP_ADJUST_DRIVE_SEAT_UPON_EXIT;
     mConfigItemProp[ConfigItem_ADP_Reset]                     = DIA_PROP_AIVI_VEHICLE_SETTING_ADP_RESET_IT;
     mConfigItemProp[ConfigItem_BCM_Complete]                  = DIA_PROP_AIVI_VEHICLE_SETTING_BCM;
     mConfigItemProp[ConfigItem_ALC_Complete]                  = DIA_PROP_AIVI_VEHICLE_SETTING_ALC;
     mConfigItemProp[ConfigItem_ADP_Complete]                  = DIA_PROP_AIVI_VEHICLE_SETTING_ADP;

      std::map<tU32,tU32>::iterator item;
      for (item = mConfigItemProp.begin(); item != mConfigItemProp.end(); item++)
      {
         addProperty(item->second);
      }

      (void) getInstanceOfConfigManager()->addConfigNotifier(this);
   }
}

//---------------------------------------------------------------------------------------
tU32 dia_ReadConfigItem::getPropItem(tU32 ConfigItem)
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::getPropItem");
   tU32 rPropId;
   std::map<tU32,tU32>::iterator it;

   it = mConfigItemProp.find(ConfigItem);
   if(it != mConfigItemProp.end())
   {
      rPropId = it->second;
   }
   else
   {
      rPropId = mConfigItemProp[ConfigItem];
   }
   return rPropId;
}

//---------------------------------------------------------------------------------------
tDiaResult
dia_ReadConfigItem::addConfigItemListener( tU32 ConfigItem, dia_IConfigItemListener* pListener )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::addConfigItemListener");
   if( (!pListener) || (!ConfigItem) ) return DIA_FAILED;

   std::map<tU32,dia_IConfigItemListener*>::iterator iter = mConfigItemListenerRep.find(ConfigItem);

   if( iter != mConfigItemListenerRep.end() ) return DIA_E_ALREADY_EXISTS;

   mConfigItemListenerRep[ConfigItem] = pListener;
   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------
tDiaResult
dia_ReadConfigItem::removeConfigItemListener( tU32 ConfigItem, dia_IConfigItemListener* pListener )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::removeConfigItemListener");
   if( (!pListener) || (!ConfigItem) ) return DIA_FAILED;

   std::map<tU32,dia_IConfigItemListener*>::iterator iter = mConfigItemListenerRep.find(ConfigItem);

   if( iter != mConfigItemListenerRep.end() ) mConfigItemListenerRep.erase(ConfigItem);

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------
tDiaResult
dia_ReadConfigItem::open ( tVoid )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::open");
   return DIA_SUCCESS ;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::close ( tVoid )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::close");
    return DIA_SUCCESS ;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::onPropertyUpdate ( tU32 propID )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::onPropertyUpdate");

   std::map<tU32,tU32>::iterator item;
   for (item = mConfigItemProp.begin(); item != mConfigItemProp.end(); item++)
   {
      if (propID == item->second)
      {
         tU32 ConfigItem = item->first;

         DIA_TR_INF("Property 0x%x is mapped to ConfigItem %d", propID, ConfigItem);

         std::map<tU32,dia_IConfigItemListener*>::iterator listener;
         for (listener = mConfigItemListenerRep.begin(); listener != mConfigItemListenerRep.end(); listener++)
         {
            if ( (listener->first == ConfigItem) /*&& listener->second*/)
            {
               listener->second->vOnConfigItemUpdate(ConfigItem);
            }
         }
      }
   }

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::getCallback ( tU32 propID, tDiaResult getStatus, tU8* buff, tU32 buffLen )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::getCallback()");

   std::map<tU32,tU32>::iterator item;
   for (item = mConfigItemProp.begin(); item != mConfigItemProp.end(); item++)
   {
      if (propID == item->second)
      {
         tU32 ConfigItem = item->first;

         std::map<tU32,Async>::iterator listener = mConfigItemRead.find(ConfigItem);

         if (listener != mConfigItemRead.end())
         {
            dia_IConfigItemListener* pListener = listener->second.pListener;
            void* cookie = listener->second.cookie;
            mConfigItemRead.erase(listener);

            if (pListener)
            {
               std::vector<tU8> data;
               if (buff && buffLen)
               {
                  data.insert(data.end(), buff, buff + buffLen);
               }
               pListener->vOnConfigItemGetComplete(ConfigItem, getStatus, data, cookie);
            }

            break;
         }
      }
   }

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::setCallbackU8 ( tU32 propID, tDiaResult setStatus )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::setCallbackU8()");

   std::map<tU32,tU32>::iterator item;
   for (item = mConfigItemProp.begin(); item != mConfigItemProp.end(); item++)
   {
      if (propID == item->second)
      {
         tU32 ConfigItem = item->first;

         std::map<tU32,Async>::iterator listener = mConfigItemWrite.find(ConfigItem);

         if (listener != mConfigItemWrite.end())
         {
            dia_IConfigItemListener* pListener = listener->second.pListener;
            void* cookie = listener->second.cookie;
            mConfigItemWrite.erase(listener);

            if (pListener)
            {
               pListener->vOnConfigItemSetComplete(ConfigItem, setStatus, cookie);
            }

            break;
         }
      }
   }

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------

tDiaResult
dia_ReadConfigItem::setCallback ( tU32 propID, tDiaResult setStatus, tU8* buff, tU32 buffLen )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::setCallback()");

   std::map<tU32,tU32>::iterator item;
   for (item = mConfigItemProp.begin(); item != mConfigItemProp.end(); item++)
   {
      if (propID == item->second)
      {
         tU32 ConfigItem = item->first;

         std::map<tU32,Async>::iterator listener = mConfigItemWrite.find(ConfigItem);

         if (listener != mConfigItemWrite.end())
         {
            dia_IConfigItemListener* pListener = listener->second.pListener;
            void* cookie = listener->second.cookie;
            mConfigItemWrite.erase(listener);

            if (pListener)
            {
               std::vector<tU8> data;
               if (buff && buffLen)
               {
                  data.insert(data.end(), buff, buff + buffLen);
               }
               pListener->vOnConfigItemSetComplete(ConfigItem, setStatus, data, cookie);
            }

            break;
         }
      }
   }

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------

void
dia_ReadConfigItem::vOnEcuSerialNumber ( const std::vector<tU8>& serialno )
{
   dia_tclFnctTrace trc("dia_ReadConfigItem::vOnEcuSerialNumber");

   (void) unsetSysAdapterListener<dia_IAIVISecurityListener>(this);

   tDiaResult retCode = DIA_FAILED;

   if (serialno.size() == 20)
   {
      std::map<tU32,tU32>::iterator item = mConfigItemProp.find(ConfigItem_DID_F18C);
      if (item != mConfigItemProp.end())
      {
         tU32 propID = item->second;
         std::vector<tU8> pSerialNo = serialno;
         retCode = DIA_SUCCESS;
         getCallback(propID, retCode, pSerialNo.data(), serialno.size());
      }
   }
}



