/**
 * \file      dia_ProcInfo.h
 *
 * \brief     Utility class for reading different information from /proc file system
 *
 *
 *  Created on: Oct 26, 2015
 *      Author: jas1hi
 *
 * \copyright Robert Bosch Car Multimedia 2015
 /*********************************************************************************************
 * REVISION HISTORY:
 * ------------------------------------------------------------------------------------------------
 * Date 				| Rev. 				| Author 		   	| Modification
 * ------------------------------------------------------------------------------------------------

 * 19.01.2017		| 1.2	| NLK1KOR (RBEI)	| Implemented feature AIVI-35187  Performance indicator (Interrupt latency and frequency)
 Connection Status
 *--------------------------------------------------------------------------------------------------------------------

 * 09.02.2017		| 1.4	| NLK1KOR (RBEI)	| updated feature AIVI-35187  Performance indicator (Interrupt latency and frequency)

 *-------------------------------------------------------------------------------------------------------------------
 *******************************************************************************************************************************/


#include <sstream>
#ifndef DIA_PROCINFO_H_
#define DIA_PROCINFO_H_

//#include <map>

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef MAX_NAME
#define MAX_NAME (tU32) 64
#endif

#define MAX_INTERRUPT_PROCESS 5

struct MemInfo {
	tULong mTotalMem;
	tULong mUsedMem;
	tULong mFreeMem;
};

struct CPUProcessMemory {
   tChar command[MAX_NAME+1];
   tULong m_maximumMemKBytes;
};

typedef struct Process_ {
//   Object super;

//   struct Settings_* settings;

//   unsigned long tLong processTime;
   pid_t pid;
//   pid_t ppid;
//   pid_t tgid;

//   tChar* comm;
//   tS32 indent;

//   tS32 basenameOffset;
//   tBool updated;

   std::string state;
//   tBool tag;
//   tBool showChildren;
//   tBool show;
//   tU32 pgrp;
//   tU32 session;
//   tU32 tty_nr;
 //  tS32 tpgid;
//   uid_t st_uid;
//   tU64 flags;
//   tS32 processor;

   tFloat percent_cpu;
//   tFloat percent_mem;
//   tChar* user;

//   tLong priority;
//   tLong nice;
//   tLong nlwp;
//   tChar starttime_show[8];
//   time_t starttime_ctime;

   tULong m_size;
   tULong m_resident;
   tULong m_peak;

//   tS32 exit_signal;

//   tULong minflt;
//   tULong majflt;
//   #ifdef DEBUG
//   tLong itrealvalue;
//   tU64 vsize;
//   tLong rss;
//   tU64 rlim;
//   tU64 startcode;
//   tU64 endcode;
//   tU64 startstack;
//   tU64 kstkesp;
//   tU64 kstkeip;
//   tU64 signal;
//   tU64 blocked;
//   tU64 sigignore;
//   tU64 sigcatch;
//   tU64 wchan;
//   tU64 nswap;
//   tU64 cnswap;
//   #endif

   tULong utime;
   tULong stime;
   tULong cutime;
   tULong cstime;
   Process_()
   :   pid(0),
       percent_cpu(0),
       m_size(0),
       m_resident(0),
       m_peak(0),

       utime(0),
       stime(0),
       cutime(0),
       cstime(0)
   {
	   state.clear();
   }

} Process;

typedef struct CPUData_ {
	tULong totalTime;
	tULong userTime;
	tULong systemTime;
	tULong systemAllTime;
	tULong idleAllTime;
	tULong idleTime;
	tULong niceTime;
	tULong ioWaitTime;
	tULong irqTime;
	tULong softIrqTime;
	tULong stealTime;
	tULong guestTime;

	tULong totalPeriod;
	tULong userPeriod;
	tULong systemPeriod;
	tULong systemAllPeriod;
	tULong idleAllPeriod;
	tULong idlePeriod;
	tULong nicePeriod;
	tULong ioWaitPeriod;
	tULong irqPeriod;
	tULong softIrqPeriod;
	tULong stealPeriod;
	tULong guestPeriod;

	CPUData_()
		: totalTime(0),
	 	  userTime(0),
		  systemTime(0),
		  systemAllTime(0),
		  idleAllTime(0),
		  idleTime(0),
		  niceTime(0),
		  ioWaitTime(0),
		  irqTime(0),
		  softIrqTime(0),
		  stealTime(0),
		  guestTime(0),

		  totalPeriod(0),
		  userPeriod(0),
		  systemPeriod(0),
		  systemAllPeriod(0),
		  idleAllPeriod(0),
		  idlePeriod(0),
		  nicePeriod(0),
		  ioWaitPeriod(0),
		  irqPeriod(0),
		  softIrqPeriod(0),
		  stealPeriod(0),
		  guestPeriod(0)
	{
	}

} CPUData;

struct DiskStats {
	tU32 totalBlocksRead;
	tU32 totalBlocksWritten;
};

typedef struct dia_ProcInterruptInfo_
{
	tULong mCPU0Frequency;
	tULong mCPU1Frequency;
	tChar mProcName[MAX_NAME];
	tChar mProcIRQNumber[MAX_NAME];
	dia_ProcInterruptInfo_():
	mCPU0Frequency(0),
	mCPU1Frequency(0),
	mProcName{},
	mProcIRQNumber{}
	{

	}

}dia_ProcInterruptInfo;

typedef struct dia_ProcInterruptSortElement_
{
	tULong mCPU0Frequencydiff;
	tULong mCPU1Frequencydiff;
	tChar mProcName[MAX_NAME];
	tChar mProcIRQNumber[MAX_NAME];
	dia_ProcInterruptSortElement_() :
		mCPU0Frequencydiff(0),
		mCPU1Frequencydiff(0),
		mProcName{},
		mProcIRQNumber{}
	{

	}

}dia_ProcInterruptSortElement;

class dia_ProcInfo {
public:
	dia_ProcInfo();
	virtual ~dia_ProcInfo();

	virtual tDiaResult scanMemInfo(tULong& usedMem);
	virtual tDiaResult updateProcessList(tDouble period);
	virtual tDiaResult scanCpuTime(tDouble& peroid); // updates all kind of total time for each CPU
	
	virtual tDiaResult scanDiskstats(tU8& eMMCState, tU8& sdaState);  // delivers the actual read/write state for eMMC and sda (USB)
	virtual std::multimap<tFloat, std::string> getTopProcessSortedViaCpuUsage(tU16 numberOfProcess);
	virtual std::map<tULong, struct CPUProcessMemory> getTopProcessSortedViaMemoryUsage(tU16 numberOfProcess);
	virtual tDiaResult getCPUIdleTime(tFloat& cpuIdleTimeInPercent);
	virtual tDiaResult getProcInterruptsInfo();
	virtual std::vector<std::string> getProcInterruptsInfoResult();

private:
	static tDouble sJiffy;
	struct MemInfo mMemInfo;
	tU32 mCpuCount;
	std::map<tS32, Process*> mProcessTable;
	typedef std::map<tS32, Process*>::iterator mProcessTableIteratorType;
	struct DiskStats mDiskStatseMMC;
	struct DiskStats mDiskStatsSDA;

	std::map<tU32, CPUData*> mCpuDataTable;
	typedef std::map<tU32, CPUData*>::iterator mCpuDataTableIteratorType;

	std::multimap<tFloat, std::string> mTopProcessSortedViaCpuUsage;
	std::multimap<tFloat, std::string>::reverse_iterator mTopProcessReverseIterator;

	std::map<tULong, struct CPUProcessMemory> mTopProcessSortedViaMemoryUsage;
	std::map<tULong, struct CPUProcessMemory>::reverse_iterator mTopProcessSortedViaMemoryUsageReverseIterator;

	tVoid init();
	tVoid clean();
	tULong adjustTimeForLinuxProcess(tULong t) const;
	Process* getProcess(tS32 pid);
	tBool readProcessStatFile(const tChar* pid, Process *process, tChar* command) const;
	tBool readProcessStatmFile(const tChar* pid, Process *process) const;
//	tBool readProcessStatusFile(const tChar* pid, Process *process);
	dia_ProcInterruptSortElement getdiffencesElement(dia_ProcInterruptInfo&, dia_ProcInterruptInfo&);
	std::vector< dia_ProcInterruptInfo> mProcInterruptPreviouslist;
	std::vector<std::string> mProcInterruptFinallist;

};

#endif /* DIA_PROCINFO_H_ */
