/*
 * dia_ProcInfo.cpp
 *
 *  Created on: Oct 26, 2015
 *      Author: jas1hi
 *********************************************************************************************
 * HISTORY:
 * ------------------------------------------------------------------------------------------------
 * Date 				| Rev. 				| Author 		   	| Modification
 *---------------------------------------------------------------------------------------------------
 * Dec 22, 2016                             | nlk1kor           | implemented 
                                                                  AIVINS2-19068 Extend the worker class 
																  for the CPU-Load to support the application status 
*---------------------------------------------------------------------------------------------------

* 19.01.2017		    | 	                | NLK1KOR (RBEI)	     | Implemented feature AIVI-35187  Performance indicator (Interrupt latency and frequency)
Connection Status
*--------------------------------------------------------------------------------------------------------------------
* 25.01.2017		    | 	                | NLK1KOR (RBEI)	     | updated feature AIVI-35187  Performance indicator (Interrupt latency and frequency)
                                                                       to handle multiple cores.
*--------------------------------------------------------------------------------------------------------------------

* 09.02.2017		| 1.4	| NLK1KOR (RBEI)	| updated feature AIVI-35187  Performance indicator (Interrupt latency and frequency)


                                                                      
*-------------------------------------------------------------------------------------------------------------------- */


#ifdef  __cplusplus
extern "C" {
#endif
#include <unistd.h>
//#include <stdlib.h>
//#include <string.h>
#include <dirent.h>
//#include <math.h>       /* isnan */
//#include <assert.h>     /* assert */
#include <errno.h>
#ifdef  __cplusplus
}
#endif


#include <fstream>

#ifndef DIA_HELPER_H_
#include "project/services/customer/MonitoringCalibration/helper/dia_Helper.h"
#endif

#include "dia_ProcInfo.h"



#ifndef PROCDIR
#define PROCDIR "/proc"
#endif

#ifndef SYSDIR
#define SYSDIR "/sys"
#endif

#ifndef PROCSTATFILE
#define PROCSTATFILE PROCDIR "/stat"
#endif

#ifndef PROCMEMINFOFILE
#define PROCMEMINFOFILE PROCDIR "/meminfo"
#endif

#ifndef PROCDISKSTATSFILE
#define PROCDISKSTATSFILE PROCDIR "/diskstats"
#endif

#ifndef SYSFSBLOCKMMCSTATSFILE
#define SYSFSBLOCKMMCSTATSFILE SYSDIR "/block/mmcblk1/stat"
#endif

#ifndef SYSFSBLOCKSDADIR
#define SYSFSBLOCKSDADIR SYSDIR "/block/sda"
#endif

#ifndef SYSFSBLOCKSDASTATSFILE
#define SYSFSBLOCKSDASTATSFILE SYSDIR "/block/sda/stat"
#endif

#ifndef PROCINTERRUPTSFILE
#define PROCINTERRUPTSFILE PROCDIR "/interrupts"
#endif

#ifndef MAX_READ
#define MAX_READ 2048
#endif

#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

#ifndef MAX
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif

#define ONE_K 1024L

#ifndef PAGE_SIZE
#define PAGE_SIZE ( sysconf(_SC_PAGESIZE) )
#endif

#define PAGE_SIZE_KB ( PAGE_SIZE / ONE_K )

#define NO_READ_WRITE_OPERATION (tU8) 0
#define INDEX_READ_OPERATION  	(tU8) 1
#define INDEX_WRITE_OPERATION	(tU8) 2


#define ONE_CORE                (tU8) 1
#define TWO_CORE 	            (tU8) 2
#define THREE_CORE	            (tU8) 3
#define FOUR_CORE	            (tU8) 4


tDouble dia_ProcInfo::sJiffy = 0.0;

struct MatchProcessIRQ
{
	MatchProcessIRQ(std::string irqno) :m_irqno(irqno)
	{}
	bool operator()(const dia_ProcInterruptInfo& value)
	{
		return (value.mProcIRQNumber == m_irqno);

	}
private:
	::std::string m_irqno;
};

dia_ProcInfo::dia_ProcInfo()
:	mMemInfo(),
	mCpuCount(0),
	mDiskStatseMMC(),
	mDiskStatsSDA()
{
	init();
}

dia_ProcInfo::~dia_ProcInfo()
{
    _BP_TRY_BEGIN
    {
    	clean();
    }
    _BP_CATCH_ALL
    {
        DIA_TR_ERR("EXCEPTION CAUGHT: dia_ProcInfo::~dia_ProcInfo !!!");
 //      NORMAL_M_ASSERT_ALWAYS();
    }
    _BP_CATCH_END
}

tVoid dia_ProcInfo::init()
{
	// update cpu count
	std::ifstream infile(PROCSTATFILE);
	DIA_ASSERT(infile.is_open());

	std::string procStatLineStr;
	tU32 cpus = 0;
	do {

	  std::getline(infile, procStatLineStr);

	  mCpuDataTable[cpus] = new CPUData(); // init with zeros
	  cpus++;
	} while (procStatLineStr.find("cpu") == 0);

	mCpuCount = (tU32)MAX(cpus-2, 1);

	DIA_TR_INF("CPU count: %u", mCpuCount);

	infile.close();
}

tVoid dia_ProcInfo::clean()
{
	DIA_TR_INF("dia_ProcInfo::clean: Cleaning up mProcessTable...");
	mProcessTableIteratorType iterProcessTable;
	for(iterProcessTable = mProcessTable.begin(); iterProcessTable != mProcessTable.end(); iterProcessTable++) {
	    delete iterProcessTable->second;
	}

	DIA_TR_INF("dia_ProcInfo::clean: Cleaning up mCpuDataTable...");
	mCpuDataTableIteratorType iterCpuDataTable;
	for(iterCpuDataTable = mCpuDataTable.begin(); iterCpuDataTable != mCpuDataTable.end(); iterCpuDataTable++) {
		delete iterCpuDataTable->second;
	}
}

tDiaResult dia_ProcInfo::scanMemInfo(tULong& usedMem)
{
	std::ifstream infile(PROCMEMINFOFILE);
	if (!infile.is_open()) {
		DIA_TR_ERR("dia_ProcInfo::scanMemInfo: Failed to open file %s", PROCMEMINFOFILE);
		return DIA_FAILED;
	}

	std::string memInfoLineStr;
	while (std::getline(infile, memInfoLineStr)) {
		switch (memInfoLineStr[0]) {
			case 'M':
				if (memInfoLineStr.find("MemTotal:") == 0)
					sscanf(memInfoLineStr.c_str(), "MemTotal: %lu kB", &mMemInfo.mTotalMem);
				else if (memInfoLineStr.find("MemFree:") == 0)
					sscanf(memInfoLineStr.c_str(), "MemFree: %lu kB", &mMemInfo.mFreeMem);
	//		 else if (String_startsWith(buffer, "MemShared:"))
	//			sscanf(buffer, "MemShared: %32llu kB", &this->sharedMem);
			 break;
	//		case 'B':
	//		 if (String_startsWith(buffer, "Buffers:"))
	//			sscanf(buffer, "Buffers: %32llu kB", &this->buffersMem);
	//		 break;
	//		case 'C':
	//		 if (String_startsWith(buffer, "Cached:"))
	//			sscanf(buffer, "Cached: %32llu kB", &this->cachedMem);
	//		 break;
	//		case 'S':
	//		 if (String_startsWith(buffer, "SwapTotal:"))
	//			sscanf(buffer, "SwapTotal: %32llu kB", &this->totalSwap);
	//		 if (String_startsWith(buffer, "SwapFree:"))
	//			sscanf(buffer, "SwapFree: %32llu kB", &swapFree);
	//		 break;
			default:
				break;
		}
   }

//	mMemInfo.mUsedMem = mMemInfo.mTotalMem - mMemInfo.mFreeMem;
	usedMem = mMemInfo.mTotalMem - mMemInfo.mFreeMem;

//	DIA_TR_INF("UsedMem: %lu TotalMem: %lu FreeMem: %lu", mMemInfo.mUsedMem, mMemInfo.mTotalMem, mMemInfo.mFreeMem);
	infile.close();

	return DIA_SUCCESS;
}

tULong dia_ProcInfo::adjustTimeForLinuxProcess(tULong t) const
{
   if(sJiffy == 0.0)
	   sJiffy = sysconf(_SC_CLK_TCK);

   tDouble jiffytime = 1.0 / sJiffy;

   return  (tULong)(t * jiffytime * 100);
}

tBool dia_ProcInfo::readProcessStatFile(const tChar* pid, Process *process, tChar* command) const
{
	// 987 (Xorg) S 946 987 987 1031 987 4202752 4815227 0 126 0 5740 21356 0 0 20 0 5 0 1921 365625344 39790 4294967295 1 1 0 0 0 0 0 4096 1367369423 4294967295 0 0 17 2 0 0 455 0 0

	tChar filename[MAX_NAME+1];
	snprintf(filename, MAX_NAME, "/proc/%s/stat", pid);

	std::ifstream infile(filename);
	if (!infile.is_open()) {
		DIA_TR_ERR("dia_ProcInfo::readProcessStatFile: Failed to open file %s", filename);
		return DIA_FAILED;
	}

	std::string statLineStr;
	if (!std::getline(infile, statLineStr)) {
		//		throw new Exception("Failed to read string line from /proc/pid/stat");
		DIA_TR_ERR("dia_ProcInfo::readProcessStatFile: Failed to read string line from '%s'", filename);
		infile.close();
		return false;
	}

	infile.close();

//	DIA_TR_INF("statLineStr: '%s'", statLineStr.c_str());

	std::vector<std::string> splittedStatLineList = dia_Helper::SplitString(statLineStr, ' ');

	if (process->pid != atoi(splittedStatLineList[0].c_str())) { // the left most item is PID
//		throw new Exception("PIDs doesn't match");
		DIA_TR_ERR("dia_ProcInfo::readProcessStatFile: PIDs doesn't match");
		return false;
	}


	std::string commandStr = splittedStatLineList[1].substr(1, splittedStatLineList[1].size()-2); // get the string between '(' and ')'
	if (commandStr.size() <= 1){ // Error
		DIA_TR_ERR("dia_ProcInfo::readProcessStatFile: PID (%p) command.size (%s) <= 1", pid, commandStr.c_str());
		return false;
	}
#ifdef VARIANT_S_FTR_ENABLE_RNAIVI
	std::string process_state;
	process_state = splittedStatLineList[2];
	process->state.clear();
	process->state.append("(");
	process->state.append(process_state);
	process->state.append(")");
#endif

//	tChar *strPos = (tChar*) strchr(statLineStr.c_str(), ' ');
//	if (!strPos)
//		return false;
//
//	strPos += 2;
//	tChar *end = strrchr(strPos, ')');
//	if (!end)
//		return false;
//
	tU32 commandSize = commandStr.size();
	if (commandSize > MAX_NAME) {
		DIA_TR_WRN("dia_ProcInfo::readProcessStatFile: commandSize (%d) > MAX_NAME (%d)", commandSize, MAX_NAME);
		commandSize = MAX_NAME;
	}
	memcpy(command, commandStr.c_str(), commandSize);
	command[commandSize] = '\0';
//	strPos = end + 2;

//	process->state = strPos[0];
//	strPos += 2;
////	process->ppid = strtol(strPos, &strPos, 10);
//	strPos += 1;
////	process->pgrp = strtoul(strPos, &strPos, 10);
//	strPos += 1;
////	process->session = strtoul(strPos, &strPos, 10);
//	strPos += 1;
////	process->tty_nr = strtoul(strPos, &strPos, 10);
//	strPos += 1;
////	process->tpgid = strtol(strPos, &strPos, 10);
//	strPos += 1;
////	process->flags = strtoul(strPos, &strPos, 10);
//	strPos += 1;
////	process->minflt = strtoull(strPos, &strPos, 10);
//	strPos += 1;
////	lp->cminflt = strtoull(linePos, &linePos, 10);
//	strPos += 1;
////	process->majflt = strtoull(linePos, &linePos, 10);
//	strPos += 1;
////	lp->cmajflt = strtoull(linePos, &linePos, 10);
//	strPos += 1;
//	process->utime = adjustTimeForLinuxProcess(strtoull(strPos, &strPos, 10)); // lp->utime = LinuxProcess_adjustTime(strtoull(linePos, &linePos, 10));
	process->utime = adjustTimeForLinuxProcess(strtoul(splittedStatLineList[13].c_str(), NULL, 10));
//	strPos += 1;
//	process->stime = adjustTimeForLinuxProcess(strtoull(strPos, &strPos, 10)); // lp->stime = LinuxProcess_adjustTime(strtoull(linePos, &linePos, 10));
	process->stime = adjustTimeForLinuxProcess(strtoul(splittedStatLineList[14].c_str(), NULL, 10));
//	strPos += 1;
////	lp->cutime = LinuxProcess_adjustTime(strtoull(strPos, &strPos, 10));
//	strPos += 1;
////	lp->cstime = LinuxProcess_adjustTime(strtoull(strPos, &strPos, 10));
//	strPos += 1;
//	process->priority = strtol(strPos, &strPos, 10);
//	strPos += 1;
//	process->nice = strtol(strPos, &strPos, 10);
//	strPos += 1;
//	process->nlwp = strtol(strPos, &strPos, 10);
//	strPos += 1;
//	for (tS32 i=0; i<17; i++) strPos = strchr(strPos, ' ')+1;
//	process->exit_signal = strtol(strPos, &strPos, 10);
//	strPos += 1;
////	assert(strPos != NULL);
//	if (strPos == NULL)
//		throw new Exception("strPos = NULL");
//	process->processor = strtol(strPos, &strPos, 10);
//
//	process->processTime = process->utime + process->stime;

//	DIA_TR_INF("Command: '%s' pid: %d  uTime: %lu  sTime: %lu", command, process->pid, process->utime, process->stime);

	return true;
}

tBool dia_ProcInfo::readProcessStatmFile(const tChar* pid, Process *process) const
{
	// 80395 30979 4094 499 0 35650 0

	tChar filename[MAX_NAME+1];
	snprintf(filename, MAX_NAME, "/proc/%s/statm", pid);

	std::ifstream infile(filename);
	if (!infile.is_open()) {
		DIA_TR_ERR("dia_ProcInfo::readProcessStatmFile: Failed to open file %s", filename);
		return DIA_FAILED;
	}

	std::string statmLineStr;
	if (!std::getline(infile, statmLineStr)) {
//		throw new Exception("Failed to read string line from /proc/pid/statm");
		DIA_TR_ERR("dia_ProcInfo::readProcessStatmFile: Failed to read string line from /proc/%s/statm", pid);
		infile.close();
		return false;
	}
	infile.close();
//	DIA_TR_INF("statmLineStr: '%s'", statmLineStr.c_str());

	tChar *p  = const_cast<tChar *>(statmLineStr.c_str());
	errno = 0;
	if (p != NULL) {
		process->m_size = (tULong)strtol(p, &p, 10); if (*p == ' ') p++;
		process->m_resident = (tULong)strtol(p, NULL, 10); //if (*p == ' ') p++;
	}

	if (process->m_peak < process->m_resident)
		process->m_peak = process->m_resident;

//	process->m_share = strtol(p, &p, 10); if (*p == ' ') p++;
//	process->m_trs = strtol(p, &p, 10); if (*p == ' ') p++;
//	process->m_lrs = strtol(p, &p, 10); if (*p == ' ') p++;
//	process->m_drs = strtol(p, &p, 10); if (*p == ' ') p++;
//	process->m_dt = strtol(p, &p, 10);

//	DIA_TR_INF("process->m_size '%d' process->m_resident: '%d'", process->m_size, process->m_resident);

    return (errno == 0);
}

//tBool dia_ProcInfo::readProcessStatusFile(const tChar* pid, Process *process)
//{
///*
//	Name:   apphmi_tuner-rn
//	State:  S (sleeping)
//	Tgid:   416
//	Ngid:   0
//	Pid:    416
//	PPid:   413
//	TracerPid:      0
//	Uid:    0       0       0       0
//	Gid:    0       0       0       0
//	FDSize: 256
//	Groups:
//	VmPeak:   506728 kB
//	VmSize:   505704 kB
//	VmLck:         0 kB
//	VmPin:         0 kB
//	VmHWM:     25896 kB
//	VmRSS:     25896 kB
//	VmData:    50840 kB
//	VmStk:       136 kB
//	VmExe:      2960 kB
//	VmLib:     64896 kB
//	VmPTE:       472 kB
//	VmSwap:        0 kB
//	Threads:        20
//	SigQ:   298/12835
//	SigPnd: 0000000000000000
//	ShdPnd: 0000000000000000
//	SigBlk: c020000000010001
//	SigIgn: 0020000000001000
//	SigCgt: c018000ffffbeeff
//	CapInh: 0000000000000000
//	CapPrm: 0000001fffffffff
//	CapEff: 0000001fffffffff
//	CapBnd: 0000001fffffffff
//	Cpus_allowed:   3
//	Cpus_allowed_list:      0-1
//	voluntary_ctxt_switches:        35
//	nonvoluntary_ctxt_switches:     97
//
//*/
//	tChar filename[MAX_NAME+1];
//	snprintf(filename, MAX_NAME, "/proc/%s/status", pid);
//	dia_FileStream filePidStatus(filename);
//	if (filePidStatus.openForRead() == DIA_STREAM_ERROR) {
//		//		throw new Exception("Failed to open /proc/pid/status");
//		DIA_TR_WRN("dia_ProcInfo::readProcessStatusFile: Failed to open file /proc/%s/status", pid);
//		return false;
//	}
//
//	std::string statusLineStr;
//	while (filePidStatus.readLine(statusLineStr))
//	{
////		DIA_TR_INF("dia_ProcInfo::readProcessStatusFile: statusLineStr: %s", statusLineStr.c_str());
//		if (statusLineStr.find("VmPeak") == 0) {
//			sscanf(statusLineStr.c_str(), "%*s	%lu", &process->m_peak);
////			DIA_TR_INF("dia_ProcInfo::readProcessStatusFile: found VmPeak: %lu", process->m_peak);
//		}
//		else if (statusLineStr.find("VmSize") == 0) {
//			sscanf(statusLineStr.c_str(), "%*s	%lu", &process->m_size);
////			DIA_TR_INF("dia_ProcInfo::readProcessStatusFile: found VmSize: %lu", process->m_size);
//		}
//		else if (statusLineStr.find("VmRSS") == 0) {
//			sscanf(statusLineStr.c_str(), "%*s	%lu", &process->m_resident);
////			DIA_TR_INF("dia_ProcInfo::readProcessStatusFile: found VmRSS: %lu", process->m_resident);
//		}
//	}
//	filePidStatus.close();
//
//	return true;
//}

Process* dia_ProcInfo::getProcess(tS32 pid)
{
	Process* process = NULL;

	mProcessTableIteratorType it = mProcessTable.find(pid);
	if(it != mProcessTable.end()) {
		process = mProcessTable[pid];
	}
	else {
//		DIA_TR_INF("dia_ProcInfo::getProcess: Adding a new process to the map with pid: %d", pid);
		process = new Process(); // zero initialized
		if (process != NULL) {
			process->pid = pid;
			mProcessTable[pid] = process;
		}
	}

	return process;
}

tDiaResult dia_ProcInfo::updateProcessList(tDouble period)
{
   DIR* dir;
   struct dirent* entry;

   dir = opendir(PROCDIR);
   if (!dir)
	   return DIA_FAILED;

   mTopProcessSortedViaCpuUsage.clear();
   mTopProcessSortedViaMemoryUsage.clear();

   while ((entry = readdir(dir)) != NULL) {
	  tChar* name = entry->d_name;

	  // Hidden files
	  if (name[0] == '.') {
//			 name++;
		  continue;
	  }

	  // Just skip all non-number directories.
	  if (name[0] < '0' || name[0] > '9') {
		 continue;
	  }

	  // filename is a number: process directory
	  tS32 pid = atoi(name);

	  if (pid <= 0)
		 continue;

	  Process* process = getProcess(pid); // gets previously saved information (or newly created entry for the process)

	  if (! readProcessStatmFile(name, process)) {
		  DIA_TR_ERR("dia_ProcInfo::updateProcessList: Failed to read process statm file for pid: '%s'", name);
		  closedir(dir);
		  return DIA_FAILED;
	  }

//	  tChar command[MAX_NAME+1];
	  struct CPUProcessMemory cpuProcessMemory;
	  tULong lastTimes = process->utime + process->stime;
	  if (! readProcessStatFile(name, process, cpuProcessMemory.command)) {
		  DIA_TR_ERR("dia_ProcInfo::updateProcessList: Failed to read process stat file for pid: '%s'", name);
		  closedir(dir);
		  return DIA_FAILED;
	  }

	 tFloat percent_cpu = (tFloat)(((process->utime + process->stime) - lastTimes) / period) * 100.0f;
	 process->percent_cpu = (tFloat)MAX(MIN(percent_cpu, mCpuCount*100.0f), 0.0f);

	 if (isnan(process->percent_cpu)) //lint !e747: Significant prototype coercion (arg. no. 1) float to double
		process->percent_cpu = 0.0f;

	 process->percent_cpu += 0.5f; // to do rounding for integer value

//	 process->percent_mem = (process->m_resident * PAGE_SIZE_KB) / (tDouble)(mMemInfo.mTotalMem) * 100.0;
	 tULong currentMemoryKBytes =  process->m_resident * (tULong)PAGE_SIZE_KB;
	 cpuProcessMemory.m_maximumMemKBytes = process->m_peak * (tULong)PAGE_SIZE_KB;
//	 DIA_TR_INF("PID: %s  command: '%s' currentMem: %lu  maxMem: %lu", name, cpuProcessMemory.command, currentMemoryKBytes, cpuProcessMemory.m_maximumMemKBytes);
//	 DIA_TR_INF("percent_cpu: %lf  process->m_peak: %lu", process->percent_cpu, process->m_peak);
#ifdef VARIANT_S_FTR_ENABLE_RNAIVI
	 std::string commandNostate;
	 commandNostate.append(cpuProcessMemory.command);
	 commandNostate.append(process->state);
	 mTopProcessSortedViaCpuUsage.insert(std::make_pair(process->percent_cpu,commandNostate.c_str()));
#else
	 mTopProcessSortedViaCpuUsage.insert(std::make_pair(process->percent_cpu, cpuProcessMemory.command));
#endif

//	 mTopProcessSortedViaCpuUsage[process->percent_cpu] = cpuProcessMemory.command;

	 mTopProcessSortedViaMemoryUsage[currentMemoryKBytes] = cpuProcessMemory;
   }

   closedir(dir);

   return DIA_SUCCESS;
}

tDiaResult dia_ProcInfo::scanCpuTime(tDouble& period)
{
//	cpu  20353 824 33751 4855526 19210 3 1963 0 0 0
//	cpu0 5104 425 12341 1198050 11277 3 1895 0 0 0
//	cpu1 4424 82 7463 1217389 4034 0 25 0 0 0
//	cpu2 6479 255 7424 1218128 2151 0 36 0 0 0
//	cpu3 4345 60 6522 1221958 1746 0 5 0 0 0

	std::ifstream infile(PROCSTATFILE);
	if (!infile.is_open()) {
		DIA_TR_ERR("dia_ProcInfo::scanCpuTime: Failed to open file %s", PROCSTATFILE);
		return DIA_FAILED;
	}

	for (tU32 i = 0; i <= mCpuCount; i++) {
		tULong usertime, nicetime, systemtime, idletime;
		tULong ioWait, irq, softIrq, steal, guest, guestnice;
		ioWait = irq = softIrq = steal = guest = guestnice = 0;

		std::string procStatLineStr;
		if (!std::getline(infile, procStatLineStr)) {
//			throw new Exception("Failed to read string line from /proc/stat");
			DIA_TR_ERR("dia_ProcInfo::scanCpuTime: Failed to read string line from '%s'", PROCSTATFILE);
			infile.close();
			return DIA_FAILED;
		}

		// Depending on the kernel version 5, 7, 8 or 9 of these fields will be set.
		// The rest will remain at zero.
		if (i == 0)
		    sscanf(procStatLineStr.c_str(), "cpu  %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu", &usertime, &nicetime, &systemtime, &idletime, &ioWait, &irq, &softIrq, &steal, &guest, &guestnice);
		else {
			tU32 cpuid;
			sscanf(procStatLineStr.c_str(), "cpu%1u %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu", &cpuid, &usertime, &nicetime, &systemtime, &idletime, &ioWait, &irq, &softIrq, &steal, &guest, &guestnice); //lint !e706: indirect object nominally inconsistent with format (unsigned int vs. unsigned long)
			if(cpuid != (i - 1)) {
				DIA_TR_ERR("dia_ProcInfo::scanCpuTime: cpudId (%u) != (%d) from '%s' doesn't match", cpuid, (i - 1), PROCSTATFILE);
				infile.close();
				return DIA_FAILED;
			}
		}
		procStatLineStr.clear();

		// Guest time is already accounted in usertime
		usertime = usertime - guest;
		nicetime = nicetime - guestnice;

		tULong idlealltime = idletime + ioWait;
		tULong systemalltime = systemtime + irq + softIrq;
		tULong virtalltime = guest + guestnice;
		tULong totaltime = usertime + nicetime + systemalltime + idlealltime + steal + virtalltime;

		// Compare with previous data
		CPUData* cpuData = mCpuDataTable[i];
		DIA_TR_INF("dia_ProcInfo::scanCpuTime: systime: %lu cpuData->sysTime: %lu", systemtime, cpuData->systemTime);

		DIA_ASSERT (systemtime >= cpuData->systemTime);
		DIA_ASSERT (idletime >= cpuData->idleTime);
		DIA_ASSERT (totaltime >= cpuData->totalTime);
		DIA_ASSERT (systemalltime >= cpuData->systemAllTime);
		DIA_ASSERT (idlealltime >= cpuData->idleAllTime);
		DIA_ASSERT (ioWait >= cpuData->ioWaitTime);
		DIA_ASSERT (irq >= cpuData->irqTime);
		DIA_ASSERT (softIrq >= cpuData->softIrqTime);
		DIA_ASSERT (steal >= cpuData->stealTime);
		DIA_ASSERT (virtalltime >= cpuData->guestTime);
		// Since we do a subtraction (usertime - guest) and cputime64_to_clock_t()
		// used in /proc/stat rounds down numbers, it can lead to a case where the
		// tS32eger overflow.
		cpuData->userPeriod = (usertime > cpuData->userTime) ? usertime - cpuData->userTime : 0;
		cpuData->nicePeriod = (nicetime > cpuData->niceTime) ? nicetime - cpuData->niceTime : 0;
		cpuData->systemPeriod = systemtime - cpuData->systemTime;
		cpuData->systemAllPeriod = systemalltime - cpuData->systemAllTime;
		cpuData->idleAllPeriod = idlealltime - cpuData->idleAllTime;
		cpuData->idlePeriod = idletime - cpuData->idleTime;
		cpuData->ioWaitPeriod = ioWait - cpuData->ioWaitTime;
		cpuData->irqPeriod = irq - cpuData->irqTime;
		cpuData->softIrqPeriod = softIrq - cpuData->softIrqTime;
		cpuData->stealPeriod = steal - cpuData->stealTime;
		cpuData->guestPeriod = virtalltime - cpuData->guestTime;
		cpuData->totalPeriod = totaltime - cpuData->totalTime;
		cpuData->userTime = usertime;
		cpuData->niceTime = nicetime;
		cpuData->systemTime = systemtime;
		cpuData->systemAllTime = systemalltime;
		cpuData->idleAllTime = idlealltime;
		cpuData->idleTime = idletime;
		cpuData->ioWaitTime = ioWait;
		cpuData->irqTime = irq;
		cpuData->softIrqTime = softIrq;
		cpuData->stealTime = steal;
		cpuData->guestTime = virtalltime;
		cpuData->totalTime = totaltime;
	}

	period = (tDouble)this->mCpuDataTable[0]->totalPeriod / mCpuCount; // First element in the table is the sum (first line of /proc/stat)

//	DIA_TR_INF("dia_ProcInfo::scanCpuTime: Period: %lf", period);

	infile.close();

	return DIA_SUCCESS;
}

//tDiaResult dia_ProcInfo::scanDiskstats(tU8& eMMCState, tU8& sdaState)
//{
//	//  179      24 mmcblk1 7964 2169 946196 25850 505 314 2804 11390 0 17390 36960
//
//	std::ifstream infile(PROCDISKSTATSFILE);
//	if (!infile.is_open()) {
//		DIA_TR_ERR("dia_ProcInfo::scanDiskstats: Failed to open file %s", PROCDISKSTATSFILE);
//		return DIA_FAILED;
//	}
//
//	std::string diskStatsLineStr;
//	tULong /*majorNumber, minorNumber, readCompleted, readsMerged, */ blocksRead, /*timeSpentReading, writesCompleted, writesMerged, */ blocksWritten;
//	tS32 numberOfItemsRead;
//	tBool mmcblk1AlreadyRead = FALSE;
//	eMMCState = NO_READ_WRITE_OPERATION;
//	sdaState  = NO_READ_WRITE_OPERATION;
//
//	while (std::getline(infile, diskStatsLineStr)) {
//		DIA_TR_INF("dia_ProcInfo::scanDiskstats: '%s'", diskStatsLineStr.c_str());
//
//		if (!mmcblk1AlreadyRead) {
//			numberOfItemsRead = sscanf(diskStatsLineStr.c_str(), "%*u %*u		mmcblk1 %*u %*u %*u %*u %*u %lu %*u %*u %*u %lu", /*&majorNumber, &minorNumber, &readCompleted, &readsMerged,*/ &blocksRead, /*&timeSpentReading, &writesCompleted, writesMerged,*/ &blocksWritten);
//			DIA_TR_INF("dia_ProcInfo::scanDiskstats: blocksRead: %lu blocksWritten: %lu, numberOfItemsRead: %d", blocksRead, blocksWritten, numberOfItemsRead);
//
//			if (numberOfItemsRead == 2) {
//				mmcblk1AlreadyRead = TRUE;
//
//				// Compare the actual results with the previous ones
//				if (mDiskStatseMMC.totalBlocksRead < blocksRead) {
//					eMMCState |= INDEX_READ_OPERATION;
//				}
//				if (mDiskStatseMMC.totalBlocksWritten < blocksWritten) {
//					eMMCState |= INDEX_WRITE_OPERATION;
//				}
//
//				DIA_TR_INF("dia_ProcInfo::scanDiskstats: mDiskStatseMMC.totalBlocksRead: %lu blocksRead: %lu", mDiskStatseMMC.totalBlocksRead, blocksRead);
//
//				// Save the actual state for the future comparison
//				mDiskStatseMMC.totalBlocksRead = (tU32)blocksRead;
//				mDiskStatseMMC.totalBlocksWritten = (tU32)blocksWritten;
//			}
//		}
//		else {  // sda is below in diskstats file
//			// 8       0 sda 937 17660 20676 4610 0 0 0 0 0 2210 4600
//			numberOfItemsRead = sscanf(diskStatsLineStr.c_str(), "%*u %*u		sda %*u %*u %*u %*u %*u %lu %*u %*u %*u %lu", /*&majorNumber, &minorNumber, &readCompleted, &readsMerged,*/ &blocksRead, /*&timeSpentReading, &writesCompleted, writesMerged,*/ &blocksWritten);
//			if (numberOfItemsRead == 2) {
//
//				// Compare the actual results with the previous ones
//				if (mDiskStatsSDA.totalBlocksRead < blocksRead) {
//					sdaState |= INDEX_READ_OPERATION;
//				}
//				if (mDiskStatsSDA.totalBlocksWritten < blocksWritten) {
//					sdaState |= INDEX_WRITE_OPERATION;
//				}
//
//				DIA_TR_INF("dia_ProcInfo::scanDiskstats: mDiskStatsSDA.totalBlocksRead: %lu blocksRead: %lu", mDiskStatsSDA.totalBlocksRead, blocksRead);
//				DIA_TR_INF("dia_ProcInfo::scanDiskstats: mDiskStatsSDA.totalBlocksWritten: %lu blocksWritten: %lu", mDiskStatseMMC.totalBlocksWritten, blocksWritten);
//
//				// Save the actual state for the future comparison
//				mDiskStatsSDA.totalBlocksRead = (tU32)blocksRead;
//				mDiskStatsSDA.totalBlocksWritten = (tU32)blocksWritten;
//
//				break;
//			}
//		}
//	}
//	infile.close();
//
//	return DIA_SUCCESS;
//}

tDiaResult dia_ProcInfo::scanDiskstats(tU8& eMMCState, tU8& sdaState)
{
    // 39507    96071  2016348    45780     3070    51737   329162   190310       93    39290   372940

/* Source: https://www.kernel.org/doc/Documentation/iostats.txt
 *
 *
	Field  1 -- # of reads completed
	    This is the total number of reads completed successfully.
	Field  2 -- # of reads merged, field 6 -- # of writes merged
	    Reads and writes which are adjacent to each other may be merged for
	    efficiency.  Thus two 4K reads may become one 8K read before it is
	    ultimately handed to the disk, and so it will be counted (and queued)
	    as only one I/O.  This field lets you know how often this was done.
	Field  3 -- # of sectors read
	    This is the total number of sectors read successfully.
	Field  4 -- # of milliseconds spent reading
	    This is the total number of milliseconds spent by all reads (as
	    measured from __make_request() to end_that_request_last()).
	Field  5 -- # of writes completed
	    This is the total number of writes completed successfully.
	Field  6 -- # of writes merged
	    See the description of field 2.
	Field  7 -- # of sectors written
	    This is the total number of sectors written successfully.
	Field  8 -- # of milliseconds spent writing
	    This is the total number of milliseconds spent by all writes (as
	    measured from __make_request() to end_that_request_last()).
	Field  9 -- # of I/Os currently in progress
	    The only field that should go to zero. Incremented as requests are
	    given to appropriate struct request_queue and decremented as they finish.
	Field 10 -- # of milliseconds spent doing I/Os
	    This field increases so long as field 9 is nonzero.
	Field 11 -- weighted # of milliseconds spent doing I/Os
	    This field is incremented at each I/O start, I/O completion, I/O
	    merge, or read of these stats by the number of I/Os in progress
	    (field 9) times the number of milliseconds spent doing I/O since the
	    last update of this field.  This can provide an easy measure of both
	    I/O completion time and the backlog that may be accumulating.
*/
	std::ifstream infileSysBlockMMCStat(SYSFSBLOCKMMCSTATSFILE);
	if (!infileSysBlockMMCStat.is_open()) {
		DIA_TR_ERR("dia_ProcInfo::scanDiskstats: Failed to open file %s", SYSFSBLOCKMMCSTATSFILE);
		return DIA_FAILED;
	}

	std::string statsLineStr;
	tULong readsCompleted;
	tULong writesCompleted;
	tS32 numberOfItemsRead;
	eMMCState = NO_READ_WRITE_OPERATION;
	sdaState  = NO_READ_WRITE_OPERATION;

	if (!std::getline(infileSysBlockMMCStat, statsLineStr)) {
		DIA_TR_ERR("dia_ProcInfo::scanDiskstats: Failed to read string line from '%s'", SYSFSBLOCKMMCSTATSFILE);
		infileSysBlockMMCStat.close();
		return DIA_FAILED;
	}
	infileSysBlockMMCStat.close();

	DIA_TR_INF("dia_ProcInfo::scanDiskstats (eMMC): '%s'", statsLineStr.c_str());

	numberOfItemsRead = sscanf(statsLineStr.c_str(), "%lu %*u %*u %*u %lu %*u %*u %*u %*u %*u %*u",  &readsCompleted, /*&readsMerged, &blocksRead, &timeSpentReading,*/ &writesCompleted /*writesMerged, &blocksWritten*/);
	DIA_TR_INF("dia_ProcInfo::scanDiskstats (eMMC): readCompleted: %lu writesCompleted: %lu", readsCompleted, writesCompleted);

	if (numberOfItemsRead == 2) {
		// Compare the actual results with the previous ones
		if (mDiskStatseMMC.totalBlocksRead < readsCompleted) {
			eMMCState |= INDEX_READ_OPERATION;
		}
		if (mDiskStatseMMC.totalBlocksWritten < writesCompleted) {
			eMMCState |= INDEX_WRITE_OPERATION;
		}

		DIA_TR_INF("dia_ProcInfo::scanDiskstats (eMMC): totalBlocksRead: %lu totalBlocksWritten: %lu", mDiskStatseMMC.totalBlocksRead, mDiskStatseMMC.totalBlocksWritten);

		// Save the actual state for the future comparison
		mDiskStatseMMC.totalBlocksRead = (tU32)readsCompleted;
		mDiskStatseMMC.totalBlocksWritten = (tU32)writesCompleted;
	}
	else {
		DIA_TR_WRN("dia_ProcInfo::scanDiskstats (eMMC): numberOfItemsRead: %d != 2", numberOfItemsRead);
	}

	// Now check USB I/O state
	if (dia_Helper::isDirectory(SYSFSBLOCKSDADIR)) { // if directory exists, then USB is mounted and thus can be analyzed

		std::ifstream infileSysBlockSDAStat(SYSFSBLOCKSDASTATSFILE);
		if (!infileSysBlockSDAStat.is_open()) {
			DIA_TR_ERR("dia_ProcInfo::scanDiskstats: Failed to open file %s", SYSFSBLOCKSDASTATSFILE);
			return DIA_FAILED;
		}

		if (!std::getline(infileSysBlockSDAStat, statsLineStr)) {
			DIA_TR_ERR("dia_ProcInfo::scanDiskstats: Failed to read string line from '%s'", SYSFSBLOCKSDASTATSFILE);
			infileSysBlockSDAStat.close();
			return DIA_FAILED;
		}
		infileSysBlockSDAStat.close();

		DIA_TR_INF("dia_ProcInfo::scanDiskstats (SDA): '%s'", statsLineStr.c_str());

		numberOfItemsRead = sscanf(statsLineStr.c_str(), "%lu %*u %*u %*u %lu %*u %*u %*u %*u %*u %*u",  &readsCompleted, /*&readsMerged, &blocksRead, &timeSpentReading,*/ &writesCompleted /*writesMerged, &blocksWritten*/);
		DIA_TR_INF("dia_ProcInfo::scanDiskstats (SDA): readCompleted: %lu writesCompleted: %lu", readsCompleted, writesCompleted);

		if (numberOfItemsRead == 2) {
			// Compare the actual results with the previous ones
			if (mDiskStatsSDA.totalBlocksRead < readsCompleted) {
				sdaState |= INDEX_READ_OPERATION;
			}
			if (mDiskStatsSDA.totalBlocksWritten < writesCompleted) {
				sdaState |= INDEX_WRITE_OPERATION;
			}

			DIA_TR_INF("dia_ProcInfo::scanDiskstats (SDA): totalBlocksRead: %lu totalBlocksWritten: %lu", mDiskStatsSDA.totalBlocksRead, mDiskStatsSDA.totalBlocksWritten);

			// Save the actual state for the future comparison
			mDiskStatsSDA.totalBlocksRead = (tU32)readsCompleted;
			mDiskStatsSDA.totalBlocksWritten = (tU32)writesCompleted;
		}
		else {
			DIA_TR_WRN("dia_ProcInfo::scanDiskstats (SDA): numberOfItemsRead: %d != 2", numberOfItemsRead);
		}
	}

	return DIA_SUCCESS;
}

std::multimap<tFloat, std::string> dia_ProcInfo::getTopProcessSortedViaCpuUsage(tU16 numberOfProcess)
{
	std::multimap<tFloat, std::string> topProcessSortedViaCpuUsage;

	if (mTopProcessSortedViaCpuUsage.size() >= numberOfProcess) {
		mTopProcessReverseIterator = mTopProcessSortedViaCpuUsage.rbegin();
		for(tU16 i=0; i<numberOfProcess; i++) {
			topProcessSortedViaCpuUsage.insert(std::make_pair(mTopProcessReverseIterator->first, mTopProcessReverseIterator->second));
			mTopProcessReverseIterator++;
		}
	}
	else { // if the size is available elements is smaller than the requested one, return empty map
		DIA_TR_WRN("dia_ProcInfo::getTopProcessSortedViaCpuUsage: numberOfProcess: %d > size: %d", mTopProcessSortedViaCpuUsage.size(), numberOfProcess);
	}

	return topProcessSortedViaCpuUsage;
}

std::map<tULong, struct CPUProcessMemory> dia_ProcInfo::getTopProcessSortedViaMemoryUsage(tU16 numberOfProcess)
{
	std::map<tULong, struct CPUProcessMemory> topProcessSortedViaMemoryUsage;

	if (mTopProcessSortedViaMemoryUsage.size() >= numberOfProcess) {
		mTopProcessSortedViaMemoryUsageReverseIterator = mTopProcessSortedViaMemoryUsage.rbegin();
		for(tU16 i=0; i<numberOfProcess; i++) {
			topProcessSortedViaMemoryUsage[mTopProcessSortedViaMemoryUsageReverseIterator->first] = mTopProcessSortedViaMemoryUsageReverseIterator->second;
			mTopProcessSortedViaMemoryUsageReverseIterator++;
		}
	}
	else { // if the size is available elements is smaller than the requested one, return empty map
		DIA_TR_WRN("dia_ProcInfo::getTopProcessSortedViaMemoryUsage: numberOfProcess: %d > size: %d", mTopProcessSortedViaMemoryUsage.size(), numberOfProcess);
	}

	return topProcessSortedViaMemoryUsage;
}

tDiaResult dia_ProcInfo::getCPUIdleTime(tFloat& cpuIdleTimeInPercent)
{
	if (this->mCpuDataTable[0]->idlePeriod == 0) {
		DIA_TR_ERR("dia_ProcInfo::getCPUIdleTime: idlePeriod = 0");
		return DIA_FAILED;
	}

	cpuIdleTimeInPercent = (this->mCpuDataTable[0]->idlePeriod / (tFloat)this->mCpuDataTable[0]->totalPeriod) * 100.0f;

	return DIA_SUCCESS;
}
/************************************************************************************************************************************************************/
/*
bash-3.2# cat /proc/interrupts
           CPU0       CPU1       CPU2       CPU3
 29:      52306      83407      68992      58574       GIC  29  twd
 34:     106135          0          0          0       GIC  34  sdma
 35:          0          0          0          0       GIC  35  VPU_JPG_IRQ
 41:        389          0          0          0       GIC  41  galcore interrupt service
 42:        348          0          0          0       GIC  42  galcore interrupt service for 2D
 44:        181          0          0          0       GIC  44  VPU_CODEC_IRQ
 54:      19612          0          0          0       GIC  54  mmc0
 56:      28442          0          0          0       GIC  56  mmc1
 58:          0          0          0          0       GIC  58  2020000.serial
 59:       3172          0          0          0       GIC  59  21e8000.serial
 60:          0          0          0          0       GIC  60  21ec000.serial
 61:       7660          0          0          0       GIC  61  21f0000.serial
 62:          0          0          0          0       GIC  62  21f4000.serial
 64:      13314          0          0          0       GIC  64  200c000.ecspi
 65:       2317          0          0          0       GIC  65  2010000.ecspi
 66:        491          0          0          0       GIC  66  2014000.ecspi
 68:       5512          0          0          0       GIC  68  21a0000.i2c
 69:      21883          0          0          0       GIC  69  21a4000.i2c
 70:       1893          0          0          0       GIC  70  21a8000.i2c
 72:         78          0          0          0       GIC  72  ci_hdrc.1
 75:      10707          0          0          0       GIC  75  ci_hdrc.0
 79:          0          0          0          0       GIC  79  BT_DAI
 81:          0          0          0          0       GIC  81  imx_thermal
 82:          0          0          0          0       GIC  82  asrc
 83:          0          0          0          0       GIC  83  esai@02024000
 87:          0          0          0          0       GIC  87  i.MX Timer Tick
112:          0          0          0          0       GIC 112  20bc000.wdog
137:          2          0          0          0       GIC 137  2101000.jr0
138:          6          0          0          0       GIC 138  2102000.jr1
164:      26627          0          0          0  gpio-mxc   4  ssi32_srq
165:          0          0          0          0  gpio-mxc   5  SCC_RSTWARN_CPU_GPIO
172:       3141          0          0          0  gpio-mxc  12  ssi32_srq
193:         84          0          0          0  gpio-mxc   1  adv7182
223:          0          0          0          0  gpio-mxc  31  gpiolib
253:          0          0          0          0  gpio-mxc  29  ssi32_srq
254:          0          0          0          0  gpio-mxc  30  MASCA_REQLINE_IRQ
293:          4          0          0          0  gpio-mxc   5  maxtouch
295:          0          0          0          0  gpio-mxc   7  1-000c
327:          0          0          0          0  gpio-mxc   7  PWR_UDROP_30_GPIO
330:          0          0          0          0  gpio-mxc  10  gpiolib
331:          0          0          0          0  gpio-mxc  11  gpiolib
384:          0          0          0          0       IPU 457  (null)
385:          3          0          0          0       IPU 451  (null)
386:          0          0          0          0       IPU 457  (null)
387:          0          0          0          0       IPU 451  (null)
388:          0          0          0          0       IPU 307  ipu-plane-di-err
389:      11482          0          0          0       IPU  23  ipu-plane
390:          0          0          0          0       IPU 308  ipu-plane-di-err
391:          0          0          0          0       IPU  28  ipu-plane
392:          0          0          0          0       IPU 307  ipu-plane-di-err
393:          0          0          0          0       IPU  23  ipu-plane
394:          0          0          0          0       IPU 308  ipu-plane-di-err
395:          0          0          0          0       IPU  28  ipu-plane
403:          0          0          0          0     dummy   7  gpiolib
408:          0          0          0          0     dummy  12  gpiolib
IPI0:          0          0          0          0  CPU wakeup interrupts
IPI1:          0          0          0          0  Timer broadcast interrupts
IPI2:      16144      28906      20354      17154  Rescheduling interrupts
IPI3:          9         13         17         13  Function call interrupts
IPI4:          1          1          0          1  Single function call interrupts
IPI5:          0          0          0          0  CPU stop interrupts
IPI6:       1199       1160        323        530  IRQ work interrupts
IPI7:          0          0          0          0  completion interrupts
Err:          0
bash-3.2#
*/
/*******************************************************************************************************************************************************/

/*******************************************************************************************************************************************************/
tBool sortdecending(dia_ProcInterruptSortElement value1, dia_ProcInterruptSortElement value2)
{

	return (value1.mCPU0Frequencydiff > value2.mCPU0Frequencydiff);
}
/*********************************************************************************************************************************************************/
dia_ProcInterruptSortElement dia_ProcInfo::getdiffencesElement(dia_ProcInterruptInfo &currentvalue,dia_ProcInterruptInfo &previousvalue)
{

	dia_ProcInterruptSortElement procInterruptInfoElement;
	tULong mCPU0FrequencyDiff = currentvalue.mCPU0Frequency - previousvalue.mCPU0Frequency;

	if(mCpuCount > ONE_CORE)
	{ 
	tULong mCPU1FrequencyDiff = currentvalue.mCPU1Frequency - previousvalue.mCPU1Frequency;
	procInterruptInfoElement.mCPU1Frequencydiff = mCPU1FrequencyDiff;
	}

	procInterruptInfoElement.mCPU0Frequencydiff = mCPU0FrequencyDiff;

	memcpy(procInterruptInfoElement.mProcName,currentvalue.mProcName,sizeof(currentvalue.mProcName));
	return procInterruptInfoElement;
}
/*******************************************************************************************************************************************************/
tDiaResult dia_ProcInfo::getProcInterruptsInfo()
{
	std::vector< dia_ProcInterruptInfo> procInterruptCurrentlist;
	std::vector< dia_ProcInterruptInfo> procInterruptInputlist;
	std::vector< dia_ProcInterruptSortElement> ProcInterruptDiffElementlist;
	ProcInterruptDiffElementlist.clear();
	mProcInterruptFinallist.clear();

	std::ifstream infile(PROCINTERRUPTSFILE);
	if (!infile.is_open())
	{
		DIA_TR_ERR("dia_ProcInfo::scanMemInfo: Failed to open file %s", PROCINTERRUPTSFILE);
		return DIA_FAILED;
	}
	else
	{
		std::string statLineStr;
		if (!std::getline(infile,statLineStr)) 
		{

			DIA_TR_ERR("dia_ProcInfo::getProcInterruptsInfo: Failed to read string line from proc/interrupts file");
			infile.close();
			return false;
		}
		else
		{
				while(std::getline(infile,statLineStr))
				{ 

					dia_ProcInterruptInfo procInterruptInfo;
					switch (mCpuCount)
					{
					case ONE_CORE :
						sscanf(statLineStr.c_str(), "%s %lu %*s %*lu %s", procInterruptInfo.mProcIRQNumber, &procInterruptInfo.mCPU0Frequency, procInterruptInfo.mProcName);
						break;
					case TWO_CORE :
						sscanf(statLineStr.c_str(), "%s %lu %lu %*s %*lu %s", procInterruptInfo.mProcIRQNumber, &procInterruptInfo.mCPU0Frequency, &procInterruptInfo.mCPU1Frequency, procInterruptInfo.mProcName);
						break;
					case THREE_CORE :
						sscanf(statLineStr.c_str(), "%s %lu %lu %*lu %*s %*lu %s", procInterruptInfo.mProcIRQNumber, &procInterruptInfo.mCPU0Frequency, &procInterruptInfo.mCPU1Frequency, procInterruptInfo.mProcName);
						break;
					case FOUR_CORE :
						sscanf(statLineStr.c_str(), "%s %lu %lu %*lu %*lu %*s %*lu %s", procInterruptInfo.mProcIRQNumber, &procInterruptInfo.mCPU0Frequency, &procInterruptInfo.mCPU1Frequency, procInterruptInfo.mProcName);
						break;
					default :
						break;
					}
					procInterruptInputlist.push_back(procInterruptInfo);
				}
			    //std::stable_sort(procInterruptInputlist.begin(), procInterruptInputlist.end(),sortdecending);

				procInterruptCurrentlist = procInterruptInputlist;
				std::vector< dia_ProcInterruptInfo>::iterator currentlist_it = procInterruptCurrentlist.begin();
				std::vector< dia_ProcInterruptInfo>::iterator currentlistEnd_it = procInterruptCurrentlist.end();
				std::vector< dia_ProcInterruptInfo>::iterator previouslist_it = mProcInterruptPreviouslist.begin();
				for (;currentlist_it != procInterruptCurrentlist.end();++currentlist_it)
				{
					previouslist_it = std::find_if(mProcInterruptPreviouslist.begin(), mProcInterruptPreviouslist.end(), MatchProcessIRQ(currentlist_it->mProcIRQNumber));
					if (previouslist_it != mProcInterruptPreviouslist.end())
					{
						    dia_ProcInterruptSortElement entry = getdiffencesElement(*currentlist_it, *previouslist_it);
							ProcInterruptDiffElementlist.push_back(entry);						
					}
				} 
				std::stable_sort(ProcInterruptDiffElementlist.begin(),ProcInterruptDiffElementlist.end(),sortdecending);
				std::vector< dia_ProcInterruptSortElement>::iterator DiffElemlist_it = ProcInterruptDiffElementlist.begin();
				for (;DiffElemlist_it != ProcInterruptDiffElementlist.end();++DiffElemlist_it)
				{
					std::ostringstream diffencesasstring;
					if (mCpuCount > ONE_CORE)
					{

						diffencesasstring << DiffElemlist_it->mCPU0Frequencydiff << "_" << DiffElemlist_it->mCPU1Frequencydiff << "_" << DiffElemlist_it->mProcName;
					}
					else
					{
						diffencesasstring << DiffElemlist_it->mCPU0Frequencydiff << "_" << DiffElemlist_it->mProcName;
					}

					mProcInterruptFinallist.push_back(diffencesasstring.str());
				}
				mProcInterruptPreviouslist = procInterruptCurrentlist;
				    
	    }


	}

	infile.close();

	return DIA_SUCCESS;
}

/*****************************************************************************************************************************************************************************/
std::vector< std::string> dia_ProcInfo::getProcInterruptsInfoResult()
{
	std::vector< std::string>::iterator finallist_it = mProcInterruptFinallist.begin();
	DIA_TR_INF("dia_ProcInfo : getProcInterruptsInfoResult");
	if(!mProcInterruptFinallist.empty())
	{ 
		for (;finallist_it != mProcInterruptFinallist.begin()+MAX_INTERRUPT_PROCESS;++finallist_it)
		{
			std::string finallistEntry = *finallist_it;
			DIA_TR_INF("mProcInterruptFinallist : %s", finallistEntry.c_str());
		}
	
	}
	return mProcInterruptFinallist;
}