/*
***********************************************************************
* FILE:           dia_Odometer.cpp
* PROJECT:        COACH MEDIA CENTER - DAIMLER EVOBUS
* SW-COMPONENT:   DIAGNOSIS
*----------------------------------------------------------------------
*
* DESCRIPTION:    VIN Odometer Handling.
*
*----------------------------------------------------------------------
* HISTORY:
* Date           | Author                            | Modification
*----------------------------------------------------------------------
* 04.03.2019     | Arjun Manjunath Sanu (RBEI/ECA2)  | Initial
***********************************************************************
*/

#ifndef __INCLUDED_DIA_VINODOMETERHANDLER__
#include "project/framework/vin/dia_Odometer.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_VIN__
#include "project/framework/vin/dia_defsVinConfig.h"
#endif

#ifndef __INCLUDED_DIA_CANSIGNALHANDLER__
#include "project/framework/vin/dia_CANSignalHandler.h"
#endif

using namespace std;

#ifndef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT 

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_Odometer);
/************************************************************************
*FUNCTION:     getInstanceOfOdometerHandler
*DESCRIPTION:  This method returns the instance of the class dia_Odometer
*PARAMETER:    void
*RETURNVALUE:  dia_Odometer* - pointer to the class dia_Odometer
*
************************************************************************/
dia_Odometer*
getInstanceOfOdometerHandler ( void )
{
   return dia_Odometer::getInstance();
}
/************************************************************************
*FUNCTION:     getInstanceOfCANSignalHandler
*DESCRIPTION:  This method deletes the instance of the class dia_Odometer
*PARAMETER:    void
*RETURNVALUE:  void
*
************************************************************************/
void
releaseInstanceOfOdometerHandler ( void )
{
   dia_Odometer::deleteInstance();
} 
/************************************************************************
*FUNCTION:     dia_Odometer
*DESCRIPTION:  Class Constructor : Initialize member variables
*PARAMETER:    void
*RETURNVALUE:  None
*
************************************************************************/
dia_Odometer::dia_Odometer(): _mOdometerLockStatus(ODOMETER_LOCKED)//,_mOdometerLimitLockStatus(ODOMETER_LIMIT_LOCKED)
{
   dia_tclFnctTrace trc("dia_Odometer::dia_Odometer");
}
/************************************************************************
*FUNCTION:     ~dia_Odometer
*DESCRIPTION:  Class Destructor
*PARAMETER:    None
*RETURNVALUE:  None 
*
************************************************************************/
dia_Odometer::~dia_Odometer ( void )
{
   DIA_TR_INF("dia_Odometer::~dia_Odometer");
}
/************************************************************************
*FUNCTION:     setup
*DESCRIPTION:  second level initialization
*PARAMETER:    None
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::setup( void )
{
   dia_tclFnctTrace trc("dia_Odometer::setup");

   _mOdometerLockStatus = getOdometerLockStatus();
   //_mOdometerLimitLockStatus = _mOdometerLockStatus;

   return DIA_SUCCESS;
}
/************************************************************************
*FUNCTION:     tearDown
*DESCRIPTION:  second level destruction
*PARAMETER:    None
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::tearDown( void )
{
   dia_tclFnctTrace trc("dia_Odometer::tearDown");
   return DIA_SUCCESS;

}
/************************************************************************
*FUNCTION:     vProcessOdometer
*DESCRIPTION:  This method processes the Odometer data 
*PARAMETER:    None
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::vProcessOdometer( void )
{
   dia_tclFnctTrace trc("dia_Odometer::vProcessOdometer");

   /* Read Odometer from the CAN Signal */
   tU32 rawOdometer = 0, refOdometer = 0;
   tU8 OdometerLimit = 0, cntrOdometer = 0;

   if(DIA_SUCCESS != readOdometerValue(&rawOdometer))
   {
      //Reading odometer failed
      return DIA_FAILED;
   }

   DIA_TR_INF( "dia_Odometer::vProcessOdometer --- Current Odometer = %d", rawOdometer);

   // Read the reference value of odometer for evaluation
   if(DIA_SUCCESS != getOdometerReference(&refOdometer))
   {
      //Reading odometer reference failed
      return DIA_FAILED;
   }

   DIA_TR_INF( "dia_Odometer::vProcessOdometer --- Reference Odometer = %d", refOdometer);

   if(DIA_SUCCESS != getOdometerLimit(&OdometerLimit))
   {
      return DIA_FAILED;
   }

   DIA_TR_INF( "dia_Odometer::vProcessOdometer --- Odometer Limit = %d", OdometerLimit);

   cntrOdometer = (tU8) (rawOdometer - refOdometer);

   DIA_TR_INF( "dia_Odometer::vProcessOdometer --- Odometer Counter = %d", cntrOdometer);

   if(cntrOdometer <= OdometerLimit)
   {
      if(DIA_SUCCESS != storeOdometerCounter(cntrOdometer))
      {
         return DIA_FAILED;
      }

      if(cntrOdometer == OdometerLimit)
      {
         // lock the odometer, odometer limit
         if(DIA_SUCCESS != lockOdometer())
         {
            return DIA_FAILED;
         }
      }
   }

   return DIA_SUCCESS;

}
/************************************************************************
*FUNCTION:     getOdometerReference
*DESCRIPTION:  This method retrieves odometer reference data from KDS
*PARAMETER:    tU32* Odometer - output variable
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::getOdometerReference( tU32* Odometer )
{
   dia_tclFnctTrace trc("dia_Odometer::getOdometerReference");

   // initialize the data vector
   std::vector<tU8> refOdo;
   // Get the property size for VIN Current
   tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW);
   // 
   refOdo.reserve(propLength);
   //refOdo.resize(propLength);
   
   if(DIA_SUCCESS == dia_getProperty (DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW, refOdo ))
   {
      *Odometer  = DIA_M_MAKE_ODOMETER_VALUE_U32(refOdo[3], refOdo[2], refOdo[1], refOdo[0]);
      return DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_Odometer::getOdometerReference - Failed to read reference value !!");
      return DIA_FAILED;
   }
}
/************************************************************************
*FUNCTION:     getOdometerLimit
*DESCRIPTION:  This method retrieves odometer limit data from KDS
*PARAMETER:    tU8* Odometer - output variable
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::getOdometerLimit( tU8* Odometer )
{
   dia_tclFnctTrace trc("dia_Odometer::getOdometerLimit");

   // initialize the data vector
   std::vector<tU8> OdoLmt;
   // Get the property size for VIN Current
   tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER_LIMIT);
   // 
   OdoLmt.reserve(propLength);
   //OdoLmt.resize(propLength);
   
   if(DIA_SUCCESS == dia_getProperty (DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER_LIMIT, OdoLmt ))
   {
      *Odometer  = OdoLmt[0];
      return DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_Odometer::getOdometerLimit - Failed to read Odometer Limit !!");
      return DIA_FAILED;
   }
}
/************************************************************************
*FUNCTION:     storeOdometerCounter
*DESCRIPTION:  This method writes odometer data into KDS
*PARAMETER:    tU8 Odometer - value to be stored in KDS
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::storeOdometerCounter( tU8 Odometer )
{
   dia_tclFnctTrace trc("dia_Odometer::storeOdometerCounter");

   tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER);
   
   // initialize the data buffer
   tU8 u8propData[propLength] = {0};
   (void) ::memset(u8propData,0,propLength);

   u8propData[0] = Odometer; 
   
   if(DIA_SUCCESS == dia_setProperty ( DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER, u8propData, propLength ))
   {
      return DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_Odometer::storeOdometerCounter - Failed to store odometer counter !!");
      return DIA_FAILED;
   }
}
/************************************************************************
*FUNCTION:     getOdometerCounter
*DESCRIPTION:  This method reads odometer data from KDS
*PARAMETER:    tU8* Odometer - output value 
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult 
dia_Odometer::getOdometerCounter( tU8* Odometer)
{
   dia_tclFnctTrace trc("dia_Odometer::getOdometerCounter");

   // initialize the data vector
   std::vector<tU8> Odo;
   // Get the property size for VIN Current
   tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER);
   // 
   Odo.reserve(propLength);
   //Odo.resize(propLength);
   
   if(DIA_SUCCESS == dia_getProperty (DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER, Odo ))
   {
      *Odometer  = Odo[0];
      return DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_Odometer::getOdometerLimit - Failed to read Odometer Counter !!");
      return DIA_FAILED;
   }
}
/************************************************************************
*FUNCTION:     lockOdometer
*DESCRIPTION:  This method writes odometer lock status as LOCKED into KDS
*PARAMETER:    void 
*RETURNVALUE:  tDiaResult 
*
************************************************************************/
tDiaResult
dia_Odometer::lockOdometer( void )
{
   dia_tclFnctTrace trc("dia_Odometer::lockOdometer");

   tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_LOCK);
   
   // initialize the data buffer
   tU8 u8propData[propLength] = {0};
   (void) ::memset(u8propData,0,propLength);

   u8propData[0] = VIN_ODOMETER_LOCKED;
      
   if(DIA_SUCCESS == dia_setProperty ( DIA_PROP_CENTER_CMC_19_VIN_LOCK, u8propData, propLength ))
   {
      DIA_TR_INF( "dia_Odometer::lockOdometer --- Odometer Locked ");
      _mOdometerLockStatus = ODOMETER_LOCKED;
      //_mOdometerLimitLockStatus = ODOMETER_LIMIT_LOCKED;
      return DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_Odometer::lockOdometer - Failed to lock odometer !!");
      return DIA_FAILED;
   }


}
/************************************************************************
*FUNCTION:     isOdometerLocked
*DESCRIPTION:  This method returns odometer lock status 
*PARAMETER:    void 
*RETURNVALUE:  tBool 
*
************************************************************************/
tBool 
dia_Odometer::isOdometerLocked( void )
{
   dia_tclFnctTrace trc("dia_Odometer::isOdometerLocked");
   _mOdometerLockStatus = getOdometerLockStatus();
   //_mOdometerLimitLockStatus = _mOdometerLockStatus;
   return _mOdometerLockStatus;
}
/************************************************************************
*FUNCTION:     isOdometerLimitLocked
*DESCRIPTION:  This method returns odometer limit's lock status 
*PARAMETER:    void 
*RETURNVALUE:  tBool 
*
************************************************************************/
/*
tBool 
dia_Odometer::isOdometerLimitLocked ( void )
{
   dia_tclFnctTrace trc("dia_Odometer::isOdometerLocked");
   _mOdometerLockStatus = getOdometerLockStatus();
   _mOdometerLimitLockStatus = _mOdometerLockStatus;
   return _mOdometerLimitLockStatus;
}
*/
/************************************************************************
*FUNCTION:     lockOdometer
*DESCRIPTION:  This method retrives odometer lock status from KDS
*PARAMETER:    void 
*RETURNVALUE:  tBool 
*
************************************************************************/
tBool
dia_Odometer::getOdometerLockStatus( void )
{
   dia_tclFnctTrace trc("dia_Odometer::getOdometerLockStatus");
   // 
   tU8 OdometerLockStatus = 0;
   // initialize the data vector
   std::vector<tU8> dataVecIn;
   // Get the property size for VIN Current
   tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_LOCK);
   // 
   dataVecIn.reserve(propLength);
   //dataVecIn.resize(propLength);
   
   if(DIA_SUCCESS == dia_getProperty (DIA_PROP_CENTER_CMC_19_VIN_LOCK, dataVecIn ))
   {
      OdometerLockStatus = dataVecIn[0];
   }
   else
   {
      DIA_TR_ERR("dia_Odometer::getOdometerLockStatus - Failed to read lock status !!");
   }
   
   return ((OdometerLockStatus == 0x01) ? ODOMETER_LOCKED : ODOMETER_NOT_LOCKED);
}


#else /* VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT  */

dia_VINOdometerHandler*
dia_VINOdometerHandler::mpInstance = 0;
/************************************************************************
*FUNCTION: 		dia_VINOdometerHandler
*DESCRIPTION: 	Class Constructor : Initialize thread data
*PARAMETER:		void
*RETURNVALUE: 	None
*
************************************************************************/
dia_VINOdometerHandler::dia_VINOdometerHandler()
   : dia_ActiveObject(strVINOdometerThreadName, DIA_PROP_VIN_ODOMETER_THREAD_PRIO, DIA_PROP_VIN_ODOMETER_THREAD_STACK_SIZE),
     mIsSetupDone(FALSE),
     mpMsgQueue(NULL)
{
   dia_tclFnctTrace trc("dia_VINOdometerHandler::dia_VINOdometerHandler");
   dia_VINOdometerHandler::mpInstance = this;
}
/************************************************************************
*FUNCTION: 		getInstance
*DESCRIPTION: 	returns the class instance
*PARAMETER:		void
*RETURNVALUE: 	dia_VINOdometerHandler* -> mpInstance   
*
************************************************************************/
dia_VINOdometerHandler* 
dia_VINOdometerHandler::getInstance()
 { 
	 dia_tclFnctTrace oTrace("dia_VINOdometerHandler::getInstance()");
	 return mpInstance;
 }
/************************************************************************
*FUNCTION: 		deleteInstance
*DESCRIPTION: 	deletes the current class instance
*PARAMETER:		None
*RETURNVALUE: 	void 
*
************************************************************************/
void
dia_VINOdometerHandler::deleteInstance()
 { 
	 dia_tclFnctTrace oTrace("dia_VINOdometerHandler::deleteInstance()");
	 if ( mpInstance )                                                                               
	 {                                                                                               
        delete mpInstance;                                                                           
        mpInstance = 0;                                                                              
     }
 }
/************************************************************************
*FUNCTION: 		~dia_VINOdometerHandler
*DESCRIPTION: 	Class Destructor
*PARAMETER:		None
*RETURNVALUE: 	None 
*
************************************************************************/
//lint -e1579 Warning: Pointer member might have been freed by a separate function but no '-sem(dia_VINOdometerHandler::tearDown,cleanup)' was seen
dia_VINOdometerHandler::~dia_VINOdometerHandler ( void )
{
   // destructors should not throw exceptions (--> lint), but functions called in the
   // could possibly throw excepections. So we catch them here and raise an assert
   _BP_TRY_BEGIN
   {
      (void) dia_VINOdometerHandler::tearDown();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_VINOdometerHandler::~dia_VINOdometerHandler !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}
//lint +e1579 Warning: Pointer member might have been freed by a separate function but no '-sem(dia_VINOdometerHandler::tearDown,cleanup)' was seen
/************************************************************************
*FUNCTION: 		setup
*DESCRIPTION: 	Method to create message queue & VIN Odomter thread
*PARAMETER:		None
*RETURNVALUE: 	None 
*
************************************************************************/
tDiaResult
dia_VINOdometerHandler::setup ( void )
{
   dia_tclFnctTrace trc("dia_VINOdometerHandler::setup");

   // we setup the object only once
   if ( mIsSetupDone )
   {
      DIA_TR_INF("### DIA_VINODOMETERHANDLER - ENGINE WAS ALREADY SET UP. RETURNING... ###");
      return DIA_SUCCESS;
   }

   //
   // create message queue
   //
   mpMsgQueue = OSAL_NEW dia_Queue<dia_VINOdometerMessage>(strVINOdometerQueueName,ODOMETER_QUEUE_SIZE);
   if ( !mpMsgQueue || (mpMsgQueue->open() != DIA_SUCCESS) ) return DIA_FAILED;

   DIA_TR_INF("### READING DIA_VINODOMETERHANDLER ENGINE CONFIGURATION (DIA_VINODOMETERHANDLER ENGINE ADDR=0x%08X)... ###",this);

   if (!(DIA_SUCCESS == startThread()))
   {
      DIA_TR_INF("### FAILED TO START VIN ODOMETER HANDLER THREAD ###");
      return DIA_FAILED;
   }

   mIsSetupDone = TRUE;

   return mIsSetupDone ? DIA_SUCCESS : DIA_FAILED;
}
/************************************************************************
*FUNCTION: 		tearDown
*DESCRIPTION: 	Method to delete message queue, VIN Odometer thread & the class instance
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult -  DIA_SUCCESS/DIA_FAILED
*
************************************************************************/
tDiaResult
dia_VINOdometerHandler::tearDown ( void )
{
   dia_tclFnctTrace oTrace("dia_VINOdometerHandler::tearDown()");

   if ( mpMsgQueue )
   {
      dia_VINOdometerMessage* pMsg = OSAL_NEW dia_VINOdometerMessage(DIA_EN_VIN_ODOMETER_HANDLING_TYPE_TERMINATE); //lint !e429: custodial pointer is freed by after engine has processed the message
      if ( pMsg )
      {
         mpMsgQueue->addElement(pMsg);

         tS32 s32WaitCount = 0;
         while ( s32WaitCount < 10 )
         {
            DIA_TR_INF("Sleep for 100ms");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
            OSAL_s32ThreadWait(100);
#else
            usleep(100*1000); // 100ms
#endif
            ++s32WaitCount;
         }

         pMsg = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pMsg'. --> lifetime is controlled by engine
      }

      mpMsgQueue->close();
   } //lint !e438 Warning: last value assigned to variable 'pMsg' not used

   OSAL_DELETE mpMsgQueue;
   mpMsgQueue = 0;
   
   dia_VINOdometerHandler::mpInstance = 0;

   return DIA_SUCCESS;
} //lint !e1578: Pointer member dia_EngineRunIn::mCmdIter neither freed nor zeroed by cleanup function. No heap allocated for iterator
/************************************************************************
*FUNCTION: 		vThreadEntrypointObject
*DESCRIPTION: 	This method processes the events posted from same/different thread.
*				As soon as the thread is created, this is the first function to hit & 
*				will stay here until thread is terminated.
*PARAMETER:		void
*RETURNVALUE: 	void
*
************************************************************************/
void
dia_VINOdometerHandler::vThreadEntrypointObject ( void )
{
   dia_tclFnctTrace oTrace("dia_VINOdometerHandler::vThreadEntrypointObject");

   dia_VINOdometerMessage* pMessage = 0;

   bool isDone = false;

   do
   {
      pMessage = 0;
      if ( (!mpMsgQueue) || (mpMsgQueue->getElement(&pMessage) != DIA_SUCCESS) )
      {
         DIA_TR_ERR("##### INVALID QUEUE POINTER, NO MESSAGE HANDLING ERROR. ABORTING #####");
         break;
      }

      DIA_TR_INF("pMessage = %s", pMessage ? "not null" : "null");

      if ( pMessage )
      {
         DIA_TR_INF("Message Type = %d", pMessage->mType);

         switch ( pMessage->mType )
         {
         case DIA_EN_VIN_ODOMETER_HANDLING_TYPE_TERMINATE:
            isDone = true;
            break;

         case DIA_EN_VIN_ODOMETER_HANDLING_TYPE_EXTERNAL_REQUEST:
            {
				// write odometer value to kds
			   dia_VINOdometerSignal* Odo = (dia_VINOdometerSignal*)pMessage->mpMsgData;
				
            /*
			   DIA_TR_INF("ODOMETER_VALUE_RECEIVED[1] = 0x%X", Odo->Odometer_Seg1);
			   DIA_TR_INF("ODOMETER_VALUE_RECEIVED[2] = 0x%X", Odo->Odometer_Seg2);
			   DIA_TR_INF("ODOMETER_VALUE_RECEIVED[3] = 0x%X", Odo->Odometer_Seg3);
			   DIA_TR_INF("ODOMETER_VALUE_RECEIVED[4] = 0x%X", Odo->Odometer_Seg4);
            */
			   
			   tU32 u32Odo = DIA_M_MAKE_ODOMETER_VALUE_RAW(Odo->Odometer_Seg4, Odo->Odometer_Seg3, Odo->Odometer_Seg2, Odo->Odometer_Seg1);
			   DIA_TR_INF("ODOMETER_VALUE_RAW = 0x%X",u32Odo);
			   u32Odo = DIA_M_MAKE_ODOMETER_VALUE_PHY(u32Odo);
			   DIA_TR_INF("ODOMETER_VALUE_PHYSICAL = %d", u32Odo);

            //Save the raw can signal value in non-volatile memory
            tU16 propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_CSM_SIGNAL_RAW);
            tU8 u8propData[propLength] = {0};
            // initialize the data memory
            (void) ::memset(u8propData,0,propLength);

            u8propData[0] = Odo->Odometer_Seg1;
            u8propData[1] = Odo->Odometer_Seg2;
            u8propData[2] = Odo->Odometer_Seg3;
            u8propData[3] = Odo->Odometer_Seg4;

            if(DIA_SUCCESS == dia_setProperty ( DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_CSM_SIGNAL_RAW, u8propData, propLength ))
            {
               DIA_TR_INF("dia_VINOdometerHandler: SUCCESS (Write Odometer Raw Value) !!");
            }
            else
            {
               DIA_TR_ERR("dia_VINOdometerHandler: FAILED (Write Odometer Raw Value) !!");
            }

            // Read the reference value of odometer for evaluation
			   propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW);
            u8propData[propLength] = {0};
            // initialize the data memory
            (void) ::memset(u8propData,0,propLength);
            if(DIA_SUCCESS == dia_getProperty ( DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_REF_RAW, u8propData, propLength ))
            {
               DIA_TR_INF("dia_VINOdometerHandler: SUCCESS (Read Odometer Ref Value) !!");
            }
            else
            {
               DIA_TR_ERR("dia_VINOdometerHandler: FAILED (Read Odometer Ref Value) !!");
            }

            tU32 u32OdoRef = DIA_M_MAKE_ODOMETER_VALUE_RAW(u8propData[3], u8propData[2], u8propData[1], u8propData[0]);
            DIA_TR_INF("ODOMETER_REF_VALUE_RAW = 0x%X",u32OdoRef);
            u32OdoRef = DIA_M_MAKE_ODOMETER_VALUE_PHY(u32OdoRef);
            DIA_TR_INF("ODOMETER_REF_VALUE_PHYSICAL = %d", u32OdoRef);

			   if((u32Odo - u32OdoRef) <= VIN_ODOMETER_LIMIT)
			   {
			   
				   propLength = dia_getPropertySize(DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER);
				
					u8propData[propLength] = {0};
					// initialize the data memory
					(void) ::memset(u8propData,0,propLength);
				
					u8propData[0] = (tU8)(u32Odo - u32OdoRef); 
					
					if(DIA_SUCCESS == dia_setProperty ( DIA_PROP_CENTER_CMC_19_VIN_ODOMETER_COUNTER, u8propData, propLength ))
					{
						DIA_TR_INF("dia_VINOdometerHandler: SUCCESS (WRITE ODOMETER DATA TO PERSISTENT MEMORY) !!");
					}
					else
					{
						DIA_TR_ERR("dia_VINOdometerHandler: FAILED (WRITE ODOMETER DATA TO PERSISTENT MEMORY) !!");
					}
			   }
			   else
			   {
				   // do nothing at the moment, at this moment the value of odometer would be mostly zero!
			   }
			   

               OSAL_DELETE pMessage;
               pMessage = 0;
            }
            break;

         default:
            DIA_TR_INF("##### INVALID MESSAGE TYPE -- IGNORED #####");
            break;
         }

         DIA_TR_INF("Message Processed !!");
         OSAL_DELETE pMessage;
         pMessage = 0;
      }
   }
   while ( !isDone ) ;

   DIA_TR_ERR("############################################################");
   DIA_TR_ERR("#");
   DIA_TR_ERR("# VIN_ODOMETER_HANDLING ENGINE RUN IN THREAD IS GETTING TERMINATED");
   DIA_TR_ERR("#");
   DIA_TR_ERR("############################################################");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   // Ask OSAL what the last ERROR to occur was
   tU32 u32ErrorReason = OSAL_u32ErrorCode();
   if ( u32ErrorReason != OSAL_E_NOERROR ) DIA_TR_ERR("msg queue error: %d", u32ErrorReason);
#endif
}
/************************************************************************
*FUNCTION: 		postEvent
*DESCRIPTION: 	This method is used to post events to the thread.
*PARAMETER:		None
*RETURNVALUE: 	None
*
************************************************************************/
void
dia_VINOdometerHandler::postEvent (dia_enVINOdometerHandlingType type, void* msgData)
{
    dia_VINOdometerMessage* pMsg = OSAL_NEW dia_VINOdometerMessage(type, msgData);  //lint !e429: custodial pointer is freed by after engine has processed the message
    
	if ( pMsg && mpMsgQueue ) 
	{
		mpMsgQueue->addElement(pMsg);
	}
    
   delete pMsg;
	pMsg = 0; //lint !e423 Warning: Creation of memory leak in assignment to 'pMsg'. --> lifetime is controlled by engine
}

#endif /* VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT  */

