/************************************************************************
* FILE:           dia_CANSignalHandler.cpp
* PROJECT:        COACH MEDIA CENTER - DAIMLER EVOBUS
* SW-COMPONENT:   DIAGNOSIS
*----------------------------------------------------------------------
*
* DESCRIPTION:    CSM-Signal-Interface for DIAGNOSIS VIN Module.
*
*----------------------------------------------------------------------
* HISTORY:
* Date           | Author                            | Modification
*----------------------------------------------------------------------
* 28.02.2019     | Arjun Manjunath Sanu (RBEI/ECA2)  | Initial
* 26.04.2019     | Harish N G                        | Added check for Can Signal Read status
************************************************************************/

#ifndef __INCLUDED_DIA_CANSIGNALHANDLER__
#include "project/framework/vin/dia_CANSignalHandler.h"
#endif

#ifndef __INCLUDED_DIA_VINODOMETERHANDLER__
#include "project/framework/vin/dia_Odometer.h"
#endif

#ifndef __INCLUDED_DIA_VIN__
#include "project/framework/vin/dia_VIN.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_VIN__
#include "project/framework/vin/dia_defsVinConfig.h"
#endif

#ifndef __INCLUDED_DIA_TEST_CONTROLLER__
#include "common/framework/test/dia_TestController.h"
#endif

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_CANSignalHandler)

csm_tclCsmAccessUser* dia_CANSignalHandler::_poCSM = NULL;
dia_CANSignalHandler::sOdometer_Pkt dia_CANSignalHandler::m_rOdometer_Pkt;
std::vector<tU8> dia_CANSignalHandler::VinMessage;

/************************************************************************
*FUNCTION: 		readOdometerValue
*DESCRIPTION: 	This method reads the odometer value from the can signal
*PARAMETER:		tU32* Odometer - output variable that holds the odometer value
*RETURNVALUE: 	tDiaResult
*
************************************************************************/
tDiaResult 
readOdometerValue(tU32* Odometer)
{
	tU8 Odo[4] = {0};
	tU32 rawOdo, phyOdo;

	if(CSM_C_NO_ERROR == getInstanceOfCANSignalHandler()->vReadSignalValue(CSM_C_SIG_RX_TotalVehDist_Cval_DIAG_CGW_C1, Odo, sizeof(Odo)))
	{
		rawOdo = DIA_M_MAKE_ODOMETER_VALUE_RAW(Odo[3],Odo[2],Odo[1],Odo[0]);
		phyOdo = DIA_M_MAKE_ODOMETER_VALUE_PHY(rawOdo);
		
		*Odometer = phyOdo;

		return DIA_SUCCESS;
	}
	else
	{
		return DIA_FAILED;
	}

}

/************************************************************************
*FUNCTION: 		getReceivedVIN
*DESCRIPTION: 	This method fills the output variable with last received VIN number
*PARAMETER:		vecVin - output variable that holds VIN
*RETURNVALUE: 	tDiaResult
*
************************************************************************/
tDiaResult
getReceivedVIN(std::vector<tU8>& vecVin)
{
	if(getInstanceOfCANSignalHandler()->VinMessage.empty())
	{
		DIA_TR_INF("getReceivedVIN : VIN Message is empty.");
		return DIA_FAILED;
	}

	vecVin = getInstanceOfCANSignalHandler()->VinMessage;

	return DIA_SUCCESS;
}

/************************************************************************
*FUNCTION: 		getInstanceOfCANSignalHandler
*DESCRIPTION: 	This method returns the instance of the class dia_CANSignalHandler
*PARAMETER:		void
*RETURNVALUE: 	dia_CANSignalHandler* - pointer to the class dia_CANSignalHandler
*
************************************************************************/
dia_CANSignalHandler*
getInstanceOfCANSignalHandler ( void )
{
   return dia_CANSignalHandler::getInstance();
}
/************************************************************************
*FUNCTION: 		getInstanceOfCANSignalHandler
*DESCRIPTION: 	This method deletes the instance of the class dia_CANSignalHandler
*PARAMETER:		void
*RETURNVALUE: 	void
*
************************************************************************/
void
releaseInstanceOfCANSignalHandler ( void )
{
   dia_CANSignalHandler::deleteInstance();
}
/************************************************************************
*FUNCTION: 		signalTable
*DESCRIPTION: 	This is the signal table for can signals which will be registered during start up
*PARAMETER:		none
*RETURNVALUE: 	none
*
************************************************************************/
static dia_CANSignalRepo signalTable[] =
{/*
    { sSignalName				, u32SignalId           				    */	
	{ "VIN"					    , CSM_C_SIG_RX_CGW_VIN   					 },
	{ "TotalVehDist_Cval_DIAG"  , CSM_C_SIG_RX_TotalVehDist_Cval_DIAG_CGW_C1 },
};

/************************************************************************
*FUNCTION: 		dia_CANSignalHandler
*DESCRIPTION: 	Constructor
*PARAMETER:		None
*RETURNVALUE: 	None
*
************************************************************************/
dia_CANSignalHandler::dia_CANSignalHandler()
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::dia_CANSignalHandler(...)");

	m_rOdometer_Pkt.Odometer_Seg1 = 0;
	m_rOdometer_Pkt.Odometer_Seg2 = 0;
	m_rOdometer_Pkt.Odometer_Seg3 = 0;
	m_rOdometer_Pkt.Odometer_Seg4 = 0; 
}
/************************************************************************
*FUNCTION: 		~dia_CANSignalHandler
*DESCRIPTION: 	Destructor
*PARAMETER:		None
*RETURNVALUE: 	None 
*
************************************************************************/
dia_CANSignalHandler::~dia_CANSignalHandler()
{
	DIA_TR_INF("dia_CANSignalHandler::~dia_CANSignalHandler(...)");
}

/************************************************************************
*FUNCTION: 		poGetCSMInstance
*DESCRIPTION: 	Returns the instance of CSM_Proxy
*PARAMETER:		None
*RETURNVALUE: 	csm_tclCsmAccessUser*
*				- Pointer to csm_tclCsmAccessUser 
*
************************************************************************/
csm_tclCsmAccessUser* dia_CANSignalHandler::poGetCSMInstance() 
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::poGetCSMInstance()");

	if (dia_CANSignalHandler::_poCSM == NULL)
	{
		_poCSM = OSAL_NEW csm_tclCsmAccessUser();
		DIA_TR_INF("dia_CANSignalHandler::poGetCSMInstance(): csm proxy instance created.");
	}
	return _poCSM;
}

/************************************************************************
*FUNCTION: 		setup
*DESCRIPTION: 	Initialize csm proxy instance & register application
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult
*
************************************************************************/
tDiaResult dia_CANSignalHandler::setup()
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::setup()");

#ifndef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT
#else 
	mTimer.s32Create();
	mTimer.addTimerListener(this);
    mTimer.s32SetTime(0,0);
    mTimer.s32SetTime(ODOMETER_CAN_SIGNAL_PROCESSING_INTERVAL, ODOMETER_CAN_SIGNAL_PROCESSING_INTERVAL);
#endif

	dia_CANSignalHandler::poGetCSMInstance()->vApplCallbackPreInit();
	DIA_TR_INF("dia_CANSignalHandler::poGetCSMInstance()->vApplCallbackPreInit() : Application Registered .");
	vRegisterCanSignals();
	
	return DIA_SUCCESS;
}

/************************************************************************
*FUNCTION: 		tearDown
*DESCRIPTION: 	Initialize csm proxy instance & register application
*PARAMETER:		None
*RETURNVALUE: 	tDiaResult
*
************************************************************************/
tDiaResult dia_CANSignalHandler::tearDown()
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::tearDown()");
	if(_poCSM != NULL)
	{
		delete _poCSM;
	}
	_poCSM = 0;	

#ifndef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT 
#else
	mTimer.s32Delete();
    mTimer.removeTimerListener(this);
#endif
	
	return DIA_SUCCESS;
}


/************************************************************************
*FUNCTION: 		vRegisterCanSignals
*DESCRIPTION: 	This method registers all required can signals with csm proxy
*PARAMETER:		None
*RETURNVALUE: 	None
*
************************************************************************/
void dia_CANSignalHandler::vRegisterCanSignals()
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::vRegisterCanSignals()");
	tU32 u32RetCode = CSM_C_NO_ERROR;
	
	for (tU32 i = 0; i <  sizeof(signalTable)/sizeof(dia_CANSignalRepo); i++)
	{
		u32RetCode = dia_CANSignalHandler::poGetCSMInstance()->s32SignalCallbackInit(signalTable[i].u32SignalId, dia_CANSignalHandler::vSignalIndCallback);
		
		if (CSM_C_NO_ERROR == u32RetCode)
		{
			DIA_TR_INF( "dia_CANSignalHandler::vRegisterCanSignals(): REGISTRATION PASSED for SIGNAL=%s with return code: %X",signalTable[i].sSignalName, u32RetCode);
		}
		else
		{
			DIA_TR_ERR( "dia_CANSignalHandler::vRegisterCanSignals(): REGISTRATION FAILED for SIGNAL=%s with return code: %X",signalTable[i].sSignalName, u32RetCode);
		}
	}
}

/************************************************************************
*FUNCTION: 		vUnregisterCanSignals
*DESCRIPTION: 	This method unregisters required can signal 
*PARAMETER:		None
*RETURNVALUE: 	None
*
************************************************************************/
void dia_CANSignalHandler::vUnregisterCanSignals()
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::vUnregisterCanSignals()");
}

/************************************************************************
*FUNCTION: 		vUnregisterAllCanSignals
*DESCRIPTION: 	This method unregisters all can signals registered 
*PARAMETER:		None
*RETURNVALUE: 	None
*
************************************************************************/
void dia_CANSignalHandler::vUnregisterAllCanSignals()
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::vRegisterCanSignals()");

	if (CSM_C_NO_ERROR > dia_CANSignalHandler::poGetCSMInstance()->s32SignalCallbackDeInitAll())
	{
		DIA_TR_ERR("dia_CANSignalHandler::vUnregisterAllCanSignals -> s32SignalCallbackDeInitAll failed");
	}
	else
	{
		DIA_TR_INF("dia_CANSignalHandler::vUnregisterAllCanSignals -> ALL SIGNALS UNREGISTERED SUCCESSFULLY");
	}
}

/************************************************************************
*FUNCTION: 		vSetSignalNewValue
*DESCRIPTION: 	This method sets the new desired value into the selected csm signal 
*PARAMETER:		u32SignalId - Signal Id
*               pvNewSignalData - pointer to the new data buffer
*				u8DataBufferLength - Length of the data buffer
*				u8TxType - transmit type
*RETURNVALUE: 	None
*
************************************************************************/
void dia_CANSignalHandler::vSetSignalNewValue(tU32 u32SignalId, tVoid* pvNewSignalData, tU8 u8DataBufferLength, tU8 u8TxType)
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::vSetSignalNewValue");

	if (CSM_C_NO_ERROR == dia_CANSignalHandler::poGetCSMInstance()->s32SignalWrite(u32SignalId, pvNewSignalData, u8DataBufferLength, u8TxType))
	{
		DIA_TR_INF( "dia_CANSignalHandler::vSetSignalNewValue(): SIGNAL WRITE PASSED");
	}
	else
	{
		DIA_TR_ERR( "dia_CANSignalHandler::vSetSignalNewValue(): SIGNAL WRITE FAILED");
	}

}
/************************************************************************
*FUNCTION: 		vSetSignalNewValue
*DESCRIPTION: 	This method sets the new desired value into the selected csm signal 
*PARAMETER:		u32SignalId - Signal Id
*               pvNewSignalData - pointer to the new data buffer
*				u8DataBufferLength - Length of the data buffer
*				u8TxType - transmit type
*RETURNVALUE: 	tDiaResult
*
************************************************************************/
tDiaResult dia_CANSignalHandler::vReadSignalValue(tU32 u32SignalId, tU8* pActSignalData, tU8 u8DataBufferLength)
{
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::vReadSignalValue");
	// Variable to store signal read status
	tU32 u32SigReadStat;

	if(CSM_C_NO_ERROR != dia_CANSignalHandler::poGetCSMInstance()->s32SignalRead(u32SignalId, pActSignalData, u8DataBufferLength, &u32SigReadStat))
	{
		DIA_TR_ERR( "dia_CANSignalHandler::vReadSignalValue(): Signal Read Failed");
		return DIA_FAILED;
	}
	else
	{
		if(CSM_C_SIGNAL_DATA_INVALID == u32SigReadStat || CSM_C_SIGNAL_TIMEOUT ==u32SigReadStat || CSM_C_SIGNAL_NOT_RECEIVED ==u32SigReadStat)
		{
			DIA_TR_ERR( "dia_CANSignalHandler::vReadSignalValue(): Signal Read status is invalid or not received or timeout");
			return DIA_FAILED;
		}
		else
		{

		}
	}
	return DIA_SUCCESS;
}

/************************************************************************
*FUNCTION: 		vSignalIndCallback
*DESCRIPTION: 	This method sets the new desired value into the selected csm signal 
*PARAMETER:		u32SignalId - Signal Id
*               pvNewSignalData - pointer to the new data buffer
*				u8DataBufferLength - Length of the data buffer
*				u8TxType - transmit type
*RETURNVALUE: 	None
*
************************************************************************/
void dia_CANSignalHandler::vSignalIndCallback(tVoid* pHandle, tU32 u32SignalId, tU32 u32State)
{
	// Variable to store signal read status
	tU32 u32SigReadStat;
	// Variable to fetch VIN data
	sVIN_Pkt rVIN_Pkt;
	static sVIN_Pkt _VinPkt;
	// variable to Store CAN signal VIN
	static sVIN_Pkt _sVIN[VIN_MAX_NUM_OF_SEGMENTS] = {0};
	// Variable to store VIN Data
	sVinMsg _VinMsg;
	_VinMsg.mVin.reserve(20);
	//tU8 u8Vin[VIN_LENGTH_MAX] = {0};
	//sOdometer_Pkt rOdometer_Pkt;
	static tU32 prevOdometer = 0;
	tU32 currOdometer = 0;
	
	dia_tclFnctTrace oTrace("dia_CANSignalHandler::vSignalIndCallback");
	
	switch(u32SignalId)
	{
		case CSM_C_SIG_RX_CGW_VIN:
		{
			DIA_TR_INF( "dia_CANSignalHandler::vSignalIndCallback(): Signal Indication called for CSM_C_SIG_RX_CGW_VIN");
			
			if(!(u32State & CSM_C_SIGNAL_TIMEOUT))
			{
				DIA_TR_INF( "dia_CANSignalHandler::vSignalIndCallback(): CSM_C_SIG_RX_CGW_VIN -> Signal Valid!! ");
				
				if(CSM_C_NO_ERROR == dia_CANSignalHandler::poGetCSMInstance()->s32SignalRead(u32SignalId, &rVIN_Pkt, sizeof(sVIN_Pkt), &u32SigReadStat))
				{
					for(tU8 i=0; i<VIN_MAX_CAN_FRAME_LENGTH; ++i)
					{
						DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> VIN[%d] = 0x%X", i, rVIN_Pkt.VIN[i]);
					}

					// Store VIN Segment
					_VinPkt = rVIN_Pkt;
					_sVIN[_VinPkt.VIN[0]] = _VinPkt;

					if(VIN_FINAL_SEGMENT == _VinPkt.VIN[0])
					{
						tU8 u8VinCnt = 0;

#ifdef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT 

						for (tU8 i=0; i<4; ++i)
						{
							_VinMsg.mVin.push_back(0);
						}

#endif /* VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT */
						
						for (tU8 i=0; i<=VIN_FINAL_SEGMENT; ++i)
						{
							for(tU8 j=1; j<VIN_MAX_CAN_FRAME_LENGTH; ++j)
							{
								if(!(VIN_LENGTH_MAX == u8VinCnt))
								{
									_VinMsg.mVin.push_back(_sVIN[i].VIN[j]);
								}
								else
								{
									break;
								}	
								++u8VinCnt;
							}
							
						}
						
						DIA_TR_INF("############################################################");
						DIA_TR_INF("#														   #");
						DIA_TR_INF("#            VEHICLE IDENTIFICATION NUMBER                 #");
						DIA_TR_INF("#                                                          #");
						DIA_TR_INF("############################################################");

#ifdef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT

						for(tU8 i=4; i<VIN_LENGTH_MAX+4; ++i)
						{
							DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> VIN[%d] = 0x%X", i-3, _VinMsg.mVin[i]);
						}

						dia_VIN* _poVin = dia_VIN::getInstance();
						if( _poVin )
						{
							_poVin->postEvent(dia_VIN::dia_enVINHandlingType::DIA_EN_VIN_HANDLING_TYPE_EXTERNAL_REQUEST, &_VinMsg);
						}

#else /* VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT */
						VinMessage.clear();
						for(tU8 i=0; i<VIN_LENGTH_MAX; ++i)
						{
							VinMessage.push_back(_VinMsg.mVin[i]);
							DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> VIN[%d] = 0x%X", i, VinMessage[i]);
						}
						
						dia_TestController* pTestController = getInstanceOfTestController();
      					if ( pTestController )
      					{
      					   //To trigger test, pass the test condition DIA_EN_TESTCONDITION_CONFIG_CALIB_DTC_UPDATE via the runTests() method to the dia_TestController singleton
      					   (void) pTestController->runTests(DIA_EN_TESTCONDITION_PROCESS_VIN);
      					}
      					else
      					{
      				   		DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback --- pTestController == NULL");
      					}


#endif /* VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT */

					}	
					else
					{
						//do nothing
					}
				}
				else
				{
					DIA_TR_ERR("dia_CANSignalHandler::vSignalIndCallback -> signal read failed for VIN");  
				}
				
			}
			else
			{
				DIA_TR_ERR( "dia_CANSignalHandler::vSignalIndCallback(): VIN -> Signal Timeout!! ");
			}
			
		}
		break;
		
		case CSM_C_SIG_RX_TotalVehDist_Cval_DIAG_CGW_C1:
		{
			DIA_TR_INF( "dia_CANSignalHandler::vSignalIndCallback(): Signal Indication called for TotalVehDist_Cval_DIAG");
			
			if(!(u32State & CSM_C_SIGNAL_TIMEOUT))
			{
				DIA_TR_INF( "dia_CANSignalHandler::vSignalIndCallback(): TotalVehDist_Cval_DIAG -> Signal Valid!! ");
				
				if(CSM_C_NO_ERROR == dia_CANSignalHandler::poGetCSMInstance()->s32SignalRead(u32SignalId, &dia_CANSignalHandler::m_rOdometer_Pkt, sizeof(dia_CANSignalHandler::sOdometer_Pkt), &u32SigReadStat))
				{
					
					/*
					DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> TotalVehDist0 = 0x%X", dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg1);
					DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> TotalVehDist1 = 0x%X", dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg2);
					DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> TotalVehDist2 = 0x%X", dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg3);
					DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> TotalVehDist3 = 0x%X", dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg4);
					*/
					

					currOdometer = DIA_M_MAKE_ODOMETER_VALUE_RAW(m_rOdometer_Pkt.Odometer_Seg4, m_rOdometer_Pkt.Odometer_Seg3, m_rOdometer_Pkt.Odometer_Seg2, m_rOdometer_Pkt.Odometer_Seg1);
					//DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> currOdometer raw  = 0x%x", currOdometer);
					currOdometer = (tU32)DIA_M_MAKE_ODOMETER_VALUE_PHY(currOdometer);

					DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> currOdometer = %d", currOdometer);
					DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback -> prevOdometer = %d", prevOdometer);


					dia_Odometer* pOdometerHandler = getInstanceOfOdometerHandler();
   					if(pOdometerHandler)
   					{
   						if (ODOMETER_LOCKED == pOdometerHandler->isOdometerLocked())
   						{
   							DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback - Odometer is locked. No further processing required. ");
   							return;
   						}
   					}

					if (currOdometer >= prevOdometer + 1)
					{
						prevOdometer = currOdometer;
						dia_TestController* pTestController = getInstanceOfTestController();
      					if ( pTestController )
      					{
      					   //To trigger test, pass the test condition DIA_EN_TESTCONDITION_PROCESS_ODOMETER via the runTests() method to the dia_TestController singleton
      					   (void) pTestController->runTests(DIA_EN_TESTCONDITION_PROCESS_ODOMETER);
      					}
      					else
      					{
      				   		DIA_TR_INF("dia_CANSignalHandler::vSignalIndCallback --- pTestController == NULL");
      					}
      				}
			
				}
				else
				{
					DIA_TR_ERR("dia_CANSignalHandler::vSignalIndCallback -> signal read failed for TotalVehDist_Cval_DIAG");  
				}
			}
			else
			{
				DIA_TR_ERR( "dia_CANSignalHandler::vSignalIndCallback(): TotalVehDist_Cval_DIAG -> Signal Timeout!! ");
			}
		}
		break;
			
		default:
		{
			DIA_TR_ERR( "dia_CANSignalHandler::vSignalIndCallback(): Signal Indication for UNKNOWN Signal!!"); 
		}
		break;
	}
}

#ifndef VARIANT_S_FTR_ENABLE_CMC19_EVOBUS_VIN_THREAD_CONCEPT 
#else
/************************************************************************
*FUNCTION: 		vOnTimerElapsed
*DESCRIPTION: 	This method is called when the cyclic odometer timer elapses.
*               Used to write the odometer can signal data to KDS memory which helps in :
*				1. To avoid load on the odometer thread because of frequent posts 
*				as the can signal si received every second  
*				2. To reduce write cycles into the KDS memory
*PARAMETER:		u32SignalId - Signal Id
*               pvNewSignalData - pointer to the new data buffer
*				u8DataBufferLength - Length of the data buffer
*				u8TxType - transmit type
*RETURNVALUE: 	None
*
************************************************************************/
void
dia_CANSignalHandler::vOnTimerElapsed ( dia_TimerID /*id*/ )
{
    dia_tclFnctTrace oTrace("dia_CANSignalHandler::vOnTimerElapsed(...)");

   static sOdometer_Pkt rPrevOdoData = {0, 0, 0, 0};

   //if(VIN_ODOMETER_NO_DATA != m_rOdometer_Pkt)
   if(( VIN_ODOMETER_NO_DATA != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg1 || \
        VIN_ODOMETER_NO_DATA != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg2 || \
        VIN_ODOMETER_NO_DATA != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg3 || \
        VIN_ODOMETER_NO_DATA != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg4   ))
   {

   		if((rPrevOdoData.Odometer_Seg1 != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg1 || \
            rPrevOdoData.Odometer_Seg2 != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg2 || \
            rPrevOdoData.Odometer_Seg3 != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg3 || \
            rPrevOdoData.Odometer_Seg4 != dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg4   ))
        {
   			dia_VINOdometerHandler* _poOdo = dia_VINOdometerHandler::getInstance();
			if( _poOdo )
			{
				_poOdo->postEvent(dia_VINOdometerHandler::dia_enVINOdometerHandlingType::DIA_EN_VIN_ODOMETER_HANDLING_TYPE_EXTERNAL_REQUEST, &dia_CANSignalHandler::m_rOdometer_Pkt);
			}

			rPrevOdoData.Odometer_Seg1 = dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg1;
			rPrevOdoData.Odometer_Seg2 = dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg2;
			rPrevOdoData.Odometer_Seg3 = dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg3;
			rPrevOdoData.Odometer_Seg4 = dia_CANSignalHandler::m_rOdometer_Pkt.Odometer_Seg4;


        }
   }
   else
   {
   		DIA_TR_INF("dia_CANSignalHandler::vOnTimerElapsed -> Odometer data empty");
   }
}
#endif


