/*
 * dia_SAFeatureTunerAMFM.cpp
 *
 *  Created on: 11.10.2012
 *      Author: shw2abt
 *
 *--------------------------------------------------------
 * Date        | Author        | Modification
 * 27.05.14    | nce5kor       | Small code change for tunerAMFM_Frequency ReadDataByIdentifier
 * 24.02.2016  | gpu2kor       | added DRM frequency set, read and Service ID set and Read functions, responses. (CMG3G-8573)
 *************************************************************************/

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#include "dia_SAFeatureTunerAMFM.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_STDVISITORS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_TYPES

#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

/// MESSAGE MAP:
// The message map maps IDs of incoming CCA messages to proper handle methods
BEGIN_MSG_MAP(dia_SAFeatureTunerAMFM,dia_SystemAdapterFeature)
   ON_MESSAGE(MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO,            vHandleTunerAMFM_ATSTATIONINFO_Results)
   ON_MESSAGE(MIDW_TUNERFI_C_U16_FID_TUN_S_ATSEEK_WITH_TUNERHEADER,  vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results)
   ON_MESSAGE(MIDW_TUNERFI_C_U16_FID_TUN_S_DIRECT_FREQUENCY,         vHandleTunerAMFMFrequency_Set)
   ON_MESSAGE(MIDW_TUNERFI_C_U16_FID_S_TUN_GET_FIELD_STRENGTH,       vHandleTunerAMFMFieldStrength_Get)
   ON_MESSAGE(MIDW_TUNERFI_C_U16_FID_TUN_G_ATSEEK,                   vHandleTunerAMFM_ATSEEK_Results)
   ON_MESSAGE(MIDW_TUNERFI_C_U16_FID_TUN_S_SELECT_DRM_AUDIOPRGM,	 vHandleTunerDRM_Staus)
END_MSG_MAP()

// Link enum and Function ID
static tU16 arAMFMFIDMapping[DIA_EN_TUNERAMFM_NOTIFICATION_COUNT] = {
        0xFFFF,
        MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO, // DIA_EN_TUNERAMFM_NOTIFICATION_FREQUENCY
        MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO, // DIA_EN_TUNERAMFM_NOTIFICATION_RDSPROGRAMINFORMATION
        MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO, // DIA_EN_TUNERAMFM_NOTIFICATION_BAND
        MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO  // DIA_EN_TUNERAMFM_NOTIFICATION_RADIO_TUNER1_STATUS
};

#define TUNER_PARAM_SIZE            288 //DP_U8_KDSLEN_AMFMTUNERPARAMETER_COMPLETE
#define TUNER_PARAM_OFFSET_LOWFM    12
#define TUNER_PARAM_OFFSET_HIGHFM   16
#define TUNER_PARAM_OFFSET_LOWMW    22
#define TUNER_PARAM_OFFSET_HIGHMW   26
#define TUNER_PARAM_OFFSET_LOWLW    32
#define TUNER_PARAM_OFFSET_HIGHLW   36

//-----------------------------------------------------------------------------

static midw_fi_tcl_e8_Tun_TunerBand::tenType getBand(tU32 frequency)
{
   // Get TunerAMFM parameter set to decide on the band
   tU32 lowFreq = 0, highFreq = 0;
   std::vector<tU8> tunerParam;
   tDiaResult retCode = dia_getProperty(DIA_PROP_AIVI_FMAM_TUNER_PARAM_SET, tunerParam);
   if ( (retCode != DIA_SUCCESS) || (tunerParam.size() < TUNER_PARAM_SIZE) )
   {
      DIA_TR_ERR("getBand: Reading TunerParam Failed (0x%08x) !!", retCode);
      return midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_INVALID;
   }

   lowFreq = tunerParam[TUNER_PARAM_OFFSET_LOWFM];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWFM+1];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWFM+2];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWFM+3];
   highFreq = tunerParam[TUNER_PARAM_OFFSET_HIGHFM];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHFM+1];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHFM+2];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHFM+3];

   DIA_TR_INF("getBand - FM: %d...%d", lowFreq, highFreq);

   if ((frequency >= lowFreq) && (frequency <= highFreq))
   {
      return midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM;
   }

   lowFreq = tunerParam[TUNER_PARAM_OFFSET_LOWMW];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWMW+1];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWMW+2];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWMW+3];
   highFreq = tunerParam[TUNER_PARAM_OFFSET_HIGHMW];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHMW+1];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHMW+2];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHMW+3];

   DIA_TR_INF("getBand - MW: %d...%d", lowFreq, highFreq);

   if ((frequency >= lowFreq) && (frequency <= highFreq))
   {
      return midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW;
   }

   lowFreq = tunerParam[TUNER_PARAM_OFFSET_LOWLW];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWLW+1];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWLW+2];
   lowFreq = (lowFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_LOWLW+3];
   highFreq = tunerParam[TUNER_PARAM_OFFSET_HIGHLW];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHLW+1];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHLW+2];
   highFreq = (highFreq << 8) | tunerParam[TUNER_PARAM_OFFSET_HIGHLW+3];

   DIA_TR_INF("getBand - LW: %d...%d", lowFreq, highFreq);

   if ((frequency >= lowFreq) && (frequency <= highFreq))
   {
      return midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW;
   }

   return midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_INVALID;
}

//------------------------------------------------------------------------------

dia_SAFeatureTunerAMFM::dia_SAFeatureTunerAMFM (
      tCString name,    // feature name
      dia_SystemAdapterServicePlugin& pSrvPlugin
   )
    : dia_SystemAdapterFeature(name,pSrvPlugin),mbUpRegATSEEKActive(false), mLastActionFreqSeekFM(DIA_EN_TUNERAMFM_ACT_STOP)
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::dia_SAFeatureTunerAMFM");
}

//-----------------------------------------------------------------------------

dia_SAFeatureTunerAMFM::~dia_SAFeatureTunerAMFM ( tVoid )
{}

//------------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_attachNotification ( dia_eTunerAMFMNotification id )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_attachNotification()");
   tDiaResult retCode = DIA_FAILED;

   tU16 fid = arAMFMFIDMapping[id];
   if ( fid != 0xFFFF )
   {
      if ( mpSrvPlugin->bRegisterForFID(fid) ) retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_detachNotification ( dia_eTunerAMFMNotification id )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_detachNotification()");
   tDiaResult retCode = DIA_FAILED;

   tU16 fid = arAMFMFIDMapping[id];
   if ( fid != 0xFFFF )
   {
      if ( mpSrvPlugin->bUnregisterForFID(fid) ) retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get ( )
{
    dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get()");
    tDiaResult retCode = DIA_FAILED;

    if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
    {
   	 DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get - send message.");
   	 gm_tclEmptyMessage oMessage (
   	           mAppID,
   	           mpSrvPlugin->getClientID(),
   	           mpSrvPlugin->getRegistrationID(),
   	           0,
   	           mpSrvPlugin->getServiceID(),
   	           MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO,
   	           CCA_C_U8_OPCODE_UPREG
   	        );

       if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ){
      	 retCode = DIA_SUCCESS;
      	 DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get - transmitMessage success.");
       } else {
      	 DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get - transmitMessage failed.");
       }

    } else {
   	 DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get - getRegistrationID failed.");
   	 DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Get - getRegistrationID == AMT_C_U16_REGID_INVALID.");
    }

    return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set ( tU32 newFrequency )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set()");
   tDiaResult retCode = DIA_FAILED;

   if (mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID)
   {
      midw_tunerfi_tclMsgFID_TUN_S_DIRECT_FREQUENCYMethodStart oData;
      oData.u32Frequency = newFrequency;
      oData.u8Band.enType = getBand(newFrequency);

      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set - frequency=%d / 0x%08x band=%d.",
            oData.u32Frequency, oData.u32Frequency, oData.u8Band.enType);

      //put the data in message
      fi_tclVisitorMessage oMessage(oData);
      mpSrvPlugin->vInitServiceData(oMessage, MIDW_TUNERFI_C_U16_FID_TUN_S_DIRECT_FREQUENCY, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

      if (mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE)
      {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set - transmitMessage success.");
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set - transmitMessage failed.");
      }
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set - getRegistrationID failed.");
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Set - getRegistrationID == AMT_C_U16_REGID_INVALID.");
   }
   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get (   )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get()");
   tDiaResult retCode = DIA_FAILED;

   if (mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID) {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get - send message.");

      gm_tclEmptyMessage oMessage(
            mAppID,
            mpSrvPlugin->getClientID(),
            mpSrvPlugin->getRegistrationID(),
            0,
            mpSrvPlugin->getServiceID(),
            MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO,
            CCA_C_U8_OPCODE_UPREG);

      if (mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE) {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get - transmitMessage success.");
      } else {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get - transmitMessage failed.");
      }
   } else {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get - getRegistrationID failed.");
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerBand_Get - getRegistrationID == AMT_C_U16_REGID_INVALID.");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get ( )
{
    dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get()");
    tDiaResult retCode = DIA_FAILED;

    if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
    {
       DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get - send message.");
       gm_tclEmptyMessage oMessage (
          mAppID,
          mpSrvPlugin->getClientID(),
          mpSrvPlugin->getRegistrationID(),
          0,
          mpSrvPlugin->getServiceID(),
          MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO,
          CCA_C_U8_OPCODE_UPREG
       );

       if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ){
          retCode = DIA_SUCCESS;
          DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get - transmitMessage success.");
       } else {
          DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get - transmitMessage failed.");
       }
    } else {
       DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get - getRegistrationID failed.");
       DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFMRDSProgramInformation_Get - getRegistrationID == AMT_C_U16_REGID_INVALID.");
    }

    return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get ( )
{
    dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get()");
    tDiaResult retCode = DIA_FAILED;

    if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
    {
       DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get - send message.");
       midw_tunerfi_tclMsgFID_S_TUN_GET_FIELD_STRENGTHMethodStart oData; //lint !e1502 Warning: defined object nonstatic data members

       //put the data in message
       fi_tclVisitorMessage oMessage(oData);
       mpSrvPlugin->vInitServiceData(oMessage, MIDW_TUNERFI_C_U16_FID_S_TUN_GET_FIELD_STRENGTH, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

       if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ){
          retCode = DIA_SUCCESS;
          DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get - transmitMessage success.");
       } else {
          DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get - transmitMessage failed.");
       }
    } else {
       DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get - getRegistrationID failed.");
       DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_TunerFieldStrength_Get - getRegistrationID == AMT_C_U16_REGID_INVALID.");
    }

    return retCode;
}
tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get()
{
	dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get()");
	tDiaResult retCode = DIA_FAILED;

	if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
	{
		DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get - send message.");
		gm_tclEmptyMessage oMessage (
				mAppID,
				mpSrvPlugin->getClientID(),
				mpSrvPlugin->getRegistrationID(),
				0,
				mpSrvPlugin->getServiceID(),
				MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO,
				CCA_C_U8_OPCODE_UPREG
		);
		if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ){
			retCode = DIA_SUCCESS;
			DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get - transmitMessage success.");
		} else {
			DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get - transmitMessage failed.");
		}
	} else {
		DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get - getRegistrationID failed.");
		DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFMWB_TunerBandFieldStrength_Get - getRegistrationID == AMT_C_U16_REGID_INVALID.");
	}

	return retCode;
}



#if 0
tDiaResult
dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek( dia_eTunerAMFMModulation modulation, dia_eTunerAMFMSeekDirection direction, dia_eTunerAMFMAction action )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek()");
   tDiaResult retCode = DIA_FAILED;

   if ((DIA_EN_TUNERAMFM_MODULATION_FM==modulation) || (DIA_EN_TUNERAMFM_MODULATION_AM==modulation))
   {
      if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - getRegistrationID OK");
         midw_tunerfi_tclMsgFID_TUN_S_ATSEEK_WITH_TUNERHEADERMethodStart oData;

         oData.u8TunerHeader.enType = midw_fi_tcl_e8_Tun_TunerHeader::FI_EN_TUN_FOREGROUND;     /* FM tuner */
         oData.u32Frequency = 0; /* start frequency = current frequency */

         if (DIA_EN_TUNERAMFM_ACT_START==action)
         {
            switch(direction)
            {
               case DIA_EN_TUNERAMFM_DIR_UP:
                  oData.u8TunerSeek.enType = midw_fi_tcl_e8_Tun_TunerSeek::FI_EN_TUN_SEEKTYPE_UP_AUTO;
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - FI_EN_TUN_SEEKTYPE_UP_AUTO.");
               break;
               case DIA_EN_TUNERAMFM_DIR_DOWN:
                  oData.u8TunerSeek.enType = midw_fi_tcl_e8_Tun_TunerSeek::FI_EN_TUN_SEEKTYPE_DOWN_AUTO;
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - FI_EN_TUN_SEEKTYPE_DOWN_AUTO.");
               break;
               default:
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - Unknown direction = 0x%02x", direction);
                  return retCode;
            }
         }
         else if (DIA_EN_TUNERAMFM_ACT_STOP==action)
         {
            DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - FI_EN_TUN_SEEKTYPE_OFF.");
            oData.u8TunerSeek.enType = midw_fi_tcl_e8_Tun_TunerSeek::FI_EN_TUN_SEEKTYPE_OFF;
         }
         else
         {
            DIA_TR_ERR("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - unknown action 0x%02x.", action);
            return DIA_FAILED;
         }

         //put the data in message
         fi_tclVisitorMessage oMessage(oData);
         mpSrvPlugin->vInitServiceData(oMessage, MIDW_TUNERFI_C_U16_FID_TUN_S_ATSEEK_WITH_TUNERHEADER, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

         if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE )
         {
            DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - transmitMessage success.");

            if (DIA_EN_TUNERAMFM_ACT_START==action)
            {
               if (DIA_SUCCESS==tunerAMFM_FreqSeekStartNotification())
               {
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - notification started.");
                  retCode = DIA_SUCCESS;
               }
               else
               {
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - notification NOT started.");
               }
            }
            else if (DIA_EN_TUNERAMFM_ACT_STOP==action)
            {
               if (DIA_SUCCESS==tunerAMFM_FreqSeekStopNotification())
               {
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - notification started.");
                  retCode = DIA_SUCCESS;
               }
               else
               {
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - notification NOT started.");
               }
            }
            else
            {
               DIA_TR_ERR("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - unknown action 0x%02x.", action);
               return DIA_FAILED;
            }
         }
         else
         {
            DIA_TR_ERR("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - transmitMessage failed.");
         }
      }
      else
      {
         DIA_TR_ERR("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - getRegistrationID failed.");
         DIA_TR_ERR("dia_SAFeatureTunerAMFM::tunerAMFM_Frequency_Seek - getRegistrationID == AMT_C_U16_REGID_INVALID.");
      }
   }
   else
   {
      /* do nothing */
   }

   return retCode;
}
#endif

#if 0
tDiaResult dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification()
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification()");
   tDiaResult retCode = DIA_FAILED;

   if (false==mbUpRegATSEEKActive)
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - UpReg not activated.");

      if (mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID)
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - send message.");
         gm_tclEmptyMessage oMessage(
               mAppID,
               mpSrvPlugin->getClientID(),
               mpSrvPlugin->getRegistrationID(),
               0,
               mpSrvPlugin->getServiceID(),
               MIDW_TUNERFI_C_U16_FID_TUN_G_ATSEEK,
               CCA_C_U8_OPCODE_UPREG
         );

         if (mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE)
         {
            mLastActionFreqSeekFM = DIA_EN_TUNERAMFM_ACT_START;
            retCode = DIA_SUCCESS;
            DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - transmitMessage success.");
         }
         else
         {
            DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - transmitMessage failed.");
         }
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - getRegistrationID failed.");
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - getRegistrationID == AMT_C_U16_REGID_INVALID.");
      }

      if (DIA_SUCCESS==retCode)
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - UpReg activation successful.");
         mbUpRegATSEEKActive = true;
      }
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStartNotification - UpReg already activated.");
      retCode = DIA_SUCCESS;
   }

   return retCode;
}
#endif

tDiaResult dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification()
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification()");
   tDiaResult retCode = DIA_FAILED;

   if (true==mbUpRegATSEEKActive)
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - UpReg activated.");

      if (mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID)
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - send message.");
         gm_tclEmptyMessage oMessage(
               mAppID,
               mpSrvPlugin->getClientID(),
               mpSrvPlugin->getRegistrationID(),
               0,
               mpSrvPlugin->getServiceID(),
               MIDW_TUNERFI_C_U16_FID_TUN_G_ATSEEK,
               CCA_C_U8_OPCODE_RELUPREG
         );

         if (mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE)
         {
            mLastActionFreqSeekFM = DIA_EN_TUNERAMFM_ACT_STOP;
            retCode = DIA_SUCCESS;
            DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - transmitMessage success.");
         }
         else
         {
            DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - transmitMessage failed.");
         }
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - getRegistrationID failed.");
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - getRegistrationID == AMT_C_U16_REGID_INVALID.");
      }

      if (DIA_SUCCESS==retCode)
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - UpReg deactivation successful.");
         mbUpRegATSEEKActive = false;
      }
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerAMFM_FreqSeekStopNotification - UpReg not activated.");
      retCode = DIA_SUCCESS;
   }

   return retCode;
}


//==============================================================================
//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set ( tU32 frequency )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set()");
   tDiaResult retCode = DIA_FAILED;

   if (mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID) {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set - send message.");

      midw_tunerfi_tclMsgFID_TUN_S_SELECT_DRM_AUDIOPRGMMethodStart oData;
      oData.u32Frequency = frequency;
      oData.u32DRMServiceId = 0;

      //put the data in message
      fi_tclVisitorMessage oMessage(oData);
      mpSrvPlugin->vInitServiceData(oMessage, MIDW_TUNERFI_C_U16_FID_TUN_S_SELECT_DRM_AUDIOPRGM, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

      if (mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE) {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set - transmitMessage success.");
      } else {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set - transmitMessage failed.");
      }
   } else {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set - getRegistrationID failed.");
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_EnsembleFrequency_Set - getRegistrationID == AMT_C_U16_REGID_INVALID.");
   }

   return retCode;
}
//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set  ( tU32 serviceID )
{
    dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set()");
   tDiaResult retCode = DIA_FAILED;

   if (mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID) {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set - send message.");

      midw_tunerfi_tclMsgFID_TUN_S_SELECT_DRM_AUDIOPRGMMethodStart oData;
      oData.u32DRMServiceId = serviceID;
      oData.u32Frequency = 0;

      //put the data in message
      fi_tclVisitorMessage oMessage(oData);
      mpSrvPlugin->vInitServiceData(oMessage, MIDW_TUNERFI_C_U16_FID_TUN_S_SELECT_DRM_AUDIOPRGM, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

      if (mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE) {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set - transmitMessage success.");
      } else {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set - transmitMessage failed.");
      }
   } else {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set - getRegistrationID failed.");
      DIA_TR_INF("dia_SAFeatureTunerAMFM::tunerDRM_ServiceID_Set - getRegistrationID == AMT_C_U16_REGID_INVALID.");
   }

   return retCode;
}

//==============================================================================

tVoid
dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results ------------------------------");

   if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results poMessage OK.");

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
      {
         dia_ITunerAMFMListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ITunerAMFMListener>(&pListener);
         if ( pListener )
         {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_tunerfi_tclMsgFID_TUN_S_ATSEEK_WITH_TUNERHEADERMethodResult oCompResp;
            (tVoid) oResultMsg.s32GetData(oCompResp);

            /* status of the client request */
            midw_fi_tcl_e8_Tun_ResponseStatus resStatus = oCompResp.TunerServer_StatusResponse.u8TunerStatus;

            /* status of tuner server */
            midw_fi_tcl_e8_Tun_ResponseState resState = oCompResp.TunerServer_StatusResponse.u8TunerState;

            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results resStatus=0x%02x", resStatus.enType);
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results resState=0x%02x", resState.enType);

            /* check status of request AND status of tuner server */
            if ( (midw_fi_tcl_e8_Tun_ResponseStatus::FI_EN_TUN_RES_REQ_SUCCEED == resStatus.enType)  &&
                 (midw_fi_tcl_e8_Tun_ResponseState::FI_EN_TUN_TUN_STATE_NORMAL == resState.enType)
               )
            {
               /* success */
               if (DIA_EN_TUNERAMFM_ACT_START==mLastActionFreqSeekFM)
               {
                  pListener->vOnTunerAMFMFrequencySeekStarted(DIA_SUCCESS);
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results started SUCCESS");
               }
               else if (DIA_EN_TUNERAMFM_ACT_STOP==mLastActionFreqSeekFM)
               {
                  pListener->vOnTunerAMFMFrequencySeekStopped(DIA_SUCCESS);
                  DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results SUCCESS");
               }
               else
               {
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results 1 unknown last action.");
               }
            }
            else
            {
               /* failure */
               if (DIA_EN_TUNERAMFM_ACT_START==mLastActionFreqSeekFM)
               {
                  pListener->vOnTunerAMFMFrequencySeekStopped(DIA_SUCCESS);
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results FAILED");
               }
               else if (DIA_EN_TUNERAMFM_ACT_STOP==mLastActionFreqSeekFM)
               {
                  pListener->vOnTunerAMFMFrequencySeekStopped(DIA_SUCCESS);
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results FAILED");
               }
               else
               {
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results 2 unknown last action.");
               }
            }
         }
         else
         {
            DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results pListener null");
         }
      }
      else if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR )
      {
           amt_tclServiceDataError oErrorMsg( poMessage );
           tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
           DIA_TR_ERR("dia_SAFeatureTunerAMFM --- opcode error received with error code: %d", u16ErrorCode);
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results UNKNOWN OPCODE==%d", poMessage->u8GetOpCode());
      }

      (tVoid) poMessage->bDelete();
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEKWITHTUNERHEADER_Results poMessage EMPTY!");
   }
}

/*
 * Handles the response for all MIDW_TUNERFI_C_U16_FID_TUN_G_ATSTATIONINFO requests
 * It sends the results to all possible Service Handler and only the active ones will work on this message
 * Implemented for:
 * - tunerAMFM_Frequency_Get
 * - tunerAMFM_TunerBand_Get
 * - tunerAMFM_TunerFMRDSProgramInformation_Get
 * - tunerAMFMWB_TunerFieldStrength_Get
 */
tVoid
dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results ------------------------------");

   if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS )
      {
         dia_ITunerAMFMListener* pListener = OSAL_NULL;
         dia_ITunerAMFMPrjListener* pListenerPrj = OSAL_NULL;
         querySysAdapterListener<dia_ITunerAMFMListener>(&pListener);
         querySysAdapterListener<dia_ITunerAMFMPrjListener>(&pListenerPrj);
         if ( pListener &&  pListenerPrj)
         {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_tunerfi_tclMsgFID_TUN_G_ATSTATIONINFOStatus oCompResp;
            (tVoid) oResultMsg.s32GetData(oCompResp);
            dia_eBandAMFM e8Band = DIA_EN_AMFMTUNER_BAND_UNKNOWN;
            dia_eBandAMFMWB e8BandPrj = DIA_EN_AMFMWBTUNER_BAND_UNKNOWN;

            // tunerAMFM_TunerBand_Get
            midw_fi_tcl_e8_Tun_TunerBand band = oCompResp.Atstationinfo.e8Band;
            if (band.enType == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM) {
            	DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND = FM.");
            	e8Band = DIA_EN_AMFMTUNER_BAND_FM;
            } else if (band.enType == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW) {
            	DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND = MW/AM.");
            	e8Band = DIA_EN_AMFMTUNER_BAND_AM;
            } else {
            	DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND UNKNOWN!");
            }
            switch(band.enType)
            {
				case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM:
				{
					DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND = FM.");
					e8BandPrj = DIA_EN_AMFMWBTUNER_BAND_FM;
				}
				break;
				case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW:
				{
					DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND = MW/AM.");
					e8BandPrj = DIA_EN_AMFMWBTUNER_BAND_AM;
				}
				break;
				case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_WB:
				{
					DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND = WB.");
					e8BandPrj = DIA_EN_AMFMWBTUNER_BAND_WB;
				}
				break;
				default:
				{
					DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results BAND UNKNOWN!");
				}
				break;
            }
            tU32 frequency = (oCompResp.Atstationinfo.u32Frequency) * (1000); // Multiplication by 1000 as Tuner provides frequency in KHz and positive response should be in Hertz
            tU16 PICode = oCompResp.Atstationinfo.u16PI;
            tU8 fieldStrength = oCompResp.Atstationinfo.u8FieldStrngth;

	        tU32 serviceID = oCompResp.Atstationinfo.u32DRMServiceId ;

            //trace
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results e8Band = 0x%02x", e8Band);
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results frequency = %d", frequency);
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results PICode = 0x%04x", PICode);
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results fieldStrength = 0x%02x", fieldStrength);
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results serviceID = 0x%04x", serviceID);

            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results e8BandPrj = 0x%02x", e8BandPrj);
            pListener->vOnTunerAMFMBand_Get(e8Band);
            pListener->vOnGetRadioTuner1Status(e8Band, frequency, fieldStrength);
			pListener->vOnTunerAMFMFieldStrength_Get(fieldStrength);
			pListenerPrj->vOnTunerAMFMFWB_BandFieldStrength_Get(e8BandPrj,fieldStrength);
            // tunerAMFM_TunerFMRDSProgramInformation_Get
            pListener->vOnTunerFMRDSProgramInformation_Get(PICode);

            // tunerAMFM_Frequency_Get
            pListener->vOnTunerAMFMFrequency_Get(frequency);

			// DRM: Frequency
            pListener->vOnTunerDRMGetFrequency(frequency);

			// DRM : Service ID
			pListener->vOnTunerDRMGetServiceID(serviceID);

            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSTATIONINFO_Results DONE!");
         }
      }
      else if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR )
      {
           amt_tclServiceDataError oErrorMsg( poMessage );
           tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
           DIA_TR_ERR("dia_SAFeatureTunerAMFM --- opcode error received with error code: %d", u16ErrorCode);
      }
      (tVoid) poMessage->bDelete();
   }
}

tVoid
dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEK_Results ( amt_tclServiceData* poMessage )
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEK_Results ------------------------------");

   if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS )
      {
         dia_ITunerAMFMListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ITunerAMFMListener>(&pListener);
         if ( pListener )
         {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_tunerfi_tclMsgFID_TUN_G_ATSEEKStatus oCompResp;
            (tVoid) oResultMsg.s32GetData(oCompResp);

            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEK_Results SeekState = 0x%02x", oCompResp.e8SeekState.enType);

            if (midw_fi_tcl_e8_Tun_TunerSeek::FI_EN_TUN_SEEKTYPE_OFF==oCompResp.e8SeekState.enType)
            {
               DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEK_Results SeekState = SEEKTYPE_OFF");
               tunerAMFM_FreqSeekStopNotification();
               pListener->vOnTunerAMFMFrequencySeekResult(DIA_SUCCESS);
            }
            else
            {
               DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFM_ATSEEK_Results SeekState NOT SEEKTYPE_OFF");

               /* ignore this state, do not send reply on CAN */
            }
         }
      }
      else if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR )
      {
           amt_tclServiceDataError oErrorMsg( poMessage );
           tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
           DIA_TR_ERR("dia_SAFeatureTunerAMFM --- opcode error received with error code: %d", u16ErrorCode);
      }
      (tVoid) poMessage->bDelete();
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set");

   if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set poMessage OK.");

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
      {
         dia_ITunerAMFMListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ITunerAMFMListener>(&pListener);
         if ( pListener )
         {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_tunerfi_tclMsgFID_TUN_S_DIRECT_FREQUENCYMethodResult oCompResp;
            (tVoid) oResultMsg.s32GetData(oCompResp);

            tU8 result = oCompResp.TunerServer_StatusResponse.u8TunerStatus.enType;
            if ( midw_fi_tcl_e8_Tun_ResponseStatus::FI_EN_TUN_RES_REQ_SUCCEED == result)
            {
               DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set SUCCESS!");
               pListener->vOnTunerAMFMFrequency_Set(DIA_SUCCESS);
            }
            else
            {
               DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set FAILED!");
               if (midw_fi_tcl_e8_Tun_ResponseStatus::FI_EN_TUN_RES_REQ_FAIL == result)
               {
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set u8TunerStatus=FI_EN_TUN_RES_REQ_FAIL.");
               }
               else
               {
                  DIA_TR_ERR("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set u8TunerStatus=%d.", result);
               }
               pListener->vOnTunerAMFMFrequency_Set(DIA_FAILED);
            }
         }
      }
      else if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR )
      {
           amt_tclServiceDataError oErrorMsg( poMessage );
           tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
           DIA_TR_ERR("dia_SAFeatureTunerAMFM --- opcode error received with error code: %d", u16ErrorCode);
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set UnknownOpCode=%d!", poMessage->u8GetOpCode());
      }

      (tVoid) poMessage->bDelete();
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFrequency_Set poMessage EMPTY!");
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get");

   if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get poMessage OK.");

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
      {
         dia_ITunerAMFMListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ITunerAMFMListener>(&pListener);
         if ( pListener )
         {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_tunerfi_tclMsgFID_S_TUN_GET_FIELD_STRENGTHMethodResult oCompResp;
            (tVoid) oResultMsg.s32GetData(oCompResp);

            tU8 tunerFieldStrength = oCompResp.Tuner_Field_Strength;

            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get tunerFieldStrength=0x%02x", tunerFieldStrength);

            pListener->vOnTunerAMFMFieldStrength_Get(tunerFieldStrength);
            pListener->vOnGetTunerAerialRadio2FieldStrength(tunerFieldStrength);
            pListener->vOnGetTunerAerialRadio1FieldStrength(tunerFieldStrength);
            DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get SUCCESS!");
         }
      }
      else if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR )
      {
           amt_tclServiceDataError oErrorMsg( poMessage );
           tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
           DIA_TR_INF( "---opcode error received with error code:%u", u16ErrorCode);
      } else {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get poMessage->u8GetOpCode() UNKNOWN => CODE==%d!", poMessage->u8GetOpCode());
      }

      (tVoid) poMessage->bDelete();
   } else {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerAMFMFieldStrength_Get poMessage EMPTY!");
   }
}

//-----------------------------------------------------------------------------

tVoid
dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus ( amt_tclServiceData* poMessage ) const
{
   dia_tclFnctTrace trc("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus");

   if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus poMessage OK.");

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
      {
         dia_ITunerAMFMListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ITunerAMFMListener>(&pListener);
         if ( pListener )
         {
            fi_tclVisitorMessage oResultMsg(poMessage);
            midw_tunerfi_tclMsgFID_TUN_S_SELECT_DRM_AUDIOPRGMMethodResult oCompResp;
            (tVoid) oResultMsg.s32GetData(oCompResp);

            tU8 result = oCompResp.TunerServer_StatusResponse.u8TunerStatus.enType;
            if ( result == midw_fi_tcl_e8_Tun_ResponseStatus::FI_EN_TUN_RES_REQ_SUCCEED ){
               pListener->vOnTunerDRMEnsembleFrequency_Set(DIA_SUCCESS);
			   pListener->vOnTunerDRMServiceID_Set(DIA_SUCCESS);
               DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus SUCCESS!");
            } else {
               pListener->vOnTunerDRMEnsembleFrequency_Set(DIA_FAILED);
			   pListener->vOnTunerDRMServiceID_Set(DIA_FAILED);
               DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus FAILED!");
               DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus u8TunerStatus==%d.",result);
            }

         }
      }
      else if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR )
      {
           amt_tclServiceDataError oErrorMsg( poMessage );
           tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
           DIA_TR_ERR("dia_SAFeatureTunerAMFM --- opcode error received with error code: %d",u16ErrorCode);
      } else {
         DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus poMessage->u8GetOpCode() UNKNOWN => CODE==%d!",poMessage->u8GetOpCode());
      }

      (tVoid) poMessage->bDelete();
   } else {
      DIA_TR_INF("dia_SAFeatureTunerAMFM::vHandleTunerDRM_Staus poMessage EMPTY!");
   }
}

//-----------------------------------------------------------------------------
