/*!
 * \file       dia_SAFeatureRVC.cpp
 *
 * \brief      Service Adapter for RVC
 *
 * \details
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON_CORE__
#include "common/depricated/dia_common_core.h"
#endif

#include "dia_SAFeatureRVC.h"

#define DIAGLIB_INCLUDE_ROUTINE_CONTROL
#include "diaglib_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

// MESSAGE MAP:
// Such a map and an own handler is needed for every service you are accessing!
// Enter the function IDs (FID) and the corresponding functions here.
// The function will be called when a message with the corresponding FID arrives

// FI is generated here:
//    <view>_GEN/ai_projects/generated/components/rvc_fi/rvc_cca_fi/source/rvc_ccafifctid.h
#define RVC_FI_S_IMPORT_INTERFACE_RVC_CCAFI_FUNCTIONIDS        /* needed by rvc_fi_if.h */
#define RVC_FI_S_IMPORT_INTERFACE_RVC_CCAFI_TYPES              /* needed by rvc_fi_if.h */
#include "rvc_fi/rvc_fi_if.h"


#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC                 /* needed for gm_tclEmptyMessage */
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

BEGIN_MSG_MAP(dia_SAFeatureRVC, dia_SystemAdapterFeature)
   ON_MESSAGE(RVC_CCAFI_C_U16_CAMERA_PWRSUPPLYSTATUS,          vHandleRVCConnectionInfo)
   ON_MESSAGE(RVC_CCAFI_C_U16_MICROPHONE_PWRSUPPLYSTATUS,      vHandleMicrophoneConnectionInfo)
   ON_MESSAGE(RVC_CCAFI_C_U16_VIDEOSIGNALSTATUS,               vHandleCVBSConnectionStatus)
END_MSG_MAP()

//------------------------------------------------------------------------------

dia_SAFeatureRVC::dia_SAFeatureRVC (
      tCString name,    // feature name
      dia_SystemAdapterServicePlugin& pSrvPlugin
   )
    : dia_SystemAdapterFeature(name,pSrvPlugin)
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SAFeatureRVC::dia_SAFeatureRVC");
#endif
}

//-----------------------------------------------------------------------------

dia_SAFeatureRVC::~dia_SAFeatureRVC ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureRVC::getRVCConnectionInfo ( void )
{
    dia_tclFnctTrace trc("dia_SAFeatureRVC::getRVCConnectionInfo()");
    tDiaResult retCode = DIA_FAILED;

    if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
    {
       DIA_TR_INF("dia_SAFeatureRVC::getRVCConnectionInfo - send message.");
       gm_tclEmptyMessage oMessage (
          mAppID,
          mpSrvPlugin->getClientID(),
          mpSrvPlugin->getRegistrationID(),
          0,
          mpSrvPlugin->getServiceID(),
          RVC_CCAFI_C_U16_CAMERA_PWRSUPPLYSTATUS,
          CCA_C_U8_OPCODE_GET
       );

       if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE )
       {
          retCode = DIA_SUCCESS;
          DIA_TR_INF("dia_SAFeatureRVC::getRVCConnectionInfo - transmitMessage success.");
       }
       else
       {
          DIA_TR_INF("dia_SAFeatureRVC::getRVCConnectionInfo - transmitMessage failed.");
       }
    }
    else
    {
       DIA_TR_INF("dia_SAFeatureRVC::getRVCConnectionInfo - getRegistrationID failed.");
       DIA_TR_INF("dia_SAFeatureRVC::getRVCConnectionInfo - getRegistrationID == AMT_C_U16_REGID_INVALID.");
    }

    return retCode;
}

tDiaResult
dia_SAFeatureRVC::getMicrophoneConnectionInfo ( tU16 micID )
{
   dia_tclFnctTrace trc("dia_SAFeatureRVC::getMicrophoneConnectionInfo()");

   //A-IVI: there is only one microphone, hence micID is ignored.
   DIA_TR_INF("dia_SAFeatureRVC::getMicrophoneConnectionInfo micID=0x%02X", micID);

   tDiaResult retCode = DIA_FAILED;

   if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
   {
      DIA_TR_INF("dia_SAFeatureRVC::getMicrophoneConnectionInfo - send message.");
      gm_tclEmptyMessage oMessage (
         mAppID,
         mpSrvPlugin->getClientID(),
         mpSrvPlugin->getRegistrationID(),
         0,
         mpSrvPlugin->getServiceID(),
         RVC_CCAFI_C_U16_MICROPHONE_PWRSUPPLYSTATUS,
         CCA_C_U8_OPCODE_GET
      );

      if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE )
      {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia_SAFeatureRVC::getMicrophoneConnectionInfo - transmitMessage success.");
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureRVC::getMicrophoneConnectionInfo - transmitMessage failed.");
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureRVC::getMicrophoneConnectionInfo - getRegistrationID failed.");
      DIA_TR_ERR("dia_SAFeatureRVC::getMicrophoneConnectionInfo - getRegistrationID == AMT_C_U16_REGID_INVALID.");
   }

   return retCode;
}

tDiaResult
dia_SAFeatureRVC::getCVBSConnectionStatus ( void )
{
   dia_tclFnctTrace trc("dia_SAFeatureRVC::vHandleCVBSConnectionStatus()");

   tDiaResult retCode = DIA_FAILED;

   if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
   {
      DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus - send message.");
      gm_tclEmptyMessage oMessage (
         mAppID,
         mpSrvPlugin->getClientID(),
         mpSrvPlugin->getRegistrationID(),
         0,
         mpSrvPlugin->getServiceID(),
         RVC_CCAFI_C_U16_VIDEOSIGNALSTATUS,
         CCA_C_U8_OPCODE_GET
      );

      if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE )
      {
         retCode = DIA_SUCCESS;
         DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus - transmitMessage success.");
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus - transmitMessage failed.");
      }
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureRVC::vHandleCVBSConnectionStatus - getRegistrationID failed.");
      DIA_TR_ERR("dia_SAFeatureRVC::vHandleCVBSConnectionStatus - getRegistrationID == AMT_C_U16_REGID_INVALID.");
   }

   return retCode;
}

//===========================================================================================================
//===========================================================================================================

void
dia_SAFeatureRVC::vHandleRVCConnectionInfo(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace trc("dia_SAFeatureRVC::vHandleRVCConnectionInfo");

   if (poMessage)
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo poMessage OK.");

      if (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS)
      {
         dia_IRVCListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_IRVCListener>(&pListener);

         if (pListener)
         {
            fi_tclVisitorMessage oResultMsg(poMessage);

            rvc_ccafi_tclMsgCamera_PwrSupplyStatusStatus oCompResp;     /* midw_ext_rvcfi_tclMsgCamera_PwrSupplyStatusStatus */
            (void) oResultMsg.s32GetData(oCompResp);

            tU32 rvcCurrent = oCompResp.current;
            DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo rvcCurrent=0x%04X", rvcCurrent);
#if 0
            rvcCurrent *= 10;
            DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo rvcCurrent (1/10mA) = %d", rvcCurrent);
#endif
            pListener->vOnRVCPowerSupplyCurrent(rvcCurrent);

            rvc_fi_tcl_e8_Camera_PowerSupplyStatus rvcPwrSupplyStat = oCompResp.status;      /* midw_ext_fi_tcl_e8_Camera_PowerSupplyStatus */
            dia_eRVC_pwrSupplyStatus pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_UNKNOWN;
            switch(rvcPwrSupplyStat.enType)
            {
               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_UNKNOWN:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_UNKNOWN;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo SUPPLY_UNKNOWN!");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_OK:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_OK;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo SUPPLY_OK!");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_OPENLOAD:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_OPENLOAD;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo SUPPLY_OPENLOAD!");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_SHORTCIRCUIT:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_SHORTCIRCUIT;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo SUPPLY_SHORTCIRCUIT!");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_DEACTIVATED:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_UNKNOWN;
                  DIA_TR_INF("SUPPLY_DEACTIVATED mapped to DIA_EN_RVC_PWR_SUPPLY_UNKNOWN");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_DEACTIVATED_LOWVOLTAGE:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_UNKNOWN;
                  DIA_TR_INF("SUPPLY_DEACTIVATED_LOWVOLTAGE mapped to DIA_EN_RVC_PWR_SUPPLY_UNKNOWN");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_DEACTIVATED_HIGHVOLTAGE:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_UNKNOWN;
                  DIA_TR_ERR("SUPPLY_DEACTIVATED_HIGHVOLTAGE mapped to DIA_EN_RVC_PWR_SUPPLY_UNKNOWN");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_SHORTTOGROUND:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_SHORTTOGROUND;
                  DIA_TR_ERR("dia_SAFeatureRVC::vHandleRVCConnectionInfo DIA_EN_RVC_PWR_SUPPLY_SHORTTOGROUND");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_SHORTTOBATTERY:
                  pwrSupplyStatus = DIA_EN_RVC_PWR_SUPPLY_SHORTTOBATTERY;
                  DIA_TR_ERR("dia_SAFeatureRVC::vHandleRVCConnectionInfo DIA_EN_RVC_PWR_SUPPLY_SHORTTOBATTERY");
               break;

               case rvc_fi_tcl_e8_Camera_PowerSupplyStatus::FI_EN_NUMBEROFITEMS:
                  DIA_TR_ERR("dia_SAFeatureRVC::vHandleRVCConnectionInfo ERROR SUPPLY_COUNT!");
               break;

               default:
                  DIA_TR_ERR("dia_SAFeatureRVC::vHandleRVCConnectionInfo CAN'T MATCH rvcPwrSupplyStat.enType(0x%02x)!", rvcPwrSupplyStat.enType);
               break;
            }
            pListener->vOnRVCPowerSupplyStatus(pwrSupplyStatus);

            DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo SUCCESS!");
         }
      }
      else if (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR)
      {
         amt_tclServiceDataError oErrorMsg(poMessage);
         tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
         DIA_TR_INF("dia_SAFeatureRVC ---opcode error received with error code 0x%04X", u16ErrorCode);
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo poMessage->u8GetOpCode() UNKNOWN => CODE==%d!", poMessage->u8GetOpCode());
      }

      (void) poMessage->bDelete();
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureRVC::vHandleRVCConnectionInfo poMessage EMPTY!");
   }
}

void
dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace trc("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo");

   if (poMessage)
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo poMessage OK.");

      if (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS)
      {
         dia_IMicrophoneListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_IMicrophoneListener>(&pListener);

         if (pListener)
         {
            tU16 micId;
            dia_MicrophoneInfo micInfo;
            fi_tclVisitorMessage oResultMsg(poMessage);

            rvc_ccafi_tclMsgMicrophone_PwrSupplyStatusStatus oCompResp;
            (void) oResultMsg.s32GetData(oCompResp);

            //parameter no. 1
            if (oCompResp.current > 0xFFFFU)
            {
               DIA_TR_ERR("### vHandleMicrophoneConnectionInfo microphoneCurrent=0x%08X Value greater than 0xFFFF.", oCompResp.current);
            }
            micInfo.mCurrent = (tU16)(0xFFFFU & oCompResp.current);
            DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo microphoneCurrent=0x%04X (dec %d)", micInfo.mCurrent, micInfo.mCurrent);

            //parameter no. 2
            rvc_fi_tcl_e8_Microphone_PowerSupplyStatus micPwrSupplyStat = oCompResp.status;
            micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN;
            switch(micPwrSupplyStat.enType)
            {
               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_UNKNOWN:
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo SUPPLY_UNKNOWN!");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_OK:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_OK;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo SUPPLY_OK!");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_OPENLOAD:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_OPEN_LOAD;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo SUPPLY_OPENLOAD!");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTCIRCUIT:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo FI_EN_SHORTCIRCUIT!");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_DEACTIVATED:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo FI_EN_DEACTIVATED!");
                  DIA_TR_INF("FI_EN_DEACTIVATED mapped to STATUS_UNKNOWN");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_DEACTIVATED_LOWVOLTAGE:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo FI_EN_DEACTIVATED_LOWVOLTAGE!");
                  DIA_TR_INF("FI_EN_DEACTIVATED_LOWVOLTAGE mapped to STATUS_UNKNOWN");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_DEACTIVATED_HIGHVOLTAGE:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_UNKNOWN;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo FI_EN_DEACTIVATED_HIGHVOLTAGE!");
                  DIA_TR_INF("FI_EN_DEACTIVATED_HIGHVOLTAGE mapped to STATUS_UNKNOWN");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTTOGROUND:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_GROUND;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo FI_EN_SHORTTOGROUND!");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_SHORTTOBATTERY:
                  micInfo.mConnectionStatus = DIA_EN_MICROPHONE_CONNECTION_STATUS_SHORT_CIRCUIT_BATTERY;
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo FI_EN_SHORTTOBATTERY!");
               break;

               case rvc_fi_tcl_e8_Microphone_PowerSupplyStatus::FI_EN_NUMBEROFITEMS:
                  DIA_TR_ERR("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo ERROR SUPPLY_COUNT!");
               break;

               default:
                  DIA_TR_ERR("### ERROR CAN'T MATCH micPwrSupplyStat.enType(0x%02x)!", micPwrSupplyStat.enType);
               break;
            }

            //parameter no. 3 (microphone id)
            micId = oCompResp.pwrSupplyID;
            DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo micId=0x%04X", micId);

            if (1!=micId)
            {
               DIA_TR_ERR("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo Error Incorrect mic ID");
               micId = 1;
            }

            //send to listener
            pListener->vOnMicrophoneConnectionInfo(micId, micInfo);

            DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo SUCCESS!");
         }
      }
      else if (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR)
      {
         amt_tclServiceDataError oErrorMsg(poMessage);
         tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
         DIA_TR_INF("dia_SAFeatureRVC ---opcode error received with error code 0x%04X", u16ErrorCode);
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo poMessage->u8GetOpCode() UNKNOWN => CODE==%d!", poMessage->u8GetOpCode());
      }

      (void) poMessage->bDelete();
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureRVC::vHandleMicrophoneConnectionInfo poMessage EMPTY!");
   }
}

void
dia_SAFeatureRVC::vHandleCVBSConnectionStatus(amt_tclServiceData* poMessage) const
{
   dia_tclFnctTrace trc("dia_SAFeatureRVC::vHandleCVBSConnectionStatus");

   if (poMessage)
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
      DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus poMessage OK.");

      if (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS)
      {
         dia_ICVBSConnectionListener* pListener = OSAL_NULL;
         querySysAdapterListener<dia_ICVBSConnectionListener>(&pListener);

         if (pListener)
         {
            dia_eCVBSConnectionInfo status = DIA_EN_CVBS_CONNECTION_INFO_UNKNOWN;
            fi_tclVisitorMessage oResultMsg(poMessage);

            rvc_ccafi_tclMsgVideoSignalStatusStatus oCompResp;
            (void) oResultMsg.s32GetData(oCompResp);

            rvc_fi_tcl_e8_RVC_SignalStatus videoSignalStatus = oCompResp.e8StatusVideoSignal;
			
            switch(videoSignalStatus.enType)
            {
               case rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_SIGNAL_OK:
			   {
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus FI_EN_SIGNAL_OK!");
				  status = DIA_EN_CVBS_CONNECTION_INFO_CONNECTED;
			   }
               break;

               case rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CONFIRMED_FAILURE:
               case rvc_fi_tcl_e8_RVC_SignalStatus::FI_EN_CURRENT_FAILURE:
			   {
                  DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus FI_EN_CONFIRMED_FAILURE or FI_EN_CURRENT_FAILURE!");
				  status = DIA_EN_CVBS_CONNECTION_INFO_NOT_CONNECTED;
			   }
               break;
			   
               default:
			   {
                  DIA_TR_ERR("### ERROR CAN'T MATCH videoSignalStatus.enType(0x%02x)!", videoSignalStatus.enType);
			   }
               break;
            }

            //send to listener
            pListener->vOnCVBSConnectionStatus(status);

            DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus SUCCESS!");
         }
      }
      else if (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_ERROR)
      {
         amt_tclServiceDataError oErrorMsg(poMessage);
         tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
         DIA_TR_INF("dia_SAFeatureRVC --- opcode error received with error code 0x%04X", u16ErrorCode);
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus poMessage->u8GetOpCode() UNKNOWN => CODE==%d!", poMessage->u8GetOpCode());
      }

      (void) poMessage->bDelete();
   }
   else
   {
      DIA_TR_INF("dia_SAFeatureRVC::vHandleCVBSConnectionStatus poMessage EMPTY!");
   }
}