/*****************************************************************************************
*
* \file    dia_SAFeatureBluetoothMost.cpp
*
* \brief   System Adapter for CCA communication with BT componenet
*
* \author  nce5kor	
*
* \date    02.12.2015
*
* (c) 2014 Robert Bosch Car Multimedia GmbH
*--------------------------------------------------------
* Date 		   | Author  	          | Modification
* 30-12-2015   | nce5kor			  | Code changes to support the new interface for BTSignalQuality
****************************************************************************************/

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSTEM_ADAPTERS__
#include "common/framework/sysadapters/dia_common_system_adapters.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_BLUETOOTH_MOST_LISTENER__
#include "project/interfaces/dia_IBluetoothMostListener.h"
#endif

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#include "dia_SAFeatureBluetoothMost.h"

#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_SERVICEINFO
#include "conn_most_fi_if.h"

std::map<most_fi_tcl_e8_BTSetDeviceDisconnectedReason::tenType, dia_eBTDeviceDisconnectedReason> dia_InternalBTDeviceDisconnectedReasonMapping;

// MESSAGE MAP:
// Such a map and an own handler is needed for every service you are accessing!
// Enter the function IDs (FID) and the corresponding functions here.
// The function will be called when a message with the corresponding FID arrives

BEGIN_MSG_MAP(dia_SAFeatureBluetoothMost, dia_SystemAdapterFeature)
	ON_MESSAGE(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED, vHandleBTDeviceList)
	ON_MESSAGE(MOST_BTSETFI_C_U16_SIGNALSTRENGTH,  vHandleBTSignalQuality)
	ON_MESSAGE(MOST_BTSETFI_C_U16_CHANGEWIFITRANSMITPOWER,  vHandleChangeWifiTransmitPower)
END_MSG_MAP()

#define MINIMUM_NUMBER_OF_CONNECTED_BT_DEVICES 1
#define FIRST_DEVICE_IN_LIST 0

/******************************************************************************
* FUNCTION:    dia_SAFeatureBluetoothMost
*
* DESCRIPTION: Constructor
*
******************************************************************************/

dia_SAFeatureBluetoothMost::dia_SAFeatureBluetoothMost (
      tCString name,    // feature name
      dia_SystemAdapterServicePlugin& pSrvPlugin
   )
    : dia_SystemAdapterFeature(name,pSrvPlugin)
{
//#ifdef VARIANT_S_FTR_ENABLE_RIVIE
	totalBTdevices = 0;
//#endif
   dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::dia_SAFeatureBluetoothMost");

   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8NOT_APPLICABLE]         = DIA_EN_BTDEVICE_DISCONNECTED_REASON_NOT_APPLICABLE;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8DEVICE_REQUEST]         = DIA_EN_BTDEVICE_DISCONNECTED_REASON_DEVICE_REQUEST;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8MODULE_REQUEST]         = DIA_EN_BTDEVICE_DISCONNECTED_REASON_MODULE_REQUEST;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8AUTOMATIC]              = DIA_EN_BTDEVICE_DISCONNECTED_REASON_AUTOMATIC;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8OUT_OF_RANGE]           = DIA_EN_BTDEVICE_DISCONNECTED_REASON_OUT_OF_RANGE;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8INTERMEDIATE]           = DIA_EN_BTDEVICE_DISCONNECTED_REASON_INTERMEDIATE;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8CONNECT_TIMEOUT]        = DIA_EN_BTDEVICE_DISCONNECTED_REASON_CONNECT_TIMEOUT;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8CONNECT_REJECTED]       = DIA_EN_BTDEVICE_DISCONNECTED_REASON_CONNECT_REJECTED;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8PROTOCOL_NOT_SUPPORTED] = DIA_EN_BTDEVICE_DISCONNECTED_REASON_PROTOCOL_NOT_SUPPORTED;
   dia_InternalBTDeviceDisconnectedReasonMapping[most_fi_tcl_e8_BTSetDeviceDisconnectedReason::FI_EN_E8CONNECT_FAILED]         = DIA_EN_BTDEVICE_DISCONNECTED_REASON_CONNECT_FAILED;
}

/******************************************************************************
* FUNCTION:    ~dia_SAFeatureBluetoothMost
*
* DESCRIPTION: Destructor
*
******************************************************************************/

dia_SAFeatureBluetoothMost::~dia_SAFeatureBluetoothMost ( void )
{}

/************************************************************************************************
* FUNCTION:    getBTMostDeviceList
*
* DESCRIPTION: This function sends the request received from tester to BT componenet through CCA
*
* PARAMETER:   void
*
* RETURNVALUE: tDiaResult - DIA_FAILED(send error), DIA_SUCCESS(send success)
*
************************************************************************************************/

tDiaResult
dia_SAFeatureBluetoothMost::getBTMostDeviceList ( void )
{
   dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::getBTMostDeviceList()");

   tDiaResult retCode = DIA_FAILED;

   if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
   {
      gm_tclEmptyMessage oMessage (
         mAppID,
         mpSrvPlugin->getClientID(),
         mpSrvPlugin->getRegistrationID(),
         0,
         mpSrvPlugin->getServiceID(),
         MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
         AMT_C_U8_CCAMSG_OPCODE_GET
      );

      if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ) 
	  {
		  retCode = DIA_SUCCESS;
	  }
	  else
	  {
		  DIA_TR_INF( "MESSAGE SENDING FAILED" );
	  }
   }
   else
   {
	   DIA_TR_INF( "Registration FAILED" );
   }

   return retCode;
}

/************************************************************************************************
* FUNCTION:    getBTSignalQuality
*
* DESCRIPTION: This function sends the request received from tester to BT componenet through CCA
*
* PARAMETER:   void
*
* RETURNVALUE: tDiaResult - DIA_FAILED(send error), DIA_SUCCESS(send success)
*
************************************************************************************************/


tDiaResult
dia_SAFeatureBluetoothMost::getBTSignalQuality ( void )
{
   dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::getBTSignalQuality()");

   tDiaResult retCode = DIA_FAILED;

   if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
   {
      gm_tclEmptyMessage oMessage (
         mAppID,
         mpSrvPlugin->getClientID(),
         mpSrvPlugin->getRegistrationID(),
         0,
         mpSrvPlugin->getServiceID(),
         MOST_BTSETFI_C_U16_SIGNALSTRENGTH,
         AMT_C_U8_CCAMSG_OPCODE_GET
      );

      if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ) 
	  {
		  retCode = DIA_SUCCESS;
	  }
	  else
	  {
		  DIA_TR_INF( "MESSAGE SENDING FAILED" );
	  }
   }
   else
   {
	   DIA_TR_INF( "Registration FAILED" );
   }

   return retCode;
}

/************************************************************************************************
* FUNCTION:    vHandleBTDeviceList
*
* DESCRIPTION: This function receives the response from BT componenet through CCA
*
* PARAMETER:   amt_tclServiceData* - CCA message pointer
*
* RETURNVALUE: void
*
************************************************************************************************/

void
dia_SAFeatureBluetoothMost::vHandleBTDeviceList ( amt_tclServiceData* poMessage ) const
{
    dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::vHandleBTDeviceList()");

	if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS )
      {
		  DIA_TR_INF(("dia_SAFeatureBluetoothMost::vHandleBTDeviceList - Received DeviceList"));

		  fi_tclVisitorMessage oStatusMessage(poMessage);
          most_btsetfi_tclMsgDeviceListExtendedStatus   oStatusMsgData;
          (void) oStatusMessage.s32GetData(oStatusMsgData);

          dia_IBluetoothMostListener* pListener = OSAL_NULL;
		  querySysAdapterListener<dia_IBluetoothMostListener>(&pListener);

          if ( pListener != OSAL_NULL )
          {
			  // Evaluate the Device List
              std::vector<most_fi_tcl_BTSetDeviceListExtendedResultItem>::const_iterator itDL = oStatusMsgData.oDeviceListExtendedResult.oItems.begin();

			  tU8 numBTDevices;
			  if(oStatusMsgData.u8NumPairedDevices > MAX_BT_DEVICE_COUNT)
			  {
				  numBTDevices = MAX_BT_DEVICE_COUNT;
			  }
			  else
			  {
				  numBTDevices = oStatusMsgData.u8NumPairedDevices;
			  }

			  dia_tBTExtendedDeviceInfo btInfo[numBTDevices];

			  DIA_TR_INF( "oDeviceListExtendedResult.oItems.size() = %d", oStatusMsgData.oDeviceListExtendedResult.oItems.size() );
			  DIA_TR_INF( "Number of Connected BT devices = %d", oStatusMsgData.u8NumPairedDevices );

			  if( !oStatusMsgData.oDeviceListExtendedResult.oItems.empty() )
               {
                  for (tU8 i=OSAL_NULL; (itDL != oStatusMsgData.oDeviceListExtendedResult.oItems.end()) && i < MAX_BT_DEVICE_COUNT; ++itDL, ++i )
                  {
                     most_fi_tcl_BTSetDeviceListExtendedResultItem oBTDevList = (*itDL);

					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - DEVICE NUMBER : %d",i);
					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - Device Name = \"%s\"", oBTDevList.sDeviceName.szValue);
					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - Device Address = \"%s\"", oBTDevList.sDeviceAddress.szValue);
					 DIA_TR_INF("dia_SAFeatureBluetoothMost  - Device Connection = \"%d\"", oBTDevList.bDeviceConnectedStatus);
					 btInfo[i].btDevAddr = strdup(oBTDevList.sDeviceAddress.szValue);
					 btInfo[i].btDevName = strdup(oBTDevList.sDeviceName.szValue);	
					 btInfo[i].btDevConnectionStatus = oBTDevList.bDeviceConnectedStatus;
					 btInfo[i].btDevDisconnectedReason = dia_InternalBTDeviceDisconnectedReasonMapping[oBTDevList.e8DeviceDisconnectedReason.enType];
				  }
				  pListener->vOnBTConnectedDeviceList(numBTDevices, btInfo);
			  }
			  else
			  {
				   DIA_TR_INF("!!!!!! DEVICE LIST EMPTY !!!!!!");
				   pListener->vOnBTConnectedDeviceList(numBTDevices, NULL);
			  }
         }
      }
	  (void) poMessage->bDelete();
   }
}
/************************************************************************************************
* FUNCTION:    registerToBTConnectionStatus
*
* DESCRIPTION: This function sends the request received from tester to BT componenet through CCA
*
* PARAMETER:   void
*
* RETURNVALUE: tDiaResult - DIA_FAILED(send error), DIA_SUCCESS(send success)
*
************************************************************************************************/

tDiaResult
dia_SAFeatureBluetoothMost::registerToBTConnectionStatus(void)
{
   dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::getBTMostDeviceConnection()");
   tDiaResult retCode = DIA_FAILED;

   if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
   {
      gm_tclEmptyMessage oMessage (
         mAppID,
         mpSrvPlugin->getClientID(),
         mpSrvPlugin->getRegistrationID(),
         0,
         mpSrvPlugin->getServiceID(),
         MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
         AMT_C_U8_CCAMSG_OPCODE_UPREG
      );

      if ( mpSrvPlugin->bTransmitMessage(&oMessage) == TRUE ) 
	  {
		  retCode = DIA_SUCCESS;
	  }
	  else
	  {
		  DIA_TR_INF( "MESSAGE SENDING FAILED" );
	  }
   }
   else
   {
	   DIA_TR_INF( "Registration FAILED" );
   }

   return retCode;
}

/************************************************************************************************
* FUNCTION:    vHandleBTSignalQuality
*
* DESCRIPTION: This function receives the response from BT componenet through CCA
*
* PARAMETER:   amt_tclServiceData* - CCA message pointer
*
* RETURNVALUE: void
*
************************************************************************************************/

void
dia_SAFeatureBluetoothMost::vHandleBTSignalQuality ( amt_tclServiceData* poMessage ) const
{
    dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::vHandleBTSignalQuality()");

	if ( poMessage )
   {
      DIA_ASSERT(poMessage->bIsServerMessage() == TRUE);
	  
      dia_IBluetoothMostListener* pListener = OSAL_NULL;
	  querySysAdapterListener<dia_IBluetoothMostListener>(&pListener);

      if ( poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_STATUS )
      {
		  DIA_TR_INF(("dia_SAFeatureBluetoothMost::vHandleBTSignalQuality - Received DeviceList"));

		  fi_tclVisitorMessage oStatusMessage(poMessage);
          most_btsetfi_tclMsgSignalStrengthStatus   oStatusMsgData;		  
          (void) oStatusMessage.s32GetData(oStatusMsgData);

          if ( pListener != OSAL_NULL )
          {		  
			   if( !(oStatusMsgData.oSignalStrengthList.oSignalStrengthItems.empty() ))
               {
				  tU32 numfConnectedDevices = oStatusMsgData.oSignalStrengthList.oSignalStrengthItems.size();
				  DIA_TR_INF( "TOTAL NUMBER OF CONNECTED DEVICES : %d",numfConnectedDevices);
#ifdef __NEW_MOST_BTSET_FI_ENABLED__		  
				  // Evaluate the Device List
			      std::vector<most_fi_tcl_BTSetSignalStrengthItem>::const_iterator itDL = oStatusMsgData.oSignalStrengthList.oSignalStrengthItems.begin();

				  most_fi_tcl_BTSetSignalStrengthItem btDevList[numfConnectedDevices];
	  
                  for (tU8 i = 0; itDL != oStatusMsgData.oSignalStrengthList.oSignalStrengthItems.end() ; ++itDL, ++i )
                  {
                     most_fi_tcl_BTSetSignalStrengthItem oBTDevice = (*itDL);
					 DIA_TR_INF( "#########  DETAILS OF DEVICE %d  #########",i);
					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - Device Handle   = %d", oBTDevice.u8DeviceHandle);
					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - Device Address  = %d", oBTDevice.sBluetoothDeviceAddress.szValue);
					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - Signal Strength = %d", oBTDevice.s8SignalStrength);
					 DIA_TR_INF( "dia_SAFeatureBluetoothMost - Link Quality    = %d", oBTDevice.u8LinkQuality);

					 btDevList[i] = oBTDevice;		
				  }

  				  most_fi_tcl_BTSetSignalStrengthItem btDeviceWithMaxStrength = btDevList[FIRST_DEVICE_IN_LIST];

				  if(numfConnectedDevices == MINIMUM_NUMBER_OF_CONNECTED_BT_DEVICES)
				  {
					  pListener->vOnCustBTLinkQualityResult(btDeviceWithMaxStrength.s8SignalStrength, btDeviceWithMaxStrength.u8LinkQuality);
				  }
				  else
				  {
					  for(tU8 i = 0; i < numfConnectedDevices; i++)
					  {
						  if( btDevList[i].s8SignalStrength > btDeviceWithMaxStrength.s8SignalStrength )
						  {
							  btDeviceWithMaxStrength = btDevList[i];
						  }
					  }
					  pListener->vOnCustBTLinkQualityResult(btDeviceWithMaxStrength.s8SignalStrength, btDeviceWithMaxStrength.u8LinkQuality);
				  }
#else
				  pListener->vOnCustBTLinkQualityResult(OSAL_NULL, OSAL_NULL);
#endif
			  }
			  else
			  {
				   DIA_TR_INF("!!!!!! DEVICE LIST EMPTY !!!!!!");
				   pListener->vOnCustBTLinkQualityResult(OSAL_NULL, OSAL_NULL);
			  }
		  }
      }
  	  else
	  {
		  pListener->vOnCustBTLinkQualityError(EN_ERROR_NO_DATA);
	  }
	  (void) poMessage->bDelete();
   }
}

/************************************************************************************************
* FUNCTION:    vChangeWifiTransmitPower
*
* DESCRIPTION: This function sends the request received from client to WiFi Most through CCA
*
* PARAMETER:   void
*
* RETURNVALUE: tDiaResult - DIA_FAILED(send error), DIA_SUCCESS(send success)
*
************************************************************************************************/

tDiaResult dia_SAFeatureBluetoothMost::vChangeWifiTransmitPower ( const tU8 wifiPowerAttenuation )
{
   dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::vChangeWifiTransmitPower()");

   tDiaResult retCode = DIA_FAILED;

   if ( mpSrvPlugin->getRegistrationID() != AMT_C_U16_REGID_INVALID )
   {
	    // Fill data with the settings
	    most_btsetfi_tclMsgChangeWifiTransmitPowerMethodStart  oMethodStart;
	    oMethodStart.u8WiFiPowerAttenuation = wifiPowerAttenuation;
	    fi_tclVisitorMessage oMsg( oMethodStart );
	    mpSrvPlugin->vInitServiceData(oMsg, MOST_BTSETFI_C_U16_CHANGEWIFITRANSMITPOWER, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

		if (mpSrvPlugin->bTransmitMessage(&oMsg) == TRUE)
		{
			DIA_TR_INF("dia_SAFeatureBluetoothMost::vChangeWifiTransmitPower. Sent message with value=%d[dB]", oMethodStart.u8WiFiPowerAttenuation);
		    retCode = DIA_SUCCESS;
		}
		else
		{
		    DIA_TR_ERR("dia_SAFeatureBluetoothMost::vChangeWifiTransmitPower(), MESSAGE SENDING FAILED");
		}
   }
   else
   {
	   DIA_TR_ERR("dia_SAFeatureBluetoothMost::vChangeWifiTransmitPower(), Registration FAILED");
   }

   return retCode;
}

/************************************************************************************************
* FUNCTION:    vHandleChangeWifiTransmitPower
*
* DESCRIPTION: This function receives the response from WiFi Most component through CCA
*
* PARAMETER:   amt_tclServiceData* - CCA message pointer
*
* RETURNVALUE: void
*
************************************************************************************************/

void dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower ( amt_tclServiceData* poMessage ) const
{
	dia_tclFnctTrace trc("dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower");

	if( poMessage )
	{
	  // Extract the operation code from the message.
	  tU8 u8OpCode = poMessage->u8GetOpCode();

	  // Switch on the opcode received
	  switch( u8OpCode )
	  {
		 case CCA_C_U8_OPCODE_METHODRESULT:
		 {
			DIA_TR_INF(( "dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower => CCA_C_U8_OPCODE_METHODRESULT" ));
			fi_tclVisitorMessage                                   oReceivedMsg( poMessage );
			most_btsetfi_tclMsgChangeWifiTransmitPowerMethodResult oReceivedData;
			(void)oReceivedMsg.s32GetData( oReceivedData );

			dia_IWifiMostListener* pListener = OSAL_NULL;
			querySysAdapterListener<dia_IWifiMostListener>(&pListener);
			if ( pListener )
			{
			   DIA_TR_INF(( "dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower: call vOnChangeWifiTransmitPower()" ));
			   tDiaResult eDiaResult = (oReceivedData.bResult) ? DIA_SUCCESS : DIA_FAILED;
			   pListener->vOnChangeWifiTransmitPower( eDiaResult );
			}
			break;
		 }
		 case AMT_C_U8_CCAMSG_OPCODE_ERROR:
		 {
		    DIA_TR_INF("dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower => AMT_C_U8_CCAMSG_OPCODE_ERROR");
			fi_tclVisitorMessage                            oReceivedMsg( poMessage );
			most_btsetfi_tclMsgChangeWifiTransmitPowerError oReceivedData;
			(void)oReceivedMsg.s32GetData( oReceivedData );

			dia_IWifiMostListener* pListener = OSAL_NULL;
			querySysAdapterListener<dia_IWifiMostListener>(&pListener);
			if ( pListener )
			{
			   tU8 errorCode = (tU8) oReceivedData.e8ErrorCode.enType;
			   DIA_TR_INF("dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower: Error code = '%d'", errorCode);
			   pListener->vOnChangeWifiTransmitPowerError( errorCode );
			}
		    break;
		 }
		 default:
			DIA_TR_INF("dia_SAFeatureBluetoothMost::vHandleChangeWifiTransmitPower - OpCode '%d' not ok", u8OpCode);
			break;
	  }
	  // Delete the message we are unable process it at this time
	  (void) poMessage->bDelete();
	}  // if( poMessage )
}

