/**
 * \file      dia_SecurityLevelAIVI.cpp
 *
 * \brief     {insert brief description here}
 *
 * \details   {insert file description here}
 *
 * \author    stc2hi
 * \date      Aug 29, 2016
 *
 * \copyright Robert Bosch Car Multimedia 2016
 */

#ifndef __INCLUDED_DIA_SECURITY__
#include "common/framework/security/dia_security.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_LEVEL_AIVI__
#include "dia_SecurityLevelAIVI.h"
#endif

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#include "common/framework/utils/dia_RandomGenerator.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_INTERFACE_AIVI_SECURITY__
#include "project/interfaces/dia_IAIVISecurity.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#ifndef __DIA_UNIT_TESTING__
#include "bpcl.h"
#endif

#if 0
#include <openssl/sha.h>
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_DIAGNOSTICS_SM
#include "trcGenProj/Header/dia_SecurityLevelAIVI.cpp.trc.h"
#endif


#define SECURED_SRV_DATA_LEN 20
#define PUN_LEN 8
#define SECRET_SEED_LEN 30
#define SECURITY_SEED_LEN 32
#define SESSION_KEY_LEN 16
#define STATIC_KEY_LEN 16 //(AES128)
#define SECRET_KEY_LEN 32
#define FINGERPRINT_LEN 16
#if 0
#define USE_ECB
#endif

//-----------------------------------------------------------------------------

dia_SecurityLevelAIVI::dia_SecurityLevelAIVI ( dia_SecurityLevelConfiguration& config )
   : dia_SecurityLevel(DIA_NAME_SECURITY_LEVEL_PROJECT_01, config),
     mSessNum(1),
     mbIsKeyInvalid(true)
{
   mNeedToRunSeedKeyPolicy = true;
   mIsLevelUnderSessionControl = true;
}

//-----------------------------------------------------------------------------

dia_SecurityLevelAIVI::~dia_SecurityLevelAIVI ( void )
{}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::initialize ( void )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::initialize");

   dia_SecurityLevel::initialize();

   (void) acceptEvent(dia_SecurityLevelFSM::evOperationModeUpdate, 0 );
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::onSessionChange ( tU8 newSession, tU8 oldSession )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelAIVI::onSessionChange");

   DIA_TR_INF("--- dia_SecurityLevelAIVI::onSessionChange: %d => %d", oldSession, newSession);

   mSessNum = newSession;

   (void) acceptEvent(dia_SecurityLevelFSM::evSessionChanged, 0);
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::vOnEcuSerialNumber ( const std::vector<tU8>& serialno )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelAIVI::vOnEcuSerialNumber");

   (void) unsetSysAdapterListener<dia_IAIVISecurityListener>(this);

   if (serialno.size() == SECURED_SRV_DATA_LEN)
   {
      mSecuredSrvData.clear();
      mSecuredSrvData = serialno;

      tChar szSecuredSrvData[SECURED_SRV_DATA_LEN+1];
      for (int i = 0; i < SECURED_SRV_DATA_LEN; i++) szSecuredSrvData[i] = mSecuredSrvData[i];
      szSecuredSrvData[SECURED_SRV_DATA_LEN] = 0;
      DIA_TR_INF("dia_SecurityLevelAIVI: SecuredSrvData = '%s'", szSecuredSrvData);

      (void) acceptEvent(dia_SecurityLevelFSM::evSeedRequested, 0 );
   }
   else
   {
      DIA_TR_ERR("!!! dia_SecurityLevelAIVI: incorrect size of ECUSerialNumber (%d)", serialno.size());
   }
}

//-----------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::vOnStaticKey ( const std::vector<tU8>& staticKey )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelAIVI::vOnStaticKey");

   (void) unsetSysAdapterListener<dia_IAIVISecurityListener>(this);

   ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vOnStaticKey: StaticKey Length: %d Data: %02x", staticKey.size(),
         ETG_LIST_LEN(staticKey.size()), ETG_LIST_PTR_T8(staticKey.data())));

   if (staticKey.size() == STATIC_KEY_LEN)
   {
      //STD-SEC-TS.0202(1): The SecuritySeed is an encryption of SecretSeed. This encryption algorithm is Encrypt StaticKey

      //Pad to a suitable length for the AES128 algorithm
      tU8 u8PadBytes = SECURITY_SEED_LEN - SECRET_SEED_LEN;

      for (tU8 i = 0; i < u8PadBytes; i++)
      {
#if 0
         //Use the randomly generated PUN to have a distinct seed at the end
         mSeed.push_back(mPUN[i]);
#else
         //The required options for the Rijndael algorithm are CBC (Cypher Block Chaining) with an initialization
         //vector to all 0, and padding according to PKCS7 (find n in the interval 1..16 that let the size of the
         //block added to n be a multiple of 16; pad with n bytes with the value n).
         mSeed.push_back(u8PadBytes);
#endif
      }

      ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vOnStaticKey: SecretSeed Length: %d Data: %02x",
            mSeed.size(), ETG_LIST_LEN(mSeed.size()), ETG_LIST_PTR_T8(mSeed.data())));

      tU8 aSecuritySeed[SECURITY_SEED_LEN];
#ifdef USE_ECB
      if (BPCL_OK == BPCL_AES_Encrypt(BPCL_AES_MODE_ECB, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
                      (tU8*) staticKey.data(), 0, (tU8*)0, mSeed.data(), aSecuritySeed, mSeed.size()))
#else
      tU8* iv = 0;
      if (BPCL_OK == BPCL_AES_Encrypt(BPCL_AES_MODE_CBC, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
                      (tU8*) staticKey.data(), 0, iv, mSeed.data(), aSecuritySeed, mSeed.size()))
#endif
      {
         //Replace the former SecretSeed with the SecuritySeed
         mSeed.clear();
         for (int i = 0; i < SECURITY_SEED_LEN; i++) mSeed.push_back(aSecuritySeed[i]);

         ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vOnStaticKey: SecuritySeed Length: %d Data: %02x",
               mSeed.size(), ETG_LIST_LEN(mSeed.size()), ETG_LIST_PTR_T8(mSeed.data())));

         //Test if the encryption is symmetric (shall result again in the original SecretSeed)
         tU8 aSecretSeed[SECURITY_SEED_LEN];
#ifdef USE_ECB
         if (BPCL_OK == BPCL_AES_Decrypt(BPCL_AES_MODE_ECB, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
                         (tU8*) staticKey.data(), 0, (tU8*)0, aSecuritySeed, aSecretSeed, /*SECRET_SEED_LEN*/SECURITY_SEED_LEN))
#else
         tU8* dec_iv = 0;
         if (BPCL_OK == BPCL_AES_Decrypt(BPCL_AES_MODE_CBC, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
                         (tU8*) staticKey.data(), 0, dec_iv, aSecuritySeed, aSecretSeed, /*SECRET_SEED_LEN*/SECURITY_SEED_LEN))
#endif
         {
            ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vOnStaticKey: SecretSeed (decrypted) Length: %d Data: %02x",
                  /*SECRET_SEED_LEN*/SECURITY_SEED_LEN, ETG_LIST_LEN(/*SECRET_SEED_LEN*/SECURITY_SEED_LEN), ETG_LIST_PTR_T8(aSecretSeed)));

            //Send the SecuritySeed
            (void) acceptEvent(dia_SecurityLevelFSM::evOnSeedAvailable,0);
         }
         else
         {
            DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vOnStaticKey: BPCL_AES_Decrypt FAILED");
         }
      }
      else
      {
         DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vOnStaticKey: BPCL_AES_Encrypt FAILED");
      }
   }
   else
   {
      DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vOnStaticKey: incorrect StaticKey len=%d", staticKey.size());
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelAIVI::getSeed ( void )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::getSeed");

   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF("mStatus = 0x%x", mStatus);

   if (mStatus == DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE)
   {
      std::vector<tU8> seedValue;
      for (unsigned i = 0; i < SECURITY_SEED_LEN; i++)
      {
         seedValue.push_back(0);
      }
      notifySeedResult(DIA_SUCCESS, seedValue);
      return DIA_SUCCESS;
   }

   // Send internal request to get the SecuredSrvData (ECUSerialNumber DID $F18C)

   dia_IAIVISecurity* pInterface = OSAL_NULL;
   if ( (querySysAdapterInterface<dia_IAIVISecurity>(&pInterface) == DIA_SUCCESS) && pInterface )
   {
      (void) setSysAdapterListener<dia_IAIVISecurityListener>(this);

      if (pInterface->getEcuSerialNumber() == DIA_SUCCESS)
      {
         retCode = DIA_SUCCESS;
      }
   }

   if (retCode != DIA_SUCCESS)
   {
      DIA_TR_ERR("!!! dia_SecurityLevelAIVI::getSeed: FAILED to get SecuredSrvData");
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelAIVI::acceptKey ( std::vector<tU8>& keyValue )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::acceptKey");

   mErrorCode = DIA_E_NOERROR;

   if (keyValue.size() != SECURITY_SEED_LEN)
   {
      DIA_TR_ERR("!!! dia_SecurityLevelAIVI::acceptKey FAILED with invalid key size = %d", keyValue.size());
      mErrorCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
   }
   else
   {
      if ((mPUN.size() == PUN_LEN) && (mSessionKey.size() == SESSION_KEY_LEN))
      {
         ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::acceptKey: SecurityKey Length: %d Data: %02x",
               keyValue.size(), ETG_LIST_LEN(keyValue.size()), ETG_LIST_PTR_T8(keyValue.data())));

         (void) acceptEvent(dia_SecurityLevelFSM::evKeyReceived, (tVoid*) &keyValue);
      }
      else
      {
         mErrorCode = DIA_FAILED;
      }
   }

   return mErrorCode;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::vFsmEvaluateOperationMode ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::vFsmEvaluateOperationMode");

#if 0
   if ( mpFSM ) mpFSM->acceptEvent(dia_SecurityLevelFSM::evOperationModeUpdate, 0 );
#endif
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::vFsmCalculateSeed ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::vFsmCalculateSeed");

   tDiaResult retCode = generateSeed();

   if (retCode == DIA_SUCCESS)
   {
      // STD-SEC-TS.0203(1): Session key is obtained from the first 16 bytes of the HASH of SecretSeed

#if 0 //Use OpenSSL in parallel
      unsigned char hash[SHA_DIGEST_LENGTH];

      (void) SHA1(mSeed.data(), mSeed.size(), hash);

      mSessionKey.clear();
      for (int i = 0; i < SHA_DIGEST_LENGTH; i++)
      {
         if (i < SESSION_KEY_LEN)
         {
            mSessionKey.push_back(hash[i]);
         }
      }

      ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vFsmCalculateSeed: SessionKey Length: %d Data: %02x",
            mSessionKey.size(), ETG_LIST_LEN(mSessionKey.size()), ETG_LIST_PTR_T8(mSessionKey.data())));
#endif

      retCode = DIA_FAILED;

      std::vector<tU8>  newBinHash;
      if (true == dia_nsUtilities::generateHash(mSeed, newBinHash))
      {
         mSessionKey.clear();
         for (int i = 0; i < SESSION_KEY_LEN; i++) mSessionKey.push_back(newBinHash[i]);

         // Send internal request to get the StaticKey needed to encrypt the SecretSeed

         dia_IAIVISecurity* pInterface = OSAL_NULL;
         if ( (querySysAdapterInterface<dia_IAIVISecurity>(&pInterface) == DIA_SUCCESS) && pInterface )
         {
            (void) setSysAdapterListener<dia_IAIVISecurityListener>(this);

            if (pInterface->getStaticKey() == DIA_SUCCESS)
            {
               retCode = DIA_SUCCESS;
            }
         }

         if (retCode != DIA_SUCCESS)
         {
            DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vFsmCalculateSeed: FAILED to get StaticKey");
         }
      }
      else
      {
         DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vFsmCalculateSeed: generateHash FAILED");
      }
   }
   else
   {
      DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vFsmCalculateSeed: generateSeed FAILED");
   }
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::vFsmSendSeed ( void* /*pArg*/ )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::vFsmSendSeed");

   // Security Seed is computed by encrypting SecretSeed by AES 128 algorithm

   // Session key is obtained from the first 16 bytes of the HASH of SecretSeed. HASH function uses SHA
   // algorithm. It is computed and stored.

#if 0
   std::vector<tU8> seedValue;
   std::vector<tU8>::iterator iter = mSeed.begin();

   for ( ; iter != mSeed.end(); iter++ )
   {
      seedValue.push_back(*iter);
   }

   notifySeedResult(DIA_SUCCESS, seedValue);
#else
   notifySeedResult(DIA_SUCCESS, mSeed);
#endif
}

//------------------------------------------------------------------------------

bool
dia_SecurityLevelAIVI::isKeyInvalid ( void* pArg )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::isKeyInvalid");

   return mbIsKeyInvalid;
}

//------------------------------------------------------------------------------

void
dia_SecurityLevelAIVI::vFsmValidateKey ( void* pArg )
{
   dia_tclFnctTrace trc("dia_SecurityLevelAIVI::vFsmValidateKey");

   mbIsKeyInvalid = true;

   if ( pArg != NULL )
   {
      std::vector<tU8>* pSecurityKey = (std::vector<tU8>*) pArg;

      //STD-SEC-TS.0205(1): The UDS_Server retrieves SecretKey by applying eSessionKey-1(SecurityKey).

      tU8 aSecretKey[SECRET_KEY_LEN];
#ifdef USE_ECB
      if (BPCL_OK == BPCL_AES_Decrypt(BPCL_AES_MODE_ECB, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
                      (tU8*) mSessionKey.data(), 0, (tU8*)0, pSecurityKey->data(), aSecretKey, /*pSecurityKey->size()*/SECRET_KEY_LEN))
#else
      tU8* iv = 0;
      if (BPCL_OK == BPCL_AES_Decrypt(BPCL_AES_MODE_CBC, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
                      (tU8*) mSessionKey.data(), 0, iv, pSecurityKey->data(), aSecretKey, /*pSecurityKey->size()*/SECRET_KEY_LEN))
#endif
      {
         ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vFsmValidateKey: SecretKey (decrypted) Length: %d Data: %02x",
               SECRET_KEY_LEN, ETG_LIST_LEN(SECRET_KEY_LEN), ETG_LIST_PTR_T8(aSecretKey)));

         //STD-SEC-TS.0206(1): The UDS_Server checks that the received PUN is the same as previously generated

         int i;
         mbIsKeyInvalid = false;
         for (i = 0; i < mPUN.size(); i++)
         {
            if (mPUN[i] != aSecretKey[i])
            {
               DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vFsmValidateKey: FAILED to compare with PUN at %d (0x%02x != 0x%02x)",
                     i, mPUN[i], aSecretKey[i]);
               mbIsKeyInvalid = true;
               break;
            }
         }

         if ((mbIsKeyInvalid == false) && (i < SECRET_KEY_LEN))
         {
            //STD-SEC-TS.0207(1): If both PUN matches, the Fingerprint is stored into non volatile UDS_Server memory associated to the SecLev.

            std::vector<tU8> fingerPrint;
            for (int j = 0; j < FINGERPRINT_LEN; j++)
            {
               fingerPrint.push_back(aSecretKey[i++]);
            }

            ETG_TRACE_COMP(("--- dia_SecurityLevelAIVI::vFsmValidateKey: Fingerprint Length: %d Data: %02x",
                  fingerPrint.size(), ETG_LIST_LEN(fingerPrint.size()), ETG_LIST_PTR_T8(fingerPrint.data())));

            tDiaResult retCode = dia_setProperty(DIA_PROP_AIVI_SECURITY_FINGERPRINT, fingerPrint);
            if (retCode != DIA_SUCCESS)
            {
               DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vFsmValidateKey: Storing of Fingerprint FAILED");
            }
         }
      }
      else
      {
         DIA_TR_ERR("!!! dia_SecurityLevelAIVI::vFsmValidateKey: BPCL_AES_Decrypt FAILED");
      }

   } //if ( pArg != NULL )

   if ( mbIsKeyInvalid == true )
   {
      DIA_TR_ERR("##### INVALID SECURITY KEY RECEIVED #####");
   }

   acceptEvent(dia_SecurityLevelFSM::evOnKeyValidationDone,0);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityLevelAIVI::generateSeed ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityLevelAIVI::generateSeed");

   tDiaResult retCode = DIA_FAILED;

   // initialize the seed attribute
   initializeSeed();

   // STD-SEC-TS.0200(1): When a SecuritySeed is requested to the UDS_Server, it generates and store a eight bytes PUN into its own memory.

   dia_RandomGenerator* pRandomGenerator = getInstanceOfRandomGenerator();
   if ( pRandomGenerator == NULL )
   {
      DIA_TR_ERR("### FAILED TO GET INSTANCE OF RANDOM GENERATOR ###");
      return retCode;
   }

   tU32 random1  = pRandomGenerator->getRandomNumber();
   OSAL_s32ThreadWait(random1 >> 24);
   tU32 random2  = pRandomGenerator->getRandomNumber();

   DIA_TR_INF("--- dia_SecurityLevelAIVI::generateSeed: rng1=0x%08x rng2=0x%08x", random1, random2);

   mPUN.clear();

   mPUN.push_back((tU8) (random1 >> 24));
   mPUN.push_back((tU8) (random1 >> 16));
   mPUN.push_back((tU8) (random1 >> 8));
   mPUN.push_back((tU8) random1);
   mPUN.push_back((tU8) (random2 >> 24));
   mPUN.push_back((tU8) (random2 >> 16));
   mPUN.push_back((tU8) (random2 >> 8));
   mPUN.push_back((tU8) random2);

   // STD-SEC-TS.0201(1): The SecretSeed is generated by concatenate PUN, SecLev, SessNum and SecuredSrvData.

   mSeed = mPUN;
   mSeed.push_back((tU8) mConfig.getAccessTypeSeed()); //SecLev
   mSeed.push_back(mSessNum);

   if (mSecuredSrvData.size() == SECURED_SRV_DATA_LEN)
   {
      for (int i = 0; i < SECURED_SRV_DATA_LEN; i++) mSeed.push_back(mSecuredSrvData[i]);
   }

   if (mSeed.size() == SECRET_SEED_LEN)
   {
      mIsSeedValid = TRUE;
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("### INVALID SEED LENGTH GENERATED ###");
   }

   return retCode;
}

