/*!
 * \file       dia_DataDownloadManagerPlugin_CMC.cpp
 *
 * \brief      Plugin for Data Download in A-IVI project
 *
 * \details    Clear following corresponding DTCs on request download:
 *             1/ logical block for fault type 0x06 (algorithm based failure)
 *             2/ logical block for fault type 0x51 (programming failure)
 *             3/ logical block for fault type 0x42 (internal failure)
 *
 * \component  Diagnosis
 *
 * \ingroup
 *
 * \copyright  (c) 2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include "common/framework/datadownload/dia_DataDownloadManager.h"
#endif

#ifndef __INCLUDED_DIA_DATASET__
#include "common/framework/datadownload/dia_Dataset.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_PROJECT__
#include "project/framework/application/dia_defsProject.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SECURITY__
#include "common/framework/security/dia_common_security.h"
#endif

#ifndef __INCLUDED_PRJ_SPECIFIC_DATA_DOWNLOAD_MANAGER_PLUGIN_CMC__
#include "dia_PrjSpecific_DataDownloadManagerPlugin_CMC.h"
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include "common/framework/application/dia_File.h"
#endif

//needed by sync
#include <unistd.h>
#include <fstream>     //lint !e537 : repeatedly included header file without standard include guard
#include <stdlib.h>    //lint !e537 !e451 : repeatedly included header file without standard include guard
#include <sys/types.h> //lint !e537 : repeatedly included header file without standard include guard
#include <sys/wait.h>  //lint !e537 : repeatedly included header file without standard include guard

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

std::unique_ptr<dia_DataDownloadManagerPlugin_CMC::DiaglogAgent>dia_DataDownloadManagerPlugin_CMC::mpDiaglogAgent = nullptr;

enum dia_ITC_value
{
   DIA_ITC_PASSED = 0,     /* =0 */
   DIA_ITC_FAILED          /* =1 */
};

#define DIA_C_U32_CMC_INSTALLATION_CONTAINER_LOGICAL_ADDRESS	(tU32)0x00000004

static std::string CIS_EXTR_SCRIPT = "/opt/bosch/base/bin/diag_doipcmcextract.sh";
static std::string AVDECC_DIR = "/var/opt/bosch/persistent/avdecc";

//-----------------------------------------------------------------------------
dia_DataDownloadManagerPlugin_CMC::dia_DataDownloadManagerPlugin_CMC(std::map<tU32,tU16>& logicAddrToDtcGroupRepo,
      std::map<tU32,tU32>& dtcProgrammingToPropertyIDRepo, const std::map<tU32, std::pair <tU32,tU32> >& startAddrToItcAndPropIDRepo)
   : mLogicAddrToDtcGroupRepo(logicAddrToDtcGroupRepo),
     mDtcProgrammingToPropertyIDRepo(dtcProgrammingToPropertyIDRepo),
     mStartAddrToItcAndPropIDRepo(startAddrToItcAndPropIDRepo)
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::dia_DataDownloadManagerPlugin_CMC()");
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManagerPlugin_CMC::prepareDownload ( dia_Dataset& dataset ) const
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::prepareDownload(dataset)");
   // make sure that the folder exists ( RTC-764575 )
   //! Create instance of the File-Directory class
   dia_FileDir avdeccDir(AVDECC_DIR);
   //! Check if the directory exists
   if(!(avdeccDir.doesExist()))
   {
	    DIA_TR_INF("directory \"%s\" does not exists. Create...", AVDECC_DIR.c_str());
		//! create the directory
		if(DIA_SUCCESS == avdeccDir.create())
		{
			DIA_TR_INF(" +--> Created Directory : \"%s\"!!!", AVDECC_DIR.c_str());
		}
   }
   else
   {
		 DIA_TR_INF("directory \"%s\"  exists. Continue...", AVDECC_DIR.c_str());
   }
   
   //run basis class method
   return dia_DataDownloadManagerPlugin::prepareDownload(dataset);
}

//-----------------------------------------------------------------------------

dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::DiaglogAgent(const dia_DataDownloadManagerPlugin_CMC* pPlugin, tU32 address)
   : mpPlugin(pPlugin),
     mpDataset(nullptr),
     mpErrorLog(nullptr)
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::DiaglogAgent");

   if ((DIA_SUCCESS != querySysAdapterInterface<dia_IErrorLog>(&mpErrorLog)) || !mpErrorLog)
   {
      DIA_ASSERT_ALWAYS();
   }

   if ((DIA_SUCCESS != getInstanceOfDataDownloadManager()->queryDataset(address, &mpDataset)) || !mpDataset)
   {
      DIA_ASSERT_ALWAYS();
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::start(const DtcClearList& dtcList)
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::start");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManagerPlugin_CMC::postprocessDownload ( dia_Dataset& dataset ) const
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::postprocessDownload(dataset)");
/*
   tU32 addressStart = dataset.getLogicalAddressStart();

   if(addressStart == DIA_C_U32_CMC_INSTALLATION_CONTAINER_LOGICAL_ADDRESS)
   {
	   //!
	   //! Execute the system extraction command
	   //!
	   int ret = system(CIS_EXTR_SCRIPT.c_str());
	   //!
	   //! Evaluate the system command execution result
	   //!
	   if ( WIFSIGNALED(ret) && ((WTERMSIG(ret) == SIGINT) || (WTERMSIG(ret) == SIGQUIT)) )
       {
		   DIA_TR_ERR("  +--> Failed to extract the container file!!! \"%s\"", CIS_EXTR_SCRIPT.c_str());
       }
	   else
	   {
		   DIA_TR_INF("  +--> Extracted container Successfully !!! \"%s\"", CIS_EXTR_SCRIPT.c_str());
	   }

	  DIA_TR_INF("Perform double sync for file system");
      sync();
      OSAL_s32ThreadWait(100);
      sync();
      OSAL_s32ThreadWait(100);

	  DIA_TR_INF("");
	  DIA_TR_INF("");
	  DIA_TR_INF("######################################################");
	  DIA_TR_INF("######################################################");
	  DIA_TR_INF("##|------------------------------------------------|##");
	  DIA_TR_INF("##| INSTALLATION CONTAINER DOWNLOADED SUCCESSFULLY |##");
	  DIA_TR_INF("##|------------------------------------------------|##");
	  DIA_TR_INF("######################################################");
	  DIA_TR_INF("######################################################");
	  DIA_TR_INF("");
	  DIA_TR_INF("");

   }
*/
   //run basis class method
   return dia_DataDownloadManagerPlugin::postprocessDownload(dataset);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::stop( const ItcReportList& itcList )
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::stop");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnControlResult ( void )
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::vOnControlResult");

}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnClearResult ( )
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnClearResult");
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnClearError(tU32 u32Error)
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnClearError");
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnDiaglogWriteTestResult( void )
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnDiaglogWriteTestResult");
}

//-----------------------------------------------------------------------------

void
dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnDiaglogWriteTestError (const tU32 u32Error )
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::DiaglogAgent::vOnDiaglogWriteTestError");
}

//-----------------------------------------------------------------------------

tBool
dia_DataDownloadManagerPlugin_CMC::IsDeviceUnlocked ( void ) const
{
   dia_tclFnctTrace trc("dia_DataDownloadManagerPlugin_CMC::IsDeviceUnlocked");

   std::list<dia_SecurityLevel*> activeSecurityLevels;
   dia_SecurityManager* pSecMgr = getInstanceOfSecurityManager();
   if ( pSecMgr )
   {
      pSecMgr->getActiveLevels(activeSecurityLevels);
      std::list<dia_SecurityLevel*>::const_iterator activeLevelIter = activeSecurityLevels.begin();
      for ( ; activeLevelIter != activeSecurityLevels.end(); ++activeLevelIter )
      {
         if ( (*activeLevelIter) && ((*activeLevelIter)->getUID() == DIA_UID_SECURITY_LEVEL_PROJECT_01) )
         {
            DIA_TR_INF("dia_DataDownloadManagerPlugin_CMC: IsDeviceUnlocked - DIA_UID_SECURITY_LEVEL_PROJECT_01 is unlocked/active!!");
            return TRUE;
         }
     }

     DIA_TR_ERR("!!! dia_DataDownloadManagerPlugin_CMC::IsDeviceUnlocked - DIA_UID_SECURITY_LEVEL_PROJECT_01 is LOCKED!!!");
     return FALSE;
   }
   else
   {
      DIA_TR_ERR("!!! dia_DataDownloadManagerPlugin_CMC::IsDeviceUnlocked => ERROR: Security manager is not available");
      return FALSE;
   }

}

