/*!
 * \file       dia_SubsystemDiagnosisStrategyDefault.cpp
 *
 * \brief      default strategy used for subsystem diagnosis
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2), Kirty Mayank (RBEI/ECA2)
 *
 * \date       23.09.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_STRATEGY__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisStrategyDefault.h"
#endif 

//! This is the helper/wrapper class which is to be implemented yet
// #ifndef __INCLUDED_DIA_DATASET__
// #include "common/framework/datadownload/dia_Dataset.h"
// #endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_MANAGER__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include <common/framework/engine/dia_EngineServer.h>
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include <common/framework/engine/dia_EngineManager.h>
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include <common/framework/engine/dia_Session.h>
#endif

//-----------------------------------------------------------------------------

dia_SubsystemDiagnosisStrategyDefault::dia_SubsystemDiagnosisStrategyDefault ( std::string& name , tU32 Id )
	: SubsystemDiagnosisStrategyCore(name, Id)
{}

//-----------------------------------------------------------------------------

dia_SubsystemDiagnosisStrategyDefault::~dia_SubsystemDiagnosisStrategyDefault ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::setup ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::setup");
	
	return registerSessionChange();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::teardown ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::teardown");
	
	return unregisterSessionChange();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::acceptEvent ( dia_SubsystemDiagnosisFSM::FsmEvent event, void* pArg )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::acceptEvent(dia_SubsystemDiagnosisFSM::FsmEvent,void*)");

	if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;
	
	mErrorCode = DIA_E_NO_ERROR;
	
	DIA_TR_INF("dia_SubsystemDiagnosisStrategyDefault::acceptEvent - State (before): %s, event = '%d'", mpFSM->getStateName(),event);
	mpFSM->acceptEvent(event,pArg);
	DIA_TR_INF("dia_SubsystemDiagnosisStrategyDefault::acceptEvent - State (after): %s", mpFSM->getStateName());
	
	return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::activateSubsystemSession (void)
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::activateSubsystemSession(void)");

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;
   
   mErrorCode = DIA_E_NO_ERROR;
   
   mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evActivate,0);
   
   return mErrorCode;
}


//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::deactivateSubsystemSession (void)
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::deactivateSubsystemSession(void)");

   if ( !mpFSM )  return DIA_E_FSM_NOT_AVAILABLE;
   
   mErrorCode = DIA_E_NO_ERROR;
   
   mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evDeactivate,0);
   
   return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::identyfyAllSubsystems ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::identyfyAllSubsystems");
	
	tDiaResult retCode = validateIdentifyEntitiesRequest();
	
	if ( retCode != DIA_SUCCESS )
	{
		DIA_TR_ERR("dia_SubsystemDiagnosisStrategyDefault: FAILED (DATA VALIDATION) !!");
		return retCode;
	}
	
	return triggerDiscover();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::identyfyAllSubsystemsRequestResult( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::identyfyAllSubsystemsRequestResult");
   
   tDiaResult retCode = validateIdentifyEntitiesRequestResult();
   
   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SubsystemDiagnosisStrategyDefault: FAILED (DATA VALIDATION) !!");
      return retCode;
   }
   
   return triggerDiscoverRequestResult();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::readSubsystemNodeList( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::readSubsystemNodeList");

   tDiaResult retCode = validateReadSubsytemNodeList();

   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SubsystemDiagnosisStrategyDefault: FAILED (DATA VALIDATION) !!");
      return retCode;
   }

   return triggerSubNodeListReadRequest();
}


//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::connectToSubsystem ( std::vector<tU8>& IP )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::connectToSubsystem");
	
	tDiaResult retCode = validateSubsystemConnectRequest(IP);
	
	if ( retCode != DIA_SUCCESS )
	{
		DIA_TR_ERR("dia_SubsystemDiagnosisStrategyDefault:: FAILED (DATA VALIDATION) !!");
		return retCode;
	}
		
	return triggerConnect();
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::routeRequestToSubsystem ( std::vector<tU8>& requestData )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::routeRequestToSubsystem");

   tDiaResult retCode = validateSubsystemRouteRequest(requestData);

   if ( retCode != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SubsystemDiagnosisStrategyDefault: FAILED (DATA VALIDATION) !!");
      return retCode;
   }
   
   mSubSysDiagReq.clear();
   
   //! copy the request data member variable
	mSubSysDiagReq = requestData;

   return triggerRoute();
}

//----------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::validateIdentifyEntitiesRequest ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::validateIdentifyEntitiesRequest");

   return DIA_E_NO_ERROR;
}

//----------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::validateIdentifyEntitiesRequestResult ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::validateIdentifyEntitiesRequestResult");

   return DIA_E_NO_ERROR;
}
//----------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::validateReadSubsytemNodeList ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::validateReadSubsytemNodeList");

   return DIA_E_NO_ERROR;
}

//----------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::validateSubsystemConnectRequest ( std::vector<tU8>& IP )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::validateSubsystemConnectRequest");
	
	tDiaResult validationResult = DIA_E_NO_ERROR;
	
	
	//! Check if the IP size is less than 4 bytes
	if(IP.size() < DIA_C_SUBSYSTEM_ENTITY_IP_ADDR_LEN)
	{
		//! precondition check failed!
		return DIA_E_ERROR;
	}
	//! Check if the IP is all zeros
	if((IP[0] == 0x00) && (IP[1] == 0x00) && (IP[2] == 0x00) && (IP[3] == 0x00))
	{
		//! precondition check failed!
		return DIA_E_ERROR;
	}
	//! Check if the IP is all 0xFF
	if((IP[0] == 0xFF) && (IP[1] == 0xFF) && (IP[2] == 0xFF) && (IP[3] == 0xFF))
	{
		//! precondition check failed!
		return DIA_E_ERROR;
	}
	//! set the requested node buffer to be used by the doip client
   DIA_TR_INF("dia_SubsystemDiagnosisStrategyDefault: validateSubsystemConnectRequest --> Requested IP : 0x%02x%02x%02x%02x", IP[0], IP[1], IP[2], IP[3]);
	mRequestedNodeAddr = ( (IP[0] << 24) | (IP[1] << 16) | (IP[2]<<8) | IP[3] );
   DIA_TR_INF("dia_SubsystemDiagnosisStrategyDefault: validateSubsystemConnectRequest --> Requested IP : 0x%4x", mRequestedNodeAddr);
	
	return validationResult;
}

//----------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::validateSubsystemRouteRequest ( std::vector<tU8>& data )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::validateSubsystemRouteRequest");

   return DIA_E_NO_ERROR;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::registerSessionChange ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::registerSessionChange");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->addListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! dia_SubsystemDiagnosisStrategyDefault::registerSessionChange => ERROR: Unable to register for Session changes");
   }

   return retVal;
}

//-----------------------------------------------------------------------------

void
dia_SubsystemDiagnosisStrategyDefault::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::vOnSessionChanged()");

   //terminate subsystem diagnosis if session is switched
   if( DIA_C_U8_UDS_SESSION_SUBSYSTEM != newSession)
   {
      if (mpFSM)
      {
         DIA_TR_INF("dia_SubsystemDiagnosisStrategyDefault session change from %d to %d in state '%s'", oldSession, newSession, mpFSM->getStateName());

         mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evDeactivate,0);
      }
      else
      {
         DIA_TR_ERR("dia_SubsystemDiagnosisStrategyDefault::vOnSessionChanged => ERROR: FSM IS NULL!!!");
      }
   }
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisStrategyDefault::unregisterSessionChange ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisStrategyDefault::unregisterSessionChange");

   tDiaResult retVal = DIA_FAILED;

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      if( pEngine->getSessionController()->removeListener(this))
      {
         retVal = DIA_SUCCESS;
      }
   }

   if(DIA_SUCCESS != retVal)
   {
      DIA_TR_ERR("!!! dia_SubsystemDiagnosisStrategyDefault::unregisterSessionChange => ERROR: Unable to deregister for Session changes");
   }

   return retVal;
}
