/*!
 * \file       dia_SubsystemDiagnosisStrategyCore.cpp
 *
 * \brief      Abstarct subsystem strategy used for subsystem diagnosis
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2), Kirty Mayank (RBEI/ECA2)
 *
 * \date       23.09.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_STRATEGY_CORE__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisStrategyCore.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_MANAGER__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisManager.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_MANAGER_PLUGIN__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisManagerPlugin.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_DOIP_CLIENT__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisDoIPClient.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define ENTITY_ID_LENGTH 6

//-----------------------------------------------------------------------------

SubsystemDiagnosisStrategyCore::SubsystemDiagnosisStrategyCore ( std::string& name, tU32 Id)
	: SubsystemDiagnosisStrategy(name, Id),
	  mpFSM(nullptr),
	  mpDoIPClient(nullptr),
	  mRequestedNodeAddr(0),
	  mErrorCode(DIA_E_NOERROR),
	  mAreConnectConditionsOk(false),
	  mAreIdentifyConditionsOk(false),
	  mAreRouteConditionsOk(false),
	  mIsActivationComplete(false),
	  mIsDeactivationComplete(false),
	  mIsEntityAlreadyConnected(false),
	  mIsEntityListEmpty(true),
	  mIsPowerModeOk(false),
	  mIsPrecondChkOk(false),
	  mIsRequestRouted(false),
	  mIsRoutingActivationOk(false),
	  mIsTCPSocketOk(false),
	  mIdentifyRoutineStatus(DIA_EN_IDENTIFY_ENTITIES_UNKNOWN)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::SubsystemDiagnosisStrategyCore");
	
	// create the state machine object
	if ( !(dia_SubsystemDiagnosisFSM::Fsm::createFSM(&mpFSM,this)) )
	{
		DIA_TR_ERR( "### STATE MACHINE INITIALIZATION FAILED ###");
	}
	
	DoIP_Cfg doipCfg;
	mpDoIPClient = new dia_SubsystemDiagnosisDoIPClient(DIA_S_SUBSYSTEM_DOIP_CLIENT_ETH_NAME, DIA_C_SUBSYSTEM_DOIP_CLIENT_PORT_NUMBER, DIA_C_SUBSYSTEM_DOIP_CLIENT_TESTER_ADDR, doipCfg);
	
	oResponseTimer.s32Create();
	mResponseTimerID = oResponseTimer.getID();

}

//-----------------------------------------------------------------------------

SubsystemDiagnosisStrategyCore::SubsystemDiagnosisStrategyCore ( char* name, tU32 Id)
	: SubsystemDiagnosisStrategy(name, Id),
	  mpFSM(nullptr),
	  mpDoIPClient(nullptr),
	  mRequestedNodeAddr(0),
	  mErrorCode(DIA_E_NOERROR),
	  mAreConnectConditionsOk(false),
	  mAreIdentifyConditionsOk(false),
	  mAreRouteConditionsOk(false),
	  mIsActivationComplete(false),
	  mIsDeactivationComplete(false),
	  mIsEntityAlreadyConnected(false),
	  mIsEntityListEmpty(true),
	  mIsPowerModeOk(false),
	  mIsPrecondChkOk(false),
	  mIsRequestRouted(false),
	  mIsRoutingActivationOk(false),
	  mIsTCPSocketOk(false),
	  mIdentifyRoutineStatus(DIA_EN_IDENTIFY_ENTITIES_UNKNOWN)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::SubsystemDiagnosisStrategyCore");

	// create the state machine object
	if ( !(dia_SubsystemDiagnosisFSM::Fsm::createFSM(&mpFSM,this)) )
	{
		DIA_TR_ERR( "### STATE MACHINE INITIALIZATION FAILED ###");
	}
	
	DoIP_Cfg cfg;
	mpDoIPClient = new dia_SubsystemDiagnosisDoIPClient(DIA_S_SUBSYSTEM_DOIP_CLIENT_ETH_NAME, DIA_C_SUBSYSTEM_DOIP_CLIENT_PORT_NUMBER, DIA_C_SUBSYSTEM_DOIP_CLIENT_TESTER_ADDR, cfg);
	
	oResponseTimer.s32Create();
	mResponseTimerID = oResponseTimer.getID();
	
}

//-----------------------------------------------------------------------------

SubsystemDiagnosisStrategyCore::~SubsystemDiagnosisStrategyCore ( void )
{
	DIA_TR_INF("SubsystemDiagnosisStrategyCore::~SubsystemDiagnosisStrategyCore");
		
	//_BP_TRY_BEGIN
	//{
		delete mpFSM;
		mpFSM = 0;	
		delete mpDoIPClient;
		mpDoIPClient = NULL;
		
		oResponseTimer.removeTimerListener(this);
		oResponseTimer.s32Delete();
		mResponseTimerID = DIA_C_INVALID_TIMER_ID;
	//}
	//_BP_CATCH_ALL
	//{
	//	DIA_TR_ERR("EXCEPTION CAUGHT: SubsystemDiagnosisStrategyCore::~SubsystemDiagnosisStrategyCore !!!");
	//	DIA_ASSERT_ALWAYS();
	//}
	//_BP_CATCH_END
	
}

//-----------------------------------------------------------------------------

tDiaResult
SubsystemDiagnosisStrategyCore::triggerDiscover ( void )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::triggerDiscover()");
	
	mErrorCode = DIA_SUCCESS;
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evIdentifyStart,0);
	
	return mErrorCode;
}


//-----------------------------------------------------------------------------

tDiaResult
SubsystemDiagnosisStrategyCore::triggerDiscoverRequestResult ( void )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::triggerDiscoverRequestResult()");
	
	mErrorCode = DIA_SUCCESS;
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evIdentifyResult,0);
	
	return mErrorCode;
}

//-----------------------------------------------------------------------------

tDiaResult
SubsystemDiagnosisStrategyCore::triggerSubNodeListReadRequest ( void )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::triggerSubNodeListReadRequest()");

	mErrorCode = DIA_SUCCESS;
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evReadSubNodeRequest,0);

	return mErrorCode;
}


//------------------------------------------------------------------------------
tDiaResult
SubsystemDiagnosisStrategyCore::triggerConnect ( void )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::triggerConnect()");

   mErrorCode = DIA_SUCCESS;
   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evConnect,0);

   return mErrorCode;
}

//------------------------------------------------------------------------------

tDiaResult
SubsystemDiagnosisStrategyCore::triggerRoute ( void )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::triggerRoute()");
   
   mErrorCode = DIA_SUCCESS;
   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRouteRequest,0);

   return mErrorCode;
}

//------------------------------------------------------------------------------

void
SubsystemDiagnosisStrategyCore::triggerReset ( void )
{
   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evReset,0);
   	   mErrorCode = DIA_E_NOERROR;
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::postprocessConnect ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::postprocessConnect()");

	if (!mpPlugin)
	{
	  DIA_TR_ERR("### postprocessConnect DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s ", (!mpPlugin? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
	}

	mErrorCode = mpPlugin->postprocessConnect();

	if (DIA_SUCCESS!=mErrorCode)
    {
      DIA_TR_ERR("### postprocessConnect mErrorCode is set to 0x%08X", mErrorCode);
    }
	
	if (mpDoIPClient) mpDoIPClient->enableTesterPresent();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::postprocessIdentify ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::postprocessIdentify()");

	if ( !mpPlugin)
	{
	  DIA_TR_ERR("### postprocessIdentify DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s ", (!mpPlugin? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
	}

	mErrorCode = mpPlugin->postprocessIdentify();

	if (DIA_SUCCESS!=mErrorCode)
    {
      DIA_TR_ERR("### postprocessIdentify mErrorCode is set to 0x%08X", mErrorCode);
    }
	
	//mpDoIPClient->powermodeReqestAll();
}

//------------------------------------------------------------------------------

void
SubsystemDiagnosisStrategyCore::vFsmOnRequestRoutingDone ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmOnRequestRoutingDone()");
	//! nothing to be done at the moment.. lets see
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::postprocessRoute ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::postprocessRoute()");

	if ( !mpPlugin )
	{
	  DIA_TR_ERR("### postprocessRoute DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s ", (!mpPlugin? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
	}

	mErrorCode = mpPlugin->postprocessRoute();

	if (DIA_SUCCESS!=mErrorCode)
    {
      DIA_TR_ERR("### postprocessRoute mErrorCode is set to 0x%08X", mErrorCode);
    }
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::preprocessConnect ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::preprocessConnect()");

	if ( !mpPlugin)
	{
	  DIA_TR_ERR("### preprocessConnect DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s ", (!mpPlugin? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
	}

	mErrorCode = mpPlugin->preprocessConnect();

	if (DIA_SUCCESS!=mErrorCode)
    {
      DIA_TR_ERR("### preprocessConnect mErrorCode is set to 0x%08X", mErrorCode);
    }

    if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRequestConnectReady,0);

}


//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::preprocessIdentify ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::preprocessIdentify()");

	if ( !mpPlugin)
	{
	  DIA_TR_ERR("### preprocessIdentify DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s ", (!mpPlugin? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
	}

	mErrorCode = mpPlugin->preprocessIdentify();

	if (DIA_SUCCESS!= mErrorCode)
    {
      DIA_TR_ERR("### preprocessIdentify mErrorCode is set to 0x%08X", mErrorCode);
    }
	
	//! Clear the doip node list
	mpDoIPClient->clearDoIPNodeList();
	//! Set the callbacks from the client
	mpDoIPClient->setCallback(this);

    if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRequestIdentifyReady,0);

}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::preprocessRoute ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::preprocessRoute()");

	if ( !mpPlugin)
	{
	  DIA_TR_ERR("### preprocessRoute DIA_E_GENERAL_PROGRAMMING_FAILURE mpPlugin=%s ", (!mpPlugin? "NULL":"OK"));
      mErrorCode = DIA_E_GENERAL_PROGRAMMING_FAILURE;
      return;
	}

	mErrorCode = mpPlugin->preprocessRoute();

	if (DIA_SUCCESS!=mErrorCode)
    {
      DIA_TR_ERR("### preprocessRoute mErrorCode is set to 0x%08X", mErrorCode);
    }
	
	//! Disable Tester Present
	if (mpDoIPClient) mpDoIPClient->disableTesterPresent();

    if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRequestRouteReady,0);

}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendIdentifyResultNokResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendIdentifyResultNokResponse()");

	mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeIEReqResListEmpty() : DIA_E_SUBSYSTEM_LIST_EMPTY;
	
	mIdentifyRoutineStatus = DIA_EN_IDENTIFY_ENTITIES_NOK;
	vStartResponseTimer();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendIdentifyResultOkResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendIdentifyResultOkResponse()");

	mErrorCode = ( mpPlugin ) ? mpPlugin->getResCodeIdentifyResultOk() : DIA_E_SUBSYSTEM_IDENTIFY_RESULT_OK;
	
	mIdentifyRoutineStatus = DIA_EN_IDENTIFY_ENTITIES_OK;
	vStartResponseTimer();
	
	/*
	dia_eRoutineStatus status = DIA_EN_RTCTRL_STATUS_UNKNOWN;
	std::list<dia_tSubsystemInfo> doipNodes;

	if( mpDoIPClient )
	{

		doipNodes = mpDoIPClient->getDoIPClientNodes();
		//status = (( doipNodes.empty() == true ) ? DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK : DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK );
		status = DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK;
	}
	else
	{
		status = DIA_EN_RTCTRL_STATUS_ABORTED;
	}

	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnIdentifySubsystemsRequestUpdate(doipNodes, status );
    }
	*/
}


//------------------------------------------------------------------------------

void
SubsystemDiagnosisStrategyCore::sendSubNodeListResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendSubNodeListResponse()");

	//mErrorCode = ( mpPlugin ) ? mpPlugin->getResCodeIdentifyResultOk() : DIA_E_SUBSYSTEM_IDENTIFY_RESULT_OK;

	//mIdentifyRoutineStatus = DIA_EN_IDENTIFY_ENTITIES_OK;
	//vStartResponseTimer();


	//dia_eRoutineStatus status = DIA_EN_RTCTRL_STATUS_UNKNOWN;
	std::list<dia_tSubsystemInfo> doipNodes;

	if( mpDoIPClient )
	{

		doipNodes = mpDoIPClient->getDoIPClientNodes();
		//status = (( doipNodes.empty() == true ) ? DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK : DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK );
		//status = DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK;
	}
	//else
	//{
		//status = DIA_EN_RTCTRL_STATUS_ABORTED;
	//}

	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnReadSubsystemsNodeListRequestUpdate(doipNodes);
    }

}
//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendIdentifyResultRunningResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendIdentifyResultRunningResponse()");

	mErrorCode = ( mpPlugin ) ? mpPlugin->getResCodeIdentifyResultRunning() : DIA_E_SUBSYSTEM_IDENTIFY_RESULT_RUNNING;
	
	mIdentifyRoutineStatus = DIA_EN_IDENTIFY_ENTITIES_RUNNING;
	vStartResponseTimer();
	
	/*
	dia_eRoutineStatus status = DIA_EN_RTCTRL_STATUS_UNKNOWN;
	std::list<dia_tSubsystemInfo> doipNodes;
	
	if( mpDoIPClient )
	{

		doipNodes = mpDoIPClient->getDoIPClientNodes();
		status = DIA_EN_RTCTRL_STATUS_IN_PROGRESS;
	}
	else
	{
		status = DIA_EN_RTCTRL_STATUS_ABORTED;
	}

	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnIdentifySubsystemsRequestUpdate(doipNodes, status );
    }*/
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendIdentifyStartNokResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendIdentifyStartNokResponse()");

	mErrorCode = ( mpPlugin ) ? mpPlugin->getResCodeIdentifyStartNok() : DIA_E_SUBSYSTEM_IDENTIFY_ENTITIES_NOK;

}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendIdentifyStartRunningResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendIdentifyStartRunningResponse()");

	mErrorCode = ( mpPlugin ) ? mpPlugin->getResCodeIdentifyStartRunning() : DIA_E_SUBSYSTEM_IDENTIFY_ENTITIES_RUNNING;
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendSubsystemConnectNegResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendSubsystemConnectNegResponse()");
	
	mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeConnectPrecondNok() : DIA_E_SUBSYSTEM_PRE_COND_CHK_NOK;
	
	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnConnectToSubsystemRequestUpdate(mErrorCode);
    }
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendSubsystemConnectPosResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendSubsystemConnectPosResponse()");
	
	mErrorCode = DIA_SUCCESS;

	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnConnectToSubsystemRequestUpdate(mErrorCode);
    }
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendSubsystemRoutingNegResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendSubsystemRoutingNegResponse()");

	mErrorCode = ( mpPlugin ) ? mpPlugin->getErrCodeRoutingFailed() : DIA_E_SUBSYSTEM_RTNG_FAILED;
	
	mSubSysDiagRes.clear();
	mSubSysDiagRes.reserve(DIA_C_SUBSYSTEM_NEGATIVE_RESPONSE_LEN);
	
	mSubSysDiagRes.push_back(DIA_C_SUBSYSTEM_NEGATIVE_RESPONSE_ID);
	mSubSysDiagRes.push_back(mSubSysDiagReq[0]);
	mSubSysDiagRes.push_back(DIA_C_SUBSYSTEM_NEGATIVE_RESPONSE_CODE_CNC);
	
	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnRouteRequestUpdate(mSubSysDiagRes);
    }
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendSubsytemSessionNegResponse ( void* /* pArg */ )
{	
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendSubsytemSessionNegResponse()");
   
   dia_ISubsystemDiagnosisListener* pListener = nullptr;
   if ( (DIA_SUCCESS == querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener)) && pListener )
   {
      pListener->vOnSessionRequestUpdate(mIsActivationComplete);
   }
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::sendSubsytemSessionPosResponse ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::sendSubsytemSessionPosResponse()");
 
   dia_ISubsystemDiagnosisListener* pListener = nullptr;
   if ( (DIA_SUCCESS == querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener)) && pListener )
   {
      pListener->vOnSessionRequestUpdate(mIsActivationComplete);
   }
	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::validateConnectRequest ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::validateConnectRequest()");
   
   mAreConnectConditionsOk = true;
 
   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRequestConnectValid, 0);	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::validateIdentifyRequest ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::validateIdentifyRequest()");
   
   mAreIdentifyConditionsOk = true;

   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRequestIdentifyValid, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::validateRouteRequest ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::validateRouteRequest()");
   
   mAreRouteConditionsOk = true;

   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRequestRouteValid, 0);
 
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::onDiagnosticResponse(tU8 const * data, tU32 U32len)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::onDiagnosticResponse(...)");
	
	if (U32len == 0) 
	{
		DIA_TR_ERR("onDiagnosticResponse: response buffer null or response length zero");
		return;
	}
	
	//! Just store the response data here
	mSubSysDiagRes.clear();
	mSubSysDiagRes.reserve(U32len);
	
	for(int i = 0; i < U32len; i++)
	{
		mSubSysDiagRes.push_back(data[i]);
	}
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evOnRoutingResponse, 0);
	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::validateRouteResponse ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::validateRouteResponse()");

   if(    /* Check if Response buffer length is 3 */
		( DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_LENGTH == mSubSysDiagRes.size() ) &&
		  /* Check if the f1rst byte of the response is 0x7F */
		( DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_ID == mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_ID_IDX] ) &&
		  /* Check if the Thi3d byte of the response is 0x78 */
		( DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC == mSubSysDiagRes[DIA_C_SUBSYSTEM_RESPONSE_TYPE_WAIT_NRC_IDX] )
     )
   {
	   /* Received response is a wait response */
	   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRouteResponseWait, 0);
   }
   else
   {
	   /* Final Response is Received */
	   if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRouteResponseFinal, 0);
   }
 
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmActivateSubsystemSession ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmActivateSubsystemSession()");
	
	mIsActivationComplete = false;
	
	if(!mpDoIPClient) return;
	
	//! Create UDP Broadcast Client Connection & Thread
	//mIsActivationComplete = (DIA_SUCCESS == mpDoIPClient->Start()) ? true : false;
	mIsActivationComplete = (true == mpDoIPClient->startDoIPComm()) ? true : false;

	//! Evaluate UDP Thread
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evEvaluateActivation, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmCheckConnectPreconditions ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmCheckConnectPreconditions()");
	
	//! initialize the flag already...
	mIsPrecondChkOk = false;
	
	if(!mpDoIPClient) return;
	
	//! Check if the Entity List is Empty
	//if((mpDoIPClient->oDoIP_Nodes.empty()))
	if((mpDoIPClient->isDoIPListEmpty()))
	{
		//! pre-condition check failed, flag already set... do nothing!
		DIA_TR_INF("vFsmCheckConnectPreconditions :: node list empty! precondition check failed! ");
	}
	
	//! List is not empty, Check if the node requested by the tester is connected/exists in the list
	//if(DIA_SUCCESS == mpDoIPClient->doesNodeExist(mRequestedNodeAddr))
	if(true == mpDoIPClient->isNodeAvailable(mRequestedNodeAddr))
	{
		mpDoIPClient->setCurrentNode(mRequestedNodeAddr);
		//! precondition check passed. set the flag!
		DIA_TR_INF("vFsmCheckConnectPreconditions :: precondition check passed! ");
		mIsPrecondChkOk = true;	
	}
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evPreCondChkDone, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmCreateTCPSocketandConnect ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmCreateTCPSocketandConnect()");
	
	//! Initialize the flag already...
	mIsTCPSocketOk = false;
	
	if(!mpDoIPClient) return;
	
	//! Try to create the TCP Socket
	//if(false == mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->createTCPSocket())
	if(false == mpDoIPClient->createNodeTCPSocket(mRequestedNodeAddr))
	{
		//! Failed to create TCP Socket, flag already set... do nothing!
		DIA_TR_INF("vFsmCreateTCPSocketandConnect :: TCP socket creation failed! ");
		return;
	}
	
	//! TCP Socket created, Now try to connect ...
	//if(true == mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->connectTCP())
	if(true == mpDoIPClient->connectNodeTCP(mRequestedNodeAddr))
	{
		//! succeeded to Connect. Set the flag!
		DIA_TR_INF("vFsmCreateTCPSocketandConnect :: Connect Done ! ");
		mIsTCPSocketOk = true;
	}
	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmDeactivateSubsystemSession ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmDeactivateSubsystemSession()");
	
	//! Initialize the flag already...
	mIsDeactivationComplete = false;
	
	if(!mpDoIPClient) return;
	
	//! Close UDP Broadcast Client Connection
	//mIsDeactivationComplete = (DIA_SUCCESS == mpDoIPClient->Stop()) ? true : false;
	mIsDeactivationComplete = (true == mpDoIPClient->stopDoIPComm()) ? true : false;
	
	//! Evaluate UDP Thread
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evEvaluateDeactivation, 0);
	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmDeactivationFailed ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmDeactivationFailed()");
	
	//! We do nothing here for now
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmDisconnect ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmDisconnect()");
	
	if(mpDoIPClient) 
	{
		// deactivate tester present
		mpDoIPClient->disableTesterPresent();
	
		// Close the current node connection
		(void)mpDoIPClient->closeNodeTCPSocket();
	}
	
	// Previous entity disconnected; Connect to newly requested entity
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evConnect, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmDiscoverEntities ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmDiscoverEntities()");
	
	
	//if( mpDoIPClient ) mpDoIPClient->vdiscoverEntities(/* const char* in_addr */);
	if( mpDoIPClient )
	{		
		mpDoIPClient->clearDoIPNodeList();
		mpDoIPClient->discoverEntities();
	}
	// I should pass the IP adress of the ecu but I don't know what it is.

	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evIdentifyStart, 0);
}

//------------------------------------------------------------------------------
void 
SubsystemDiagnosisStrategyCore::vFsmOnPMRTimeout ( void* /* pArg */ )
{
	// Nothing to do here, negative response will be set by the fsm
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmOnPMRTimeout()");
}

//------------------------------------------------------------------------------
void 
SubsystemDiagnosisStrategyCore::vFsmOnRATimeout ( void* /* pArg */ )
{
	// Nothing to do here, negative response will be set by the fsm
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmOnRATimeout()");
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmOnSessionDeactivated ( void* /* pArg */ )
{
   dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmOnSessionDeactivated()");
   
   dia_ISubsystemDiagnosisListener* pListener = nullptr;
   if ( (DIA_SUCCESS == querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener)) && pListener )
   {
      (mIsDeactivationComplete == true) ? pListener->vOnSessionRequestUpdate(false) : pListener->vOnSessionRequestUpdate(true);
   }
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmOnSubsystemWaitResponse ( void* /* pArg */ )
{
	// Only info. Nothing to do here...
	DIA_TR_INF("SubsystemDiagnosisStrategyCore::vFsmOnSubsystemWaitResponse - Received a wait response ");  
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmReset ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmReset()");
	
	mIsActivationComplete = FALSE;

	// At the moment I am not sure what all parameters to reset,
	// lets keep this for the end
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmRouteRcvdRequest ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmRouteRcvdRequest()");

	if ( mpDoIPClient ) 
	{
		//mIsRequestRouted = mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->diagnosticReq(mSubSysDiagReq);
		mIsRequestRouted = mpDoIPClient->diagSendRequest(mRequestedNodeAddr, mSubSysDiagReq);
	}

	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evEvalRoutedReq, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmRouteRcvdResponse ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmRouteRcvdResponse()");
	
	dia_ISubsystemDiagnosisListener* pListener = nullptr;
	if ( (DIA_SUCCESS == querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener)) && pListener )
	{
		pListener->vOnRouteRequestUpdate(mSubSysDiagRes);
	}
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmSendReqSessionSwitch2Extended ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmSendReqSessionSwitch2Extended()");	

	mSubSysDiagReq.clear();
	mSubSysDiagReq.push_back(DIA_C_SUBSYSTEM_EXTENDED_SESSION_REQUEST_SRV_ID);
	mSubSysDiagReq.push_back(DIA_C_SUBSYSTEM_EXTENDED_SESSION_REQUEST_SES_ID);

	if ( mpDoIPClient )
	{
		//(void) mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->diagnosticReq(mSubSysDiagReq);
		DIA_TR_INF("SubsystemDiagnosisStrategyCore::vFsmSendReqSessionSwitch2Extended - Sending Extended session switch request ");
		mIsRequestRouted = mpDoIPClient->diagSendRequest(mRequestedNodeAddr, mSubSysDiagReq);
	}	
	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStartDEDelayTimer ( void* /* pArg */ )
{
   dia_tclFnctTrace trc("SubsystemDiagnosisStrategyCore::vFsmStartDEDelayTimer");
   
   if ( mpDoIPClient ) mpDoIPClient->startDiscoverEntitiesDelayTimer();  
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStartPMRDelayTimer ( void* /* pArg */ )
{
   dia_tclFnctTrace trc("SubsystemDiagnosisStrategyCore::vFsmStartPMRDelayTimer");
   
   if ( mpDoIPClient ) mpDoIPClient->startPowerModeRequestDelayTimer();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStartRADelayTimer ( void* /* pArg */ )
{
   dia_tclFnctTrace trc("SubsystemDiagnosisStrategyCore::vFsmStartRADelayTimer");
   
   if ( mpDoIPClient ) mpDoIPClient->startRoutingActivationDelayTimer();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStartSubsystemRoutingDelayTimer ( void* /* pArg */ )
{
   dia_tclFnctTrace trc("SubsystemDiagnosisStrategyCore::vFsmStartSubsystemRoutingDelayTimer");
   
   if ( mpDoIPClient ) mpDoIPClient->startSubsystemRoutingDelayTimer();
   
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStopDEDelayTimer ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmStopDEDelayTimer()");
	
	if ( mpDoIPClient ) mpDoIPClient->stopDiscoverEntitiesDelayTimer();
	
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStopPMRDelayTimer ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmStopPMRDelayTimer()");
	
	if ( mpDoIPClient ) mpDoIPClient->stopPowerModeRequestDelayTimer();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStopRADelayTimer ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmStopRADelayTimer()");
	
	if ( mpDoIPClient ) mpDoIPClient->stopRoutingActivationDelayTimer();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmStopSubsystemRoutingDelayTimer ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmStopSubsystemRoutingDelayTimer()");
	
	if ( mpDoIPClient ) mpDoIPClient->stopSubsystemRoutingDelayTimer();
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmTriggerESR ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmTriggerESR()");
	/*
	if ( mpDoIPClient ) 
	{
		//! iSocket_TCP_Client is the socket descriptor returned,
		//! if we able to successfully create socket. 
		//! This integer value or socket descriptor also know as sockfd in short form. 
		//mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->entityStatusReq(iSocket_TCP_Client);
		(void) mpDoIPClient->diagSendEntityStatusReq(mRequestedNodeAddr);
	}*/
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmTriggerPMR ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmTriggerPMR()");
	
	if ( mpDoIPClient ) 
	{
		//! iSocket_TCP_Client is the socket descriptor returned,
		//! if we able to successfully create socket. 
		//! This integer value or socket descriptor also know as sockfd in short form. 
		//mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->powermodeRequest(iSocket_TCP_Client);
		
		//! mIsPowerModeOk will be updated when the powermode response is received
		//! this is handled in callback fn from client - onPMRResponse(tU8 pwrMode)
		(void) mpDoIPClient->diagSendPowerModeReq(mRequestedNodeAddr);
	}

}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vFsmTriggerRAR ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vFsmTriggerRAR()");
	
	if ( mpDoIPClient ) 
	{ 
		//! Activation Type indicates the specific type of routing activation that may require 
		//! different types of authentication and/or confirmation. Defined values are:
		//!
		//! 0	    Default.
		//! 1	    WWH-OBD (worldwide harmonized onboard diagnostic).
		//! 0xE0	Use an OEM-pecific central security approach.
		//!
		//! Values 2 to 0xDF are reserved. Values 0xE0 to 0xFF are OEM specific.
		//mIsRoutingActivationOk = mpDoIPClient->vGetNodeByIP(mRequestedNodeAddr)->routingActivationReq(0 /* RA Type */, nullptr /* OEM Specific Data */);
		
		//! mIsRoutingActivationOk will be updated when the activation response is received
		//! this is handled in callback fn from client - onRAResponse(tU8 respCode)
		(void) mpDoIPClient->diagSendRoutingActivationReq(mRequestedNodeAddr);	
	}
}

void
SubsystemDiagnosisStrategyCore::vfsmCloseCurrentConnectiontoNode ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vfsmCloseCurrentConnectiontoNode()");
	
	bool res = false;
	
	if ( mpDoIPClient ) res = mpDoIPClient->closeNodeTCPSocket();
	
	if (res)  DIA_TR_INF("SubsystemDiagnosisStrategyCore:vfsmCloseCurrentConnectiontoNode - Closed");
}


//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::areConnectConditionsOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::areConnectConditionsOk()");
	
	return mAreConnectConditionsOk;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::areIdentifyConditionsOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::areIdentifyConditionsOk()");
	
	return mAreIdentifyConditionsOk;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::areRouteConditionsOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::areRouteConditionsOk()");
	
	return mAreRouteConditionsOk;
}


//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsActivationComplete ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsActivationComplete()");
	
	return mIsActivationComplete;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsDeactivationComplete ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsDeactivationComplete()");
	
	return mIsDeactivationComplete;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsEntityAlreadyConnected ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsEntityAlreadyConnected()");
	
	mIsEntityAlreadyConnected = false;
	
	if( mpDoIPClient )
	{
		mIsEntityAlreadyConnected = mpDoIPClient->isNodeAlreadyConnected(mRequestedNodeAddr);
	}
	
	DIA_TR_INF("bIsEntityAlreadyConnected:: Requested Node %s Connected", (mIsEntityAlreadyConnected == true) ? "already" : "not");
	
	return mIsEntityAlreadyConnected;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsEntityListEmpty ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsEntityListEmpty()");

	std::list<dia_tSubsystemInfo> doipNodes;
	mIsEntityListEmpty = true;

	if( mpDoIPClient )
	{
		doipNodes = mpDoIPClient->getDoIPClientNodes();
		mIsEntityListEmpty = (( doipNodes.empty() == true ) ? true : false );
	}
	
	return mIsEntityListEmpty;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsPowerModeOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsPowerModeOk()");
	
	return mIsPowerModeOk;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsPreCondChkOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsPreCondChkOk()");
	
	return mIsPrecondChkOk;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsRequestRouted ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsRequestRouted()");
	
	return mIsRequestRouted;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsRoutingActivationOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsRoutingActivationOk()");
	
	return mIsRoutingActivationOk;
}

//------------------------------------------------------------------------------

bool 
SubsystemDiagnosisStrategyCore::bIsTCPSocketOk ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::bIsTCPSocketOk()");
	
	return mIsTCPSocketOk;
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::onTimeout ( tU32 timerID )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::onTimeout(...)");
	
	if(mpDoIPClient->getTimerId_IE() == timerID)
	{
		DIA_TR_INF("### Identify Entity Timer Elapsed --- ID = %d ###", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evDEDTimeout, 0);
	}
	else if(mpDoIPClient->getTimerId_PMR() == timerID)
	{
		DIA_TR_INF("### Power Mode Request Timer Elapsed --- ID = %d ###", timerID);
		// This snippet is disabled for now. Once we have concrete idea of powermode request and response from the subsystem
		// We should enable this back
		 if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evPMRDelayTimeout, 0);
	}
	else if(mpDoIPClient->getTimerId_RA() == timerID)
	{
		DIA_TR_INF("### Routing Activation Timer Elapsed --- ID = %d ###", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evRADelayTimeout, 0);
	}
	else if(mpDoIPClient->getTimerId_RR() == timerID)
	{
		DIA_TR_INF("### Routing Request Timer Elapsed --- ID = %d ###", timerID);
		if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evOnRoutingDelayTimeout, 0);
	}
	else
	{
		DIA_TR_INF("### Unknown Timer --- ID = %d ###", timerID);
	}
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::onRAResponse(tU8 respCode)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::onRAResponse(...)");
	
	DIA_TR_INF("Routing Activation Response Code = %d", respCode);
	mIsRoutingActivationOk = (( DOIP_RA_RESPCODE_SUCCESS == respCode ) ? true : false );
	
	if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evOnRAResponse, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::onPMRResponse(tU8 pwrMode)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::onPMRResponse(...)");
	
	DIA_TR_INF("Power Mode Response Code = %d", pwrMode);
	mIsPowerModeOk = (( DIA_C_SUBSYSTEM_POWER_MODE_READY == pwrMode ) ? true : false );
	
	// This snippet is disabled for now. Once we have concrete idea of powermode request and response from the subsystem
	// We should enable this back
	 if ( mpFSM ) mpFSM->acceptEvent(dia_SubsystemDiagnosisFSM::evOnPMResponse, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vStartResponseTimer(void)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vStartResponseTimer(...)");
	
	oResponseTimer.addTimerListener(this);
	oResponseTimer.s32SetTime(0,0);
	oResponseTimer.s32SetTime(DIA_C_SUBSYSTEM_IDENTIFY_ENTITIES_INTERNAL_RESPONSE_DELAY, 0);
}

//------------------------------------------------------------------------------

void 
SubsystemDiagnosisStrategyCore::vStopResponseTimer(void)
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vStopResponseTimer(...)");
	
	oResponseTimer.s32SetTime(0,0);
    oResponseTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
SubsystemDiagnosisStrategyCore::vOnTimerElapsed ( dia_TimerID  id )
{
    dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::vOnTimerElapsed(...)");

	std::list<dia_tSubsystemInfo> doipNodes;
	
	if( mpDoIPClient )
	{
		doipNodes = mpDoIPClient->getDoIPClientNodes();
	}
	else
	{
		mIdentifyRoutineStatus = DIA_EN_IDENTIFY_ENTITIES_ABORTED;
	}

	dia_ISubsystemDiagnosisListener* pListener = OSAL_NULL;
    querySysAdapterListener<dia_ISubsystemDiagnosisListener>(&pListener);
    if ( pListener )
    {
    	pListener->vOnIdentifySubsystemsRequestUpdate(doipNodes, mIdentifyRoutineStatus);
    }   
}

//------------------------------------------------------------------------------

void
SubsystemDiagnosisStrategyCore::prepareDeactivate ( void* /* pArg */ )
{
	dia_tclFnctTrace oTrace("SubsystemDiagnosisStrategyCore::prepareDeactivate(...)");
	
	if(mpDoIPClient) 
	{
		// deactivate tester present
		mpDoIPClient->disableTesterPresent();
	
		// Close the current node connection
		(void)mpDoIPClient->closeNodeTCPSocket();
	}
}
