/*!
 * \file       dia_SubsystemDiagnosisRequestRouter.cpp
 *
 * \brief      router class (service handler) that routes the diagnostic commands from
 *             tester to the connected subsystem and routes the response from 
 *             connected subsystem back to the tester
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       17.10.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_REQUEST_ROUTER__
#include "dia_SubsystemDiagnosisRequestRouter.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include "common/framework/engine/dia_LookupKey.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_MANAGER__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisManager.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_SUBSYSTEM_DIAGNOSIS__
#include <project/framework/cis/subsystem/dia_defsSubsystemDiagnosis.h>
#endif

#define MSG_DATA_OFFSET 1

//-----------------------------------------------------------------------------

dia_SubsystemDiagnosisRequestRouter::dia_SubsystemDiagnosisRequestRouter( void )
	: dia_ServiceHandlerUDS("SUBSYSTEM_DIAGNOSIS_ROUTER")
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisRequestRouter::dia_SubsystemDiagnosisRequestRouter ( CONSTRUCTOR )");
}

//-----------------------------------------------------------------------------

dia_SubsystemDiagnosisRequestRouter::~dia_SubsystemDiagnosisRequestRouter( void )
{
}

//-----------------------------------------------------------------------------

void
dia_SubsystemDiagnosisRequestRouter::vProcessRequest ( const std::vector<tArgsType>& /* vecArgs */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisRequestRouter::vProcessRequest()");

	//! extract the data from the received message
	tU8 const* pU8 = oDiagMsgBuffer().u8GetBuffer();
	
	//! copy the request data into local buffer
	std::vector<tU8> requestData;
	for ( tU16 len = 0; len < (oDiagMsgBuffer().u16GetDataLength()- MSG_DATA_OFFSET); len++ )
	{
		requestData.push_back(pU8[len+MSG_DATA_OFFSET]);
	}
	
	// if at all it was there previously!
	(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisListener>(this);
	
	//! get the instance of subsystem manager 
	dia_SubsystemDiagnosisManager* pManager = getInstanceOfSubsystemDiagnosisManager();
	if( !pManager )
	{
		//! No manager available, impossible to process the request; send a -ve response.
		DIA_TR_ERR("dia_SubsystemDiagnosisRequestRouter: FAILED (NO ACCESS TO SUBSYSTEM MANAGER) !!");
		vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
		return;
	}
	
	//!  set the listener function to get an update later when the processing from the framework is done
	if (setSysAdapterListener<dia_ISubsystemDiagnosisListener>(this) != DIA_SUCCESS)
	{
		DIA_TR_ERR("dia_SubsystemDiagnosisRequestRouter: FAILED (NO REG AS LISTENER) !!");
		vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
		return;
	}

	tDiaResult retCode = pManager->routeRequestToSubsystem(requestData);
	if ( retCode != DIA_SUCCESS )
	{
		//! Sending back the Negative answer
		DIA_TR_ERR("dia_SubsystemDiagnosisRequestRouter: FAILED (SUBSYSTEM DIAG REQUEST) !!");
		
		(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisListener>(this);
		
		vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
	}
	
	//! Wait for update on listener interface
   
}

//----------------------------------------------------------------------

void
dia_SubsystemDiagnosisRequestRouter::vOnRouteRequestUpdate( std::vector<tU8>& responseData )
{
   dia_tclFnctTrace trc("dia_SubsystemDiagnosisRequestRouter::vOnRouteRequestUpdate");

   (void) unsetSysAdapterListener<dia_ISubsystemDiagnosisListener>(this);

   //! Sending back the positive answer
   oDiagMsgBuffer().vSetDataLength((tU16) (responseData.size()));
   for ( tU16 j=0; j<((tU16) responseData.size()); ++j )
   {
      (void) oDiagMsgBuffer().vSetDataU8(tU16(j), responseData.at(j));
   }

   vResReadyAndQuit();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SubsystemDiagnosisRequestRouter::makeLookupKeys ( std::vector<dia_LookupKey*>& keys )
{
   dia_tclFnctTrace trc("dia_SubsystemDiagnosisRequestRouter::makeLookupKeys");
   
   if ( mLookupKeys.empty() )
   {
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_AVBOS_VERSION,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_GEN_AVB_VERSION,     			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CONFIG_READ,     			    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_SPARE_PART_NUMBER,     			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_SERIAL_NUMBER,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_SW_VERSION,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_HW_VERSION,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_NAME,                      		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SYSTEM_IDENTIFICATION,          	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SW_FINGERPRINT,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CFG_FINGERPRINT,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AVB_ENTITY_ID,      				DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWITCH_ATU,            				DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DOIP_UDS_STACK_VERSIONS,            DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AVB_GRANDMASTER_ID,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CFG,            					DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWIVEL_SW_VERSION,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWIVEL_UUID,            		    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_VIN_ORIGINAL,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_VIN_CURRENT,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEVICE_TEMPERATURE,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_TEMPERATURE_STATISTICS,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SYSTEM_OPERATION_TIME,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SYSTEM_UPDATE,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_OABR_CONNECTION_STATUS_PORT_1,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_OABR_CONNECTION_STATUS_PORT_2,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_OABR_CONNECTION_STATUS_PORT_3,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_OABR_CONNECTION_STATUS_PORT_4,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_OABR_CONNECTION_STATUS_PORT_5,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECUREBOOT_SRK_FUSES,               DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECUREBOOT_ENV_ENCRYPTED,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECURITYROOT_FS_VERIFIED,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECURITYDATA_IS_ENCRYPTED,          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECUREBOOT_UBOOT_SRK_HASH,          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECUREBOOT_DEVICETREE_SRK_HASH,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECUREBOOT_IMXKERNEL_SRK_HASH,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_MAC_ADDRESS_IMX6,            DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_CPU_ID,                      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_POWER_SUPPLY_VOLTAGE,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AVG_CPU_LOAD,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_MAX_RAM_CONSUMPTION,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_STARTUP_TIME_STATISTIC,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); 
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_LOUD_SPEAKER_STATUS,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_STATUS,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_MICROPHONES_STATUS,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_WAKEUP_LINES_STATUS,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DIGITAL_OUTPUTS_STATUS,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_CONNECTION_STATUS,         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_CON_STATUS_HANDHELDS,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_CONVERSATION_COUNTER,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ACTIVE_DIAGNOSTIC_SESSION,          DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_USB_DEVICEINFO,              DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_INPUTLINES_STATUS,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_GNSS_RECEIVER_STATUS,        DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_GNSS_ANTENNA_STATUS,         DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_MOBILE_NW_ANTENNA_STATUS,    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_MOBILE_NW_FIELDSTRENGTH,     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEVICE_PLACE_OF_PRODUCTION,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, 		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEVICE_PRODUCTION_TIME,            	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CAM_RELAY_ACTIVATION,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AMP_SHUTDOWN_TIMEOUT,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_THRESHOLDS,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_RELEASETIME,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_WAKEUP_ALIVE_CONFIGURATION,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWIVEL_CONFIGURATION,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DISPLAY_SETTINGS,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_OFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_IN_OFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_MIC_OFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_OFFSET_GAIN_MIC,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_OFFSET_GAIN_DIGITAL,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_AVAILABLE  ,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_MIXOFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_PHONEBOX_VOLUME,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CAMERA_USERNAME,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CAMERA_PASSWORD,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_MIC_OFFSET_GAIN_TO_ICOM_DIGITAL,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DISPLAY_TEST_SCREEN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );//Diagnosis Changes- new DID (0xC280) via CAN diagnostic-porting to Iguacu branch
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_PSK_ADMIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WEB_PASSWORD_ADMIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); 
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_NODE_ID,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_MODE,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_PSK_USER,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );

      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_BOOT_CONFIG,                   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_REVOKE_SRK_REG,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_MQTT_PASSWORD,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECURE_JTAG,                   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_SSID_ADMIN,       DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_SSID_PASSENGER,   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_MAC_2G,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_MAC_5G,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_WRITE_KEY_FOR_SECURITY_ACCESS, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AVB_ENTITY_ID,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AVB_GRANDMASTER_ID,					DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CFG,            					DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWIVEL_SW_VERSION,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWIVEL_UUID,            		    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_VIN_ORIGINAL,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_VIN_CURRENT,            			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER, 	(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CONFIG_WRITE,            		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CAM_RELAY_ACTIVATION,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_AMP_SHUTDOWN_TIMEOUT,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_THRESHOLDS,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_RELEASETIME,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_WAKEUP_ALIVE_CONFIGURATION,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SWIVEL_CONFIGURATION,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DISPLAY_SETTINGS,                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_OFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_IN_OFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_MIC_OFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_OFFSET_GAIN_MIC,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_OFFSET_GAIN_DIGITAL,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_AVAILABLE  ,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_OUT_MIXOFFSET_GAIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_INTERCOM_PHONEBOX_VOLUME,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CAMERA_USERNAME,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_CAMERA_PASSWORD,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_IP_MIC_OFFSET_GAIN_TO_ICOM_DIGITAL,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_PSK_ADMIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WEB_PASSWORD_ADMIN,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_NODE_ID,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_MODE,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,  (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_PSK_USER,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_BOOT_CONFIG,                   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_REVOKE_SRK_REG,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_MQTT_PASSWORD,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SECURE_JTAG,                   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_SSID_ADMIN,       DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_SSID_PASSENGER,   DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_MAC_2G,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DEFAULT_WIFI_MAC_5G,           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_WRITE_DATA_BY_IDENTIFIER,     (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_WRITE_KEY_FOR_SECURITY_ACCESS, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_START_SW_UPDATE,                    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,         		(tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_PREPARE_SW_UPDATE,      			    DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_RELOAD_SUPPLIER_DELIVERYSTATE,      DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ROUTER_GNSS_RECEIVER_STATUS,        DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CABIN_IN_OFFSET_GAIN,               DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );

      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_START_REMOTE_CONSOLE,               DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ENABLE_UART_CONSOLE,                DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DID_TC_ID_413_SECUREBOOT_ACTIVATOR, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_DID_TC_ID_406_AVB_TEST,             DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_RCID_TC_ID_403_GPS,                 DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ROUTINE_CONTROL,             (tU16) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_USB_TEST,                           DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
		
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_SECURITY_ACCESS,              (tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SAC_LVL_PGMNG_3_SEED,              DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
     mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_SECURITY_ACCESS,              (tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SAC_LVL_PGMNG_3_KEY,                  DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );  
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_SECURITY_ACCESS,         		(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SAC_LVL_CFG_5_SEED,      			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_SECURITY_ACCESS,         		(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_SAC_LVL_CFG_5_KEY,      				DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
		
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_COMMUNICATION_CONTROL,    	(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_COM_CTRL_ENABLE_TX,      			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_COMMUNICATION_CONTROL,    	(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_COM_CTRL_DISABLE_TX,      			DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
		
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_ECU_RESET,         			(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_ECU_HARD_RST,      					DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
		
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_CONTROL_DTC_SETTING,      	(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CTRL_DTC_SETTING_ACTIVATE,      		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_CONTROL_DTC_SETTING,      	(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_CTRL_DTC_SETTING_SUPPRESS,      		DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
		
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_CLEAR_DIAG_INFO,         		(tU8) UDS_U8_PARAMETER_NOT_USED,      										DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
		
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DTC_INFO,         		(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_FM_READ_DTC_RPRT_NUM_BY_SM,      	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DTC_INFO,         		(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_FM_READ_DTC_RPRT_DTC_BY_SM,      	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
	  mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DTC_INFO,         		(tU8) DIA_C_U16_DID_SUBSYSTEM_DIAG_REQ_FM_READ_DTC_RPRT_DTC_SPRTD,      	DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );

}

   keys = mLookupKeys;

   return DIA_SUCCESS;
}
