/*!
 * \file       dia_SubsystemDiagnosisDoIPClient.cpp
 *
 *
 * \brief      ...
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Kirty Mayank (RBEI/ECA2), Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       25.09.2019
 *
 * \copyright  (c) 2019 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SUBSYSTEM_DIAGNOSIS_DOIP_CLIENT__
#include "project/framework/cis/subsystem/dia_SubsystemDiagnosisDoIPClient.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define ENTITY_ID_LENGTH 6

dia_SubsystemDiagnosisDoIPClient::dia_SubsystemDiagnosisDoIPClient(const char* pszDevice, tU16 u16Port, tU16 testerAddress, DoIP_Cfg &doipCfg)
	: 	DoIP_Tester(pszDevice, u16Port, testerAddress, doipCfg),
		mpCallback(nullptr),
		mSizeInBytes(7),
		pmActiveDoIP_Node(nullptr),
		mIsTesterPresentActive(false),
		poNode(NULL)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::dia_SubsystemDiagnosisDoIPClient ( const char* pszDevice, tU16 u16Port, tU16 testerAddress, DoIP_Cfg &doipCfg )");
	
	oIEDelayTimer.s32Create();
	mIEDelayTimerID = oIEDelayTimer.getID();
	
	oPMRDelayTimer.s32Create();
	mPMRDelayTimerID = oPMRDelayTimer.getID();
	
	oRADelayTimer.s32Create();
	mRADelayTimerID = oRADelayTimer.getID();
	
	oRRDelayTimer.s32Create();
	mRRDelayTimerID = oRRDelayTimer.getID();
	
	oTesterPresentTimer.s32Create();
	mTesterPresentTimerID = oTesterPresentTimer.getID();
	
	oSlaveProcessTimer.s32Create();
	mSlaveProcessTimerID = oSlaveProcessTimer.getID();
}

//------------------------------------------------------------------------------

dia_SubsystemDiagnosisDoIPClient::~dia_SubsystemDiagnosisDoIPClient(void)
{
	DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::~dia_SubsystemDiagnosisDoIPClient ( void )");

	//_BP_TRY_BEGIN
	 //  {
			oIEDelayTimer.removeTimerListener(this);
			oIEDelayTimer.s32Delete();
			mIEDelayTimerID = DIA_C_INVALID_TIMER_ID;

			oPMRDelayTimer.removeTimerListener(this);
			oPMRDelayTimer.s32Delete();
			mPMRDelayTimerID = DIA_C_INVALID_TIMER_ID;

			oRADelayTimer.removeTimerListener(this);
			oRADelayTimer.s32Delete();
			mRADelayTimerID = DIA_C_INVALID_TIMER_ID;

			oRRDelayTimer.removeTimerListener(this);
			oRRDelayTimer.s32Delete();
			mRRDelayTimerID = DIA_C_INVALID_TIMER_ID;
			
			oTesterPresentTimer.removeTimerListener(this);
			oTesterPresentTimer.s32Delete();
			mTesterPresentTimerID = DIA_C_INVALID_TIMER_ID;
			
			oSlaveProcessTimer.removeTimerListener(this);
			oSlaveProcessTimer.s32Delete();
			mSlaveProcessTimerID = DIA_C_INVALID_TIMER_ID;
			
	//   	}
	 //  _BP_CATCH_ALL
	 //  {
	 //     DIA_TR_ERR("EXCEPTION CAUGHT: dia_SubsystemDiagnosisDoIPClient::~dia_SubsystemDiagnosisDoIPClient !!!");
	 //     DIA_ASSERT_ALWAYS();
	 //  }
	 //  _BP_CATCH_END

	   delete poNode;
	   poNode = NULL;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::startDoIPComm( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::startDoIPComm()");
	
	//return ((DIA_SUCCESS == Start()) ? true : false);

	dia_ISubsystemDiagnosisProxy* pInterface = nullptr;
	tDiaResult retCode = queryInterface<dia_ISubsystemDiagnosisProxy>(&pInterface);
	
	if ((retCode == DIA_SUCCESS) && pInterface)
	{
		(void) setSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);

		if (DIA_SUCCESS == pInterface->vStartUdpBroadcastClient())
		{
			DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient --> Request to vStartUdpBroadcastClient sent.");
		}
		else
		{
			DIA_TR_ERR("dia_SubsystemDiagnosisDoIPClient --> Request to vStartUdpBroadcastClient failed.");

			(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		}
	}
	
	return true;
	/*
	if(DIA_SUCCESS == Start())
	{
		DIA_TR_INF("startDoIPComm :: STARTED "); return true;;
	}
	else
	{
		DIA_TR_INF("startDoIPComm :: COULD NOT START "); return false;
	}
	*/
}

void
dia_SubsystemDiagnosisDoIPClient::onStartUDPComm ( tDiaResult result )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::onStartUDPComm");

   (void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);

   if (result == DIA_SUCCESS)
   {
      DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::onStartUDPComm result DIA_SUCCESS.");
	  DIA_TR_INF(" startDoIPComm :: %s", ((DIA_SUCCESS == Start()) ? "STARTED" : "COULD NOT START"));
   }
   else
   {
      DIA_TR_ERR("dia_SubsystemDiagnosisDoIPClient::onStartUDPComm result DIA_FAILURE.");
   }  
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::stopDoIPComm( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::stopDoIPComm()");
	
	return ((DIA_SUCCESS == Stop()) ? true : false);

}

//------------------------------------------------------------------------------

bool 
dia_SubsystemDiagnosisDoIPClient::isNodeAvailable(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::isNodeAvailable(...)");
	
	return ((DIA_SUCCESS == doesNodeExist(mNodeIP)) ? true : false);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::discoverEntities(void)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::discoverEntities(...)");
	
	//this->vdiscoverEntities(DIA_S_SUBSYSTEM_DOIP_BROADCAST_ADDR);
	
	dia_ISubsystemDiagnosisProxy* pInterface = nullptr;
	tDiaResult retCode = queryInterface<dia_ISubsystemDiagnosisProxy>(&pInterface);
	
	if ((retCode == DIA_SUCCESS) && pInterface)
	{
		(void) setSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		if (DIA_SUCCESS == pInterface->vBroadcastVehIdRequest())
		{
			DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient --> Request to discover sent.");
		}
		else
		{
			DIA_TR_ERR("dia_SubsystemDiagnosisDoIPClient --> Request to discover failed.");

			(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		}
	}
	
	// start the timer to send the request results message to slave process
	oSlaveProcessTimer.addTimerListener(this);
	oSlaveProcessTimer.s32SetTime(0,0);
	oSlaveProcessTimer.s32SetTime(DIA_C_SUBSYSTEM_SLAVE_PROCESS_TIMER_VAL, 0);
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::vFetchNodeListFromSlaveProcess(void)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::vFetchNodeListFromSlaveProcess(...)");
		
	dia_ISubsystemDiagnosisProxy* pInterface = nullptr;
	tDiaResult retCode = queryInterface<dia_ISubsystemDiagnosisProxy>(&pInterface);
	
	if ((retCode == DIA_SUCCESS) && pInterface)
	{
		(void) setSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		if (DIA_SUCCESS == pInterface->vBroadcastVehIdResults())
		{
			DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient --> Request to discover results sent.");
		}
		else
		{
			DIA_TR_ERR("dia_SubsystemDiagnosisDoIPClient --> Request to discover results failed.");

			(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		}
	}
	
}

void
dia_SubsystemDiagnosisDoIPClient::onBroadcastVehIdResults ( tDiaResult /*result*/, ::std::vector< subsystem > nodeList )
{
   dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::onBroadcastVehIdResults");

   (void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
   
  // for(std::vector<subsystem>::iterator doip_node = std::begin(nodeList); doip_node != std::end(nodeList); ++doip_node)
	for(auto doip_node : nodeList )
   {
		struct sockaddr_in remoteaddr;
		remoteaddr.sin_family = doip_node.sin_family;
		remoteaddr.sin_port = doip_node.sin_port;
		remoteaddr.sin_addr.s_addr = doip_node.s_addr;
		for(int i=0; i<8; i++)
		{
			remoteaddr.sin_zero[i] = doip_node.sin_zero[i];
		}
		
		vAddNode(0xFF1, remoteaddr, doip_node.logAdr, doip_node.EID, doip_node.GID, doip_node.VIN, DOIP_NODE, 0);
		//vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, const tU8* EID, const tU8* GID, const tU8* VIN, DoIP_NodeType nodeType, tU32 maxDataSize)
		
   }
   
   
}

//------------------------------------------------------------------------------

DoIP_Node* 
dia_SubsystemDiagnosisDoIPClient::vAddNode(DoIP_Node* node)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::vAddNode(DoIP_Node* node)");
	return NULL;
}

//------------------------------------------------------------------------------

DoIP_Node* 
dia_SubsystemDiagnosisDoIPClient::vAddNode(tU16 /*testerAddress*/, sockaddr_in /*IP*/, tU16 /*logAddr*/, tU32 /*maxDataSize*/)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, tU32 maxDataSize)");
	
	DoIP_Node* poNode = NULL;
	/*
	DoIP_NodeType nodeType;

	poNode = new DoIP_Node(testerAddress,IP,logAddr,nullptr,nullptr,nullptr,nodeType,maxDataSize,nullptr);
	
	if(!poNode){
	oDoIP_Nodes.push_back(poNode);}
	*/
	return (poNode);
}

//------------------------------------------------------------------------------

DoIP_Node* 
dia_SubsystemDiagnosisDoIPClient::vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, const tU8* EID, const tU8* GID, const tU8* VIN, DoIP_NodeType nodeType, tU32 maxDataSize)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, const tU8* EID, const tU8* GID, const tU8* VIN, DoIP_NodeType nodeType, tU32 maxDataSize)");
	DoIP_Node* poNode = NULL;
	
	DoIP_Cfg doipCfg;
	bool nodeAlreadyPresent = false;

	poNode = new DoIP_Node(testerAddress,IP,logAddr,EID,GID,VIN,nodeType,maxDataSize,doipCfg);
	
	DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient:vAddNode -- 0x%08X", IP.sin_addr.s_addr);

	std::list<DoIP_Node*>::iterator itNode;
	for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
	{
		if( (*itNode)->getIPAddr() == IP.sin_addr.s_addr )
		{
			DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::vAddNode - Node already present (IP check)");
			nodeAlreadyPresent = true;
			break;
		}
	}

	if(!nodeAlreadyPresent)
	{
		DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::vAddNode - New Node Added ");
		oDoIP_Nodes.push_back(poNode);
	}

	return (poNode);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::onNodeDiagnosticResponse(tU8 const * data, tU32 U32len)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::onNodeDiagnosticResponse(...)");
	
	// Check for tester present response and if yes, ignore the same
	if((data[DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SRV_ID) && 
	   (data[DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SUB_FUNC_ID_IDX] == DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SUB_FUNC_ID))
	{
		DIA_TR_INF("### Tester Present Response ###");
		return;
	}
	
	// Check for session change response and if yes, ignore the same
	if((data[DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SRV_ID) && 
	   (data[DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SES_ID_IDX] == DIA_C_SUBSYSTEM_EXTENDED_SESSION_POS_RESP_SES_ID))
	{
		DIA_TR_INF("### Extended Session Response ###");
		return;
	}
	
	//! call back to parent to validate the response
	if (mpCallback) mpCallback->onDiagnosticResponse(data, U32len);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::onRoutingActivationResponse(tU8 respCode)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::onRoutingActivationResponse(...)");

	if(!pmActiveDoIP_Node) return;
	
	if(DOIP_RA_RESPCODE_SUCCESS != respCode) 
	{
		//! reset the callback to client as the routing activation request failed.
		pmActiveDoIP_Node->setDoIPCallback(nullptr);
	} 
	
	//! call back to parent to validate the response
	if (mpCallback) mpCallback->onRAResponse(respCode);
}

//------------------------------------------------------------------------------
/*	
void 
dia_SubsystemDiagnosisDoIPClient::diagnosticACK(tU8 ackCode, tU8* data, tU32 U32len)
{
	//! Can be ignored
}
	
//------------------------------------------------------------------------------
	
void 
dia_SubsystemDiagnosisDoIPClient::diagnosticNACK(tU8 nackCode, tU8* data, tU32 U32len)
{
	//! Can be ignored
}

//------------------------------------------------------------------------------
*/
	
void
dia_SubsystemDiagnosisDoIPClient::onPowerModeRequestResponse(tU8 pwrMode)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::onPowerModeRequestResponse(tU8 pwrMode)");
	
	if(!pmActiveDoIP_Node) return;
	
	if(DIA_C_SUBSYSTEM_POWER_MODE_READY != pwrMode) 
	{
		//! reset the callback to client as the power mode request failed.
		pmActiveDoIP_Node->setDoIPCallback(nullptr);
	}
	
	//! call back to parent to validate the response
	if (mpCallback) mpCallback->onPMRResponse(pwrMode);
}

//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::stopDiscoverEntitiesDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::stopDiscoverEntitiesDelayTimer(...)");
	
	oIEDelayTimer.s32SetTime(0,0);
    oIEDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::startDiscoverEntitiesDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::startDiscoverEntitiesDelayTimer(...)");
	
	oIEDelayTimer.addTimerListener(this);
	oIEDelayTimer.s32SetTime(0,0);
	oIEDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_IDENTIFY_ENTITIES_DELAY, 0);
}


//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::stopPowerModeRequestDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::stopPowerModeRequestDelayTimer(...)");
		
	oPMRDelayTimer.s32SetTime(0,0);
    oPMRDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::startPowerModeRequestDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::startPowerModeRequestDelayTimer(...)");
	   
	oPMRDelayTimer.addTimerListener(this);
	oPMRDelayTimer.s32SetTime(0,0);
	oPMRDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_POWER_MODE_REQUEST_DELAY, 0);
}


//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::stopRoutingActivationDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::stopRoutingActivationDelayTimer(...)");
		
	oRADelayTimer.s32SetTime(0,0);
    oRADelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::startRoutingActivationDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::startRoutingAactivationDelayTimer(...)");
	   
	oRADelayTimer.addTimerListener(this);
	oRADelayTimer.s32SetTime(0,0);
	oRADelayTimer.s32SetTime(DIA_C_SUBSYSTEM_ROUTING_ACTIVATION_DELAY, 0);
}


//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::stopSubsystemRoutingDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::stopSubsystemRoutingDelayTimer(...)");
		
	oRRDelayTimer.s32SetTime(0,0);
    oRRDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::startSubsystemRoutingDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::startSubsystemRoutingDelayTimer(...)");
	
	oRRDelayTimer.addTimerListener(this);
	oRRDelayTimer.s32SetTime(0,0);
	oRRDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_REQUEST_ROUTING_DELAY, 0);
}

//------------------------------------------------------------------------------

void
dia_SubsystemDiagnosisDoIPClient::vOnTimerElapsed ( dia_TimerID id )
{
    // dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::vOnTimerElapsed(...)");

	tU32 timerID = (tU32) id;
	
	if( mTesterPresentTimerID == timerID )
	{
		if ( mIsTesterPresentActive )
		{
			DIA_TR_INF("### Tester Present Active ###");
			sendTesterPresentRequest();
		}
		else
		{
			DIA_TR_INF("### Tester Present Inactive ###");
		}
	}
	else if(mSlaveProcessTimerID == timerID)
	{
		DIA_TR_INF("### Slave Timer Elapsed ###");
		vFetchNodeListFromSlaveProcess();
	}
	else
	{
		DIA_TR_INF("### dia_SubsystemDiagnosisDoIPClient::vOnTimerElapsed --- ID = %d ###", timerID);
		//! call back to parent to validate the response
		if (mpCallback) mpCallback->onTimeout(timerID);		   
	}
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::setCurrentNode(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::setCurrentNode(tU32 mNodeIP)");
	
	//! retrieve the pointer to the doip node
	pmActiveDoIP_Node =	this->vGetNodeByIP(mNodeIP);
	
	if(pmActiveDoIP_Node)
	{
		DIA_TR_INF("### dia_SubsystemDiagnosisDoIPClient::setCurrentNode --- Active Node : 0x%02x", pmActiveDoIP_Node->logicalAddr );
	}
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemDiagnosisDoIPClient::createNodeTCPSocket( tU32 /* mNodeIP */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::createTCPSocket(...)");
	
	if (pmActiveDoIP_Node) 
	{
		return (pmActiveDoIP_Node->createTCPSocket());
	}
	
	return false;
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemDiagnosisDoIPClient::connectNodeTCP( tU32 /* mNodeIP */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::connectNodeTCP(...)");
	
	if (pmActiveDoIP_Node) 
	{
		return (pmActiveDoIP_Node->connectTCP());
	}
	
	return false;
	
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemDiagnosisDoIPClient::closeNodeTCPSocket()
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::closeNodeTCPSocket()");
	
	if (pmActiveDoIP_Node)
	{
		return (pmActiveDoIP_Node->closeTCPSocket());
	}
	
	return false;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::diagSendRequest(tU32 /* mNodeIP */, std::vector<tU8> diagRequest)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::diagSendRequest(...)");
	
	if (pmActiveDoIP_Node)
	{
		return (pmActiveDoIP_Node->diagnosticReq(diagRequest));
	}
	
	return false;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::diagSendEntityStatusReq(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::diagSendEntityStatusReq(...)");
	
	int iSocket_TCP_Client;
	
	if (pmActiveDoIP_Node) 
	{
		iSocket_TCP_Client = pmActiveDoIP_Node->getTCPClientSocket();
		DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::diagSendEntityStatusReq : TCP Socket = 0x%02x", iSocket_TCP_Client);
		pmActiveDoIP_Node->entityStatusReq(iSocket_TCP_Client);
	}
	
	return true;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::diagSendPowerModeReq(tU32 /* mNodeIP */ )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::diagSendPowerModeReq(...)");
	
	int iSocket_UDP_Client;
	
	if (pmActiveDoIP_Node) 
	{
		//! set the callback to client
		pmActiveDoIP_Node->setDoIPCallback(this);
		//! retrieve the socket id of the doip tester
		iSocket_UDP_Client = getUDPClientSocket();
		//! send power mode request
		pmActiveDoIP_Node->powermodeRequest(iSocket_UDP_Client);
	}
	
	return true;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::diagSendRoutingActivationReq(tU32 /* mNodeIP */)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::diagSendRoutingActivationReq(...)");
	
	if (!pmActiveDoIP_Node) return false;
	
	pmActiveDoIP_Node->setDoIPCallback(this); 
	
	return (pmActiveDoIP_Node->routingActivationReq( DIA_C_SUBSYSTEM_ROUTING_ACTIVATION_TYPE /* RA Type */, nullptr /* OEM Specific Data */));
}

//------------------------------------------------------------------------------

bool
dia_SubsystemDiagnosisDoIPClient::isDoIPListEmpty( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::isDoIPListEmpty(...)");
	
	return (this->oDoIP_Nodes.empty());
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemDiagnosisDoIPClient::isNodeAlreadyConnected(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::isNodeAlreadyConnected(tU32 mNodeIP)");
	
	tU32 ConnectedNodeIP;
	DIA_TR_INF("isNodeAlreadyConnected:: Requested Node IP = 0x%08X ", mNodeIP);
	if(!pmActiveDoIP_Node) return false;
	
	ConnectedNodeIP = pmActiveDoIP_Node->getIPAddr();
	
	DIA_TR_INF("isNodeAlreadyConnected:: Connected Node IP = 0x%08X ", ConnectedNodeIP);
	
	return ( ConnectedNodeIP == mNodeIP ) ? true : false;
	
	
}

//------------------------------------------------------------------------------

std::list<dia_tSubsystemInfo> 
dia_SubsystemDiagnosisDoIPClient::getDoIPClientNodes(void)
{
    dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::getDoIPClientNodes");

    mDoipNodes.clear();

    if(!(isDoIPListEmpty()))
    {
    	std::list<DoIP_Node*>::iterator it = oDoIP_Nodes.begin();

    	for( ; it != oDoIP_Nodes.end(); ++it)
    	{

    		DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::getDoIPClientNodes DOIP NODES AVAILABLE");

    		tU32 IPAddr =  ((*it)->getIPAddr());
    		const tU8* cGID = ((*it)->getGID());
    		const tU8* cEID = ((*it)->getEID());

    		tU16 entityName = ((cGID[0]) << 8)|(cGID[1]);
    		tU16 swVariant = ((cGID[4]) << 8)|(cGID[5]);

    		dia_tSubsystemInfo tSubsystemInfo;
    		tSubsystemInfo.subsystemEntityName = entityName;
    		tSubsystemInfo.subsystemSupplierName = cGID[2];
    		tSubsystemInfo.subsystemHWVariant = cGID[3];
    		tSubsystemInfo.subsystemSWVariant = swVariant;
    		tSubsystemInfo.subsystemIPAddress = IPAddr;
    		for(int i = 0; i<ENTITY_ID_LENGTH; ++i)
    		{
    			tSubsystemInfo.subsystemEntityID[i] = *(cEID+i);
    		}

    		mDoipNodes.push_back(tSubsystemInfo);
    	}
    }
    else
    {
    	DIA_TR_INF("dia_SubsystemDiagnosisDoIPClient::getDoIPClientNodes DOIP LIST IS EMPTY");
    }
    return mDoipNodes;
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::clearDoIPNodeList( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::clearDoIPNodeList");
	
	oDoIP_Nodes.clear();
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::enableTesterPresent( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::enableTesterPresent");
	
	oTesterPresentTimer.addTimerListener(this);
	oTesterPresentTimer.s32SetTime(0,0);
	oTesterPresentTimer.s32SetTime(DIA_C_SUBSYSTEM_TESTER_PRESENT_TIMER_VAL_FIRST, 
								   DIA_C_SUBSYSTEM_TESTER_PRESENT_TIMER_VAL);
								   
	mIsTesterPresentActive = true;
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::disableTesterPresent( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::disableTesterPresent");
	
	oTesterPresentTimer.s32SetTime(0,0);
    oTesterPresentTimer.removeTimerListener(this);
	
	mIsTesterPresentActive = false;
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemDiagnosisDoIPClient::sendTesterPresentRequest( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemDiagnosisDoIPClient::sendTesterPresentRequest");
	
	std::vector<tU8> testerPresent;
	testerPresent.clear();
	testerPresent.reserve(DIA_C_SUBSYSTEM_TESTER_PRESENT_REQ_LEN);
	
	testerPresent.push_back(DIA_C_SUBSYSTEM_TESTER_PRESENT_REQ_SRV_ID);
	testerPresent.push_back(DIA_C_SUBSYSTEM_TESTER_PRESENT_REQ_SUB_FUNC_ID);
	
	if (pmActiveDoIP_Node)
	{
		pmActiveDoIP_Node->diagnosticReq(testerPresent);
	}
	
}
