/*!
 * \file       dia_SubsystemLogoInstallationDoIPClient.cpp
 *
 *
 * \brief      ...
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    subsystem diagnosis
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       17.03.2020
 *
 * \copyright  (c) 2020 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_DOIP_CLIENT__
#include "project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationDoIPClient.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#define ENTITY_ID_LENGTH 6

dia_SubsystemLogoInstallationDoIPClient::dia_SubsystemLogoInstallationDoIPClient(const char* pszDevice, tU16 u16Port, tU16 testerAddress, DoIP_Cfg &doipCfg)
	: 	DoIP_Tester(pszDevice, u16Port, testerAddress, doipCfg),
		mpCallback(nullptr),
		mSizeInBytes(7),
		pmActiveDoIP_Node(nullptr),
		mIsTesterPresentActive(false),
		poNode(NULL)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::dia_SubsystemLogoInstallationDoIPClient ( const char* pszDevice, tU16 u16Port, tU16 testerAddress, DoIP_Cfg &doipCfg )");
	
	oIEDelayTimer.s32Create();
	mIEDelayTimerID = oIEDelayTimer.getID();
	
	oPMRDelayTimer.s32Create();
	mPMRDelayTimerID = oPMRDelayTimer.getID();
	
	oRADelayTimer.s32Create();
	mRADelayTimerID = oRADelayTimer.getID();
	
	oRRDelayTimer.s32Create();
	mRRDelayTimerID = oRRDelayTimer.getID();
	
	oTesterPresentTimer.s32Create();
	mTesterPresentTimerID = oTesterPresentTimer.getID();
	
	oSlaveProcessTimer.s32Create();
	mSlaveProcessTimerID = oSlaveProcessTimer.getID();
	
	oLTDelayTimer.s32Create();
	mLTDelayTimerID = oLTDelayTimer.getID();
}

//------------------------------------------------------------------------------

dia_SubsystemLogoInstallationDoIPClient::~dia_SubsystemLogoInstallationDoIPClient(void)
{
	DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::~dia_SubsystemLogoInstallationDoIPClient ( void )");

	delete poNode;
	poNode = NULL;
	_BP_TRY_BEGIN
	   {
			oIEDelayTimer.removeTimerListener(this);
			oIEDelayTimer.s32Delete();
			mIEDelayTimerID = DIA_C_INVALID_TIMER_ID;

			oPMRDelayTimer.removeTimerListener(this);
			oPMRDelayTimer.s32Delete();
			mPMRDelayTimerID = DIA_C_INVALID_TIMER_ID;

			oRADelayTimer.removeTimerListener(this);
			oRADelayTimer.s32Delete();
			mRADelayTimerID = DIA_C_INVALID_TIMER_ID;

			oRRDelayTimer.removeTimerListener(this);
			oRRDelayTimer.s32Delete();
			mRRDelayTimerID = DIA_C_INVALID_TIMER_ID;
			
			oTesterPresentTimer.removeTimerListener(this);
			oTesterPresentTimer.s32Delete();
			mTesterPresentTimerID = DIA_C_INVALID_TIMER_ID;
			
			oSlaveProcessTimer.removeTimerListener(this);
			oSlaveProcessTimer.s32Delete();
			mSlaveProcessTimerID = DIA_C_INVALID_TIMER_ID;
			
			oLTDelayTimer.removeTimerListener(this);
			oLTDelayTimer.s32Delete();
			mLTDelayTimerID = DIA_C_INVALID_TIMER_ID;
			
	   	}
	   _BP_CATCH_ALL
	   {
	      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SubsystemLogoInstallationDoIPClient::~dia_SubsystemLogoInstallationDoIPClient !!!");
	      DIA_ASSERT_ALWAYS();
	   }
	   _BP_CATCH_END
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::startDoIPClient( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::startDoIPClient()");
	
	//return ((DIA_SUCCESS == Start()) ? true : false);

	dia_ISubsystemDiagnosisProxy* pInterface = nullptr;
	tDiaResult retCode = queryInterface<dia_ISubsystemDiagnosisProxy>(&pInterface);
	
	if ((retCode == DIA_SUCCESS) && pInterface)
	{
		(void) setSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);

		if (DIA_SUCCESS == pInterface->vStartUdpBroadcastClient())
		{
			DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient --> Request to vStartUdpBroadcastClient sent.");
		}
		else
		{
			DIA_TR_ERR("dia_SubsystemLogoInstallationDoIPClient --> Request to vStartUdpBroadcastClient failed.");

			(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		}
	}
	
	return true;
	/*
	if(DIA_SUCCESS == Start())
	{
		DIA_TR_INF("startDoIPComm :: STARTED "); return true;;
	}
	else
	{
		DIA_TR_INF("startDoIPComm :: COULD NOT START "); return false;
	}
	*/
}

void
dia_SubsystemLogoInstallationDoIPClient::onStartUDPComm ( tDiaResult result )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::onStartUDPComm");

   (void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);

   if (result == DIA_SUCCESS)
   {
      DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::onStartUDPComm result DIA_SUCCESS.");
	  //DIA_TR_INF(" startDoIPComm :: %s", ((DIA_SUCCESS == Start()) ? "STARTED" : "COULD NOT START"));
   }
   else
   {
      DIA_TR_ERR("dia_SubsystemLogoInstallationDoIPClient::onStartUDPComm result DIA_FAILURE.");
   }  
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::stopDoIPClient( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::stopDoIPClient()");
	
	return ((DIA_SUCCESS == Stop()) ? true : false);

}

//------------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationDoIPClient::isNodeAvailable(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::isNodeAvailable(...)");
	
	return ((DIA_SUCCESS == doesNodeExist(mNodeIP)) ? true : false);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::discoverEntities(void)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::discoverEntities(...)");
	
	// Since Vehicle Identification Request needs ROOT access to send via the broadcast address
	// 255.255.255.255, So we make use of a slave process with root rights that can do this activity
	
	dia_ISubsystemDiagnosisProxy* pInterface = nullptr;
	tDiaResult retCode = queryInterface<dia_ISubsystemDiagnosisProxy>(&pInterface);
	
	if ((retCode == DIA_SUCCESS) && pInterface)
	{
		(void) setSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		if (DIA_SUCCESS == pInterface->vBroadcastVehIdRequest())
		{
			DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient --> Vehicle Identification Request Sent.");
		}
		else
		{
			DIA_TR_ERR("dia_SubsystemLogoInstallationDoIPClient --> Failed to send Vehicle Identification Request.");

			(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		}
	}
	
	// start the timer to send the request results message to slave process
	oSlaveProcessTimer.addTimerListener(this);
	oSlaveProcessTimer.s32SetTime(0,0);
	oSlaveProcessTimer.s32SetTime(DIA_C_SUBSYSTEM_SLAVE_PROCESS_TIMER_VAL, 0);
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::vFetchNodeListFromSlaveProcess(void)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::vFetchNodeListFromSlaveProcess(...)");
		
	dia_ISubsystemDiagnosisProxy* pInterface = nullptr;
	tDiaResult retCode = queryInterface<dia_ISubsystemDiagnosisProxy>(&pInterface);
	
	if ((retCode == DIA_SUCCESS) && pInterface)
	{
		(void) setSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		if (DIA_SUCCESS == pInterface->vBroadcastVehIdResults())
		{
			DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient --> Request to discover results sent.");
		}
		else
		{
			DIA_TR_ERR("dia_SubsystemLogoInstallationDoIPClient --> Request to discover results failed.");

			(void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
		}
	}
	
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::onBroadcastVehIdResults ( tDiaResult /*result*/, ::std::vector< subsystem > nodeList )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::onBroadcastVehIdResults");

   (void) unsetSysAdapterListener<dia_ISubsystemDiagnosisProxyListener>(this);
   
	for(auto doip_node : nodeList )
    {
		struct sockaddr_in remoteaddr;
		remoteaddr.sin_family = doip_node.sin_family;
		remoteaddr.sin_port = doip_node.sin_port;
		remoteaddr.sin_addr.s_addr = doip_node.s_addr;
		
		for(int i=0; i<8; i++)
		{
			remoteaddr.sin_zero[i] = doip_node.sin_zero[i];
		}
		
		vAddNode(DIA_C_SUBSYSTEM_DOIP_CLIENT_TESTER_ADDR, remoteaddr, doip_node.logAdr, doip_node.EID, doip_node.GID, doip_node.VIN, DOIP_NODE, 0);
		
    }
}

//------------------------------------------------------------------------------

DoIP_Node* 
dia_SubsystemLogoInstallationDoIPClient::vAddNode(DoIP_Node* node)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::vAddNode(DoIP_Node* node)");
	return NULL;
}

//------------------------------------------------------------------------------

DoIP_Node* 
dia_SubsystemLogoInstallationDoIPClient::vAddNode(tU16 /*testerAddress*/, sockaddr_in /*IP*/, tU16 /*logAddr*/, tU32 /*maxDataSize*/)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, tU32 maxDataSize)");
	
	//DoIP_Node* poNode = NULL;
	poNode = NULL;
	/*
	DoIP_NodeType nodeType;

	poNode = new DoIP_Node(testerAddress,IP,logAddr,nullptr,nullptr,nullptr,nodeType,maxDataSize,nullptr);
	
	if(!poNode){
	oDoIP_Nodes.push_back(poNode);}
	*/
	return (poNode);
}

//------------------------------------------------------------------------------

DoIP_Node* 
dia_SubsystemLogoInstallationDoIPClient::vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, const tU8* EID, const tU8* GID, const tU8* VIN, DoIP_NodeType nodeType, tU32 maxDataSize)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::vAddNode(tU16 testerAddress, sockaddr_in IP, tU16 logAddr, const tU8* EID, const tU8* GID, const tU8* VIN, DoIP_NodeType nodeType, tU32 maxDataSize)");
	//DoIP_Node* poNode = NULL;
	
	DoIP_Cfg doipCfg;
	bool nodeAlreadyPresent = false;

	poNode = new DoIP_Node(testerAddress,IP,logAddr,EID,GID,VIN,nodeType,maxDataSize,doipCfg);
	
	DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient:vAddNode [IP : 0x%08X ] [ LOGICAL ADR : 0x%02X ] ", IP.sin_addr.s_addr, poNode->logicalAddr);

	//if(poNode)
	//{
		//DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient:vAddNode [ LOGICAL ADR : 0x%02X ]", poNode->logicalAddr);
		
		std::list<DoIP_Node*>::iterator itNode;
		for (itNode=oDoIP_Nodes.begin(); itNode != oDoIP_Nodes.end(); itNode++)
		{
			if( (*itNode)->getIPAddr() == IP.sin_addr.s_addr )
			{
				DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::vAddNode - Node already present (IP check)");
				nodeAlreadyPresent = true;
				break;
			}
		}
		
		if(!nodeAlreadyPresent)
		{
			DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::vAddNode - New Node Added ");
			oDoIP_Nodes.push_back(poNode);
		}
	//}

	return (poNode);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::onNodeDiagnosticResponse(tU8 const * data, tU32 U32len)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::onNodeDiagnosticResponse(...)");
	
	// Check for tester present response and if yes, ignore the same
	if((data[DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SRV_ID_IDX] == DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SRV_ID) && 
	   (data[DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SUB_FUNC_ID_IDX] == DIA_C_SUBSYSTEM_TESTER_PRESENT_POS_RESP_SUB_FUNC_ID))
	{
		DIA_TR_INF("### Tester Present Response ###");
		return;
	}
	
	//! call back to parent to validate the response
	if (mpCallback) mpCallback->onDiagnosticResponse(data, U32len);
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::onRoutingActivationResponse(tU8 respCode)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::onRoutingActivationResponse(...)");

	if(!pmActiveDoIP_Node) return;
	/*
	if(DOIP_RA_RESPCODE_SUCCESS != respCode) 
	{
		//! reset the callback to client as the routing activation request failed.
		pmActiveDoIP_Node->setDoIPCallback(nullptr);
	} 
	*/
	//! call back to parent to validate the response
	if (mpCallback) mpCallback->onRAResponse(respCode);
}

//------------------------------------------------------------------------------
/*	
void 
dia_SubsystemLogoInstallationDoIPClient::diagnosticACK(tU8 ackCode, tU8* data, tU32 U32len)
{
	//! Can be ignored
}
	
//------------------------------------------------------------------------------
	
void 
dia_SubsystemLogoInstallationDoIPClient::diagnosticNACK(tU8 nackCode, tU8* data, tU32 U32len)
{
	//! Can be ignored
}

//------------------------------------------------------------------------------
*/
	
void
dia_SubsystemLogoInstallationDoIPClient::onPowerModeRequestResponse(tU8 pwrMode)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::onPowerModeRequestResponse(tU8 pwrMode)");
	
	if(!pmActiveDoIP_Node) return;
	
	/*
	if(DIA_C_SUBSYSTEM_POWER_MODE_READY != pwrMode) 
	{
		//! reset the callback to client as the power mode request failed.
		pmActiveDoIP_Node->setDoIPCallback(nullptr);
	}
	*/
	
	//! call back to parent to validate the response
	if (mpCallback) mpCallback->onPMRResponse(pwrMode);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::stopDiscoverEntitiesDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::stopDiscoverEntitiesDelayTimer(...)");
	
	oIEDelayTimer.s32SetTime(0,0);
    oIEDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::startDiscoverEntitiesDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::startDiscoverEntitiesDelayTimer(...)");
	
	oIEDelayTimer.addTimerListener(this);
	oIEDelayTimer.s32SetTime(0,0);
	oIEDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_IDENTIFY_ENTITIES_DELAY, 0);
}


//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::stopPowerModeRequestDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::stopPowerModeRequestDelayTimer(...)");
		
	oPMRDelayTimer.s32SetTime(0,0);
    oPMRDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::startPowerModeRequestDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::startPowerModeRequestDelayTimer(...)");
	   
	oPMRDelayTimer.addTimerListener(this);
	oPMRDelayTimer.s32SetTime(0,0);
	oPMRDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_POWER_MODE_REQUEST_DELAY, 0);
}


//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::stopRoutingActivationDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::stopRoutingActivationDelayTimer(...)");
		
	oRADelayTimer.s32SetTime(0,0);
    oRADelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::startRoutingActivationDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::startRoutingAactivationDelayTimer(...)");
	   
	oRADelayTimer.addTimerListener(this);
	oRADelayTimer.s32SetTime(0,0);
	oRADelayTimer.s32SetTime(DIA_C_SUBSYSTEM_ROUTING_ACTIVATION_DELAY, 0);
}


//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::stopSubsystemRoutingDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::stopSubsystemRoutingDelayTimer(...)");
		
	oRRDelayTimer.s32SetTime(0,0);
    oRRDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::startSubsystemRoutingDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::startSubsystemRoutingDelayTimer(...)");
	
	oRRDelayTimer.addTimerListener(this);
	oRRDelayTimer.s32SetTime(0,0);
	oRRDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_REQUEST_ROUTING_DELAY, 0);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::vOnTimerElapsed ( dia_TimerID id )
{
    // dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::vOnTimerElapsed(...)");

	tU32 timerID = (tU32) id;
	
	if( mTesterPresentTimerID == timerID )
	{
		if ( mIsTesterPresentActive )
		{
			DIA_TR_INF("### Tester Present Active ###");
			sendTesterPresentRequest();
		}
		else
		{
			DIA_TR_INF("### Tester Present Inactive ###");
		}
	}
	else if(mSlaveProcessTimerID == timerID)
	{
		DIA_TR_INF("### Slave Timer Elapsed ###");
		vFetchNodeListFromSlaveProcess();
	}
	else
	{
		DIA_TR_INF("### dia_SubsystemLogoInstallationDoIPClient::vOnTimerElapsed --- ID = %d ###", timerID);
		//! call back to parent to validate the response
		if (mpCallback) mpCallback->onTimeout(timerID);		   
	}
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::setCurrentNode(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::setCurrentNode(tU32 mNodeIP)");
	
	//! retrieve the pointer to the doip node
	pmActiveDoIP_Node =	this->vGetNodeByIP(mNodeIP);
	
	if(pmActiveDoIP_Node)
	{
		DIA_TR_INF("### dia_SubsystemLogoInstallationDoIPClient::setCurrentNode --- Active Node : 0x%02x", pmActiveDoIP_Node->logicalAddr );
	}
}

tU32
dia_SubsystemLogoInstallationDoIPClient::getCurrentNodeIP()
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::getCurrentNodeIP()");
	
	tU32 IP = DIA_C_SUBSYSTEM_LOGO_INSTALLATION_INVALID_IP;
	
	if(pmActiveDoIP_Node)
	{
		IP = pmActiveDoIP_Node->getIPAddr();
	}
	
	return IP;
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationDoIPClient::createNodeTCPSocket( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::createTCPSocket(...)");
	
	if (pmActiveDoIP_Node) 
	{
		return (pmActiveDoIP_Node->createTCPSocket());
	}
	
	return false;
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationDoIPClient::connectNodeTCP( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::connectNodeTCP(...)");
	
	if (pmActiveDoIP_Node) 
	{
		return (pmActiveDoIP_Node->connectTCP());
	}
	
	return false;
	
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationDoIPClient::closeNodeTCPSocket()
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::closeNodeTCPSocket()");
	
	if (pmActiveDoIP_Node)
	{
		return (pmActiveDoIP_Node->closeTCPSocket());
	}
	
	return false;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::diagSendRequest(std::vector<tU8> diagRequest)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::diagSendRequest(...)");
	
	if (pmActiveDoIP_Node)
	{
		return (pmActiveDoIP_Node->diagnosticReq(diagRequest));
	}
	
	return false;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::diagSendEntityStatusReq(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::diagSendEntityStatusReq(...)");
	
	int iSocket_TCP_Client;
	
	if (pmActiveDoIP_Node) 
	{
		iSocket_TCP_Client = pmActiveDoIP_Node->getTCPClientSocket();
		DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::diagSendEntityStatusReq : TCP Socket = 0x%02x", iSocket_TCP_Client);
		pmActiveDoIP_Node->entityStatusReq(iSocket_TCP_Client);
	}
	
	return true;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::diagSendPowerModeReq( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::diagSendPowerModeReq(...)");
	
	int iSocket_UDP_Client;
	
	if (pmActiveDoIP_Node) 
	{
		//! set the callback to client
		pmActiveDoIP_Node->setDoIPCallback(this);
		//! retrieve the socket id of the doip tester
		iSocket_UDP_Client = getUDPClientSocket();
		//! send power mode request
		pmActiveDoIP_Node->powermodeRequest(iSocket_UDP_Client);
	}
	
	return true;
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::diagSendRoutingActivationReq( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::diagSendRoutingActivationReq(...)");
	
	if (!pmActiveDoIP_Node) return false;
	
	pmActiveDoIP_Node->setDoIPCallback(this); 
	
	return (pmActiveDoIP_Node->routingActivationReq( DIA_C_SUBSYSTEM_ROUTING_ACTIVATION_TYPE /* RA Type */, nullptr /* OEM Specific Data */));
}

//------------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDoIPClient::isDoIPListEmpty( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::isDoIPListEmpty(...)");
	
	return (this->oDoIP_Nodes.empty());
}

//------------------------------------------------------------------------------

bool 
dia_SubsystemLogoInstallationDoIPClient::isNodeAlreadyConnected(tU32 mNodeIP)
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::isNodeAlreadyConnected(tU32 mNodeIP)");
	
	tU32 ConnectedNodeIP;
	DIA_TR_INF("isNodeAlreadyConnected:: Requested Node IP = 0x%08X ", mNodeIP);
	if(!pmActiveDoIP_Node) return false;
	
	ConnectedNodeIP = pmActiveDoIP_Node->getIPAddr();
	
	DIA_TR_INF("isNodeAlreadyConnected:: Connected Node IP = 0x%08X ", ConnectedNodeIP);
	
	return ( ConnectedNodeIP == mNodeIP ) ? true : false;
	
	
}

//------------------------------------------------------------------------------

std::list<dia_tSubsystemInfo> 
dia_SubsystemLogoInstallationDoIPClient::getDoIPClientNodes(void)
{
    dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::getDoIPClientNodes");

    mDoipNodes.clear();

    if(!(isDoIPListEmpty()))
    {
    	std::list<DoIP_Node*>::iterator it = oDoIP_Nodes.begin();

    	for( ; it != oDoIP_Nodes.end(); ++it)
    	{

    		DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::getDoIPClientNodes DOIP NODES AVAILABLE");

    		tU32 IPAddr =  ((*it)->getIPAddr());
    		const tU8* cGID = ((*it)->getGID());
    		const tU8* cEID = ((*it)->getEID());

    		tU16 entityName = ((cGID[0]) << 8)|(cGID[1]);
    		tU16 swVariant = ((cGID[4]) << 8)|(cGID[5]);

    		dia_tSubsystemInfo tSubsystemInfo;
    		tSubsystemInfo.subsystemEntityName = entityName;
    		tSubsystemInfo.subsystemSupplierName = cGID[2];
    		tSubsystemInfo.subsystemHWVariant = cGID[3];
    		tSubsystemInfo.subsystemSWVariant = swVariant;
    		tSubsystemInfo.subsystemIPAddress = IPAddr;
    		for(int i = 0; i<ENTITY_ID_LENGTH; ++i)
    		{
    			tSubsystemInfo.subsystemEntityID[i] = *(cEID+i);
    		}

    		mDoipNodes.push_back(tSubsystemInfo);
    	}
    }
    else
    {
    	DIA_TR_INF("dia_SubsystemLogoInstallationDoIPClient::getDoIPClientNodes DOIP LIST IS EMPTY");
    }
    return mDoipNodes;
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::clearDoIPNodeList( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::clearDoIPNodeList");
	
	oDoIP_Nodes.clear();
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::enableTesterPresent( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::enableTesterPresent");
	
	oTesterPresentTimer.addTimerListener(this);
	oTesterPresentTimer.s32SetTime(0,0);
	oTesterPresentTimer.s32SetTime(DIA_C_SUBSYSTEM_TESTER_PRESENT_TIMER_VAL_FIRST, 
								   DIA_C_SUBSYSTEM_TESTER_PRESENT_TIMER_VAL);
								   
	mIsTesterPresentActive = true;
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::disableTesterPresent( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::disableTesterPresent");
	
	oTesterPresentTimer.s32SetTime(0,0);
    oTesterPresentTimer.removeTimerListener(this);
	
	mIsTesterPresentActive = false;
	
}

//------------------------------------------------------------------------------

void 
dia_SubsystemLogoInstallationDoIPClient::sendTesterPresentRequest( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::sendTesterPresentRequest");
	
	std::vector<tU8> testerPresent;
	testerPresent.clear();
	testerPresent.reserve(DIA_C_SUBSYSTEM_TESTER_PRESENT_REQ_LEN);
	
	testerPresent.push_back(DIA_C_SUBSYSTEM_TESTER_PRESENT_REQ_SRV_ID);
	testerPresent.push_back(DIA_C_SUBSYSTEM_TESTER_PRESENT_REQ_SUB_FUNC_ID);
	
	if (pmActiveDoIP_Node)
	{
		pmActiveDoIP_Node->diagnosticReq(testerPresent);
	}
	
}

void
dia_SubsystemLogoInstallationDoIPClient::stopLogoTransferDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::stopLogoTransferDelayTimer(...)");
	
	oLTDelayTimer.s32SetTime(0,0);
    oLTDelayTimer.removeTimerListener(this);
}

//------------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDoIPClient::startLogoTransferDelayTimer ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDoIPClient::startLogoTransferDelayTimer(...)");
	
	oLTDelayTimer.addTimerListener(this);
	oLTDelayTimer.s32SetTime(0,0);
	oLTDelayTimer.s32SetTime(DIA_C_SUBSYSTEM_LOGO_INSTALLATION_DELAY, 0);
}