/*!
 * \file       dia_SubsystemLogoInstallationDataDownload.cpp
 *
 * \brief      Data Download Strategy used for subsystem logo installation
 *
 * \details    ...
 *
 * \component  Diagnostics
 *
 * \ingroup    cis subsystem hmi logo installation
 *
 * \author     Arjun Manjunath Sanu (RBEI/ECA2)
 *
 * \date       20.03.2020
 *
 * \copyright  (c) 2020 Robert Bosch Engineering & Business Solutions Ltd.
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
 
#ifndef __INCLUDED_DIA_SUBSYSTEM_LOGO_INSTALLATION_DATA_DOWNLOAD__
#include <project/framework/cis/logoinstallation/dia_SubsystemLogoInstallationDataDownload.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include <common/framework/config/dia_defsConfig.h>
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include "common/framework/application/dia_File.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include <common/framework/config/dia_ConfigManager.h>
#endif

#include <fstream>

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationDataDownload::dia_SubsystemLogoInstallationDataDownload(const dia_DatasetConfigLogo& cfgObj, tU8 DataFormatIdentifier, tU8 AddressAndLengthFormatIDentifier )
   : mName(cfgObj.mName),
     mStartAddr(cfgObj.mAddrStart),
     mSizeInBytes(cfgObj.mSizeInBytes),
     mSizeInBytesRequested(cfgObj.mSizeInBytes),
     mSettingsBitmask(cfgObj.mSettings),
     mPropID(cfgObj.mPropID),
     mBSN(DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER),
     mIsDataTransferActive(false),
     mIsDataTransferComplete(false),
	 mBlockSize(0),
	 mBlockSizeInBytes(0),
	 mpByteBuffer(0),
	 mTotalNumberOfBlocks(0),
	 mRemainingBlocks(0),
	 mNumOfBytesTransmitted(0),
	 mDataFmtId(DataFormatIdentifier),
	 mAddrAndLenFmtId(AddressAndLengthFormatIDentifier),
	 mCurrentDataBlockIndex(0),
	 mCurrentBlockSize(0),
	 mIsLoaded(false)
{}

//-----------------------------------------------------------------------------

dia_SubsystemLogoInstallationDataDownload::~dia_SubsystemLogoInstallationDataDownload( void )
{
   mName  = 0;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationDataDownload::load ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::load()");

   if ( !mpByteBuffer )
   {
      mpByteBuffer = OSAL_NEW tU8[mSizeInBytes];
      if ( !mpByteBuffer ) return DIA_E_DATASET_INVALID_DATA_POINTER;
   }

   tU32 sizeInBytes = mSizeInBytes;
   
   DIA_TR_INF("::load() Max. number of bytes that can be read : %d", sizeInBytes);
   
   tDiaResult retCode = dia_getProperty(mPropID, mpByteBuffer,&sizeInBytes);
   
   DIA_TR_INF("::load() Actual Size of the property : %d", sizeInBytes);
   
   if ( retCode != DIA_SUCCESS )
   {
      return retCode;
   }
   
   if ( (!hasVariableLength()) && (sizeInBytes != mSizeInBytes) )
   {
      return DIA_FAILED;
   }
   
   if ( mSizeInBytes > sizeInBytes ) 
   {
      mSizeInBytes = sizeInBytes;
   }
   
   DIA_TR_INF("::load() Number of bytes that were read : %d", mSizeInBytes);
   
   
   /*
   ifstream logo(logopath.c_str(),ios::binary);
   logo.seekg((long long) 0,ios::end);
   sizeInBytes = (tU32) logo.tellg();
   
   DIA_TR_INF("::load() Actual Size of the property : %d", sizeInBytes);

   logo.seekg((long long) 0, ios::beg);

   logo.read((char*)mpByteBuffer, (int) sizeInBytes);
   
   if ( mSizeInBytes > sizeInBytes ) 
   {
      mSizeInBytes = sizeInBytes;
   }
   
   DIA_TR_INF("::load() Number of bytes that were read : %d", mSizeInBytes);
   */

   mIsLoaded = true;  // data is loaded

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationDataDownload::split ( void )
{
   dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::split");

   DIA_TR_INF("::split --> mBlockSizeInBytes = %d", mBlockSizeInBytes);
   
   if( 0 == mBlockSizeInBytes )
   {
	   return DIA_FAILED;
   }

   mBSN = 0;
   mBlockData.clear();

   if ( mSizeInBytes )
   {
      bool isDone   = false;
      size_t position = 0;
      tU16 i = 0;
	  
	  /* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
	  tU16 blocks = 0;
	  size_t remainder = mSizeInBytes % mBlockSizeInBytes ;
	  blocks =  mSizeInBytes / mBlockSizeInBytes ;
	  if(remainder != 0)
	  {
		  blocks += 1;
	  }
	  DIA_TR_INF("::split --> Total Number Of Blocks = %d", blocks);
	  /* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

      while ( !isDone )
      {
         size_t remainingLength = mSizeInBytes - position;
		 
         size_t blockSize = mBlockSizeInBytes;
		 
         if ( remainingLength > mBlockSizeInBytes )
         {
            std::vector<tU8> dataVec;
            dataVec.insert(dataVec.end(), &mpByteBuffer[position], &mpByteBuffer[position+mBlockSizeInBytes]);

            mBlockData.push_back(dataVec);
            position += mBlockSizeInBytes;
         }
         else
         {
            std::vector<tU8> dataVec;
            dataVec.insert(dataVec.end(), &mpByteBuffer[position], &mpByteBuffer[position+remainingLength]);
            mBlockData.push_back(dataVec);
            isDone = true;
            position = mSizeInBytes;
            blockSize = remainingLength;
         }
		 // since 'i' is the index, I am doing +1 for print...
         DIA_TR_INF("::split --> Created Block #%02d (size is %d bytes)", (i+1), blockSize);
         DIA_TR_INF("::split --> Block Data: '%s'", dia::utils::bin2str(mBlockData[i].data(),(int) mBlockData[i].size(),' ').c_str());
         i++;
      }
	  
	  mTotalNumberOfBlocks = mBlockData.size();
	  DIA_TR_INF("::split --> Total Number Of Blocks (2) = %d", mTotalNumberOfBlocks);
	  mRemainingBlocks = mTotalNumberOfBlocks;
   }
   else
   {
	   DIA_FAILED;
   }
   
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationDataDownload::setBlockSize( tU16 bs )
{
	DIA_TR_INF("::setBlockSize --> Block Size Set To : %d ", bs);
	mBlockSizeInBytes = bs;
	
	if(mBlockSizeInBytes > 4096)
	{
		mBlockSizeInBytes = 4096;
	}


	return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationDataDownload::setNextBlock( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::setNextBlock");

	// increment block sequence number
	mBSN += 1;
	
	if( mBSN > 255 )
	{
		// Reset the Block Sequence Number
		mBSN = 0;
	}

	DIA_TR_INF("::setNextBlock --> Block Sequence Number = %d", mBSN);
	
	mCurrentDataBlockIndex = mTotalNumberOfBlocks - mRemainingBlocks;
	// decrement remaining blocks
	mRemainingBlocks -= 1;

    /* we dont need this here. we will check the status only after tranmission is complete
	if ( 0 == mRemainingBlocks )
	{
		DIA_TR_INF("INFO --> LOGO HAS BEEN COMPLETELY TRANSFERRED TO SUBSYSTEM! ");
		// data transmission done
		mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
		mIsDataTransferComplete = true;
	}
	*/
	
	return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SubsystemLogoInstallationDataDownload::assembleData( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::assembleData");
	
	if( ( mBSN > 255 ) || ( mCurrentDataBlockIndex > mTotalNumberOfBlocks ) )
	{
		return DIA_FAILED;
	}
	
	// Clear the existing data from the vector 
	mNextTransferData.clear();
	// insert the block sequence number 
	mNextTransferData.push_back(mBSN);
	// insert the block data
	mNextTransferData.insert(	
								mNextTransferData.end(), 
								mBlockData[mCurrentDataBlockIndex].begin(), 
								mBlockData[mCurrentDataBlockIndex].end()
							);
							
	mCurrentBlockSize = mBlockData[mCurrentDataBlockIndex].size();
	
	DIA_TR_INF("::assembleData --> Current Block Size = %d", mCurrentBlockSize);
																		
	return DIA_SUCCESS;
	
}

//-----------------------------------------------------------------------------

tDiaResult 
dia_SubsystemLogoInstallationDataDownload::getNextTransferDataRequest(std::vector<tU8>& TransferDataRequest )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::getNextTransferDataRequest(...)");
	
	if(mNextTransferData.empty())
	{
		return DIA_FAILED;
	}
	
	// Clear the existing data from the vector
	TransferDataRequest.clear();
	//
	TransferDataRequest.push_back(DIA_C_SUBSYSTEM_TRANSFER_DATA_REQUEST_SID);
	// insert the block data
	TransferDataRequest.insert(	
								TransferDataRequest.end(), 
								mNextTransferData.begin(), 
								mNextTransferData.end()
							  );
							
	return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult 
dia_SubsystemLogoInstallationDataDownload::getDownloadRequest(std::vector<tU8>& DownloadRequest )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::getDownloadRequest(...)");
	
	if( (mStartAddr == 0) || (mSizeInBytes == 0) )
	{
		return DIA_FAILED;
	}
	
	// Clear the existing data from the vector
	DownloadRequest.clear();
	// SID
	DownloadRequest.push_back(DIA_C_SUBSYSTEM_REQUEST_DOWNLOAD_REQUEST_SID);
	// Data Format IDentifier
	DownloadRequest.push_back(mDataFmtId);
	// Address&LengthFormatIdentifier
	DownloadRequest.push_back(mAddrAndLenFmtId);
	// Logical Address
	DownloadRequest.push_back( mStartAddr >> 24 ); 
	DownloadRequest.push_back( mStartAddr >> 16 ); 
	DownloadRequest.push_back( mStartAddr >> 8 ); 
	DownloadRequest.push_back( mStartAddr ); 
	// Total Size in number of bytes
	DownloadRequest.push_back( mSizeInBytes >> 24 ); 
	DownloadRequest.push_back( mSizeInBytes >> 16 ); 
	DownloadRequest.push_back( mSizeInBytes >> 8 ); 
	DownloadRequest.push_back( mSizeInBytes ); 
	
							
	return DIA_SUCCESS;
	
	
}

//-----------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDataDownload::hasVariableLength ( void ) const
{
   return (mSettingsBitmask & DIA_C_U16_DATASET_CFG_VARIABLE_LENGTH) ? true : false;
}

//-----------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDataDownload::isDataTransferActive ( void ) const
{
   return mIsDataTransferActive;
}

//-----------------------------------------------------------------------------

bool
dia_SubsystemLogoInstallationDataDownload::isDataTransferComplete ( void ) const
{
	DIA_TR_INF("::isDataTransferComplete --> %s", (mIsDataTransferComplete) ? "YES" : "NO" );
   return mIsDataTransferComplete;
}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDataDownload::postprocessTransferData ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::postprocessTransferData(...)");
	
    mNumOfBytesTransmitted = tU32(mNumOfBytesTransmitted + mCurrentBlockSize);
	
	DIA_TR_INF("######################################################################################################");
	DIA_TR_INF("######################################################################################################");
	DIA_TR_INF("######################################################################################################");
	DIA_TR_INF("::postprocessTransferData --> Total number of bytes = %d", mSizeInBytes);
	DIA_TR_INF("::postprocessTransferData --> Number of bytes Transmitted = %d", mNumOfBytesTransmitted);
	DIA_TR_INF("::postprocessTransferData --> Number of bytes Remaining = %d", (mSizeInBytes - mNumOfBytesTransmitted));
	
	DIA_TR_INF("::postprocessTransferData --> Total number of Blocks = %d", mTotalNumberOfBlocks);
	DIA_TR_INF("::postprocessTransferData --> Number of blocks Transmitted = %d", (mTotalNumberOfBlocks - mRemainingBlocks));
	DIA_TR_INF("::postprocessTransferData --> Number of blocks Remaining = %d", mRemainingBlocks);
	DIA_TR_INF("######################################################################################################");
	DIA_TR_INF("######################################################################################################");
	DIA_TR_INF("######################################################################################################");
	
	if((mNumOfBytesTransmitted == mSizeInBytes) || ( 0 == mRemainingBlocks ) || ( mCurrentDataBlockIndex == (mTotalNumberOfBlocks-1)))
	{
		DIA_TR_INF("INFO --> LOGO HAS BEEN COMPLETELY TRANSFERRED TO SUBSYSTEM! ");
		mIsDataTransferComplete = true;
		mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
	}	
}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDataDownload::initializeDataDownload ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::initializeDataDownload(...)");
	
    mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
    mIsDataTransferActive = true;
    mIsDataTransferComplete = false;
	
	mBlockSize = 0;
	mBlockSizeInBytes = 0;
	mTotalNumberOfBlocks = 0;
	mRemainingBlocks = 0;
	mNumOfBytesTransmitted = 0;
	mIsLoaded = false; 
}

//-----------------------------------------------------------------------------

void
dia_SubsystemLogoInstallationDataDownload::finaliseDataDownload ( void )
{
	dia_tclFnctTrace oTrace("dia_SubsystemLogoInstallationDataDownload::finaliseDataDownload(...)");
	
	mBSN = DIA_C_U16_INVALID_BLOCK_SEQUENCE_NUMBER;
    mIsDataTransferActive = false;
	mIsDataTransferComplete = false;

	mBlockSize = 0;
	mBlockSizeInBytes = 0;
	mTotalNumberOfBlocks = 0;
	mRemainingBlocks = 0;
	mNumOfBytesTransmitted = 0;
	mIsLoaded = false; 
}


