/*!
 * \file       dia_SrvHandler_ReadEmmcJedecCidReg.cpp
 *
 * \brief      CM Diag Servicehandler to read CID and CSD data for eMMC
 *
 * \component  Diagnostics
 *
 * \ingroup    diaServicesCommon
 *
 * \copyright  (c) 2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
#ifndef __INCLUDED_DIA_SRVHANDLER_READEMMCJEDECCIDREG__
#include "dia_SrvHandler_ReadEmmcJedecCidReg.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include <project/framework/config/dia_defsProjectConfig.h>
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include "common/framework/engine/dia_LookupKey.h"
#endif

#ifndef DIA_BASHCOMMAND_H_
#include "common/framework/application/dia_BashCommand.h"
#endif

#include <fstream>
#include <sstream>

//-----------------------------------------------------------------------------

dia_SrvHandler_ReadEmmcJedecCidReg::dia_SrvHandler_ReadEmmcJedecCidReg()
    : dia_ServiceHandlerUDS("dia_SrvHandler_ReadEmmcJedecCidReg")
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadEmmcJedecCidReg::dia_SrvHandler_ReadEmmcJedecCidReg()");
}

//-----------------------------------------------------------------------------

dia_SrvHandler_ReadEmmcJedecCidReg::~dia_SrvHandler_ReadEmmcJedecCidReg()
{
}

// return emmc data file name based on DID
std::string dia_SrvHandler_ReadEmmcJedecCidReg::getFilename() const
{
   tU16 did = oDiagMsgBuffer().u16GetSubServiceId();
   std::string name;
   if(did == DIA_C_U16_DID_RBCM_EMMC_JEDEC_CID_REG)
   {
      name = "emmc_cid";
   }
   else if(did == DIA_C_U16_DID_RBCM_EMMC_JEDEC_CSD_REG)
   {
      name = "emmc_csd";
   }
   return name;
}

#ifndef __DIA_UNIT_TESTING__
// concat file and path based on mmc number
std::string dia_SrvHandler_ReadEmmcJedecCidReg::getFilePath(std::string filename) const
{
   return std::string("/var/opt/bosch/dynamic/diagnosis/" + filename);
}
#endif

void
dia_SrvHandler_ReadEmmcJedecCidReg::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   dia_tclFnctTrace trc("dia_SrvHandler_ReadEmmcJedecCidReg::vProcessRequest");

   std::string filename = getFilename();
   std::string generateLinkCmd = "/opt/bosch/base/bin/diag_generate_link.sh " + filename + " /var/opt/bosch/dynamic/diagnosis";
#ifndef __DIA_UNIT_TESTING__
   uint8_t generateLinkReturn = dia_BashCommand::executeCommand(generateLinkCmd);

   if (generateLinkReturn != 0) {
      DIA_TR_ERR("dia_SrvHandler_ReadEmmcJedecCidReg: Creation of link %s FAILED!", filename.c_str());
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      return;
   }
#endif
   std::string path = getFilePath(filename);
   DIA_TR_INF("dia_SrvHandler_ReadEmmcJedecCidReg: Reading %s", path.c_str());
   std::string db;
   std::ifstream cidFile(path.c_str());
   cidFile >> db;
   if(db.size() == registerSize)
   {
      DIA_TR_INF("dia_SrvHandler_ReadEmmcJedecCidReg: READ EMMC REG SUCCESS!");
      const tU16 DATA_START = 3;
      oDiagMsgBuffer().vSetPosResp();
      oDiagMsgBuffer().vSetDataLength(static_cast<tU16>((DATA_START + registerSize/2) & 0xffff));
      std::stringstream ss;
      for(size_t i = 0; i < db.size(); i += 2)
      {
         ss << db.substr(i, 2);
         int uByte =0;
         ss >> std::hex >> uByte;
         oDiagMsgBuffer().setDataU8(static_cast<tU16>(DATA_START + i/2), static_cast<tU8>(uByte & 0xff));
         ss.clear();
      }
      vResReadyAndQuit();
   }
   else
   {
      DIA_TR_ERR("dia_SrvHandler_ReadEmmcJedecCidReg: READ EMMC REG FAILED!");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   }

}

//-----------------------------------------------------------------------------

tDiaResult dia_SrvHandler_ReadEmmcJedecCidReg::makeLookupKeys ( std::vector<dia_LookupKey*>& keys )
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandler_ReadEmmcJedecCidReg::makeLookupKeys");
#endif

   keys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_RBCM_EMMC_JEDEC_CID_REG, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - CID reg
   keys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16) DIA_C_U16_DID_RBCM_EMMC_JEDEC_CSD_REG, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) ); // ReadDataByIdentifier - csd reg
   return DIA_SUCCESS;
}
