/*!
 * \file       dia_SrvHandler_ODXFileIdentifier.cpp
 *
 * \brief      Read ODX FileIdentifier with dynamic length
 *
 * \component  Diagnostics
 *
 * \ingroup    diaServicesCommon
 *
 * \copyright  (c) 2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */
#ifndef __INCLUDED_DIA_ODXFILEIDENTIFIER_H__
#include "dia_SrvHandler_ODXFileIdentifier.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include <project/framework/config/dia_defsProjectConfig.h>
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include "common/framework/utils/dia_utilities.h"
#endif

#include <errno.h>
#include <sys/time.h>

static const uint16_t DID_RESPONSE_DATA_OFFSET = 3;

/**
 * @brief Process incoming diagnostic request for ODX FileIdentifier read
 * @details ServiceHandler tries to read ODXFileIdentifier from
 *          ConfigManager, reduce the size if zero terminated and
 *          sends response to tester.
 *
 * @param  vecArgs Unused
 * @return void
 */
void dia_SrvHandler_ODXFileIdentifier::vProcessRequest(const std::vector<tArgsType>& /*vecArgs*/)
{
    dia_tclFnctTrace trc(__PRETTY_FUNCTION__);

    std::vector<uint8_t> odxFileIdentifier;
    tDiaResult result =  dia_getProperty(DIA_PROP_CM_ODX_FILE_IDENTIFIER, odxFileIdentifier);

    if (result == DIA_SUCCESS)
    {
        std::vector<uint8_t>::iterator it = std::find(odxFileIdentifier.begin(), odxFileIdentifier.end(), 0x00);
        if(it != odxFileIdentifier.end()) {
            odxFileIdentifier.erase(it, odxFileIdentifier.end());
        }

        DIA_TR_INF("ODX FileIdentifier: %s", dia::utils::bin2hexstr(odxFileIdentifier).c_str());
        oDiagMsgBuffer().vSetPosResp();
        oDiagMsgBuffer().vSetDataLength((uint16_t)(DID_RESPONSE_DATA_OFFSET + odxFileIdentifier.size()));
        for (size_t i = 0; i < odxFileIdentifier.size(); ++i)
        {
            (void)oDiagMsgBuffer().vSetDataU8((uint16_t)(DID_RESPONSE_DATA_OFFSET + i), odxFileIdentifier[i]);
        }
        vResReadyAndQuit();
    }
    else
    {
        DIA_TR_ERR("ODX FileIdentifier NOT available in ConfigManager. Send negative response to tester.");
        vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
    }
}

/**
 * @brief Generates and stores lookup keys for ServiceHandler
 *
 * @param keys Lookup-key storage to be extended by function
 * @return DIA_SUCCESS
 */
tDiaResult dia_SrvHandler_ODXFileIdentifier::makeLookupKeys(std::vector<dia_LookupKey*>& keys)
{
#ifdef __DIA_UNIT_TESTING__
    dia_tclFnctTrace trc(__PRETTY_FUNCTION__);
#endif
    keys.push_back(new dia_LookupKey(DIA_C_U8_UDS_SID_READ_DATA_BY_IDENTIFIER, (tU16)DIA_C_U16_DID_RBCM_ODX_FILE_IDENTIFIER,
                                     DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED));
    return DIA_SUCCESS;
}
