#ifndef __INCLUDED_DIA_ROUTINE_VERIFY_SECURITY_SETTINGS_SCC__
#include <common/services/uds/production/dia_RoutineCtrlVerifySecuritySettingsSCC.h>
#endif

#ifndef __INCLUDED_DIA_FILE__
#include <common/framework/application/dia_File.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

static const std::string fullCmdName("/opt/bosch/base/bin/swu_common_v850_app_out.out");

const int SWUPDATE_EXIT_CODE_OK                                               = 0;
const int SWUPDATE_EXIT_CODE_SECURITY_BYTE_READ_ERROR                         = 18;
const int SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH                                 = 19;
const int SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED                            = 20;
const int SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH_SECURITY_BYTE_READ_ERROR        = 38;
const int SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED_SECURITY_BYTE_READ_ERROR   = 37;

namespace dia
{

const tU16 RoutineCtrlVerifySecuritySettingsSCC::mSizeOfJtagCRC = 4;
const tU16 RoutineCtrlVerifySecuritySettingsSCC::mSizeOfParams  = 4;

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlVerifySecuritySettingsSCC::RoutineCtrlVerifySecuritySettingsSCC ( void )
   : RoutineCtrlGenericSecuritySettingsSCC("RoutineCtrlVerifySecuritySettingsSCC", DIA_C_U16_ID_RTCTRL_JTAG_ID_AND_SECURITY_BYTE_VERIFY, fullCmdName),
     mSecurityByte(0x00)
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlVerifySecuritySettingsSCC::RoutineCtrlVerifySecuritySettingsSCC");
   mJtagCRC.clear();
}

//---------------------------------------------------------------------------------------------------------------------

RoutineCtrlVerifySecuritySettingsSCC::~RoutineCtrlVerifySecuritySettingsSCC()
{}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
RoutineCtrlVerifySecuritySettingsSCC::evaluateStartParameters ( std::vector<tU8>& params )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlVerifySecuritySettingsSCC::evaluateStartParameters");

   if ( params.size() != mSizeOfParams )
   {
      DIA_TR_INF("RoutineCtrlVerifySecuritySettingsSCC::evaluateStartParameters INVALID PARAMETERS (expected = %u, actually = %zu)", mSizeOfParams, params.size());
      return DIA_FAILED;
   }

   // extract JTAG ID
   mJtagCRC.clear();
   mJtagCRC.insert(mJtagCRC.end(),params.begin(),params.begin()+mSizeOfJtagCRC);
   DIA_TR_INF("RoutineCtrlVerifySecuritySettingsSCC::evaluateStartParameters JTAG-CRC = '%s'", dia::utils::bin2str(mJtagCRC,' ').c_str());

   return DIA_SUCCESS;
}

//---------------------------------------------------------------------------------------------------------------------

tDiaResult
RoutineCtrlVerifySecuritySettingsSCC::getCommandArguments ( std::vector<std::string>& argv )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlVerifySecuritySettingsSCC::getCommandArguments");

   // generate filename for temporary file used to store the results
   char filename[] = "/tmp/XXXXXX";
   int fd = mkstemp(filename);
   if(fd == -1)
   {
      return DIA_FAILED;
   }

   mTmpFileName = filename;

   // CRC for the JTAG-ID to be checked
   argv.push_back("-jverify"); argv.push_back(dia::utils::bin2hexstr(mJtagCRC, true));
   // security bytes shall be read from SCC
   argv.push_back("-sread");
   // cookie to be store in the output file for associating the data with this request
   argv.push_back("-token"); argv.push_back(mTmpFileName);
   // output file to be used for storing the security byte
   argv.push_back("-o"); argv.push_back(mTmpFileName);

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlVerifySecuritySettingsSCC::evaluateCommandResponse ( int exitCode )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlVerifySecuritySettingsSCC::evaluateCommandResponse");

   tDiaResult retCode = DIA_SUCCESS;

   bool isJtagCrcMatched = false;
   bool isJtagCrcSupported = true;
   bool isSecurityByteAvailable = false;

   mSecurityByte = 0;

   switch ( exitCode )
   {
   case SWUPDATE_EXIT_CODE_OK:
      {
         // CRC matches and security byte successfully
         DIA_TR_INF("EXIT_CODE = SWUPDATE_EXIT_CODE_OK (%d)", exitCode);
         isSecurityByteAvailable = true;
         isJtagCrcMatched = true;
      }
      break;

   case SWUPDATE_EXIT_CODE_SECURITY_BYTE_READ_ERROR:
      {
         // CRC matches but reading security byte failed
         DIA_TR_INF("EXIT_CODE = SWUPDATE_EXIT_CODE_SECURITY_BYTE_READ_ERROR (%d)", exitCode);
         isJtagCrcMatched = true;
      }
      break;

   case SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH:
   case SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED:
      {
         // CRC does not match or fingerprint calculation not implemented (old SCC)
         if (SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH==exitCode)             DIA_TR_INF("EXIT_CODE = SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH (%d)", exitCode);
         else if (SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED==exitCode)   DIA_TR_INF("EXIT_CODE = SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED (%d)", exitCode);

         isSecurityByteAvailable = true;
         if (exitCode == SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED) isJtagCrcSupported = false;
      }
      break;

   case SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH_SECURITY_BYTE_READ_ERROR:
   case SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED_SECURITY_BYTE_READ_ERROR:
      {
         if (SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH_SECURITY_BYTE_READ_ERROR==exitCode)             DIA_TR_INF("EXIT_CODE = SWUPDATE_EXIT_CODE_JTAGCRC_MISMATCH_SECURITY_BYTE_READ_ERROR (%d)", exitCode);
         else if (SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED_SECURITY_BYTE_READ_ERROR==exitCode)   DIA_TR_INF("EXIT_CODE = SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED_SECURITY_BYTE_READ_ERROR (%d)", exitCode);

         // CRC does not match or fingerprint calculation not implemented (old SCC) and reading security byte failed
         if (exitCode == SWUPDATE_EXIT_CODE_JTAGCRC_NOT_SUPPORTED_SECURITY_BYTE_READ_ERROR) isJtagCrcSupported = false;
      }
      break;

   default:
      DIA_TR_INF("EXIT_CODE = UNKNOWN (%d)", exitCode);
      break;
   }

   if ( isSecurityByteAvailable )
   {
      tU8 secByte = 0;
      if ( readSecurityByte(secByte) == DIA_SUCCESS )
      {
         mSecurityByte = secByte;
      }
   }
   else
   {
      retCode = DIA_FAILED;
   }

   mResults.push_back((isJtagCrcSupported) ? ((isJtagCrcMatched) ? 0x00 : 0x01) : 0x02);
   mResults.push_back(mSecurityByte);

   for ( tU16 i=0; i<mResults.size(); i++ )
   {
      DIA_TR_INF("mResults[%02d] = 0x%02x",i,mResults[i]);
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
RoutineCtrlVerifySecuritySettingsSCC::readSecurityByte ( tU8& securityByte )
{
   dia_tclFnctTrace oTrace("dia::RoutineCtrlVerifySecuritySettingsSCC::readSecurityByte");

   tDiaResult retCode = DIA_FAILED;

   dia_File tmpFile(mTmpFileName);
   if ( !tmpFile.doesExist() ) return DIA_E_FILE_DOES_NOT_EXIST;
   if ( tmpFile.open("r") != DIA_SUCCESS ) return DIA_E_FILE_OPEN_FAILED;

   // read and verify cookie
   std::string cookie;
   if ( tmpFile.getLine(cookie) == DIA_SUCCESS )
   {
      if ( cookie[cookie.size()-1] == '\n' ) cookie.erase(cookie.size()-1);

      DIA_TR_INF("##### COOKIE = \"%s\", TMPFILE = \"%s\" #####",cookie.c_str(),mTmpFileName.c_str());
      if ( cookie == mTmpFileName )
      {
         DIA_TR_INF("##### COOKIE MATCHES TMPFILE !! #####");
         // cookie matches so we can read the security byte
         std::string secByte;
         if ( tmpFile.getLine(secByte) == DIA_SUCCESS )
         {
            if ( secByte[secByte.size()-1] == '\n' ) secByte.erase(secByte.size()-1);
            securityByte = dia::utils::hexstr2byte(secByte);
            DIA_TR_INF("##### SUCCESSFULLY READ SECURITY BYTE (BYTE = 0x%02x) #####",securityByte);
            retCode = DIA_SUCCESS;
         }
         else
         {
            DIA_TR_INF("##### FAILED TO READ SECURITY BYTE #####");
         }
      }
   }

   (void) tmpFile.close();

   return retCode;
}

}
