/*
 * dia_RoutineCtrlChecksum.cpp
 *
 *  Created on: 19.02.2014
 *      Author: kaa1hi
 */
// DIA_REQ UDS 05 31 01 30 60/61/62/63/64/65/66 -> start routine
//       pos rep  71 01 30 60 00    OK
//       pos rep  71 01 30 60 01    NOK
//       pos rep  71 01 30 60 02    ABORTED
//       pos rep  71 01 30 60 03    RUNNING
//
// DIA_REQ UDS 05 31 03 30 60/61/62/63/64/65/66 -> get status
//       pos rep  71 03 30 60 AA [16 bytes] BB     (AA - status, BB - verification result)
//

#ifndef DIA_ROUTINECTRLCHECKSUM_H_
#include "dia_RoutineCtrlChecksum.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG_PROJECT__
#include "project/framework/config/dia_defsProjectConfig.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_ROUTINE_CONTROL_MANAGER__
#include "common/framework/protocols/uds/rtctrl/dia_RoutineCtrlManager.h"
#endif

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#include "common/framework/utils/dia_RandomGenerator.h"
#endif

#include "common/framework/application/dia_File.h"

#include <errno.h>
#include <ctime>        //needed for time()

#define DIA_CHECKSUM_MAX_SIZE_OF_THEAD_NAME      ((tU8)32)
#define DIA_CHECKSUM_SIZE_OF_CHECKSUM_IN_CHARS   (32)
#define DIA_CHECKSUM_SIZE_OF_CHECKSUM_IN_BYTES   (16)

#if (DIA_CHECKSUM_SIZE_OF_CHECKSUM_IN_BYTES<<1) != (DIA_CHECKSUM_SIZE_OF_CHECKSUM_IN_CHARS)
#error "Incorrect value of checksum size."
#endif

#ifndef DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_NAVI_RAW
#define DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_NAVI_RAW     ((tU16)0x0)
#endif

//Script macros
#define DIA_CHECKSUM_GENERIC_SCRIPT_PATH               "/opt/bosch/base/bin/diag_checksumproxy.sh"
#define DIA_CHECKSUM_GENERIC_SCRIPT_MAX_LEN            256
#define DIA_CHECKSUM_NAV_SCRIPT_PATH           "/opt/bosch/checksum/fingerprint_fs.sh"
#define DIA_CHECKSUM_NAV_SCRIPT_NAME           "fingerprint_fs.sh"
#define DIA_CHECKSUM_NAV_SCRIPT_MAX_LEN        256

//Result file macros
#define DIA_CHECKSUM_RESULT_FILE_GENERIC_PATH         "/tmp/chksums_check/"  //added forward slash at the end
#define DIA_CHECKSUM_RESULT_FILE_NAVI_FINGERPRINT_PATH         "/tmp/fingerprint/"

#define DIA_CHECKSUM_RESULT_FILE_NAME_RAW_NOR            "raw_nor.chk"              /* RAW */   /* routineID=0x3060 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_RAW_EMMC_USER      "raw_emmc_user.chk"        /* RAW */   /* routineID=0x3062 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_RAW_BOOT           "raw_boot.chk"             /* RAW */   /* routineID=0x3064 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_FFS_ROOT           "filesystem_root.chk"      /* FFS */   /* routineID=0x3063 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_FFS_STATIC         "filesystem_static.chk"    /* FFS */   /* routineID=0x3067 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_FFS_MAP_DATA       "filesystem_navdata.chk"   /* FFS */   /* routineID=0x3068 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_FFS_SDB            "filesystem_sdb.chk"       /* FFS */   /* routineID=0x306B */
#define DIA_CHECKSUM_RESULT_FILE_NAME_FFS_GRACENOTE      "filesystem_gracenote.chk" /* FFS */   /* routineID=0x306C */
#define DIA_CHECKSUM_RESULT_FILE_NAME_V850_BOOTLOADER    "v850_bootloader.chk"      /* V850 */  /* routineID=0x3065 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_V850_APPLICATION   "v850_application.chk"     /* V850 */  /* routineID=0x3066 */
#define DIA_CHECKSUM_RESULT_FILE_NAME_RAW_NAV_SW         "navdata.chk"              /* NAV */   /* routineID=0x3068 */  /* deprecated, info from SwUpdate team: please do not use this */

//Module macros
#define DIA_CHECKSUM_MODULE_NAME_RAW         "raw"
#define DIA_CHECKSUM_MODULE_NAME_FFS         "filesystem"
#define DIA_CHECKSUM_MODULE_NAME_V850        "v850"

//Area macros
#define DIA_CHECKSUM_AREA_NAME_NOR           "nor"          /* RAW */   /* routineID=0x3060 */
#define DIA_CHECKSUM_AREA_NAME_EMMC_USER     "emmc_user"    /* RAW */   /* routineID=0x3062 */
#define DIA_CHECKSUM_AREA_NAME_BOOT          "boot"         /* RAW */   /* routineID=0x3064 */
#define DIA_CHECKSUM_AREA_NAME_ROOT          "root"         /* FFS */   /* routineID=0x3063 */
#define DIA_CHECKSUM_AREA_NAME_STATIC        "static"       /* FFS */   /* routineID=0x3067 */
#define DIA_CHECKSUM_AREA_NAME_MAP_DATA      "navdata"      /* FFS */   /* routineID=0x3068 */
#define DIA_CHECKSUM_AREA_NAME_SDB           "sdb"          /* FFS */   /* routineID=0x306B */
#define DIA_CHECKSUM_AREA_NAME_GRACENOTE     "gracenote"    /* FFS */   /* routineID=0x306C */
#define DIA_CHECKSUM_AREA_NAME_BOOTLOADER    "bootloader"   /* V850 */  /* routineID=0x3065 */
#define DIA_CHECKSUM_AREA_NAME_APPLICATION   "application"  /* V850 */  /* routineID=0x3066 */
#define DIA_CHECKSUM_AREA_NAME_NAV           "navdata"      /* NAV */   /* routineID=0x3068 */     /* deprecated, info from SwUpdate team: please do not use this */


#define DIA_CHECKSUM_V850_SHORT_LEN_IN_CHARS    16
#define DIA_CHECKSUM_LEN_IN_CHARS               32

static tU8 sRoutineResultMapping[DIA_EN_RTCTRL_STATUS_COUNT] = {
      0xFF, // DIA_EN_RTCTRL_STATUS_UNKNOWN
      0xFF, // DIA_EN_RTCTRL_STATUS_IDLE
      0x03, // DIA_EN_RTCTRL_STATUS_IN_PROGRESS
      0x00, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK
      0x01, // DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK
      0x02, // DIA_EN_RTCTRL_STATUS_ABORTED
      0xFF  // DIA_EN_RTCTRL_STATUS_TIMED_OUT
};

static tU8 sVerificationResultMapping[DIA_EN_ERROR_CODE_COUNT] = {
      0x00, // DIA_EN_ERROR_CODE_OK
      0x01, // DIA_EN_ERROR_CODE_CHECK_PERFORMED_CHECKSUM_DO_NOT_MATCH
      0x01, // DIA_EN_ERROR_CODE_CHECK_PERFORMED_CHECKSUM_DO_NOT_MATCH_BUT_SELFCHECK_OK
      0x01, // DIA_EN_ERROR_CODE_CHECK_CANT_BE_EXECUTED
      0x02, // DIA_EN_ERROR_CODE_NO_CHECKSUM_FILE_DIRECTORY
      0x02, // DIA_EN_ERROR_CODE_NO_CHECKSUM_FILE
      0x01, // DIA_EN_ERROR_CODE_PARAMETER_ERROR
      0x01, // DIA_EN_ERROR_CODE_NO_VALID_BOOTCHAIN_FOUND
      0x02, // DIA_EN_ERROR_CODE_NOT_APPLICABLE
     };

dia_chksum_attr
aDiaChkSumMapTable[DIA_EN_CAV_MEM_AREA_COUNT] = {
   {NULL,                           NULL,                                NULL                                           }, /* DIA_EN_CAV_MEM_AREA_UNKNOWN */
   {DIA_CHECKSUM_MODULE_NAME_RAW,   DIA_CHECKSUM_AREA_NAME_NOR,          DIA_CHECKSUM_RESULT_FILE_NAME_RAW_NOR          }, /* DIA_EN_CAV_MEM_AREA_RAW_NOR */              /* routineID=0x3060 */
   {DIA_CHECKSUM_MODULE_NAME_RAW,   DIA_CHECKSUM_AREA_NAME_EMMC_USER,    DIA_CHECKSUM_RESULT_FILE_NAME_RAW_EMMC_USER    }, /* DIA_EN_CAV_MEM_AREA_RAW_EMMC_USER */        /* routineID=0x3062 */
   {DIA_CHECKSUM_MODULE_NAME_RAW,   DIA_CHECKSUM_AREA_NAME_BOOT,         DIA_CHECKSUM_RESULT_FILE_NAME_RAW_BOOT         }, /* DIA_EN_CAV_MEM_AREA_RAW_BOOT */             /* routineID=0x3064 */
   {DIA_CHECKSUM_MODULE_NAME_FFS,   DIA_CHECKSUM_AREA_NAME_ROOT,         DIA_CHECKSUM_RESULT_FILE_NAME_FFS_ROOT         }, /* DIA_EN_CAV_MEM_AREA_FFS_ROOT */             /* routineID=0x3063 */
   {DIA_CHECKSUM_MODULE_NAME_FFS,   DIA_CHECKSUM_AREA_NAME_STATIC,       DIA_CHECKSUM_RESULT_FILE_NAME_FFS_STATIC       }, /* DIA_EN_CAV_MEM_AREA_FFS_STATIC */           /* routineID=0x3067 */
   {DIA_CHECKSUM_MODULE_NAME_FFS,   DIA_CHECKSUM_AREA_NAME_MAP_DATA,     DIA_CHECKSUM_RESULT_FILE_NAME_FFS_MAP_DATA     }, /* DIA_EN_CAV_MEM_AREA_FFS_MAP_DATA */         /* routineID=0x3068 */
   {DIA_CHECKSUM_MODULE_NAME_FFS,   DIA_CHECKSUM_AREA_NAME_SDB,          DIA_CHECKSUM_RESULT_FILE_NAME_FFS_SDB          }, /* DIA_EN_CAV_MEM_AREA_FFS_SDB */              /* routineID=0x306B */
   {DIA_CHECKSUM_MODULE_NAME_FFS,   DIA_CHECKSUM_AREA_NAME_GRACENOTE,    DIA_CHECKSUM_RESULT_FILE_NAME_FFS_GRACENOTE    }, /* DIA_EN_CAV_MEM_AREA_FFS_GRACENOTE */        /* routineID=0x306C */
   {DIA_CHECKSUM_MODULE_NAME_V850,  DIA_CHECKSUM_AREA_NAME_BOOTLOADER,   DIA_CHECKSUM_RESULT_FILE_NAME_V850_BOOTLOADER  }, /* DIA_EN_CAV_MEM_AREA_V850_BOOTLOADER */      /* routineID=0x3065 */
   {DIA_CHECKSUM_MODULE_NAME_V850,  DIA_CHECKSUM_AREA_NAME_APPLICATION,  DIA_CHECKSUM_RESULT_FILE_NAME_V850_APPLICATION }, /* DIA_EN_CAV_MEM_AREA_V850_APPLICATION */     /* routineID=0x3066 */
   {DIA_CHECKSUM_MODULE_NAME_RAW,   DIA_CHECKSUM_AREA_NAME_NAV,          DIA_CHECKSUM_RESULT_FILE_NAME_RAW_NAV_SW       }  /* DIA_EN_CAV_MEM_AREA_NAV */                  /* routineID=0x3068 */   /*deprecated, info from SwUpdate team: please do not use this*/
   };

tU8* dia_RoutineCtrlChecksum::sFullPathScript = const_cast<tU8*>((const tU8*)DIA_CHECKSUM_GENERIC_SCRIPT_PATH);

// Constructor to use standard script file
dia_RoutineCtrlChecksum::dia_RoutineCtrlChecksum (tCString name, tU16 udsID, dia_eRoutineID routineID)
   : dia_Routine(name, udsID, routineID, DIA_EN_RTCTRL_TYPE_LONG_TERM),
     mInOutParam(name),
     mDID(udsID),
#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
     mThreadID(0)
#else
     mThreadID(OSAL_ERROR)
#endif
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::dia_RoutineCtrlChecksum(tCString, tU16, dia_eRoutineID)");
   if(udsID == DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_NAVI_RAW)
   {
      dia_RoutineCtrlChecksum::sFullPathScript = const_cast<tU8*>((const tU8*)DIA_CHECKSUM_NAV_SCRIPT_PATH);
   }
   else
   {
      dia_RoutineCtrlChecksum::sFullPathScript = const_cast<tU8*>((const tU8*)DIA_CHECKSUM_GENERIC_SCRIPT_PATH);
   }

}

//Constructor to provide alternative script file
dia_RoutineCtrlChecksum::dia_RoutineCtrlChecksum (tCString name, tU16 udsID, dia_eRoutineID routineID, tCString executable)
   : dia_Routine(name, udsID, routineID, DIA_EN_RTCTRL_TYPE_LONG_TERM),
     mInOutParam(name),
     mDID(udsID),
#ifdef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
     mThreadID(0)
#else
     mThreadID(OSAL_ERROR)
#endif
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::dia_RoutineCtrlChecksum(tCString, tU16, dia_eRoutineID)");
   setFullPathScript(const_cast<tU8*>((const tU8*)executable));
}

//------------------------------------------------------------------------------

dia_RoutineCtrlChecksum::~dia_RoutineCtrlChecksum ( void )
{
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlChecksum::start ( std::vector<tU8>& params, tU8 /*timerValue*/ )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::start()");
   tDiaResult retCode = DIA_FAILED;

   // prepare processing of the routine
   vInitialize();

   // trace
   std::vector<tU8>::iterator iter = params.begin();
   for (int i=0; iter != params.end(); iter++, i++)
   {
      DIA_TR_INF("dia_RoutineCtrlChecksum::start params[%d] = 0x%02x", i, (*iter) );
   }

   dia_File scriptFile((tCString)sFullPathScript);

   if ( !scriptFile.doesExist() )
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::start Script CANNOT be started. ABORTED");

      /* retCode is already set to DIA_FAILED */
   }
   else if (0!=params.size())
   {
      /* check input data (expected 0 bytes of routine control option) */
      DIA_TR_ERR("dia_RoutineCtrlChecksum::start incorrect size of params = 0x%08zx", params.size());

      /* retCode is already set to DIA_FAILED */
   }
   else
   {
      dia_eCAV_Mem_Area checksumArea = DIA_EN_CAV_MEM_AREA_UNKNOWN;

      switch(mDID)
      {
            /*
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_NOR_FFS:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_NOR_FFS");
            checksumArea = DIA_EN_CAV_AREA_NOR_FFS;
            break;
         */
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_NOR_RAW:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_RAW_NOR");
            checksumArea = DIA_EN_CAV_MEM_AREA_RAW_NOR;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_RAW:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_RAW_EMMC_USER");
            checksumArea = DIA_EN_CAV_MEM_AREA_RAW_EMMC_USER;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_FFS_ROOT:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_FFS_ROOT");
            checksumArea = DIA_EN_CAV_MEM_AREA_FFS_ROOT;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_FASTBOOT:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_RAW_BOOT");
            checksumArea = DIA_EN_CAV_MEM_AREA_RAW_BOOT;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_V850_BOOT:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_V850_BOOTLOADER");
            checksumArea = DIA_EN_CAV_MEM_AREA_V850_BOOTLOADER;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_V850_APP:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_V850_APPLICATION");
            checksumArea = DIA_EN_CAV_MEM_AREA_V850_APPLICATION;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_FFS_STATIC:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_FFS_STATIC");
            checksumArea = DIA_EN_CAV_MEM_AREA_FFS_STATIC;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_FFS_MAP_DATA:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_FFS_MAP_DATA");
            checksumArea = DIA_EN_CAV_MEM_AREA_FFS_MAP_DATA;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_FFS_SDB:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_FFS_SDB");
            checksumArea = DIA_EN_CAV_MEM_AREA_FFS_SDB;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_EMMC_FFS_GRACENOTE:
            DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_FFS_GRACENOTE");
            checksumArea = DIA_EN_CAV_MEM_AREA_FFS_GRACENOTE;
            break;
         case DIA_C_U16_DID_RBCM_RTCTRL_CAV_CHECKSUM_NAVI_RAW:
             DIA_TR_INF("dia_RoutineCtrlChecksum::start area = CHECKSUM_NAV_SW");
             checksumArea = DIA_EN_CAV_MEM_AREA_NAV;
            break;
         default:
            DIA_TR_ERR("dia_RoutineCtrlChecksum::start area(0x%04x) UNKNOWN", mDID);
            break;
      }

      if (DIA_EN_CAV_MEM_AREA_UNKNOWN!=checksumArea)
      {
         /* start a new thread */
         retCode = startScript(checksumArea);
      }
      else
      {
         DIA_TR_ERR("dia_RoutineCtrlChecksum::start Script will be NOT started.");
      }

      if ( DIA_SUCCESS == retCode )
      {
         eSetStatus(DIA_EN_RTCTRL_STATUS_IN_PROGRESS);
         DIA_TR_INF("dia_RoutineCtrlChecksum::start RTCTRL_STATUS_IN_PROGRESS");
      }
      else if (DIA_E_CONDITIONS_NOT_CORRECT == retCode)
      {
         // Send negative response "Conditions not correct"
         eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK);
         DIA_TR_INF("dia_RoutineCtrlChecksum::start DIA_E_CONDITIONS_NOT_CORRECT");
      }
      else
      {
         eSetStatus(DIA_EN_RTCTRL_STATUS_ABORTED);
         DIA_TR_INF("dia_RoutineCtrlChecksum::start CalculateAndVerifyChecksum FAILED.");
      }

      //negative answer doesn't need any bytes in output buffer
      if (DIA_FAILED!=retCode)
      {
         // one byte expected
         mResults.push_back(sRoutineResultMapping[eGetStatus()]);

         mIsResultReady = TRUE;
      }
   }

   DIA_TR_INF("dia_RoutineCtrlChecksum::start retCode = 0x%08X", retCode);

   return retCode;
}

//------------------------------------------------------------------------------
//! Return value: If false, negative response. If true, positive response;

tDiaResult
dia_RoutineCtrlChecksum::requestResult( std::vector<tU8>& results )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::requestResult()");
   tDiaResult retCode = DIA_E_SEQUENCE_ERROR;

   results.clear();
   if ( !(mResults.empty()) )
   {
      /* this code is executed in case of reply for start request only */
      DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult --- 1");
      std::vector<tU8>::iterator iter = mResults.begin();
      for ( ; iter != mResults.end(); iter++ )
      {
         results.push_back(*iter);
      }

      // clear buffer, such reply is necessary once
      mResults.clear();

      // keep the result until new calculation is started
      retCode = DIA_SUCCESS;
   }
   else
   {
      /* trace */
      switch ( mStatus )
      {
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK:
            DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult --- COMPLETED_AND_OK");
         break;

         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK:
            DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult --- COMPLETED_AND_NOK");
         break;

         case DIA_EN_RTCTRL_STATUS_IN_PROGRESS:
            DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult --- IN_PROGRESS");
         break;

         case DIA_EN_RTCTRL_STATUS_ABORTED:
            DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult --- ABORTED");
         break;

         default:
            DIA_TR_ERR("dia_RoutineCtrlChecksum::requestResult --- default mStatus = 0x%02X", mStatus);
         break;
      }

      switch ( mStatus )
      {
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK:
         case DIA_EN_RTCTRL_STATUS_COMPLETED_AND_NOK:
         case DIA_EN_RTCTRL_STATUS_IN_PROGRESS:
         {
            bool bThreadStartedCopy = mInOutParam.mbThreadStarted;   /* read is allowed everytime */

            /* script execution not allowed during report parsing */
            if ( (false==bThreadStartedCopy) &&
                 (DIA_SUCCESS==mInOutParam.mScriptExitCode)  )
            {
               //take always bigger buffer
               tU8 buffRespConvStr[DIA_CHECKSUM_SIZE_OF_CHECKSUM_IN_BYTES] = {0};
               tU8 verificationResult = 0;

               const char* resultFilePath = (mInOutParam.mReqDiagArea == DIA_EN_CAV_MEM_AREA_NAV)
              										? DIA_CHECKSUM_RESULT_FILE_NAVI_FINGERPRINT_PATH
              										: DIA_CHECKSUM_RESULT_FILE_GENERIC_PATH;

                             retCode = parseReport(aDiaChkSumMapTable, resultFilePath, buffRespConvStr, sizeof(buffRespConvStr), &verificationResult);


               if (DIA_E_BUSY_REPEAT==retCode)
               {
                  /* report file couldn't be opened */
                  DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult DIA_E_BUSY_REPEAT at first request");

                  /* this is exception when we send positive answer */
                  retCode = DIA_SUCCESS;

                  /* send positive response with null checksum */
                  vposResponse(results, NULL, DIA_EN_RTCTRL_STATUS_IN_PROGRESS, DIA_EN_ERROR_CODE_CHECK_CANT_BE_EXECUTED);

                  mIsResultReady = TRUE;
               }
               else if (DIA_E_ROUTINE_NOT_COMPLETE==retCode)
               {
                  /* either checksum or return code cannot be found, or cookies do not match */
                  DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult parseReport returned DIA_E_ROUTINE_NOT_COMPLETE");

                  /* this is exception when we send positive answer */
                  retCode = DIA_SUCCESS;

                  /* send positive response with null checksum */
                  vposResponse(results, NULL, DIA_EN_RTCTRL_STATUS_ABORTED, DIA_EN_ERROR_CODE_NO_CHECKSUM_FILE);

                  mIsResultReady = TRUE;
               }
               else if (DIA_SUCCESS==retCode)
               {
                  eSetStatus(DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK);

                  DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult parseReport returned DIA_SUCCESS");

                  /* send positive response with non-zero checksum*/
                  vposResponse(results, buffRespConvStr, DIA_EN_RTCTRL_STATUS_COMPLETED_AND_OK, static_cast<dia_eScriptErrorCode>(verificationResult));

                  mIsResultReady = TRUE;
               }
               else
               {
                  /* do nothing */
               }
            }
            else
            {
               if (true==bThreadStartedCopy)
               {
                  /* positive answer */
                  retCode = DIA_SUCCESS;

                  /* send positive response with zero checksum*/
                  vposResponse(results, NULL, DIA_EN_RTCTRL_STATUS_IN_PROGRESS, DIA_EN_ERROR_CODE_NO_CHECKSUM_FILE);

                  mIsResultReady = TRUE;
               }
               else
               {
                  DIA_TR_INF("dia_RoutineCtrlChecksum::requestResult mScriptExitCode is NOK, report parsing makes no sense.");
               }
            }
         }
         break;

         default:
            /* do nothing */
         break;
      }
   }

   return retCode;
}

void dia_RoutineCtrlChecksum::vposResponse ( std::vector<tU8>& results, tU8* inBuff, dia_eRoutineStatus routineResultInx, dia_eScriptErrorCode verificationResultInx )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::vposResponse()");

   DIA_TR_INF("dia_RoutineCtrlChecksum::vposResponse routineResultInx=0x%02X verificationResultInx=0x%02X", routineResultInx, verificationResultInx );

   // first byte of answer: status
   results.push_back(sRoutineResultMapping[routineResultInx]);

   //configuration hash
   for (tU8 i=0; i<DIA_CHECKSUM_SIZE_OF_CHECKSUM_IN_BYTES; i++)
   {
      tU8 param = (inBuff!=NULL) ? (inBuff[i]): (0x00);
      results.push_back(param);
   }

   // last byte of answer: verification result
   results.push_back(sVerificationResultMapping[verificationResultInx]);
}

//------------------------------------------------------------------------------

void
dia_RoutineCtrlChecksum::vOnServiceTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::vOnServiceTimeout()");

   eSetStatus(DIA_EN_RTCTRL_STATUS_TIMED_OUT);

#ifndef __DIA_UNIT_TESTING__
   dia_RoutineCtrlManager* pRTCtrlMgr = getInstanceOfRoutineControlManager();

   if ( pRTCtrlMgr )
   {
       pRTCtrlMgr->vOnRoutineUpdate(*this);
   }
#endif
}

//------------------------------------------------------------------------------

tDiaResult
dia_RoutineCtrlChecksum::convertString ( const tU8* inConfigHash, tU8* outConvStr, tU8 numCharsToBeConverted )
{
   dia_tclFnctTrace oTrace("dia_RoutineCtrlChecksum::convertString()");
   tDiaResult retVal = DIA_SUCCESS;

   if (numCharsToBeConverted & 0x01)
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::convertString numCharsToBeConverted = %u is odd", numCharsToBeConverted);
      return DIA_FAILED;
   }

   bool error = false;
   tU8 tmpOutBuff[(numCharsToBeConverted>>1)];
   memset(tmpOutBuff, 0, sizeof(tmpOutBuff) );

   DIA_TR_INF("dia_RoutineCtrlChecksum::convertString configHash=%s", inConfigHash);

   // set configuration hash
   for(int i=0; i < numCharsToBeConverted; i+=2)
   {
      tU8 tmpMSN = (tU8)tolower(inConfigHash[i+0]);  //most significant nibble
      tmpMSN = (tmpMSN>'9') ? static_cast<tU8>((tmpMSN -'a') + 10): static_cast<tU8>(tmpMSN - '0');
      tU8 tmpLSN = (tU8)tolower(inConfigHash[i+1]);  //less significant nibble
      tmpLSN = (tmpLSN>'9') ? static_cast<tU8>((tmpLSN -'a') + 10): static_cast<tU8>(tmpLSN - '0');

      DIA_TR_INF("dia_RoutineCtrlChecksum::convertString tmpMSN = 0x%02X tmpLSN = 0x%02X", tmpMSN, tmpLSN);

      /* check converted values */
      if ((tmpMSN>0x0F) || (tmpLSN>0x0F))
      {
         error = true;
         DIA_TR_INF("dia_RoutineCtrlChecksum::convertString - Conversion STR2HEX failure.");
         break;
      }

      tmpOutBuff[(i>>1)] = static_cast<tU8>((0xFF & (tmpMSN<<4)) + tmpLSN);

      DIA_TR_INF("dia_RoutineCtrlChecksum::convertString array[0x%02X]= 0x%02X", (i>>1), tmpOutBuff[(i>>1)]);
   }

   if (true==error)
   {
      DIA_TR_INF("dia_RoutineCtrlChecksum::convertString Error occurred.");
      retVal = DIA_FAILED;
   }

   // write to output buffer in case of success
   if (DIA_SUCCESS==retVal)
   {
      (void)memcpy(outConvStr, tmpOutBuff, (numCharsToBeConverted>>1));
   }

   return retVal;
}

tDiaResult
dia_RoutineCtrlChecksum::startScript ( dia_eCAV_Mem_Area diag_area)
{
   dia_tclFnctTrace trc("dia_RoutineCtrlChecksum::startScript()");
   tDiaResult retCode = DIA_FAILED;

   DIA_TR_INF("dia_RoutineCtrlChecksum::startScript: diag_area=%d", diag_area);

   if (false==mInOutParam.mbThreadStarted)
   {
      mInOutParam.mReqDiagArea = diag_area;

      mInOutParam.mbThreadStarted = true;

      uint32_t cookie = getInstanceOfRandomGenerator()->getRandomNumber();
      snprintf(mInOutParam.mCookieString, COOKIE_STR_LEN, "%lu", (unsigned long)cookie);
      DIA_TR_INF("dia_RoutineCtrlChecksum::startScript: mCookieString = %s", mInOutParam.mCookieString);

      tDiaResult startedResult = vThreadRequest();

      if (DIA_FAILED==startedResult)
      {
         DIA_TR_INF( "dia_RoutineCtrlChecksum::startScript vThreadRequest failed" );
         mInOutParam.mbThreadStarted = false;
      }
      else
      {
         retCode = DIA_SUCCESS;
         DIA_TR_INF( "dia_RoutineCtrlChecksum::startScript vThreadRequest succeeded" );
      }
   }
   else
   {
      retCode = DIA_SUCCESS;
      DIA_TR_INF( "dia_RoutineCtrlChecksum::startScript thread already started");
   }

   DIA_TR_INF( "dia_RoutineCtrlChecksum::startScript retCode=0x%08X", retCode);

   return retCode;
}

void
dia_RoutineCtrlChecksum::convertToLower(std::string& str) const
{
   for ( std::string::iterator it = str.begin(); it != str.end(); ++it )
   {
      *it = (char) tolower(*it);
   }
}

bool
dia_RoutineCtrlChecksum::getVal(std::string& inLine, const char* key, std::string& outStrVal)
{
   if (NULL==key)
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::getVal: key is NULL");
      return false;
   }

   bool retVal = false;

   std::string strKey(key);
   strKey += "=\"";

   size_t lineLen = inLine.size();
   size_t firstRes = inLine.find(strKey);

   if ((size_t)-1!=firstRes)
   {
      DIA_TR_INF("dia_RoutineCtrlChecksum::getVal: key=%s found! firstRes=%zu", strKey.c_str(), firstRes);

      //check length of line
      size_t posString = firstRes + strKey.size();

      if (lineLen > posString)
      {
         size_t secondRes = inLine.find('\"', posString );

         if ((size_t)-1!=secondRes)
         {
            DIA_TR_INF("quotation mark found at: firstPos=%zu secondPos=%zu", (posString-1), secondRes);

            if (posString!=secondRes)
            {
               size_t valLen = (secondRes-posString);
               outStrVal.assign(inLine, posString, valLen);
               retVal = true;

               DIA_TR_INF("Found (key=%s,value=%s) valLen=%zu", key, outStrVal.c_str(), valLen);
            }
            else
            {
               DIA_TR_INF("Empty string for value (key=%s)", key);
            }
         }
      }
      else
      {
         DIA_TR_ERR("Line incorrect. lineLen=%zu posString=%zu", lineLen, posString);
      }
   }

   return retVal;
}

tDiaResult
dia_RoutineCtrlChecksum::parseReport(dia_chksum_attr* pInMapTable, const char* pInPathCompReport, tU8* pOutBuffRespConvStr, tU32 outBuffSize, tU8* pOutVerificationResult)
{
   dia_tclFnctTrace trc("dia_RoutineCtrlChecksum::parseReport()");

   const char* checksumFileName = NULL;

   DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport: mInOutParam.mReqDiagArea = %d", mInOutParam.mReqDiagArea);

   if (mInOutParam.mReqDiagArea < DIA_EN_CAV_MEM_AREA_COUNT)
   {
      checksumFileName = pInMapTable[mInOutParam.mReqDiagArea].ResultFileName;
   }
   else
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: mInOutParam.mReqDiagArea=%d incorrect", mInOutParam.mReqDiagArea );

      // incorrect input - please send request again
      return DIA_E_BUSY_REPEAT;
   }

   if (NULL!=checksumFileName)
   {
      DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport: Try to open: %s", checksumFileName);
   }
   else
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: Requested diag_area not found.");

      // incorrect input - please send request again
      return DIA_E_BUSY_REPEAT;
   }

   std::string compRepFullPath(pInPathCompReport);
   compRepFullPath += checksumFileName;

   DIA_TR_INF("compRepFullPath = %s", compRepFullPath.c_str());

   dia_File report(compRepFullPath.c_str());

   std::string cookieVal, fingerprintVal, checksumstoredVal, codeVal;
   bool cookieFound=false, fingerprintFound=false, checksumstoredFound=false, codeFound=false;
   bool isNavDiagArea = (mInOutParam.mReqDiagArea == DIA_EN_CAV_MEM_AREA_NAV) ? true : false;

   // open and parse report
   if (DIA_SUCCESS==report.open("r"))
   {
      std::string line;
      DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport: report opened successfully.");

      //parse report (search for keys line by line)
      while(DIA_SUCCESS==report.getLine(line))
      {
         /* convert to lowercase */
         convertToLower(line);

         if (!cookieFound)
         {
            if (true==getVal(line, "cookie", cookieVal))
            {
               DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport cookie=%s", cookieVal.c_str());
               cookieFound = true;
            }
         }

         if (!fingerprintFound)
         {
            if (true==getVal(line, "fingerprint", fingerprintVal))
            {
               DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport fingerprint=%s", fingerprintVal.c_str());
               fingerprintFound = true;
            }
         }

         if (!checksumstoredFound)
         {
            if (true==getVal(line, "v850_crc", checksumstoredVal))
            {
               DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport v850_crc=%s", checksumstoredVal.c_str());
               checksumstoredFound = true;
            }
         }

         if (!codeFound)
         {
            if (true==getVal(line, "code", codeVal))
            {
               DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport code=%s", codeVal.c_str());
               codeFound = true;
            }
         }

         if (cookieFound && codeFound && (fingerprintFound || checksumstoredFound))
         {
            DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport all keys found");
            break;
         }
      }

      if (DIA_SUCCESS!=report.close())
      {
         DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: close() failed for %s", report.getFileName());
      }
   }
   else
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: UNABLE TO OPEN THE REPORT");

      // incorrect input - please send request again
      return DIA_E_BUSY_REPEAT;
   }

   if (!cookieFound || !codeFound || (!fingerprintFound && !checksumstoredFound))
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: at least one key is missing");

      if (!cookieFound) DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: cookie is missing");
      if (!codeFound)   DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: code is missing");
      if (!fingerprintFound && !checksumstoredFound)
      {
         DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: fingerprint and checksumstored are missing");
      }

      // incorrect input - please send request again
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   if (fingerprintFound && checksumstoredFound)
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: ERROR! Both fingerprint and checksumstored have been read.");

      //send positive response with aborted (0x02)
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   std::string hash = fingerprintFound ? fingerprintVal : checksumstoredVal;

   //check cookie
   if (cookieVal.compare(mInOutParam.mCookieString)!=0)
   {
      DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: ERROR Cookie NOT Matched" );
      DIA_TR_ERR("expected cookie = %s, actual cookie = %s",  mInOutParam.mCookieString, cookieVal.c_str()  );

      //send positive response with aborted (0x02)
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport: Cookie Matched" );

   //interpret code
   dia_eScriptErrorCode codeResult = static_cast<dia_eScriptErrorCode>(codeVal.at(0) - '0');

   switch(codeResult)
   {
      case DIA_EN_ERROR_CODE_OK:
      case DIA_EN_ERROR_CODE_CHECK_PERFORMED_CHECKSUM_DO_NOT_MATCH:
      case DIA_EN_ERROR_CODE_CHECK_PERFORMED_CHECKSUM_DO_NOT_MATCH_BUT_SELFCHECK_OK:
      case DIA_EN_ERROR_CODE_CHECK_CANT_BE_EXECUTED:
      case DIA_EN_ERROR_CODE_NO_CHECKSUM_FILE_DIRECTORY:
      case DIA_EN_ERROR_CODE_NO_CHECKSUM_FILE:
      case DIA_EN_ERROR_CODE_PARAMETER_ERROR:
      case DIA_EN_ERROR_CODE_NO_VALID_BOOTCHAIN_FOUND:
      case DIA_EN_ERROR_CODE_OTHER_ERROR:
         /* do nothing */
      break;

      default:
         DIA_TR_ERR("dia_RoutineCtrlChecksum::parseReport: ERROR Unexpected CodeResult=0x%02X", codeResult);
         codeResult = DIA_EN_ERROR_CODE_OTHER_ERROR;
      break;
   }

   DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport: Result Code is %d", codeResult);

   switch (mInOutParam.mReqDiagArea)
   {
      case DIA_EN_CAV_MEM_AREA_V850_BOOTLOADER:
      case DIA_EN_CAV_MEM_AREA_V850_APPLICATION:
         if (DIA_CHECKSUM_V850_SHORT_LEN_IN_CHARS==hash.size())
         {
            hash += "0000000000000000";

            DIA_TR_INF("added zero padding checksumstored = %s", hash.c_str());
         }
         else
         {
            DIA_TR_INF("zero padding not necessary");
         }
      break;

      default:
         /* do nothing */
         break;
   }

   //check size of checksum
   if (DIA_CHECKSUM_LEN_IN_CHARS!=hash.size())
   {
      DIA_TR_ERR("parseReport: Incorrect checksum length. Expected=%d Actual=%zu", DIA_CHECKSUM_LEN_IN_CHARS, hash.size());

      // incorrect input - please send request again
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   if (NULL!=pOutVerificationResult)
   {
	   if(isNavDiagArea)
	   	   {
	   		   *pOutVerificationResult = static_cast<tU8>(DIA_EN_ERROR_CODE_NOT_APPLICABLE);
	   	   }
	   	   else
	   	   {
	   		   *pOutVerificationResult = static_cast<tU8>(codeResult);
	   	   }
   }
   else
   {
      DIA_TR_ERR("pOutVerificationResult is NULL");

      // incorrect input - please send request again
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   if (NULL==pOutBuffRespConvStr)
   {
      DIA_TR_ERR("pOutBuffRespConvStr is NULL");

      // incorrect input - please send request again
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   if ((outBuffSize<(hash.size()/2)) || (0xFF<hash.size()))
   {
      DIA_TR_ERR("Buff too short or hash too long: outBuffSize=%u, hash in bytes=%zu", outBuffSize, (size_t)(hash.size()/2));

      // incorrect input - please send request again
      return DIA_E_ROUTINE_NOT_COMPLETE;
   }

   //convert hex string to byte array
   if (DIA_FAILED==convertString((const tU8*)hash.c_str() /*input*/, pOutBuffRespConvStr /*output*/, (tU8)hash.size()))
   {
      /* conversion error occurred */
      if (DIA_EN_ERROR_CODE_OK==codeResult)
      {
         DIA_TR_ERR("convertString failed. Set verification result to DIA_EN_ERROR_CODE_OTHER_ERROR");

         /* update it if not set before */
         *pOutVerificationResult = DIA_EN_ERROR_CODE_OTHER_ERROR;
      }
      else
      {
         DIA_TR_ERR("convertString failed. Verification result has been set before conversion.");
      }
   }

   DIA_TR_INF("dia_RoutineCtrlChecksum::parseReport() returned DIA_SUCCESS");

   return DIA_SUCCESS;
}

tDiaResult
dia_RoutineCtrlChecksum::vThreadRequest()
{
   dia_tclFnctTrace trc("dia_RoutineCtrlChecksum::vThreadRequest()");

   char nameThread[DIA_CHECKSUM_MAX_SIZE_OF_THEAD_NAME] = {0};

   //create a unique name for the thread
   std::string(std::string(mName) + "Thread").copy(nameThread, DIA_CHECKSUM_MAX_SIZE_OF_THEAD_NAME-1);

   DIA_TR_INF( "ENTER dia_RoutineCtrlChecksum::vThreadRequest nameThread=%s", nameThread);

   tDiaResult retVal = mInOutParam.startThread();
   mThreadID = mInOutParam.getThreadID();

   DIA_TR_INF( "vThreadRequest retVal=0x%08X mThreadID=0x%08X nameThread=%s", (uint32_t)retVal, (uint32_t)mThreadID, nameThread);

   return retVal;
}

