/*
 * dia_IOCtrlWifiMode.cpp
 *
 *  Created on: 04.02.2019
 *      Author: kaa1hi
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#include "common/services/uds/production/dia_IOCtrlWifiMode.h"
#include "common/services/uds/generic/dia_SrvHandlerGenericIOCtrlByIdentifier.h"

const tU32 dia_IOCtrlWifiMode::IOCTRL_WIFI_MODE_PARAMETER_SIZE = 1;

//------------------------------------------------------------------------------

dia_IOCtrlWifiMode::dia_IOCtrlWifiMode ( void )
     : dia_IOCtrlSignal("dia_IOCtrlWifiMode", DIA_C_U16_DID_RBCM_IOCTRL_WIFI_MODE, IOCTRL_WIFI_MODE_PARAMETER_SIZE),
       mLastWifiMode("")
{
}

//------------------------------------------------------------------------------

dia_IOCtrlWifiMode::~dia_IOCtrlWifiMode ( void )
{
   _BP_TRY_BEGIN
   {
      (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
   }
   _BP_CATCH_ALL
   {
       DIA_TR_ERR("EXCEPTION CAUGHT: dia_IOCtrlWifiMode::~dia_IOCtrlWifiMode !!!");
       NORMAL_M_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlWifiMode::handleRequest ( tU8 /*timerValue*/, std::vector<tU8>* ctrlValue )
{
   dia_tclFnctTrace oTrace("dia_IOCtrlWifiMode::handleRequest()");

   if (NULL==ctrlValue)
   {
      DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest ctrlValue is NULL.");
      return DIA_E_INVALID_POINTER;
   }

   if (IOCTRL_WIFI_MODE_PARAMETER_SIZE!=ctrlValue->size())
   {
      DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest Invalid size of param: exp %u, act %zu.", IOCTRL_WIFI_MODE_PARAMETER_SIZE, ctrlValue->size());
      return DIA_E_INVALID_LENGTH;
   }

   dia_enWifiMode wifiModeParam = DIA_ENUM_WIFI_MODE_UNKNOWN;
   std::map<tU8, dia_enWifiMode >::const_iterator i = mMapByteToWifiMode.find(ctrlValue->at(0));
   if ( i != mMapByteToWifiMode.end() )
   {
      DIA_TR_INF("dia_IOCtrlWifiMode::handleRequest Byte 0x%02X converted to 0x%02X.", ctrlValue->at(0), (tU8)i->second);
      wifiModeParam = i->second;
   }
   else
   {
      DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest Invalid param 0x%02X.", ctrlValue->at(0));
      setErrorInfo(DIA_E_OUT_OF_RANGE);
      return DIA_E_NOT_SUPPORTED;
   }

   std::map<dia_enWifiMode, const std::string >::const_iterator it = mMapWifiModeToString.find(wifiModeParam);
   if ( it != mMapWifiModeToString.end() )
   {
      DIA_TR_INF("dia_IOCtrlWifiMode::handleRequest WifiMode 0x%02X converted to '%s'.", wifiModeParam, it->second.c_str());
   }
   else
   {
      DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest Invalid param 0x%02X.", wifiModeParam);
      std::map<dia_enWifiMode, const std::string >::const_iterator itt = mMapWifiModeToString.begin();
      tU32 i = 0;
      while (itt != mMapWifiModeToString.end())
      {
         DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest Valid param #%d 0x%02X '%s'.", i, itt->first, itt->second.c_str());
         ++itt;
         ++i;
      }
      return DIA_E_NOT_SUPPORTED;
   }

   if (DIA_ENUM_WIFI_MODE_OFF_STA==wifiModeParam)
   {
      if (DIA_SUCCESS!=unsetWifiMode(it->second))
      {
         DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest setWifiMode failed, param %s.", it->second.c_str());
         return DIA_FAILED;
      }
   }
   else
   {
      if (DIA_SUCCESS!=setWifiMode(it->second))
      {
         DIA_TR_ERR("### dia_IOCtrlWifiMode::handleRequest setWifiMode failed, param %s.", it->second.c_str());
         return DIA_FAILED;
      }
   }

   mLastWifiMode = it->second;

   // now initialize the counter for the timer
   vSetTimer(DIA_C_U8_UDS_IOCTRL_TIMER_VALUE_INFINITE);
   mIsResultReady = FALSE;

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_IOCtrlWifiMode::vOnTerminate ( dia_eIOCtrlStatus status )
{
   dia_tclFnctTrace oTrace("dia_IOCtrlWifiMode::vOnTerminate()");
   tDiaResult retCode = DIA_E_NOERROR;

   if (eGetStatus() == DIA_EN_IOCTRL_STATUS_ACTIVE)
   {
      if (status == DIA_EN_IOCTRL_STATUS_INACTIVE)
      {
         // we have to wait for the response
         mIsResultReady = FALSE;
      }

      if (DIA_SUCCESS!=unsetWifiMode(mLastWifiMode))
      {
         DIA_TR_ERR("### dia_IOCtrlWifiMode::vOnTerminate unsetWifiMode failed.");
         retCode = DIA_FAILED;
      }
      else
      {
         DIA_TR_INF("dia_IOCtrlWifiMode::vOnTerminate unsetWifiMode success.");
      }
   }

   // this ioctrl is no longer active
   eSetStatus(status);

   return retCode;
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlWifiMode::onPrepareSetup ( tDiaResult response )
{
   dia_tclFnctTrace trc("dia_IOCtrlWifiMode::onPrepareSetup()");

   DIA_TR_INF("dia_IOCtrlWifiMode::onPrepareSetup response=0x%02X.", response);

   (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
   mIsResultReady = TRUE;
   if (DIA_SUCCESS==response)
   {
      DIA_TR_INF("dia_IOCtrlWifiMode::onPrepareSetup success.");
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      DIA_TR_ERR("dia_IOCtrlWifiMode::onPrepareSetup DIA_FAILED !!!");
   }

   dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlWifiMode::onDeActivateSetup ( tDiaResult response )
{
   dia_tclFnctTrace trc("dia_IOCtrlWifiMode::onDeActivateSetup()");

   DIA_TR_INF("dia_IOCtrlWifiMode::onDeActivateSetup response=0x%02X.", response);

   (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
   mIsResultReady = TRUE;
   if (DIA_SUCCESS==response)
   {
      DIA_TR_INF("dia_IOCtrlWifiMode::onDeActivateSetup success.");
   }
   else
   {
      setErrorInfo(DIA_E_CONDITIONS_NOT_CORRECT);
      DIA_TR_ERR("dia_IOCtrlWifiMode::onDeActivateSetup DIA_FAILED !!!");
   }

   dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate(this);
}

//-----------------------------------------------------------------------------

void
dia_IOCtrlWifiMode::handleTimeout ( void )
{
   dia_tclFnctTrace trc("dia_IOCtrlWifiMode::handleTimeout");
   (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlWifiMode::setWifiMode ( const std::string& mode )
{
   dia_tclFnctTrace trc("dia_IOCtrlWifiMode::setWifiMode( const std::string& mode )");

   if (mode.empty())
   {
      DIA_TR_ERR("### dia_IOCtrlWifiMode::setWifiMode empty string mode");
      return DIA_FAILED;
   }

   bool errorDetected = true;
   DIA_TR_INF("dia_IOCtrlWifiMode::setWifiMode mode '%s'.", mode.c_str());

   dia_IWifiSetups* pInterface = nullptr;
   if (querySysAdapterInterface<dia_IWifiSetups>(&pInterface) == DIA_SUCCESS)
   {
      if (pInterface)
      {
         (void) setSysAdapterListener<dia_IWifiSetupsListener>(this);

         if (pInterface->prepareSetup(mode) == DIA_SUCCESS)
         {
            DIA_TR_INF("dia_IOCtrlWifiMode::setWifiMode prepareSetup success.");
            errorDetected = false;
         }
      }
      else
      {
         DIA_TR_ERR("### dia_IOCtrlWifiMode::setWifiMode pInterface is NULL.");
      }
   }

   if (errorDetected)
   {
      (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
      DIA_TR_ERR("dia_IOCtrlWifiMode::setWifiMode --- SEND TO WIFI SERVER FAILED!!!!");
      return DIA_FAILED;
   }

   DIA_TR_INF("dia_IOCtrlWifiMode::setWifiMode return DIA_SUCCESS for mode '%s'.", mode.c_str());
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_IOCtrlWifiMode::unsetWifiMode ( const std::string& mode )
{
   dia_tclFnctTrace trc("dia_IOCtrlWifiMode::unsetWifiMode( const std::string& mode )");

   if (mode.empty())
   {
      DIA_TR_ERR("### dia_IOCtrlWifiMode::unsetWifiMode empty string mode");
      return DIA_FAILED;
   }

   bool errorDetected = true;

   DIA_TR_INF("dia_IOCtrlWifiMode::unsetWifiMode mode '%s'.", mode.c_str());

   dia_IWifiSetups* pInterface = nullptr;
   if (querySysAdapterInterface<dia_IWifiSetups>(&pInterface) == DIA_SUCCESS)
   {
      if (pInterface)
      {
         (void) setSysAdapterListener<dia_IWifiSetupsListener>(this);

         if (pInterface->deActivateSetup(mode) == DIA_SUCCESS)
         {
            DIA_TR_INF("dia_IOCtrlWifiMode::unsetWifiMode deActivateSetup success.");
            errorDetected = false;
         }
      }
      else
      {
         DIA_TR_ERR("### dia_IOCtrlWifiMode::unsetWifiMode pInterface is NULL.");
      }
   }

   if (errorDetected)
   {
      (void) unsetSysAdapterListener<dia_IWifiSetupsListener>(this);
      DIA_TR_ERR("dia_IOCtrlWifiMode::unsetWifiMode --- SEND TO WIFI SERVER FAILED!!!!");
      return DIA_FAILED;
   }

   DIA_TR_INF("dia_IOCtrlWifiMode::unsetWifiMode return DIA_SUCCESS for mode '%s'.", mode.c_str());
   return DIA_SUCCESS;
}
