/*!
 * \file       dia_SrvHandlerGenericWIFINameWrite.cpp
 *
 * \brief      Service handler for writing Wifi name consists from two parts.
 *
 * \details    First param is Property ID of first part of wifi name.
 *             Second param is length of first part of wifi name.
 *             Third param is Property ID of second part of wifi name.
 *             Fourth param is length of second part of wifi name.
 *
 * \component  Diagnostics
 *
 * \ingroup
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include "dia_SrvHandlerGenericWIFINameWrite.h"

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework/engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#define DATA_START        (tU16)3
using namespace dia;
//-----------------------------------------------------------------------------

dia_SrvHandlerGenericWIFINameWrite::dia_SrvHandlerGenericWIFINameWrite (tCString name, tU8 sid, tU16 did )
   : dia_ServiceHandlerUDS(name, sid, did)
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandlerGenericWIFINameWrite::dia_SrvHandlerGenericWIFINameWrite(tCString name, tU8 sid, tU16 did)");
#endif
}

//-----------------------------------------------------------------------------

dia_SrvHandlerGenericWIFINameWrite::~dia_SrvHandlerGenericWIFINameWrite( void )
{
}

//-------------------------------------------------------------------------------

void
dia_SrvHandlerGenericWIFINameWrite::vProcessRequest ( const std::vector<tArgsType>& vecArgs )
{
   dia_tclFnctTrace trc("dia_SrvHandlerGenericWIFINameWrite::vOnProcessRequest");

   /* SrvHandler Defines according to CDD-File */
   #define MSG_DATA_OFFSET 4                        // Offset from oDiagMsgBuffer-start to 1st data-byte (e.g. MemorySize-Byte)

   DIA_TR_INF("Received data buffer length %d !!", oDiagMsgBuffer().u16GetDataLength());

   if ( (0==vecArgs.size()) || (4!=vecArgs.size()) )
   {
      // Sending back the Negative answer
      DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: FAILED (Property ID Not Available) vecArgs.size()=%zu!!", vecArgs.size());
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      return;
   }

   tU32 wifi1NamePropID = VoidCast<tU32>(vecArgs[0]);
   tU32 wifi1NameLen    = VoidCast<tU32>(vecArgs[1]);
   tU32 wifi2NamePropID = VoidCast<tU32>(vecArgs[2]);
   tU32 wifi2NameLen    = VoidCast<tU32>(vecArgs[3]);

   // Check length of Wifi name
   if ( 
      (oDiagMsgBuffer().u16GetDataLength() <= MSG_DATA_OFFSET) 
      || 
      static_cast<tU32>(oDiagMsgBuffer().u16GetDataLength()-MSG_DATA_OFFSET) != (wifi1NameLen + wifi2NameLen)
      )
   {
      DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: WiFi name length=%d is different than (%d + %d).", (oDiagMsgBuffer().u16GetDataLength()-MSG_DATA_OFFSET), wifi1NameLen, wifi2NameLen);
      DIA_TR_ERR(" ### dia_SrvHandlerGenericWIFINameWrite: FAILED (DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT) !!");
      vSendNegativeResponse(DIA_E_U8_UDS_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT);
      return;
   }

   // extract the data from the received message
   tU8 const* pU8 = oDiagMsgBuffer().u8GetBuffer();
   std::vector<tU8> wifiName(wifi1NameLen, 0);

   tU16 u16EleCount = 0;
   //WIFI Name1
   for ( ; (u16EleCount < (oDiagMsgBuffer().u16GetDataLength()-MSG_DATA_OFFSET)) && (u16EleCount < wifi1NameLen); u16EleCount++)
   {
      if ( u16EleCount < DIA_PROP_LENGTH_MAX )
      {
         wifiName[u16EleCount] = pU8[u16EleCount+MSG_DATA_OFFSET];
         DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite::u8propData[%d] 0x%02x", u16EleCount, wifiName[u16EleCount]);
      }
   }

   tDiaResult retCode = validateData(&wifiName[0], u16EleCount);

   if ( retCode != DIA_SUCCESS )
   {
     // Sending back the Negative answer
     DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: FAILED (DATA VALIDATION) !!");
     vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
     return;
   }

   retCode = dia_setProperty ( wifi1NamePropID, &wifiName[0], u16EleCount );

   if ( retCode != DIA_SUCCESS )
   {
      // Sending back the Negative answer
      DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: FAILED (WRITE DATA TO PERSISTENT MEMORY) !!");
      vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
      return;
   }

   wifiName.clear();

   //WIFI Name2
   tU16 u16EleCountWifi2 = 0;
   wifiName.resize(wifi2NameLen);

   for ( ; ((u16EleCount < oDiagMsgBuffer().u16GetDataLength()-MSG_DATA_OFFSET) && (u16EleCountWifi2 < wifi2NameLen)); u16EleCount++, u16EleCountWifi2++)
   {
     if ( u16EleCount < DIA_PROP_LENGTH_MAX )
     {
       wifiName[u16EleCountWifi2] = pU8[u16EleCount+MSG_DATA_OFFSET];
       DIA_TR_INF("dia_SrvHandlerGenericIdentWrite::u8propData[%d] 0x%02x", u16EleCountWifi2, wifiName[u16EleCountWifi2]);
     }
   }

   retCode = validateData(&wifiName[0], (tU16)wifi2NameLen);

   if ( retCode != DIA_SUCCESS )
   {
     // Sending back the Negative answer
     DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: FAILED (DATA VALIDATION) !!");
     vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
     return;
   }

   retCode = dia_setProperty ( wifi2NamePropID, &wifiName[0],(tU16)wifi2NameLen );

   if ( retCode != DIA_SUCCESS )
   {
     // Sending back the Negative answer
     DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: FAILED (WRITE DATA TO PERSISTENT MEMORY) !!");
     vSendNegativeResponse(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
     return;
   }

   // Sending back the positive answer
   DIA_TR_INF("dia_SrvHandlerGenericWIFINameWrite: SUCCEEDED");
   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength(DATA_START);
   vResReadyAndQuit();
}
