/*!
 * \file       dia_SrvHandlerGenericRequestFileTransfer.cpp
 *
 * \brief      Service handler for service $38 (RequestFileTransfer)
 *
 * \details    Service handler for service $38 (RequestFileTransfer)
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreUploadDownload
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_GENERIC_SERVICE_HANDLER_REQUEST_FILE_TRANSFER__
#include <common/services/uds/generic/dia_SrvHandlerGenericRequestFileTransfer.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include <common/framework/application/dia_Factory.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY_METHODS_PROJECT__
#include <project/framework/factory/dia_factoryMethodsProject.h>
#endif

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_DATA_DOWNLOAD_MANAGER__
#include <common/framework/datadownload/dia_DataDownloadManager.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#define MSG_DATA_OFFSET 1

#define DIA_C_U8_REQFTRANS_BYTEPOS_SRV_ID                         ((tU8) 0)
#define DIA_C_U8_REQFTRANS_BYTEPOS_MODE_OF_OPERATION              ((tU8) 1)
#define DIA_C_U8_REQFTRANS_BYTEPOS_FILE_PATH_AND_NAME_LENGTH      ((tU8) 2)
#define DIA_C_U8_REQFTRANS_BYTEPOS_FILE_PATH_AND_NAME             ((tU8) 4)

namespace dia {

//-----------------------------------------------------------------------------

SrvHandlerGenericRequestFileTransfer::SrvHandlerGenericRequestFileTransfer ( void )
   : dia_ServiceHandlerUDS("SrvHandlerGenericRequestFileTransfer",DIA_C_U8_UDS_SID_REQUEST_FILE_TRANSFER)
{
   ScopeTrace trc("dia_SrvHandlerGenericRequestDownload::dia_SrvHandlerGenericRequestDownload()");
}

//-----------------------------------------------------------------------------

SrvHandlerGenericRequestFileTransfer::~SrvHandlerGenericRequestFileTransfer ( void )
{}

//----------------------------------------------------------------------

void
SrvHandlerGenericRequestFileTransfer::vProcessRequest ( const std::vector<tArgsType>& /*vecArgs*/ )
{
   ScopeTrace trc("SrvHandlerGenericRequestFileTransfer::vProcessRequest");

   // extract the data from the received message
   tU8* pMessageBuffer = oDiagMsgBuffer().u8GetBuffer();

   std::vector<tU8> requestData;
   requestData.reserve(oDiagMsgBuffer().u16GetDataLength());
   std::copy(&pMessageBuffer[MSG_DATA_OFFSET], &pMessageBuffer[oDiagMsgBuffer().u16GetDataLength()], std::back_inserter(requestData));

   dia_DataDownloadManager* pDnlManager = getInstanceOfDataDownloadManager();
   if ( !pDnlManager )
   {
      // Sending back the Negative answer
      DIA_TR_INF("SrvHandlerGenericRequestFileTransfer: FAILED (NO ACCESS TO DOWNLOAD MANAGER) !!");
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
      return;
   }

   DIA_TR_INF("##### PROCESSING TRANSFER FILE REQUEST: \"%s\"",dia::utils::bin2str(requestData,' ').c_str());

   tDiaResult retCode = pDnlManager->requestFileTransfer(requestData);
   if ( retCode != DIA_SUCCESS )
   {
      // Sending back the Negative answer
      DIA_TR_INF("SrvHandlerGenericRequestFileTransfer: FAILED (FILE TRANSFER REQUEST) !!");
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(retCode));
      return;
   }

   // Sending back the positive answer
   DIA_TR_INF("SrvHandlerGenericRequestFileTransfer: SUCCEEDED");
   const std::vector<tU8>* pResponseData = pDnlManager->getResponseData();
   if ( !pResponseData )
   {
      // Sending back the Negative answer
      DIA_TR_INF("SrvHandlerGenericRequestFileTransfer: NO RESPONSE DATA AVAILABLE !!");
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(DIA_E_CONDITIONS_NOT_CORRECT));
      return;
   }

   oDiagMsgBuffer().vSetPosResp();
   oDiagMsgBuffer().vSetDataLength((tU16) (pResponseData->size() + 1));
   for ( size_t j=0; j<pResponseData->size(); j++ )
   {
      (void) oDiagMsgBuffer().vSetDataU8(tU16(j+1), pResponseData->at(j));
   }

   vResReadyAndQuit();
}

}




