/*
 * dia_SrvHandlerGenericIOCtrlByIdentifier.cpp
 *
 *  Created on: 19.07.2012
 *      Author: gib2hi
 */

#ifndef __INCLUDED_DIA_DEFINES_UDS__
#include <common/framework/protocols/uds/dia_defsUds.h>
#endif

#ifndef __INCLUDED_DIA_LOOKUPKEY__
#include <common/framework/engine/dia_LookupKey.h>
#endif

#ifndef __INCLUDED_DIA_FACTORY__
#include "common/framework/application/dia_Factory.h"
#endif

#ifndef __INCLUDED_DIA_IOCTRL_MANAGER__
#include "common/framework/protocols/uds/ioctrl/dia_IOCtrlManager.h"
#endif

#ifndef __INCLUDED_DIA_IOCTRL_SIGNAL__
#include "common/framework/protocols/uds/ioctrl/dia_IOCtrlSignal.h"
#endif

// include own class
#include "dia_SrvHandlerGenericIOCtrlByIdentifier.h"

#define DIA_C_U16_IOCTRL_MAX_PROCESSING_TIME_DEFAULT        ((tU16) 10000)
#define DIA_C_U16_IOCTRL_MIN_MESSAGE_LENGTH                 ((tU16)     5)

using namespace std;

//-----------------------------------------------------------------------------

dia_SrvHandlerGenericIOCtrlByIdentifier::dia_SrvHandlerGenericIOCtrlByIdentifier ( void )
   : dia_ServiceHandlerUDS("dia_SrvHandlerGenericIOCtrlByIdentifier",DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER),
     mpIOCtrlMgr(getInstanceOfIOCtrlManager()), // IOCtrlManager is instantiated before any service handler ...
     mSignalID(0),
     mpSignal(0),
     mCtrlState(0xFF),
     mIsDone(false)
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::dia_SrvHandlerGenericIOCtrlByIdentifier()");
}

//------------------------------------------------------------------------------

dia_SrvHandlerGenericIOCtrlByIdentifier::~dia_SrvHandlerGenericIOCtrlByIdentifier ( void )
{
   mpIOCtrlMgr = 0;
   mpSignal = 0;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vInitialize ( void )
{
   mIsDone   = false;
   mSignalID = 0;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vFinalize ( void )
{
   if ( mIsDone && (mSignalID != 0) )
   {
      mpIOCtrlMgr->removeIOControlListener(mSignalID,this);
   }
}

//------------------------------------------------------------------------------

tU32
dia_SrvHandlerGenericIOCtrlByIdentifier::getTimeoutValue ( void ) const
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::getTimeoutValue()");

   tU32 maxTime = DIA_C_U16_IOCTRL_MAX_PROCESSING_TIME_DEFAULT;

   // extract the identifier from the message buffer and convert to the corresponding signal type
   tU16 subID = oDiagMsgBuffer().u16GetSubServiceId();

   dia_IOCtrlSignal* pSignal = 0;
   if ( ( mpIOCtrlMgr->queryIOControlSignalByDID(subID,&pSignal) == DIA_SUCCESS ) && pSignal )
   {
      maxTime = pSignal->getMaxProcessingTime();
   }

   DIA_TR_INF("##### IOCONTROL TIMEOUT VALUE SET TO %d ms", maxTime);

   return maxTime;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vProcessRequest(const std::vector<tArgsType>& /*vecArgs*/)
{
    dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::vProcessRequest()");
    tDiaResult retCode = DIA_FAILED;

    vInitialize();

    // check if we have received the right length
    tU16 dataLength = oDiagMsgBuffer().u16GetDataLength();

    if (dataLength < DIA_C_U16_IOCTRL_MIN_MESSAGE_LENGTH)
    {
        vSendIOCtrlNegativeResponse(DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT);
        return;
    }

    // extract UDS identifier from received message buffer and query for associated signal
    tU16 subID = oDiagMsgBuffer().u16GetSubServiceId();
    if (mpIOCtrlMgr)
    {
        dia_IOCtrlSignal* pSignal = 0;
        if ((mpIOCtrlMgr->queryIOControlSignalByDID(subID, &pSignal) != DIA_SUCCESS) || (!pSignal) || (pSignal->getUID() == 0))
        {
            DIA_TR_INF("UNKNOWN SIGNAL !!");
            vSendIOCtrlNegativeResponse(DIA_E_OUT_OF_RANGE);
            return;
        }

        mSignalID = pSignal->getUID();

        retCode = pSignal->checkPreconditions();
        if (retCode != DIA_SUCCESS)
        {
            DIA_TR_INF("UNABLE TO PROCESS SIGNAL (PRECONDITIONS NOT MATCHED) !!");
            vSendIOCtrlNegativeResponse(retCode);
            return;
        }

        // here we know that all conditions are matched to process the request, so we start processing now

        mpIOCtrlMgr->addIOControlListener(mSignalID, this);

        // extract control state 1
        mCtrlState = oDiagMsgBuffer().u8GetData(4);
        DIA_TR_INF("ControlState1: 0x%02x", (tU32)mCtrlState);

        switch (mCtrlState)
        {
            case DIA_C_U8_UDS_IOCTRL_RESET_OR_INIT:
            {
                DIA_TR_INF("Resetting/Initializing Signal \"%s\"", pSignal->getName());
                retCode = resetOrInit(*pSignal);
            }
            break;

            case DIA_C_U8_UDS_IOCTRL_SHORT_TERM_ADJUSTMENT:
            {
                DIA_TR_INF("Performing ShortTermAdjustment on Signal \"%s\"", pSignal->getName());
                retCode = shortTermAdjustment(*pSignal);
            }
            break;

            case DIA_C_U8_UDS_IOCTRL_FREEZE_CURRENT_STATE:
            {
                DIA_TR_INF("Freezing Current State of Signal \"%s\"", pSignal->getName());
                retCode = freezeCurrentState(*pSignal);
            }
            break;

            case DIA_C_U8_UDS_IOCTRL_RETURN_CTRL_TO_ECU:
            {
                DIA_TR_INF("Returning Control to ECU for Signal \"%s\"", pSignal->getName());
                retCode = returnControlToECU(*pSignal);
            }
            break;

            default:
            {
                DIA_TR_INF("UNKNOWN CONTROL STATE !!");
                retCode = DIA_E_OUT_OF_RANGE;
            }
            break;
        }
    }
    else
    {
        DIA_TR_ERR("ERROR: mpIOCtrlMgr is not initialized yet. IOControl will fail.");
    }

    if (retCode != DIA_SUCCESS)
    {
        vSendIOCtrlNegativeResponse(retCode);
    }

    vFinalize();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericIOCtrlByIdentifier::resetOrInit ( dia_IOCtrlSignal& /*signal*/ ) const
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::resetOrInit(signal)");
   return DIA_E_OUT_OF_RANGE;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericIOCtrlByIdentifier::freezeCurrentState ( dia_IOCtrlSignal& signal )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::freezeCurrentState(signal)");

   tDiaResult retCode = signal.checkFreezeLength(oDiagMsgBuffer().u16GetDataLength());

   if (  retCode == DIA_SUCCESS )
   {
      retCode = mpIOCtrlMgr->freeze(signal);
      if ( retCode == DIA_SUCCESS && signal.isResultReady() )
      {
         mpIOCtrlMgr->removeIOControlListener(signal.getUID(),this);
         vSendIOCtrlPositiveResponse(signal,mCtrlState);
      }
   }

   else
   {
      DIA_TR_ERR("Incorrect message length passed!!!");
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericIOCtrlByIdentifier::shortTermAdjustment ( dia_IOCtrlSignal& signal )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::shortTermAdjustment(signal)");

   tDiaResult retCode = DIA_SUCCESS;

   tU8  payloadOffset = 0;
   tU8  timerValue    = 0xFF;
   tU16 dataLength    = oDiagMsgBuffer().u16GetDataLength();

   if ( signal.getMonitoringMode() != DIA_EN_IOCTRL_MONITORING_MODE_NOT_SUPPORTED )
   {
      // timer mode is supported
      timerValue = oDiagMsgBuffer().u8GetData(DIA_C_U16_IOCTRL_PAYLOAD_START_BYTE);
      payloadOffset = 1;
   }

   // copy ctrlValue data from received message to vector
   std::vector<tU8> ctrlValue;
   for ( int i = DIA_C_U16_IOCTRL_PAYLOAD_START_BYTE + payloadOffset; i < dataLength; i++ )
   {
      ctrlValue.push_back(oDiagMsgBuffer().u8GetData(static_cast<tU16>(i)));
   }

   if ( signal.checkPayloadCorrectness(dataLength,&ctrlValue) != DIA_SUCCESS )
   {
      // Prints are send in checkPayloadCorrectness
      retCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
   }
   else
   {
      // check if the signal was frozen before
      if ( signal.isFrozen() )
      {
         retCode = mpIOCtrlMgr->shortTermAdjustment(signal,timerValue,&ctrlValue);
         if ( retCode == DIA_SUCCESS && signal.isResultReady() )
         {
            mpIOCtrlMgr->removeIOControlListener(signal.getUID(),this);
            vSendIOCtrlPositiveResponse(signal,mCtrlState,timerValue,&ctrlValue);
         }
      }
      else
      {
         // signal was not frozen before. send sequence error
         DIA_TR_INF("SIGNAL NOT FROZEN !!");
         retCode = DIA_E_SEQUENCE_ERROR;
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericIOCtrlByIdentifier::returnControlToECU ( dia_IOCtrlSignal& signal )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::returnControlToECU(signal)");

   tDiaResult retCode = DIA_SUCCESS;

   if ( oDiagMsgBuffer().u16GetDataLength() != DIA_C_U16_IOCTRL_MIN_MESSAGE_LENGTH )
   {
      retCode = DIA_E_INVALID_MESSAGE_LENGHT_OR_INVALID_FORMAT;
   }
   else
   {
      retCode = mpIOCtrlMgr->returnControlToECU(signal);
      if ( ( retCode == DIA_SUCCESS ) && signal.isResultReady() )
      {
         mpIOCtrlMgr->removeIOControlListener(signal.getUID(),this);
         vSendIOCtrlPositiveResponse(signal,mCtrlState);
      }
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate ( dia_IOCtrlSignal* pSignal )
{
   if ( pSignal ) getInstanceOfIOCtrlManager()->vOnSignalUpdate(*pSignal);
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vOnIOCtrlSignalUpdate ( dia_IOCtrlSignal& signal )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::vOnSignalUpdate()");

   tDiaResult retCode = signal.getErrorInfo();

   if ( signal.getUID() == mSignalID )
   {
      switch ( mCtrlState )
      {
      case DIA_C_U8_UDS_IOCTRL_FREEZE_CURRENT_STATE:
         // we have to handle freeze separately, because we might not have an active signal
         if ( signal.isResultReady() && ( retCode == DIA_E_NOERROR) )
         {
            vSendIOCtrlPositiveResponse(signal,mCtrlState);
         }
         break;

      case DIA_C_U8_UDS_IOCTRL_SHORT_TERM_ADJUSTMENT:
         // create positive response (always for short term adjustment
         if ( signal.isResultReady() && ( retCode == DIA_E_NOERROR) )
         {
            vSendIOCtrlPositiveResponse (signal,mCtrlState,signal.getTimerValue(),&(signal.getControlValue()));
         }
         break;

      case DIA_C_U8_UDS_IOCTRL_RETURN_CTRL_TO_ECU:
         if ( signal.isResultReady() && ( retCode == DIA_E_NOERROR) )
         {
            vSendIOCtrlPositiveResponse (signal,mCtrlState);
         }
         break;

      default:
         DIA_TR_INF("dia_SrvHandlerGenericIOCtrlByIdentifier::vOnIOCtrlSignalUpdate UNKNOWN CONTROL STATE !!");
         retCode = DIA_E_OUT_OF_RANGE;
         break;
      }
   }

   if ( retCode != DIA_SUCCESS )
   {
      vSendIOCtrlNegativeResponse(retCode);
   }

   vFinalize();
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vSendIOCtrlPositiveResponse (
      dia_IOCtrlSignal& signal,
      tU8 ctrlState1,
      tU8 timerValue,         // optional: only for short time adjustment
      const vector<tU8>* ctrlValue  // optional: only for short time adjustment
      )
{
   if ( !mpIOCtrlMgr ) return;

   if ( mIsDone )
   {
      DIA_TR_INF("dia_SrvHandlerGenericIOCtrlByIdentifier::vSendIOCtrlPositiveResponse: Response Was Already Sent. Done. !!!!!");
      return;
   }

   tU16 subID = signal.getDID();

   if ( subID != 0xFFFF )
   {
      // results for this request
      vector<tU8> results;
      results.push_back(static_cast<tU8>(subID >> 8));
      results.push_back(static_cast<tU8>(subID));
      results.push_back(ctrlState1);

      switch ( ctrlState1 )
      {
      case DIA_C_U8_UDS_IOCTRL_RESET_OR_INIT:
         DIA_TR_INF("dia_SrvHandlerGenericIOCtrlByIdentifier::vSendIOCtrlPositiveResponse: DIA_C_U8_UDS_IOCTRL_RESET_OR_INIT (YOU_SHOULD_NEVER_SEE_THIS_TRACE) !!!!!");
         // not implemented yet. we should never arrive at this point
         return;

      case DIA_C_U8_UDS_IOCTRL_SHORT_TERM_ADJUSTMENT:
         {
            if ( signal.getMonitoringMode() != DIA_EN_IOCTRL_MONITORING_MODE_NOT_SUPPORTED )
            {
               results.push_back(timerValue);
            }

            if ( ctrlValue )
            {
               for ( tU16 i=0; i<ctrlValue->size(); i++ )
               {
                  results.push_back(ctrlValue->at(i));
               }
            }
         }
         break;

      case DIA_C_U8_UDS_IOCTRL_FREEZE_CURRENT_STATE:
      case DIA_C_U8_UDS_IOCTRL_RETURN_CTRL_TO_ECU:
         {
            signal.getCurrentState(results);
         }
         break;

      default:
         // Ouups. we should never arrive at this point
         return;
      }

      vSendPositiveResponse(static_cast<tU16>(results.size() + 1), &results);
      mIsDone = true;
   }
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vSendIOCtrlNegativeResponse ( tDiaResult errCode )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::vSendIOCtrlNegativeResponse()");

   if ( !mIsDone )
   {
      vSendNegativeResponse(getInstanceOfFactory()->makeNRC(errCode));
      mIsDone = true;
   }
}

//------------------------------------------------------------------------------

void
dia_SrvHandlerGenericIOCtrlByIdentifier::vOnTimeout ( void )
{
   dia_tclFnctTrace oTrace("dia_SrvHandlerGenericIOCtrlByIdentifier::vOnTimeout()");

   if ( mpIOCtrlMgr )
   {
      dia_UID signalID = mpIOCtrlMgr->getActiveSignalUID();

      if ( signalID )
      {
          DIA_TR_INF("vOnTimeout: Forwarding Timeout... !!");
          mpIOCtrlMgr->handleTimeout();
      }
   }

   oDiagMsgBuffer().vSetNegResp(DIA_E_U8_UDS_CONDITIONS_NOT_CORRECT);
   vResReadyAndQuit();
}

//------------------------------------------------------------------------------

tDiaResult
dia_SrvHandlerGenericIOCtrlByIdentifier::makeLookupKeys ( std::vector<dia_LookupKey*>& keys )
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace trc("dia_SrvHandlerGenericIOCtrlByIdentifier::makeLookupKeys");
#endif

   if ( mLookupKeys.empty() )
   {
      const std::map<dia_UID,dia_IOCtrlSignal*>& signalRep = dia_IOCtrlManager::getInstance()->getSignals();
      std::map<dia_UID,dia_IOCtrlSignal*>::const_iterator cIter = signalRep.begin();
      for ( ; cIter != signalRep.end(); cIter++ )
      {
         tU16 did = cIter->second->getDID();
         mLookupKeys.push_back( OSAL_NEW dia_LookupKey( DIA_C_U8_UDS_SID_IOCONTROL_BY_IDENTIFIER, (tU16) did, DIA_C_U16_SRVDISPATCHER_KEY_LENGTH_NOT_USED) );
      }
   }

   keys = mLookupKeys;

   return DIA_SUCCESS;
}
