#ifndef __INCLUDED_DIA_UTILITIES__
#include "dia_utilities.h"
#endif

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_FILE__
#include <common/framework/application/dia_File.h>
#endif

#ifndef __INCLUDED_DIA_FILE_DIR__
#include <common/framework/application/dia_FileDir.h>
#endif

#ifndef __DIA_UNIT_TESTING__
#include "bpcl.h"
#endif

//#include <sstream>
#include <iomanip>

namespace dia_nsUtilities
{
   bool m_crcEnabled = true;

   bool toggleCrc()
   {
       m_crcEnabled = !m_crcEnabled;
       return m_crcEnabled;
   }

   void generateCrc(std::vector<tU8> const& data, tU16& crc, tU32 len /*= 0*/)
   {
       dia_tclFnctTrace trc("dia_nsUtilities::generateCrc");
       if(!m_crcEnabled)
       {
           crc = 0xf0b8;
           return;
       }
       static const tU16 auwcrc_table[] =   {
                                 0x0000, 0x1081, 0x2102, 0x3183,
                                 0x4204, 0x5285, 0x6306, 0x7387,
                                 0x8408, 0x9489, 0xa50a, 0xb58b,
                                 0xc60c, 0xd68d, 0xe70e, 0xf78f
                                 };
      crc = 0xffff;
       if(len == 0)
           len = static_cast<tU32>(data.size());
      for(tU32 i = 0; i < len; ++i)
      {
         tU8 d = data[i];
         crc = static_cast<tU16>((crc >> 4) ^ auwcrc_table[(crc ^ d) & 0x000f]);
         crc = static_cast<tU16>((crc >> 4) ^ auwcrc_table[(crc ^ (d >> 4)) & 0x000f]);
      }
      DIA_TR_INF("dia_nsUtilities::generateCrc: CRC value=0x%04x", crc);
   }

#ifndef __DIA_UNIT_TESTING__
   bool generateHash(std::vector<tU8> const& data, std::vector<tU8>& u8Hash)
   {
      dia_tclFnctTrace trc("dia_nsUtilities::generateHash");

      size_t inputDataSize = data.size();
      DIA_TR_INF("dia_nsUtilities:generateHash data.size=%zu", inputDataSize);
      if (inputDataSize>=2)
      {
         DIA_TR_INF("data begin: data[0]=0x%02X data[1]=0x%02X", data[0], data[1] );
         DIA_TR_INF("data end:   data[%zu]=0x%02X data[%zu]=0x%02X", inputDataSize-1, data[inputDataSize-1], inputDataSize-2, data[inputDataSize-2] );
      }

      // create hashing context
      tsSHA1Context shaCtx;
      tErrCode err = BPCL_SHA1_Init(&shaCtx);
      if(BPCL_OK != err)
      {
         DIA_TR_ERR("dia_nsUtilities: BPCL_SHA1_Init failed");
         return false;
      }
      // Load content for hashing
      err = BPCL_SHA1_Update(&shaCtx, const_cast<tU8*>(&data[0]), static_cast<tU32>(data.size()));
      if(BPCL_OK != err)
      {
         DIA_TR_ERR("dia_nsUtilities: BPCL_SHA1_Update failed");
         return false;
      }
      // get hash
      const int cHashSize = 5;  // 5 x U32
      std::vector<tU32> hash(cHashSize, 0);
      err = BPCL_SHA1_Finish(&shaCtx, &hash[0]);

      if(BPCL_OK != err)
      {
         DIA_TR_ERR("dia_nsUtilities: BPCL_SHA1_Finish failed");
         return false;
      }

      u8Hash.clear();
      u8Hash.reserve(cHashSize * sizeof(tU32));
      for(std::vector<tU32>::const_iterator it = hash.begin(); it != hash.end(); ++it)
      {
         tU8 byte1 = (tU8) ((*it >> 24) & 0xff);
         u8Hash.push_back(byte1);
         tU8 byte2 = (tU8) ((*it >> 16) & 0xff);
         u8Hash.push_back(byte2);
         tU8 byte3 = (tU8) ((*it >> 8) & 0xff);
         u8Hash.push_back(byte3);
         tU8 byte4 = (tU8) (*it & 0xff);
         u8Hash.push_back(byte4);
         DIA_TR_INF("dia_nsUtilities: Hash value=0x%02x%02x%02x%02x", byte1, byte2, byte3, byte4);
      }

      return true;
   }
#endif

   tU8 getBinaryFromHexString(char sHex)
   {
      dia_tclFnctTrace trc("dia_nsUtilities::getBinaryFromHexString");
      tU8 binary = 0xFF;
      if (sHex >= 0x30 && sHex <= 0x39) {
         binary = static_cast<tU8>(sHex - 0x30);
      } else if (sHex >= 0x41 && sHex <= 0x46) {
         binary = static_cast<tU8>((sHex - 0x41) + 0x0A);
      } else if (sHex >= 0x61 && sHex <= 0x66) {
         binary = static_cast<tU8>((sHex - 0x61) + 0x0A);
      } else {
         DIA_TR_ERR("dia_nsUtilities::getBinaryFromHexString %c (0x%02x) is no HEX value (0...F).", sHex, sHex);
      }
      return binary;
   }

   //! @brief     Convert 20-bytes array to specific format ASCII string
   //!            (e.g. "{00112233-00112233-00112233-00112233-00112233}")
   //!            If input vector length is different than 20, it
   //!
   //! @param[in]  vectSha1  Input vector of 20 bytes (sha1)
   //! @param[out] strHash   Specific HEX format of SHA1
   //!
   void convertToHashSpecificFormat ( std::vector<tU8>& vectSha1, std::string& strHash )
   {
      if (DIA_CONFIG_HASH_BINARY_SIZE!=vectSha1.size())
      {
         DIA_TR_ERR("### Error. ConvertToHashSpecificFormat input vector incorrect size=%zu ###", vectSha1.size());
         strHash.append("{00000000-00000000-00000000-00000000-00000000}");
         return;
      }

      strHash.append("{");

      for (tU16 i=0; i<vectSha1.size(); i++)
      {
         if((i != 0) && ((i % 4) == 0))
         {
            strHash.append("-");
         }

         char tmp[3] = {0};
         int numBytes = snprintf( tmp, sizeof(tmp), "%02X", (unsigned int)vectSha1[i]);
         if (2!=numBytes)
         {
            DIA_TR_ERR("convertToHashSpecificFormat: ERROR numBytes=%d", numBytes);
            break;
         }

         strHash.append(tmp);
      }

      strHash.append("}");
   }

   //! @brief  Convert array of 8-bit elements to HEX string
   //!         E.g. array[2]={1,2} shall be converted to string "0102".
   //!
   //! @param[in] data
   //! @return    HEX string
   std::string convertArray2HexString ( std::vector<tU8>& data )
   {
      std::string retString;

      DIA_TR_INF("convertArray2HexString: data.size()=%zu", data.size() );

      for (tU16 i=0; i<data.size(); i++)
      {
         char tmp[3] = {0};
         int numBytes = snprintf( tmp, sizeof(tmp), "%02X", (unsigned int)data[i]);
         if (2!=numBytes)
         {
            DIA_TR_ERR("convertArray2HexString: ERROR numBytes=%d", numBytes);
            break;
         }

         retString += std::string(tmp);
      }

      return retString;
   }

   //! @brief        Read data of configuration item.
   //! @return       DIA_E_INVALID_LENGTH if size of read property is not available
   //! @return       DIA_FAILED if reading was unsuccessful
   //! @return       DIA_SUCCESS otherwise
   //! @param[in]    id      Property ID to be read.
   //! @param[out]   data    Buffer for read data.
   tDiaResult readConfigItem ( tU32 id, std::vector<tU8>& data)
   {
       auto cfgDataLen = dia_getPropertySize(id);

       if (0==cfgDataLen)
       {
          DIA_TR_ERR("readConfigItem: ERROR. Size of configuration item 0x%08x is null!!", id);
          return DIA_E_INVALID_LENGTH;
       }

       std::vector<tU8> cfgData;

       if ( DIA_SUCCESS != dia_getProperty(id, cfgData) )
       {
           DIA_TR_ERR("readConfigItem: Reading configuration item 0x%08x FAILED", id);
           return DIA_FAILED;
       }

       DIA_TR_INF("readConfigItem PropID=0x%08X RealSize=%zu decimal", id, cfgData.size() );

       // reading was successful so append the data to the configuration data vector
       data.insert(data.end(), cfgData.begin(), cfgData.end()); //lint !e864 Info: expression possibly depends on order of evaluation (but standard std use case)
       return DIA_SUCCESS;
   }

   //! @brief        Read set of configuration items. If at least one of configuration item is
   //!               not available, the function will return DIA_FAILED and skip this incorrect configuration item (it continues reading for next items)
   //! @param[in]    vPropData   Vector of ID properties to be read.
   //! @param[out]   data        Output vector of read data.
   //!
   tDiaResult readConfiguration ( std::vector<tU32>& vPropData, std::vector<tU8>& data )
   {
      dia_tclFnctTrace oTrace("readConfiguration");
      tDiaResult retVal = DIA_SUCCESS;

      DIA_TR_INF("readConfiguration: vPropData.size=%zu", vPropData.size() );

      DIA_TR_INF("#######################################################################################");

      //clean output buffer
      data.clear();
      for (std::vector<tU32>::iterator it = vPropData.begin(); it!=vPropData.end(); ++it)
      {
         if (DIA_SUCCESS==readConfigItem((*it), data))
         {
            DIA_TR_INF("readConfiguration: reading successful for propData[i]=0x%08X", (*it) );
         }
         else
         {
            retVal = DIA_FAILED;
            DIA_TR_ERR("readConfiguration: reading FAILED for propData[i]=0x%08X", (*it) );
         }
      }

      DIA_TR_INF("#######################################################################################");

      return retVal;
   }

   //! @brief        Write hash long format (47 bytes) to registry.
   //!               If input buffer is not defined (pointer set to NULL), write zeros "{00000000-...-00000000}" instead of real hash.
   //!
   //! @param[in]    buffer         NULL or pointer to string of long hash format (e.g. "{AABBCCDD-...-00000000}")
   //! @param[in]    mPropertyID    Property ID of registry to be written
   //!
   tDiaResult writeHashLongFormatToRegistry ( const tU8* buffer, tU32 propertyID )
   {
      dia_tclFnctTrace oTrace("writeHashLongFormatToRegistry()");
      tU8 buffNewHash[DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE] = {0};
      const tU8 zeroPattern[DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE] = "{00000000-00000000-00000000-00000000-00000000}";

      if (NULL==buffer)
      {
         DIA_TR_INF("writeHashLongFormatToRegistry() buffer is NULL");
         ::memcpy(buffNewHash, zeroPattern, sizeof(zeroPattern));
      }
      else
      {
         size_t size = strnlen((const char*)buffer, (size_t)DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE);

         if ((DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE-1)==size)
         {
            DIA_TR_INF("writeHashLongFormatToRegistry() expected length of input string");
            ::memcpy(buffNewHash, buffer, sizeof(buffNewHash));
         }
         else
         {
            DIA_TR_INF("writeHashLongFormatToRegistry() length of input string is different (%zu)", size);
            ::memcpy(buffNewHash, zeroPattern, sizeof(zeroPattern));
         }
      }

      DIA_TR_INF("writeHashLongFormatToRegistry() buffNewHash=%s", buffNewHash);

      // write to registry
      tDiaResult result = dia_setProperty( propertyID, buffNewHash, DIA_CONFIG_HASH_STRING_LONG_FORMAT_SIZE);
      if ( DIA_SUCCESS!=result )
      {
         DIA_TR_ERR("writeHashLongFormatToRegistry() ERROR: dia_setProperty FAILED (%d)", result);
      }

      return result;
   }

}


namespace dia {
namespace utils {

   std::string bin2str ( const unsigned char* data, int len, char separator )
   {
      std::stringstream ss;

      ss << std::hex;
      for ( int i=0; i<len; ++i)
      {
         ss << std::setw(2) << std::setfill('0') << (int) data[i];
         if ( separator && ((i+1) < len) ) ss << separator;
      }

      return ss.str();
   }

   std::string bin2str ( const std::vector<tU8>& data, char separator )
   {
      std::stringstream ss;

      ss << std::hex;
      for ( tU32 i=0; i < ((tU32) data.size()); ++i )
      {
         ss << std::setw(2) << std::setfill('0') << (int) data[i];
         if ( separator && ((i+1) < data.size()) ) ss << separator;
      }

      return ss.str();
   }

   /**
    * @brief byte vector to string
    *
    * Example:
    * {0x37, 0x20, 0x41} -> "7 A"
    *
    * @param data byte vector
    * @return resulting string
    */
   std::string vector2string ( const std::vector<tU8>& data )
   {
      std::stringstream ss;

      for ( tU32 i=0; i < ((tU32) data.size()); ++i )
      {
         ss << (char) data[i];
      }

      return ss.str();
   }

   /**
    * @brief byte vector to hex string
    *
    * Example:
    * {0x12, 0xab, 0x05} -> "12ab05"
    * {0x12, 0xab, 0x05} -> "0x12ab05" (hasPrefix == true)
    *
    * @param data byte vector
    * @param hexPrefix Whether or not 0x prefix shall be used
    * @return resulting string
    */
   std::string bin2hexstr ( const std::vector<tU8>& data, bool hexPrefix )
   {
      std::stringstream ss;

      if (hexPrefix)
      {
         ss << "0x";
      }

      ss << std::hex;
      for ( tU32 i=0; i < ((tU32) data.size()); ++i )
      {
         ss << std::setw(2) << std::setfill('0') << (int) data[i];
      }

      return ss.str();
   }

   std::string byte2hexstr ( const tU8 byte, bool hexPrefix )
   {
      std::stringstream ss;
      if (hexPrefix)
      {
         ss << "0x";
      }
      ss << std::hex << std::setw(2) << std::setfill('0') << (int) byte;
      return ss.str();
   }

   tU8 hexstr2byte ( const std::string& str )
   {
       std::istringstream iss(str);
       int temp;
       iss >> std::hex >> temp;
       return static_cast<tU8>(temp);
   }

   tU16 hexstr2word(const std::string& str)
   {
      size_t size = str.size();
      if (4!=size) return 0;
      std::istringstream iss(str);
      int temp;
      iss >> std::hex >> temp;
      return static_cast<tU16>(temp);
   }

   static inline uint8_t hexNibbleToBin(char chr, tDiaResult &err) {
      if (chr >='0' && chr <= '9') {
         return (uint8_t)(chr - '0');
      }
      else if (chr >='a' && chr <= 'f') {
         return (uint8_t)(chr +10 - 'a');

      }
      else if (chr >='A' && chr <= 'F') {
         return (uint8_t)(chr + 10 - 'A');
      }
      err=DIA_FAILED;
      return 0;
   }


   tDiaResult hexstr2bin(char const *str, uint8_t *bin, size_t binLen)
   {
      tDiaResult retCode = DIA_SUCCESS;

      if (!str) {
         return DIA_FAILED;
      }
      uint32_t strLen=(uint32_t)strlen(str);
      if (strLen%2) {
         return DIA_FAILED;   //hex incorrect (not even letters)
      }
      if (strLen > binLen * 2) {
         return DIA_FAILED;   //hex incorrect (string too long)
      }
      for (uint32_t i=0; i<binLen;++i) {
         uint8_t nibbleHigh=hexNibbleToBin(*str++, retCode);
         uint8_t nibbleLow=hexNibbleToBin(*str++, retCode);
         *bin++=(uint8_t)((nibbleHigh << 4) | nibbleLow);
      }
      return retCode;
   }


   tDiaResult hexstr2bin(const std::string& str, std::vector<tU8>& data)
   {
      size_t size = str.size();
      if (size%2>0) return DIA_FAILED;   //hex incorrect (not even letters)
      if (size>0) data.clear();
      for (size_t i=0; i<size; i=i+2)
      {
         std::string s = str.substr(i, 2);   //2-chars string from index i
         std::istringstream iss(s);
         int temp;
         iss >> std::hex >> temp;
         if (temp>0xFF)
         {
            DIA_TR_ERR("### hexstr2bin failed temp=%d is greater than 0xFF.", temp);
            data.clear();
            return DIA_FAILED;
         }
         data.push_back(static_cast<tU8>(temp));
      }
      return (size>0)? DIA_SUCCESS : DIA_FAILED;
   }

   uint16_t getU16(uint8_t const *data) {
      return (uint16_t) getUxx(data, 2);
   }

   uint32_t getU32(uint8_t const *data) {
      return (uint32_t) getUxx(data, 4);
   }

   uint64_t getUxx(uint8_t const *data, uint8_t numBytes) {
      uint64_t res;
      if (!data || (numBytes>8) || numBytes==0) {
         return 0;
      }
      res=*data;
      for (uint8_t i=0; i<(numBytes-1);++i) {
         data++;
         res<<=8;
         res|=*data;
      }
      return res;      
   }

   void setUxx(uint64_t val, uint8_t numBytes, uint8_t *data) {
      if (!data || (numBytes>8) || numBytes==0) {
         return;
      }
      
      memset(data, 0, numBytes);
      for (int i=numBytes-1; i>=0;--i) {
         *data=((val >> (i*8)) & 0xFF) ;
         data++;
      }
   }

   void addUxx(uint64_t val, uint8_t numBytes, std::vector<uint8_t> &data) {
      DIA_TR_ERR("addUxx: val=%llu numBytes=%u", val, numBytes);
   
      if ((numBytes>8) || numBytes==0) {
         return;
      }
      
      for (int i=numBytes-1; i>=0;--i) {
         uint8_t curVal=((val >> (i*8)) & 0xFF);
         DIA_TR_ERR("addUxx: curVal=%02x", curVal);
         data.push_back(curVal);
      }
   }


   std::string tail(std::string const& source, size_t const length)
   {
     if (length >= source.size()) { return source; }
     return source.substr(source.size() - length);
   }

   //search a file with given key (e.g. "*.pdx") in specific directory, result is stored in fileRep
   tDiaResult scanFilesInDir(const std::string& folderName, const std::string& key, std::list<std::string>& fileRep)
   {
      ScopeTrace oTrace("dia::utils::scanFilesInDir(std::string&, std::string, std::string)");
      const static tU16 MAX_BUFFER_SIZE = 1024;

      dia_FileDir dir(folderName);
      if ( !dir.doesExist() )
      {
         DIA_TR_ERR("### DIRECTORY %s DOESN'T EXIST.", folderName.c_str());
         return DIA_E_FILE_DOES_NOT_EXIST;
      }

      std::string cmdString = "ls " + folderName + key; // command used to retrieve the list of shared libraries in the given folder
      FILE* cmdPipe = ::popen(cmdString.c_str(), "r");
      if ( !cmdPipe )
      {
         ::perror("popen");
         return DIA_FAILED;
      }

      char buffer [MAX_BUFFER_SIZE];
      char pdxFileName[MAX_BUFFER_SIZE];

      DIA_TR_INF("Scanning folder '%s' for '%s' files", folderName.c_str(), key.c_str());

      while ( ::fgets(buffer, MAX_BUFFER_SIZE, cmdPipe) )
      {
         // trim off the whitespace
         char* whitespace = ::strpbrk(buffer," \t\n");
         if ( whitespace ) *whitespace = '\0';
         // append "$folderName/" to the front of the library name
         ::sprintf(pdxFileName, "%s", buffer);

         std::string fullPdxFileName(pdxFileName);
         size_t found = fullPdxFileName.find_last_of("/\\");
         std::string strippedPdxFileName(fullPdxFileName.substr(found+1));
         DIA_TR_INF("  + Found PDX container '%s'",strippedPdxFileName.c_str());
         fileRep.push_back(fullPdxFileName);
      }

      (void) ::pclose(cmdPipe);
      return DIA_SUCCESS;
   }

} // utils
} // dia
