/**
 * \file      dia_RandomGenerator.h
 *
 * \brief     Singleton source for random integers
 *
 * \details   A singleton that provides possibility
 *            to get random integer values.
 *
 * \author    gib2hi
 * \date      Nov 4, 2013
 *
 * \copyright Robert Bosch Car Multimedia 2013-2018
 */

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#define __INCLUDED_DIA_RANDOM_GENERATOR__

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#include <cstdint>
#include <random>

class dia_RandomGenerator
{
    DECL_SINGLETON_CONSTRUCTOR_AND_DESTRUCTOR(dia_RandomGenerator);

  public:
    ~dia_RandomGenerator(void);

    uint32_t getRandomNumber(uint32_t min = 0, uint32_t max = std::numeric_limits<uint32_t>::max());

    /**
     * @brief Returns uniformly distributed pseudo random numbers in given range
     * @details Uses Pseudo random engine to generate a random numer in given range
     *          Generated numbers are uniformly distributed
     *
     * @param[in] min Minimum value of random numer (defaults to min for used type)
     * @param[in] max Maximum value of random numer (defaults to min for used type)
     *
     * @return Random number in given range with template type
     */
    template <typename T>
    T getRangedRandomNumber(T min = std::numeric_limits<T>::min(), T max = std::numeric_limits<T>::max())
    {
        std::uniform_int_distribution<T> dist(min, max);
        return dist(_randengine);
    }

    tDiaResult generateNonPseudoRandomNumber(uint32_t& randomNumber, uint32_t min = 0, uint32_t max = std::numeric_limits<uint32_t>::max());

  protected:
    dia_RandomGenerator(void);

  private:
    std::random_device _randdev;
    std::default_random_engine _randengine;
};

dia_RandomGenerator* getInstanceOfRandomGenerator(void);

#endif /* __INCLUDED_DIA_RANDOM_GENERATOR__ */
