/**
 * \file      dia_RandomGenerator.h
 *
 * \brief     Singleton source for random integers
 *
 * \details   A singleton that provides possibility
 *            to get random integer values.
 *
 * \author    gib2hi
 * \date      Nov 4, 2013
 *
 * \copyright Robert Bosch Car Multimedia 2013-2018
 */

#ifndef __INCLUDED_DIA_RANDOM_GENERATOR__
#include "common/framework/utils/dia_RandomGenerator.h"
#endif

DIA_IMPL_SINGLETON(dia_RandomGenerator)

#ifndef __DIA_UNIT_TESTING__

/**
 * @brief Returns pointer to singleton instance
 * @return Pointer to singleton instance
 */
dia_RandomGenerator* getInstanceOfRandomGenerator(void) { return dia_RandomGenerator::getInstance(); }

#endif

/**
 * @brief Constructor of singleton instance
 * @details Create instance, random device and PRNG engine. Seeds PRNG engine with real random number
 */
dia_RandomGenerator::dia_RandomGenerator(void) : _randdev(), _randengine() { _randengine.seed(_randdev()); }

/**
 * @brief Destructor
 */
dia_RandomGenerator::~dia_RandomGenerator(void) {}

/**
 * @brief Returns uniformly distributed pseudo random numbers in given range
 * @details Uses Pseudo random engine to generate a random numer in given range
 *          Generated numbers are uniformly distributed
 *
 * @param[in] min Minimum value of random numer (defaults to 0)
 * @param[in] max Maximum value of random numer (defaults to max for uint32_t)
 *
 * @return Random number in given range
 */
uint32_t dia_RandomGenerator::getRandomNumber(uint32_t min, uint32_t max)
{
    return getRangedRandomNumber<uint32_t>(min, max);
}

/**
 * @brief Returns uniformly distributed random numbers in given range
 * @details Uses real random engine to generate a random numer in given range
 *          Generated numbers are uniformly distributed
 *          USAGE OF THIS FUNCTION IS USUALLY NOT REQUIRED FOR STANDARD USE-
 *          CASES. IT SHOULD ONLY BE USED IF REAL RANDOM NUMBERS ARE REQUIRED
 *          INSTEAD OF ONES GENERATED BY PSEUDO RNG ENGINES. CALL getRandomNumber
 *          FOR ALL OTHER USE-CASES
 *
 * @param[out] randomNumber will be used to store the generated random number
 * @param[in] min Minimum value of random numer (defaults to 0)
 * @param[in] max Maximum value of random numer (defaults to max for uint32_t)
 *
 * @return DIA_SUCCESS in case real random device is available and random
 *                     number was generated
 *         DIA_FAILED in case no real random device is available
 */
tDiaResult dia_RandomGenerator::generateNonPseudoRandomNumber(uint32_t& randomNumber, uint32_t min, uint32_t max)
{
    tDiaResult result = DIA_SUCCESS;

    if (_randdev.entropy() != 0.0) // entropy returns 0.0 in case no real random device is available
    {
        std::uniform_int_distribution<uint32_t> dist(min, max);
        randomNumber = dist(_randdev);
    }
    else
    {
        result = DIA_FAILED;
    }

    return result;
}