/*!
 * \file       dia_CommChannelTCP.cpp
 *
 * \brief      Communication channel used for TCP/IP communication
 *
 * \details    Communication channel used for Ethernet communication
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreUDD
 *
 * \copyright  (c) 2015-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>

#ifndef __INCLUDED_DIA_COMMCHANNEL_TCP__
#include "dia_CommChannelTCP.h"
#endif

#ifndef __INCLUDED_DIA_APPCONTROLLER__
#include "common/framework/application/dia_AppController.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include "common/framework/engine/dia_EngineManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __INCLUDED_DIA_SESSION_CONTROLLER__
#include "common/framework/application/dia_SessionController.h"
#endif

#ifndef __INCLUDED_DIA_SESSION__
#include "common/framework/engine/dia_Session.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_COMMON_SECURITY__
#include "common/framework/security/dia_common_security.h"
#endif
#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER__
#include "common/framework//engine/dia_MessageBuffer.h"
#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include "common/framework/protocols/uds/dia_MessageBufferUDS.h"
#endif

#ifndef __INCLUDED_DIA_DIAGSESSION_UDS__
#include "common/depricated/dia_tclDiagSessionUds.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include "common/framework/config/dia_defsConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#define DIA_THREAD_NAME                         "DIA_TCPIP_THREAD"
//#define DIA_THREAD_PRIO                         35
//#define DIA_THREAD_STACKSIZE                    5000
#define DEFAULT_MSGSZ                           4096
#define AF_BOSCH_INC_AUTOSAR                    PF_INET //41 /* AF_INC */
//#define PORT_OFFSET                             0xC700
#define DIA_UDD_PORT                            13400           //(0x05 | PORT_OFFSET)
#define PAYLOAD_LEN_OFFSET                      4   //prot_version = 1 byte,inv_prot_version = 1 byte, payload type = 2bytes
#define DOIP_ACK_FRAME_LENGTH                   12
#define PAYLOAD_TYPE_DIAG_DATA                   0x8001
#define PAYLOAD_TYPE_POS_ACK                    0x8002
//#define PAYLOAD_TYPE_NACK                       0x8003
#define DOIP_MSG_HEADER_LEN                       12

//using namespace std;
//using namespace udd;

DIA_IMPL_SINGLETON(dia_CommChannelTCP)

//-----------------------------------------------------------------------------

int dia_CommChannelTCP::m_nSocketFd = 0;
int dia_CommChannelTCP::client_socketFd = 0;
unsigned char dia_CommChannelTCP::m_au8RecvBuffer[DEFAULT_MSGSZ];
unsigned char dia_CommChannelTCP::m_au8SendBuffer[DEFAULT_MSGSZ];
diag_doip_info dia_CommChannelTCP::frame_info;

//-----------------------------------------------------------------------------

dia_CommChannelTCP*
getInstanceOfCommChannelTCP ( void )
{
   return dia_CommChannelTCP::getInstance();
}

void
releaseInstanceOfCommChannelTCP ( void )
{
   dia_CommChannelTCP::deleteInstance();
}

//-----------------------------------------------------------------------------

dia_CommChannelTCP::dia_CommChannelTCP ( void ): 
   dia_ActiveObject(DIA_THREAD_NAME, DIA_PROP_TCP_THREAD_PRIO, DIA_PROP_TCP_THREAD_STACKSIZE),
   udd_CommChannel("dia_CommChannelTCP",UDD_CHANNEL_UID_UDS_TCPIP)
{
}

//-----------------------------------------------------------------------------

dia_CommChannelTCP::~dia_CommChannelTCP ( void )
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelTCP::connect ( udd_UID /*uid*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::connect");

   tDiaResult retCode = DIA_SUCCESS;
/*      char threadName[] = DIA_THREAD_NAME;
//      tString threadName = DIA_THREAD_NAME;

      OSAL_trThreadAttribute oThreadAttribute = {
           threadName, //(char*) DIA_THREAD_NAME,
           DIA_THREAD_PRIO,
           DIA_THREAD_STACKSIZE,
           (OSAL_tpfThreadEntry) ThreadFunction,
           NULL //this
        };

      m_oThread = OSAL_ThreadSpawn(&oThreadAttribute);

   if(m_oThread == OSAL_ERROR)
   {
      DIA_TR_INF( "!!! dia_CommChannelTCP::connect => ERROR: Unable to create a new OSAL thread." );
      return DIA_FAILED;
   }*/
    if ( DIA_SUCCESS != startThread())
    {
        DIA_TR_INF("!!! dia_CommChannelTCP::connect => ERROR: start thread failed.");
      retCode = DIA_FAILED;
    }

    DIA_TR_INF(("--- dia_CommChannelTCP::connect => Successfully spawn the thread."));

   dia_AppController* pAppCtrl = getInstanceOfAppController();
   if (pAppCtrl)
   {
      pAppCtrl->addRunlevelListener(this);
   }
   else
   {
      DIA_TR_INF( "!!! dia_CommChannelTCP::connect => ERROR: Unable to register for RunLevel changes" );
     retCode = DIA_FAILED;
   }

   dia_EngineServer* pEngine = 0;
   if (( getInstanceOfEngineManager()->queryEngineServer(DIA_UID_ENGINE_CUSTOMER_UDS,&pEngine) == DIA_SUCCESS ) && pEngine)
   {
      pEngine->getSessionController()->addListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelTCP::connect => ERROR: Unable to register for Session changes");
     retCode = DIA_FAILED;
   }

/*   dia_SecurityManager* pSecMgr = getInstanceOfSecurityManager();
   if ( pSecMgr )
   {
      pSecMgr->addListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelTCP::connect => ERROR: Security manager is not available");
   }*/
   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelTCP::disconnect ( udd_UID /*uid*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::disconnect");

   tDiaResult retCode = DIA_SUCCESS;

#if 0 //STC2HI: TODO
   if ( hDevice != DIA_OSAL_ERROR )
   {
      // we have a connection so we disconnect now
      dia_Engine* pEngine = 0;
      if ( dia_EngineController::getInstance()->queryEngine(DIA_EN_ENGINE_ID_CUSTOMER,&pEngine) != DIA_SUCCESS )
      {
         DIA_TR_ERR("##### UNABLE TO RETRIEVE POINTER TO DIAGNOSIS ENGINE #####");
         return DIA_FAILED;
      }

      pEngine->getSessionController()->removeListener(this);

      if ( dia_OSAL_s32IOClose(hDevice) == DIA_OSAL_ERROR )
      {
         DIA_TR_INF( "dia_CommChannelTCP::disconnect - OSAL returned an error when closing the UDS/CAN driver" );
         retCode = DIA_FAILED;
      }
      else
      {
         DIA_TR_INF( "dia_CommChannelTCP::disconnect - Successfully closed UDS/CAN driver" );
      }

      hDevice = DIA_OSAL_ERROR;
   }
#endif //STC2HI: TODO

/*   dia_SecurityManager* pSecMgr = getInstanceOfSecurityManager();
   if ( pSecMgr )
   {
      pSecMgr->removeListener(this);
   }
   else
   {
      DIA_TR_INF("!!! dia_CommChannelTCP::disconnect => ERROR: Security manager is not available");
   } */

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_CommChannelTCP::onDiagnosisResponse ( const tU8 au8MsgBuffer[], tU16 u16Length, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("--> dia_CommChannelTCP::onDiagnosisResponse");

   if ( !au8MsgBuffer )
   {
      DIA_TR_INF( "!!! dia_CommChannelTCP::onDiagnosisResponse => ERROR: au8MsgBuffer == NULL" );
      return;
   }

   DIA_TR_SM("msg TX via TCPIP: Length: %d Data: %s", u16Length, dia::utils::bin2str(au8MsgBuffer,u16Length,' ').c_str());

   tS32 iMsgLength = 0;
   //tU16 iNumberOfBytes = 0;

   frame_info.payload_len = u16Length + PAYLOAD_LEN_OFFSET;
   //copy message data to the buffer
   m_au8SendBuffer[iMsgLength++] = frame_info.prot_version;
   m_au8SendBuffer[iMsgLength++] = frame_info.inv_prot_version;
   m_au8SendBuffer[iMsgLength++] = (PAYLOAD_TYPE_DIAG_DATA & 0xFF00) >> 8;  //payload type is
   m_au8SendBuffer[iMsgLength++] = (PAYLOAD_TYPE_DIAG_DATA & 0x00FF);   //diagnositc data
   m_au8SendBuffer[iMsgLength++] = tU8((frame_info.payload_len & 0xFF000000) >> 24);
   m_au8SendBuffer[iMsgLength++] = tU8((frame_info.payload_len & 0x00FF0000) >> 16);
   m_au8SendBuffer[iMsgLength++] = tU8((frame_info.payload_len & 0x0000FF00) >> 8);
   m_au8SendBuffer[iMsgLength++] = (frame_info.payload_len & 0x000000FF);
   m_au8SendBuffer[iMsgLength++] = frame_info.ecu_logical_addr[1];
   m_au8SendBuffer[iMsgLength++] = frame_info.ecu_logical_addr[0];
   m_au8SendBuffer[iMsgLength++] = frame_info.tester_logical_addr[1];
   m_au8SendBuffer[iMsgLength++] = frame_info.tester_logical_addr[0];

   memcpy(&m_au8SendBuffer[iMsgLength], au8MsgBuffer, u16Length);
   iMsgLength += u16Length;
   // send data to client
 //  DLT_LOG(DiagLog,DLT_LOG_FATAL,DLT_STRING( "msg sent length:"),DLT_UINT16(iMsgLength),DLT_STRING ("Data: "),DLT_RAW(reinterpret_cast<tChar*>(const_cast<tU8*>(m_au8SendBuffer)),iMsgLength) );
   //dia_uds_Response_Trace(TR_CLASS_DIAGNOSTICS_SM , "msg sent length:" , iMsgLength, (const_cast<tU8*>(m_au8SendBuffer) ) );
   //iNumberOfBytes = write(client_socketFd, m_au8SendBuffer, iMsgLength);

   //check if all the bytes are sent
   if (iMsgLength != write(client_socketFd, m_au8SendBuffer, iMsgLength))
   {
      DIA_TR_INF( "!!! dia_CommChannelTCP::onDiagnosisResponse => ERROR: Not all bytes written" );

   }

#if 0 //TODO: move to the receiption of SCC_DIAGNOSIS_R_DIAG_RESPONSE (0x33)
   dia_Application::getInstance()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventConfTxOk()); //lint !e429: custodial pointer is freed by after engine has processed the message
#endif

}


void dia_CommChannelTCP::OnDiagRequest (const unsigned char* in_pTxBuffer, tU16 iLength)
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::OnDiagRequest");

   if ( NULL==in_pTxBuffer )
   {
     DIA_TR_INF("!!! dia_CommChannelTCP::OnDiagRequest => ERROR: in_pTxBuffer == NULL");
     return;
   }

   DIA_TR_SM("msg RX via TCP: Length: %d Data: %s", iLength, dia::utils::bin2str(in_pTxBuffer,iLength,' ').c_str());

   // Create an object containing the DoIP msg. It is deleted by dia_tclDiagSessionUds

   /*lint -save -e429 Lifetime is controlled by FSM of diagnostic engine (it is deleted by dia_tclDiagSessionUds)*/
   dia_MessageBufferUDS* pMsgBuff = OSAL_NEW dia_MessageBufferUDS (
         in_pTxBuffer,
         (tU16) iLength,
         dia_CommChannelTCP::onDiagnosisResponse,
         dia_MessageBuffer::holds_request,
         dia_MessageBuffer::format_raw
         );

   if ( pMsgBuff && (handleMessage(*pMsgBuff) == DIA_E_NOT_AVAILABLE) )
   {
#ifndef __DIA_UNIT_TESTING__
      dia_Application::getInstance()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuff));
      DIA_TR_INF( "--- dia_CommChannelTCP::OnDiagRequest => Posted as UDS message" );
#endif
   }
//   if ( pMsgBuff )
//   {
//      dia_Application::getInstance()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuff));
//
//      DIA_TR_INF( "--- dia_CommChannelTCP::OnDiagRequest => Posted as UDS message" );
//   }
//   else
//   {
//      DIA_TR_INF( "dia_CommChannelTCP::OnDiagRequest - OSAL_NEW failed" );
//      FATAL_M_ASSERT_ALWAYS();
//   }

   /*lint -restore */
}

//-----------------------------------------------------------------------------

void
dia_CommChannelTCP::vThreadEntrypointObject ( void )
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::vThreadEntrypointObject");

   struct sockaddr_in serv_addr, client_addr;
   std::array<unsigned char,DEFAULT_MSGSZ> rec_buffer;
   rec_buffer.fill(0);

   do
   {
      /* create socket */
      m_nSocketFd = socket(AF_BOSCH_INC_AUTOSAR, (int)SOCK_STREAM, IPPROTO_TCP);
      if (m_nSocketFd < 0)
      {
      DIA_TR_INF( "!!! dia_CommChannelTCP::vThreadEntrypointObject => ERROR: Cannot create socket" );
      break ;
      }

     DIA_TR_INF( "!!! dia_CommChannelTCP::vThreadEntrypointObject => created socket" );
      /* bind to socket */
      /* Construct the server sockaddr_in structure */
      memset(&serv_addr, 0, sizeof(serv_addr));       /* Clear struct */
      serv_addr.sin_family = AF_INET;                 /* Internet/IP */
      serv_addr.sin_addr.s_addr = INADDR_ANY;         /* server addr */
      serv_addr.sin_port = htons(DIA_UDD_PORT);       /* server port */

      /* Bind to a local port to receive response*/
     DIA_TR_INF( "!!! dia_CommChannelTCP::vThreadEntrypointObject => binding the socket" );

      if (bind(m_nSocketFd, (const struct sockaddr *) ((void*)&serv_addr) , (socklen_t)sizeof(serv_addr)) < 0)
      {
       DIA_TR_INF("!!! dia_CommChannelTCP::vThreadEntrypointObject => ERROR: Binding to port failed");
       break;
      }
      /* listen to port */
     DIA_TR_INF( "!!! dia_CommChannelTCP::vThreadEntrypointObject => listen to port" );

      if (listen(m_nSocketFd, 0) < 0)   // num of clients to be changed (gpu2kor: 10feb)
      {
       DIA_TR_INF("!!! dia_CommChannelTCP::vThreadEntrypointObject => ERROR: Listening to port failed");
       break;
      }

      socklen_t client_len = sizeof(client_addr);

      while(1)  //lint !e716 Put constant value "1" intentionally
      {
       client_socketFd = accept(m_nSocketFd, (struct sockaddr *) ((void*)&client_addr), &client_len);

       if(client_socketFd < 0)
       {
           DIA_TR_INF( "!!! dia_CommChannelTCP::vThreadEntrypointObject => ERROR: accept request failed" );
       }

       ssize_t iNumberOfBytesReceived = 0;

       while( 0 < (iNumberOfBytesReceived = recv(client_socketFd, rec_buffer.data(), DEFAULT_MSGSZ,0)))
       {
         if(iNumberOfBytesReceived <= DEFAULT_MSGSZ)
         {
            tU16 num = get_diag_data(&rec_buffer[0], iNumberOfBytesReceived, &m_au8RecvBuffer[0]);
            if (num > 0 ){
            //send ack to tester
            sendDoIPAck();

            OnDiagRequest(&m_au8RecvBuffer[0], num );
            }
         }
       }

       shutdown(client_socketFd,2);
       (void) close(client_socketFd);
      }

   }while(0); //lint !e717 do..while(0) used deliberately to jump without goto instruction
}

tU16 dia_CommChannelTCP::get_diag_data(const unsigned char* in_buffer,ssize_t numRecvBytes, unsigned char* res_buffer) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::get_diag_data");
   if ( numRecvBytes < DOIP_MSG_HEADER_LEN )
   {
      return 0;
   }
   frame_info.prot_version = (unsigned char)in_buffer[0];
   frame_info.inv_prot_version = (unsigned char)in_buffer[1];
   frame_info.payload_len = ((((unsigned int)in_buffer[4])<<24) | (((unsigned int)in_buffer[5])<<16) | (((unsigned int)in_buffer[6])<<8) | (unsigned int)in_buffer[7]);
   frame_info.tester_logical_addr[1] = in_buffer[8];
   frame_info.tester_logical_addr[0] = in_buffer[9];
   frame_info.ecu_logical_addr[1] = in_buffer[10];
   frame_info.ecu_logical_addr[0] = in_buffer[11];

   tU16 data_len = (tU16)(frame_info.payload_len - PAYLOAD_LEN_OFFSET);
   if ( numRecvBytes != DOIP_MSG_HEADER_LEN+frame_info.payload_len )
   {
      return 0;
   }

   for (tU16 i = 0; i<data_len; i++)
   {
      if (
         ((data_len - i - 1) <= DEFAULT_MSGSZ) 
         && 
         ((DOIP_MSG_HEADER_LEN + data_len - i - 1) <= DEFAULT_MSGSZ)
         )
      {
         res_buffer[(data_len -i) -1] =  in_buffer[((DOIP_MSG_HEADER_LEN + data_len) -i) -1];
      }
   }

   return data_len;
}

tVoid dia_CommChannelTCP::sendDoIPAck(tVoid) const
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::sendDoIPAck");

   unsigned char m_au8SendAckBuffer[DEFAULT_MSGSZ];  //buffer to send Ack to tester
   //int iNumberOfBytes = 0;
   frame_info.payload_len = PAYLOAD_LEN_OFFSET;
   //TODO: check the case for sending negative ack to tester, presently send only pos ack.
   m_au8SendAckBuffer[0] = frame_info.prot_version;
   m_au8SendAckBuffer[1] = frame_info.inv_prot_version;
   m_au8SendAckBuffer[2] = (PAYLOAD_TYPE_POS_ACK & 0xFF00) >> 8;
   m_au8SendAckBuffer[3] = (PAYLOAD_TYPE_POS_ACK & 0x00FF);
   m_au8SendAckBuffer[4] = static_cast<tU8>((frame_info.payload_len & 0xFF000000) >> 24);
   m_au8SendAckBuffer[5] = static_cast<tU8>((frame_info.payload_len & 0x00FF0000) >> 16);
   m_au8SendAckBuffer[6] = static_cast<tU8>((frame_info.payload_len & 0x0000FF00) >> 8);
   m_au8SendAckBuffer[7] = (frame_info.payload_len & 0x000000FF);
   m_au8SendAckBuffer[8] = frame_info.ecu_logical_addr[1];
   m_au8SendAckBuffer[9] = frame_info.ecu_logical_addr[0];
   m_au8SendAckBuffer[10] = frame_info.tester_logical_addr[1];
   m_au8SendAckBuffer[11] = frame_info.tester_logical_addr[0];

   //send the ack to tester
   ssize_t iNumberOfBytes = write(client_socketFd, m_au8SendAckBuffer, DOIP_ACK_FRAME_LENGTH);

   DIA_TR_INF("sendDoIPAck -- %zd Bytes send to tester: ", iNumberOfBytes);

   //check if all the bytes are sent
   if (iNumberOfBytes != DOIP_ACK_FRAME_LENGTH)
   {
      DIA_TR_INF("!!! dia_CommChannelTCP::sendDoIPAck => ERROR: Not all bytes written");
      DIA_TR_INF("sendDoIPAck -- not all bytes sent to tester");
   }
}


//------------------------------------------------------------------------------

tVoid
dia_CommChannelTCP::vOnSessionChanged ( tU8 newSession, tU8 /*oldSession*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::vOnSessionChanged");

   DIA_TR_INF("--- dia_CommChannelTCP::vOnSessionChanged => newSession = %d", newSession);

}

//-----------------------------------------------------------------------------

tVoid
dia_CommChannelTCP::vOnRunLevelChanged ( dia_enRunlevel newLevel, dia_enRunlevel /*oldLevel*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::vOnRunLevelChanged");

   DIA_TR_INF("--- dia_CommChannelTCP::vOnRunLevelChanged => newLevel = %d", newLevel);

}

//-----------------------------------------------------------------------------

void
dia_CommChannelTCP::vOnSecurityLevelChanged ( dia_SecurityLevel* /*pNewLevel*/, dia_SecurityLevel* /*pOldLevel*/ )
{
   dia_tclFnctTrace oTrace("dia_CommChannelTCP::vOnSecurityLevelChanged");
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommChannelTCP::OnTesterDisconnected ( void ) const
{
   static tU8 tempBuffer[] = { 3, DIA_C_U8_UDS_SID_SESSION_CONTROL, 0x00 /* project specific default session ID will be available later */ };

   dia_tclFnctTrace oTrace("dia_CommChannelTCP::OnTesterDisconnected");
   tU8 defSession = 0;

   tU32 retCode = dia_getProperty(DIA_PROP_DEFAULT_SESSION, defSession);
   if (DIA_SUCCESS == retCode)
   {
      tempBuffer[2] = defSession;

      DIA_TR_INF("dia_CommChannelTCP::OnTesterDisconnected: default session ID is 0x%02X", defSession);
   }
   else
   {
      DIA_TR_INF("!!!  dia_CommChannelTCP::OnTesterDisconnected => ERROR: dia_getProperty FAILED retCode=0x%08X", retCode);
      return DIA_FAILED;
   }

   /*lint -save -e429 Lifetime is controlled by FSM of diagnostic engine (it is deleted by dia_tclDiagSessionUds)*/
   dia_MessageBufferUDS* pMsgBuff = OSAL_NEW dia_MessageBufferUDS (
      &tempBuffer[0],
      tU16(sizeof tempBuffer),
      dia_tclDiagSessionUds::vNullResponse
   );

#ifndef __DIA_UNIT_TESTING__
   if ( pMsgBuff )
   {
      // Create and send a ReqRx event to the uds session queue
      dia_Application::getInstance()->postMessage(OSAL_NEW dia_tclDiagSession::tclEventReqRx(pMsgBuff));
   }
   else
   {
      DIA_TR_INF("dia_CommChannelTCP::OnTesterDisconnected ERROR. NO MEMORY for dia_MessageBufferUDS object.");
      FATAL_M_ASSERT_ALWAYS();
   }
#endif

   return DIA_SUCCESS;
   /*lint -restore */
}

//tU16
//dia_CommChannelTCP::getSecLevelMask (const dia_SecurityLevel* pSecLevel) const
//{
//   dia_tclFnctTrace oTrace("dia_CommChannelTCP::getSecLevelMask");
//}


