/*!
 * \file       dia_trace.cpp
 *
 * \brief      Diagnostic trace macros, functions, ...
 *
 * \details    Diagnostic trace macros, functions, ...
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreTrace
 *
 * \copyright  (c) 2012-2018 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include <common/framework/application/dia_ThreadMonitor.h>
#endif

// we have undefined the etrace macro definition for "_" in dia_trace.h, but as we use some etrace within this file we
// need to redefine here locally
#ifndef _
#define _ ,
#endif

//---------------------------------------------------------------------------------------------------------------------

#define DIA_M_WRITE_TRACE(traceClass,traceLevel)                                                                     \
                                                                                                                     \
   vDoTrace(traceClass, (tU16) traceLevel, (const tS8*) pBuffer, ::strlen(pBuffer)+1)

//---------------------------------------------------------------------------------------------------------------------

#define DIA_M_WRITE_TRACE_TO_ERROR_MEMORY(traceClass,traceLevel)                                                     \
                                                                                                                     \
   vDoTraceErrmem(traceClass, traceLevel, static_cast<tU16>(::strlen(pBuffer)+1), (const tU8*) pBuffer)

//---------------------------------------------------------------------------------------------------------------------

namespace dia
{
   namespace trace
   {
#ifdef __DIA_EXTENDED_TRACE_OUTPUT_DISABLED__
      bool gEnableExtendedTraceOutput = false;
#else
      bool gEnableExtendedTraceOutput = true;
#endif
   }

   void createTraceHeader ( tChar* pBuffer, tU16 /*maxLength*/, int& index );
}


#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
#include <unistd.h>        //lint !e537  kaa1hi: repeatedly included header file without standard include guard
#include <pthread.h>       //lint !e537  kaa1hi: repeatedly included header file without standard include guard
pthread_mutex_t gRecMutex = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;  //lint !e708  kaa1hi: this macro has been used to avoid call of pthread_mutexattr_init
#else
dia::Lock gTraceMutex("DIA_MUTEX_TRACE");
#endif

#ifdef __cplusplus
extern "C"
{
#endif

extern OSAL_tIODescriptor et_gfdTrace;

//---------------------------------------------------------------------------------------------------------------------

static void
vLockTrace ( void )
{
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_ASSERT(0==pthread_mutex_lock (&gRecMutex));
#else
   (void) gTraceMutex.lock();
#endif
}

//---------------------------------------------------------------------------------------------------------------------

static void
vUnlockTrace ( void )
{
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_ASSERT(0==pthread_mutex_unlock (&gRecMutex));
#else
   (void) gTraceMutex.unlock();
#endif
}

//---------------------------------------------------------------------------------------------------------------------

static void
vDoTrace ( tU16 compId, tU16 levelID, const tS8* pBuffer, size_t length )
{
   if ( (et_gfdTrace != 0) && (et_gfdTrace != OSAL_ERROR) )
   {
      OSAL_trIOWriteTrace trace;

      trace.u32Class    = (tU32) compId;
      trace.u32Level    = (tU32) levelID;
      trace.pcos8Buffer = pBuffer;
      trace.u32Length   = (tU32) length;

      (void) OSAL_s32IOWrite(et_gfdTrace, (const tS8*)&trace, sizeof(trace));
   }
}

//---------------------------------------------------------------------------------------------------------------------

static void
vDoTraceErrmem ( tU16 entry, tInt entryType, tU16 entryLength, const tU8* pBuffer )
{
   OSAL_tIODescriptor fd_errmem = OSAL_IOOpen(OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_READWRITE);

   if ( fd_errmem != OSAL_ERROR )
   {
      trErrmemEntry  rErrmemEntry;

      rErrmemEntry.u16Entry       = entry;
      rErrmemEntry.eEntryType     = (tenErrmemEntryType) entryType;
      rErrmemEntry.u16EntryLength = entryLength;

      (void) OSAL_s32ClockGetTime(&rErrmemEntry.rEntryTime);
      if ( rErrmemEntry.u16EntryLength > ERRMEM_MAX_ENTRY_LENGTH )
      {
         // ensure no memory writer
         rErrmemEntry.u16EntryLength = ERRMEM_MAX_ENTRY_LENGTH;
      }
      (void) ::memcpy((void*) rErrmemEntry.au8EntryData, (void*) pBuffer, rErrmemEntry.u16EntryLength);
      (void) OSAL_s32IOWrite(fd_errmem, (tPCS8)&rErrmemEntry, sizeof(rErrmemEntry));
      (void) OSAL_s32IOClose(fd_errmem);
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceInfo ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_STRING,TR_LEVEL_USER_1);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceWarning ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_STRING,TR_LEVEL_COMPONENT);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceError ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_STRING,TR_LEVEL_ERRORS);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceErrorMemory ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE_TO_ERROR_MEMORY(OSAL_C_TR_CLASS_DEV_ERRMEM,eErrmemEntryFatal);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceSmall ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_SM,TR_LEVEL_USER_1);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceBase ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_BASE,TR_LEVEL_USER_1);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceCustomer ( const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_CUST,TR_LEVEL_COMPONENT);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_TraceGeneric ( tU16 classID, tU16 level, const char* format, ...)
{
   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(classID,level);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

#ifdef __DIA_EXTENDED_DEBUG_TRACES__

//---------------------------------------------------------------------------------------------------------------------

void
dia_DebugTraceInfo ( const char* format, ...)
{
   if ( !dia::trace::gEnableExtendedTraceOutput ) return;

   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_STRING,TR_LEVEL_USER_1);
   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_DebugTraceError ( const char* format, ...)
{
   if ( !dia::trace::gEnableExtendedTraceOutput ) return;

   vLockTrace();
   DIA_M_FILL_TRACE_BUFFER(format);
   DIA_M_WRITE_TRACE(DIA_TRACE_CLASS_DIAGNOSTICS_STRING,TR_LEVEL_ERRORS);
   vUnlockTrace();
}

#else

void
dia_DebugTraceInfo ( const char* /*format*/, ...)
{}

void
dia_DebugTraceError ( const char* /*format*/, ...)
{}

#endif

//---------------------------------------------------------------------------------------------------------------------

bool
dia_isTraceActive ( tU16 compID, tU16 levelID )
{
   vLockTrace();

   bool isActive = false;

   if ( (et_gfdTrace != 0) && (et_gfdTrace != OSAL_ERROR) )
   {
      OSAL_trIOCtrlActivTrace rActive;

      rActive.enTraceClass = (TR_tenTraceClass) compID;
      rActive.enTraceLevel = (TR_tenTraceLevel) levelID;
      rActive.bIsActive    = FALSE;

      if ( OSAL_OK == OSAL_s32IOControl(et_gfdTrace,OSAL_C_S32_IOCTRL_ISACTIVE, reinterpret_cast<intptr_t>(&rActive)) )
      {
         isActive = (rActive.bIsActive == TRUE) ? true : false;
      }
   }


   vUnlockTrace();

   return isActive;
}

#ifdef __cplusplus
}
#endif

namespace dia {

//---------------------------------------------------------------------------------------------------------------------

ScopeTrace::ScopeTrace ( tCString data )
   : mTraceData(data),
     mClassID(DIA_TRACE_CLASS_DIAGNOSTICS_BASE),
     mIsTracingActivated(false)
{
   vLockTrace();

   if ( dia_isTraceActive(mClassID, (tU16)TR_LEVEL_USER_4) )
   {
      std::string trcData = std::string(">>>> ") + mTraceData;
      DIA_M_FILL_TRACE_BUFFER_WITH_DATA("%s",trcData.c_str());
      DIA_M_WRITE_TRACE(mClassID,TR_LEVEL_USER_4);
   }

   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

ScopeTrace::ScopeTrace ( const std::string& data )
   : mTraceData(data.c_str()),
     mClassID(DIA_TRACE_CLASS_DIAGNOSTICS_BASE),
     mIsTracingActivated(false)

{
   vLockTrace();

   if ( dia_isTraceActive(mClassID, (tU16)TR_LEVEL_USER_4) )
   {
      std::string trcData = std::string(">>>> ") + mTraceData;
      DIA_M_FILL_TRACE_BUFFER_WITH_DATA("%s",trcData.c_str());
      DIA_M_WRITE_TRACE(mClassID,TR_LEVEL_USER_4);
   }

   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

ScopeTrace::ScopeTrace ( tU16 classID, tCString data )
   : mTraceData(data),
     mClassID(classID),
     mIsTracingActivated(false)

{
   vLockTrace();

   if ( dia_isTraceActive(mClassID, (tU16)TR_LEVEL_USER_4) )
   {
      std::string trcData = std::string(">>>> ") + mTraceData;
      DIA_M_FILL_TRACE_BUFFER_WITH_DATA("%s",trcData.c_str());
      DIA_M_WRITE_TRACE(mClassID,TR_LEVEL_USER_4);
   }

   vUnlockTrace();
}

//---------------------------------------------------------------------------------------------------------------------

ScopeTrace::~ScopeTrace()
{
   _BP_TRY_BEGIN
   {
      vLockTrace();

      if ( dia_isTraceActive(mClassID, (tU16)TR_LEVEL_USER_4) )
      {
         std::string trcData = std::string("<<<< ") + mTraceData;
         DIA_M_FILL_TRACE_BUFFER_WITH_DATA("%s",trcData.c_str());
         DIA_M_WRITE_TRACE(mClassID,TR_LEVEL_USER_4);
      }

      mTraceData = 0;

      vUnlockTrace();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: ScopeTrace::~ScopeTrace !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//---------------------------------------------------------------------------------------------------------------------

dia_ScopeTraceVarg::dia_ScopeTraceVarg(const char* format, ...):
   scopeTrace(0) {
   if ( dia_isTraceActive(DIA_TRACE_CLASS_DIAGNOSTICS_BASE, (tU16)TR_LEVEL_USER_4) ) {
      tChar pBuffer[DIA_C_TRACE_MAX];
      int index = 0;
      ::va_list argList;
      ::va_start(argList, format);
         index += ::vsnprintf(&pBuffer[index], DIA_C_TRACE_MAX-index, format, argList);
         ::va_end(argList);
         pBuffer[DIA_C_TRACE_MAX-1] = 0;
         traceString=pBuffer;
         scopeTrace=new  ScopeTrace(traceString);
   }
}

//---------------------------------------------------------------------------------------------------------------------

dia_ScopeTraceVarg::~dia_ScopeTraceVarg() {
   delete scopeTrace;
   scopeTrace=0;
}


//---------------------------------------------------------------------------------------------------------------------

void
createTraceHeader ( tChar* pBuffer, tU16 /*maxLength*/, int& index )
{
   int threadID = dia::getCurrentThreadID();
   if ( threadID >= 0 )
   {
      index = ::sprintf(pBuffer,"[%d]: ",threadID);
   }
   else
   {
      pthread_t threadID = dia::getCurrentSystemThreadID();
      index = ::sprintf(pBuffer,"[?,0x%0lx]: ", threadID);
   }
}

}
