/*!
 * \file       dia_SystemSettingsPlugin.h
 *
 * \brief      Abstract base class for all kinds of diagnosis internal system setting plugins.
 *
 * \details    Class dia_SystemSetPluginPM is the abstract base class for all kinds of diagnosis
 *             internal system setting plugins. In contrast to system setting modules the
 *             methods called on system setting plugins are processed in a synchronous way, so
 *             no interaction with other applications is involved, except of drivers or platform
 *             components like the authorization level daemon. In each system setting level
 *             (prepare, execute and finalize) system setting plugins are always processed before
 *             any system setting module is processed.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_PLUGIN__
#define __INCLUDED_DIA_SYSTEM_SETTINGS_PLUGIN__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSSET__
#include <common/framework/sysset/dia_common_sysset.h>
#endif

/*!
 * \enum dia_eSystemSetPluginID
 *
 * \brief Enumeration used to provide a unique identifier for system setting plugins
 *
 * This enumeration is used to provide a unique identifier for system setting plugins.
 * Do not extend this enumeration, as this enumeration is deprecated. because we now use a string/hash code based policy
 *
 * \deprecated This enumeration is obsolete and shall not be extended
 *
 * \todo       Remove this enum once all projects are migrated to the hash code policy}
 */

enum dia_eSystemSetPluginID
{
   DIA_EN_SYSSET_PLUGIN_ID_UNKNOWN = 0,
   DIA_EN_SYSSET_PLUGIN_ID_DEFAULT_CONFIG_CUSTOMER,
   DIA_EN_SYSSET_PLUGIN_ID_DEFAULT_CALIB_CUSTOMER,
   DIA_EN_SYSSET_PLUGIN_ID_LOCK_DEVICE,
   DIA_EN_SYSSET_PLUGIN_ID_PM_DATAPOOL,
   DIA_EN_SYSSET_PLUGIN_ID_COUNT
};

/*!
 * \class      dia_SystemSetPluginPM.h
 *
 * \brief      Abstract base class for all kinds of diagnosis internal system setting plugins.
 *
 * \details    Class dia_SystemSetPluginPM is the abstract base class for all kinds of diagnosis
 *             internal system setting plugins. In contrast to system setting modules the
 *             methods called on system setting plugins are processed in a synchronous way, so
 *             no interaction with other applications is involved, except of drivers or platform
 *             components like the authorization level daemon. In each system setting level
 *             (prepare, execute and finalize) system setting plugins are always processed before
 *             any system setting module is processed.
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 */

class dia_SystemSettingsPlugin
{
public:
   //! default class constructor
   dia_SystemSettingsPlugin ( tCString name );
   //! class destructor
   virtual ~dia_SystemSettingsPlugin ( void );

   //! return the plugin's name
   virtual tCString getName ( void ) const;
   //! return the plugin's UID which the hash code retrieved from the name
   dia_UID getUID ( void ) const;

   //! prepare for the upcoming system setting
   virtual tDiaResult prepare ( dia_UID sysSetTypeUID );
   //! perform the system set operation
   virtual tDiaResult execute ( dia_UID sysSetTypeUID );
   //! finalize the system set operation
   virtual tDiaResult finalize ( dia_UID sysSetTypeUID );
   //! perform the system set operation
   virtual tDiaResult runToCompletion ( dia_UID sysSetTypeUID );

   //! prepare the module for the system setting
   virtual tDiaResult prepare ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& extData );
   //! execute the system setting
   virtual tDiaResult execute ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& extData  );
   //! finalize the system setting
   virtual tDiaResult finalize ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& extData  );
   //! perform the system set operation
   virtual tDiaResult runToCompletion ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& extData  );

   //! add a supported system setting type to the plugin by type UID
   virtual tDiaResult addSupportedSysSetType ( dia_UID sysSetTypeUID );
   //! add a supported system setting type to the plugin by type name
   virtual tDiaResult addSupportedSysSetType ( tCString sysSetTypeName );

   //! check whether the plugin supports specified sysset type
   virtual bool isSystemSettingTypeSupported ( dia_UID typeUID );

protected:
    //! default class constructor without implementation
   dia_SystemSettingsPlugin ( void );
public:
protected:
   //##########################################################################
   //#
   //# DEPRECATED METHODS (DO NOT USE IN NEW SOURCE CODE !!)
   //#
   //##########################################################################

   //! default class constructor
   dia_SystemSettingsPlugin ( tCString name, dia_eSystemSetPluginID id );

protected:
   //! name of the plugin
   tCString mName;
   //! unique system setting plugin identifier
   dia_UID mUID;
   //! type info lookup table
   std::map<dia_UID,tCString> mTypeLookupRep;
};

#endif /* __INCLUDED_DIA_SYSTEM_SETTINGS_PLUGIN__ */
