/**
 * \file       dia_SystemSettingsPlugin.cpp
 *
 * \brief      Abstract base class for all kinds of diagnosis internal modules need to be triggered by a system setting operation
 *
 * \details    Abstract base class for all kinds of diagnosis internal modules need to be triggered by a system setting operation
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#include "dia_SystemSettingsPlugin.h"

#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_MANAGER__
#include <common/framework/sysset/dia_SystemSettingsManager.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_CONFIGURATION__
#include <common/framework/sysset/dia_SystemSettingsConfiguration.h>
#endif

//#################################################################################
//#
//# The following mapping table is used for backward compatibility only. Do not edit
//#
//#################################################################################

static tCString arSystemSettingsPluginNameMapping[DIA_EN_SYSSET_PLUGIN_ID_COUNT] = {
   "DIA_EN_SYSSET_PLUGIN_ID_UNKNOWN",
   "DIA_EN_SYSSET_PLUGIN_ID_DEFAULT_CONFIG_CUSTOMER",
   "DIA_EN_SYSSET_PLUGIN_ID_DEFAULT_CALIB_CUSTOMER",
   "DIA_EN_SYSSET_PLUGIN_ID_LOCK_DEVICE",
   "DIA_EN_SYSSET_PLUGIN_ID_PM_DATAPOOL"
};


//-----------------------------------------------------------------------------

dia_SystemSettingsPlugin::dia_SystemSettingsPlugin ( tCString name )
   : mName(name)
{
   mUID = dia_getHashCodeFromString(mName);
}

//-----------------------------------------------------------------------------

dia_SystemSettingsPlugin::dia_SystemSettingsPlugin ( tCString name, dia_eSystemSetPluginID id )
   : mName(name)
{
   mUID = dia_getHashCodeFromString(arSystemSettingsPluginNameMapping[id]);
}

//-----------------------------------------------------------------------------

dia_SystemSettingsPlugin::~dia_SystemSettingsPlugin ( void )
{
   mName = 0;
}

//-----------------------------------------------------------------------------

tCString
dia_SystemSettingsPlugin::getName ( void ) const
{
   return mName;
}

//-----------------------------------------------------------------------------

dia_UID
dia_SystemSettingsPlugin::getUID ( void ) const
{
   return mUID;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::prepare ( dia_UID sysSetTypeUID )
{
   dia_SystemSettingExtendedData extData;
   return prepare(sysSetTypeUID,extData);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::execute ( dia_UID sysSetTypeUID )
{
   dia_SystemSettingExtendedData extData;
   return execute(sysSetTypeUID,extData);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::finalize ( dia_UID sysSetTypeUID )
{
   dia_SystemSettingExtendedData extData;
   return finalize(sysSetTypeUID,extData);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::prepare ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& /*extData*/ )
{
   DIA_TR_INF("dia_SystemSettingsPlugin::prepare(): name=\"%s\", UID=0x%08x", mName, sysSetTypeUID);
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::execute ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& /*extData*/ )
{
   DIA_TR_INF("dia_SystemSettingsPlugin::execute(): name=\"%s\", UID=0x%08x", mName, sysSetTypeUID);
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::finalize ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& /*extData*/ )
{
   DIA_TR_INF("dia_SystemSettingsPlugin::finalize(): name=\"%s\", UID=0x%08x", mName, sysSetTypeUID);
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::runToCompletion ( dia_UID sysSetTypeUID )
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsPlugin::runToCompletion(dia_UID)");
   dia_SystemSettingExtendedData extData;
   return runToCompletion(sysSetTypeUID,extData);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::runToCompletion ( dia_UID sysSetTypeUID, const dia_SystemSettingExtendedData& extData )
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsPlugin::runToCompletion(dia_UID,const dia_SystemSettingExtendedData&)");

   if ( prepare(sysSetTypeUID,extData) != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SystemSettingsPlugin(\"%s\"): prepare() failed !!", getName());
      return DIA_E_SYSSET_PLUGIN_PREPARATION_FAILED;
   }

   if ( execute(sysSetTypeUID,extData) != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SystemSetPlugin(\"%s\"): execute() failed !!", getName());
      return DIA_E_SYSSET_PLUGIN_EXECUTION_FAILED;
   }

   if ( finalize(sysSetTypeUID,extData) != DIA_SUCCESS )
   {
      DIA_TR_ERR("dia_SystemSetPlugin(\"%s\"): finalize() failed !!", getName());
      return DIA_E_SYSSET_PLUGIN_FINALIZATION_FAILED;
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::addSupportedSysSetType ( dia_UID sysSetTypeUID )
{
   dia_tclFnctTrace trc("dia_SystemSettingsManager::addSysSetPluginSupportType(dia_UID)");

   tDiaResult retCode = DIA_FAILED;

   std::map<dia_UID,tCString>::iterator iter = mTypeLookupRep.find(sysSetTypeUID);
   if (iter == mTypeLookupRep.end())
   {
      dia_SystemSettingsConfiguration* pSysSetConfig = getInstanceOfSystemSettingsManager()->getConfiguration();
      mTypeLookupRep[sysSetTypeUID] = pSysSetConfig->getSystemSettingTypeName(sysSetTypeUID);
      retCode = DIA_SUCCESS;
   }
   else
   {
      retCode = DIA_E_KEY_ALREADY_EXISTS;
   }

   return retCode;

}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsPlugin::addSupportedSysSetType ( tCString sysSetTypeName )
{
   dia_tclFnctTrace trc("dia_SystemSettingsManager::addSysSetPluginSupportType(tCString)");

   tDiaResult retCode = DIA_FAILED;
   dia_UID sysSetTypeUID = dia_getHashCodeFromString(sysSetTypeName);

   std::map<dia_UID,tCString>::iterator iter = mTypeLookupRep.find(sysSetTypeUID);
   if (iter == mTypeLookupRep.end())
   {
      mTypeLookupRep[sysSetTypeUID] = sysSetTypeName;
      retCode = DIA_SUCCESS;
   }
   else
   {
      retCode = DIA_E_KEY_ALREADY_EXISTS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsPlugin::isSystemSettingTypeSupported ( dia_UID typeUID  )
{

   std::map<dia_UID,tCString>::iterator iter = mTypeLookupRep.find(typeUID);
   if ( iter == mTypeLookupRep.end() )
   {
      DIA_TR_INF("dia_SystemSettingsPlugin:System setting type (UID = 0x%08x) is NOT supported!",typeUID);
      return false;
   }

   DIA_TR_INF("dia_SystemSettingsPlugin:System setting type \"%s\" (UID = 0x%08x) is supported!",iter->second,typeUID);
   return true;
}
