/**
 * \file       dia_SystemSettingsConfiguration.h
 *
 * \brief      Used to configure the behaviour of dia_SystemSettingsManager
 *
 * \details    Used to configure the behaviour of dia_SystemSettingsManager
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_CONFIGURATION__
#define __INCLUDED_DIA_SYSTEM_SETTINGS_CONFIGURATION__

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_FUNCTOR__
#include <common/framework/application/dia_Functor.h>
#endif

#ifndef __INCLUDED_DIA_COMMON_SYSSET__
#include <common/framework/sysset/dia_common_sysset.h>
#endif

struct dia_SystemSettingTypeInfo
{
   //! name of the system setting type as string
   tCString mName;
   //! hash code of the name
   dia_UID  mUID;
   //! flag to indicate if we have a specific (applies to a subset of components) or general type (applies to all components)
   bool     mIsSpecific;
};

struct dia_SystemSettingsConfigurationASF
{
   std::string  mPortName;
   std::string  mSAServicePluginName;
   std::string  mSAFeatureName;
   std::string  mSystemSettingID;
   tU32         mSystemSettingTypeMask;
   dia_Functor* pFunctor;
};

//-----------------------------------------------------------------------------

class dia_SystemSettingsConfiguration
{
public:
   //! class constructor
   dia_SystemSettingsConfiguration ( tCString name );
   //! copy constructor
   dia_SystemSettingsConfiguration ( const dia_SystemSettingsConfiguration& obj );
   //! assignment operator
   dia_SystemSettingsConfiguration& operator = ( const dia_SystemSettingsConfiguration& obj );

   //! virtual class destructor
   virtual ~dia_SystemSettingsConfiguration ( void );

   //! retrieve the configuration's name
   virtual tCString getName ( void ) const { return mName.c_str(); }

   //! load a set of properties
   virtual tDiaResult loadTimeoutValues ( dia_enSystemSettingLevel level, const tU32 timeoutData[], tU16 numOfTimeouts );
   //! load the set of supported system setting types
   virtual tU32 getMaxRetries ( dia_enSystemSettingLevel level ) const;
   virtual tU32 getTimeoutValue ( dia_enSystemSettingLevel level, tU16 retryCounter ) const;

   virtual tDiaResult loadSystemSettingTypeInformation ( const dia_SystemSettingTypeInfo typeInfoData[], tU16 numOfTypeInfos );
   //! load the set of supported system setting types
   virtual tDiaResult resetSystemSettingTypeInformation ( void );
   //! check if a system setting with the given type name is supported
   virtual bool isSystemSettingTypeSupported ( tCString typeName ) const;
   //! check if a system setting with the given type name UID is supported
   virtual bool isSystemSettingTypeSupported ( dia_UID uid ) const;
   //!
   virtual tCString getSystemSettingTypeName ( dia_UID uid ) const;

   //! prototype pattern used to create clone of this object on the heap
   virtual dia_SystemSettingsConfiguration* clone ( void );

   //! load a set of properties
   tDiaResult loadConfigurationASF ( const dia_SystemSettingsConfigurationASF cfgData[], tU16 count );
   //! get the return load a set of properties
   const std::list<dia_SystemSettingsConfigurationASF*>& getConfigurationASF ( void ) const { return mConfigASF; }

protected:
   //! deprecated default class constructor without implementation
   dia_SystemSettingsConfiguration ( void );

protected:
   //! name of the configuration object
   std::string mName;
   //! repository of retry values
   std::map<dia_enSystemSettingLevel,std::vector<tU32> > mTimeoutValueRep;
   //! repository of retry values
   std::map<dia_enSystemSettingLevel,tU32> mMaxRetryRep;
   //! repository of system setting types
   std::map<dia_UID,dia_SystemSettingTypeInfo> mTypeInfoRep;
   //! repository of system setting types
   std::list<dia_SystemSettingsConfigurationASF*> mConfigASF;
};

#endif /* __INCLUDED_DIA_SYSTEM_SETTINGS_CONFIGURATION__ */
