/**
 * \file       dia_SystemSettingsConfiguration.cpp
 *
 * \brief      Used to configure the behaviour of dia_SystemSettingsManager
 *
 * \details    Used to configure the behaviour of dia_SystemSettingsManager
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSystemSettings
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

#ifndef __INCLUDED_DIA_SYSTEM_SETTINGS_CONFIGURATION__
#include <common/framework/sysset/dia_SystemSettingsConfiguration.h>
#endif

#ifndef __INCLUDED_DIA_HASH_CALCULATOR__
#include <common/framework/utils/dia_HashCalculator.h>
#endif

// Enter timeouts for each retry. There must be Number-of-retries + 1 entries in the array.
static const tU32 arDefaultPrepTimeout[] = {  7500, 2500 };
static const tU32 arDefaultExecTimeout[] = { 45000, 4800 };

static const tCString strUnknownType = "DIA_SYSSET_TYPE_UNKNOWN";

//
// Default system setting types (no UID specified, will be generated
//

static const dia_SystemSettingTypeInfo
arDefaultSystemSettingTypes[] =
{
//    name                                          UID                 isSpecific
      { "DIA_SYSSET_TYPE_UNKNOWN",                  0,                  true  },
      { "DIA_SYSSET_TYPE_PRODUCTION",               0,                  false },
      { "DIA_SYSSET_TYPE_CUSTOMER",                 0,                  false },
      { "DIA_SYSSET_TYPE_ALIGNMENT",                0,                  false },
      { "DIA_SYSSET_TYPE_CALIBRATION",              0,                  false },
      { "DIA_SYSSET_TYPE_USER_PROFILE",             0,                  false },
      { "DIA_SYSSET_TYPE_LANGUAGE",                 0,                  false },
      { "DIA_SYSSET_TYPE_UNITS",                    0,                  false },
      { "DIA_SYSSET_TYPE_HMI",                      0,                  false },
      { "DIA_SYSSET_TYPE_HMI_NAVIGATION",           0,                  true  },
      { "DIA_SYSSET_TYPE_HMI_TUNER",                0,                  true  },
      { "DIA_SYSSET_TYPE_HMI_SYSTEM",               0,                  true  },
      { "DIA_SYSSET_TYPE_HMI_AUDIO",                0,                  true  },
      { "DIA_SYSSET_TYPE_HMI_MEDIA",                0,                  true  },
      { "DIA_SYSSET_TYPE_DIAGNOSTICS_INTERNAL",     0,                  true  }
};

//-----------------------------------------------------------------------------

dia_SystemSettingsConfiguration::dia_SystemSettingsConfiguration ( tCString name )
   : mName(name)
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::dia_SystemSettingsConfiguration(tCString)");

   // load the default preparation timeouts
   std::vector<tU32> prepTimeoutVec;
   tU16 numOfPrepTimeouts = static_cast<tU16>(sizeof(arDefaultPrepTimeout)/sizeof(tU32));
   prepTimeoutVec.insert(prepTimeoutVec.end(), &arDefaultPrepTimeout[0], &arDefaultPrepTimeout[numOfPrepTimeouts]);
   mTimeoutValueRep[DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE] = prepTimeoutVec;
   mMaxRetryRep[DIA_EN_SYSTEM_SETTING_LEVEL_PREPARE] = ( numOfPrepTimeouts > 1 ) ? (numOfPrepTimeouts - 1) : 0;

   // load the default execution timeouts
   std::vector<tU32> execTimeoutVec;
   tU16 numOfExecTimeouts = static_cast<tU16>(sizeof(arDefaultExecTimeout)/sizeof(tU32));
   execTimeoutVec.insert(execTimeoutVec.end(), &arDefaultExecTimeout[0], &arDefaultExecTimeout[numOfExecTimeouts]);
   mTimeoutValueRep[DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE] = execTimeoutVec;
   mMaxRetryRep[DIA_EN_SYSTEM_SETTING_LEVEL_EXECUTE] = ( numOfExecTimeouts > 1 ) ? (numOfExecTimeouts - 1) : 0;

   // as default no retries for the finalization
   mMaxRetryRep[DIA_EN_SYSTEM_SETTING_LEVEL_FINALIZE] = 0;

   // load the default system setting types
   tU16 numOfTypes = static_cast<tU16>(sizeof(arDefaultSystemSettingTypes)/sizeof(dia_SystemSettingTypeInfo));
   for ( tU16 i=0; i<numOfTypes; i++ )
   {
      dia_UID uid = dia_getHashCodeFromString(arDefaultSystemSettingTypes[i].mName);
      mTypeInfoRep[uid] = arDefaultSystemSettingTypes[i];
      mTypeInfoRep[uid].mUID = uid;
   }

   mConfigASF.clear();
}

//-----------------------------------------------------------------------------

dia_SystemSettingsConfiguration::dia_SystemSettingsConfiguration ( const dia_SystemSettingsConfiguration& obj )
   : mName(obj.mName)
{
   mTimeoutValueRep = obj.mTimeoutValueRep;
   mConfigASF       = obj.mConfigASF;

   std::map<dia_enSystemSettingLevel,tU32>::const_iterator retryIter = obj.mMaxRetryRep.begin();
   for ( ; retryIter != obj.mMaxRetryRep.end(); retryIter++ )
   {
      mMaxRetryRep[retryIter->first] = retryIter->second;
   }

   std::map<dia_UID,dia_SystemSettingTypeInfo>::const_iterator typeIter = obj.mTypeInfoRep.begin();
   for ( ; typeIter != obj.mTypeInfoRep.end(); typeIter++ )
   {
      mTypeInfoRep[typeIter->first] = typeIter->second;
   }
}

//-----------------------------------------------------------------------------

dia_SystemSettingsConfiguration&
dia_SystemSettingsConfiguration::operator = ( const dia_SystemSettingsConfiguration& obj )
{
   if ( this != &obj )
   {
      mName = obj.mName;
      mTimeoutValueRep = obj.mTimeoutValueRep;
      mConfigASF       = obj.mConfigASF;

      std::map<dia_enSystemSettingLevel,tU32>::const_iterator retryIter = obj.mMaxRetryRep.begin();
      for ( ; retryIter != obj.mMaxRetryRep.end(); retryIter++ )
      {
         mMaxRetryRep[retryIter->first] = retryIter->second;
      }

      std::map<dia_UID,dia_SystemSettingTypeInfo>::const_iterator typeIter = obj.mTypeInfoRep.begin();
      for ( ; typeIter != obj.mTypeInfoRep.end(); typeIter++ )
      {
         mTypeInfoRep[typeIter->first] = typeIter->second;
      }
   }
   return *this;
}

//-----------------------------------------------------------------------------

dia_SystemSettingsConfiguration::~dia_SystemSettingsConfiguration ( void )
{
   _BP_TRY_BEGIN
   {
      DIA_IMPL_LIST_REPOSITORY_TEAR_DOWN(dia_SystemSettingsConfigurationASF,mConfigASF);
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SystemSettingsConfiguration::~dia_SystemSettingsConfiguration !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

dia_SystemSettingsConfiguration*
dia_SystemSettingsConfiguration::clone ( void )
{
   return OSAL_NEW dia_SystemSettingsConfiguration(*this);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsConfiguration::loadTimeoutValues ( dia_enSystemSettingLevel level, const tU32 timeoutData[], tU16 numOfTimeouts )
{
   std::map<dia_enSystemSettingLevel,std::vector<tU32> >::iterator iter = mTimeoutValueRep.find(level);
   if ( iter != mTimeoutValueRep.end() )
   {
      iter->second.clear();
   }

   std::vector<tU32> timeoutVec;
   timeoutVec.insert(timeoutVec.end(), timeoutData, timeoutData + numOfTimeouts);
   mTimeoutValueRep[level] = timeoutVec;
   mMaxRetryRep[level] = ( numOfTimeouts > 1 ) ? (numOfTimeouts - 1) : 0;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tU32
dia_SystemSettingsConfiguration::getTimeoutValue ( dia_enSystemSettingLevel level, tU16 retryCounter ) const
{
   std::map<dia_enSystemSettingLevel,std::vector<tU32> >::const_iterator iter = mTimeoutValueRep.find(level);
   if ( ( iter != mTimeoutValueRep.end() ) && ( retryCounter < iter->second.size() ) )
   {
      return iter->second.at(retryCounter);
   }

   // either we have no timeouts configured for the given level or the value of retryCounter is out of range
   return 0;
}

//-----------------------------------------------------------------------------

tU32
dia_SystemSettingsConfiguration::getMaxRetries ( dia_enSystemSettingLevel level ) const
{
   std::map<dia_enSystemSettingLevel,tU32>::const_iterator iter = mMaxRetryRep.find(level);
   if ( iter != mMaxRetryRep.end() )
   {
      return iter->second;
   }

   // seems that we have not configured retries for the given level
   return 0;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsConfiguration::resetSystemSettingTypeInformation ( void )
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::resetSystemSettingTypeInformation(...)");
   mTypeInfoRep.clear();
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SystemSettingsConfiguration::loadSystemSettingTypeInformation ( const dia_SystemSettingTypeInfo typeInfoData[], tU16 numOfTypeInfos )
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::loadSystemSettingTypeInformation(...)");

   tDiaResult retCode = DIA_FAILED;

   if ( typeInfoData && ( numOfTypeInfos > 0 ) )
   {
      for ( tU16 i=0; i<numOfTypeInfos; ++i )
      {
         dia_UID key = typeInfoData[i].mUID;
         if ( !key )
         {
            key = dia_getHashCodeFromString(typeInfoData[i].mName);
         }

         mTypeInfoRep[key] = typeInfoData[i];
         mTypeInfoRep[key].mUID = key;

         DIA_TR_INF("Added System Setting Type \"%s\" (UID=0x%08x)", mTypeInfoRep[key].mName, mTypeInfoRep[key].mUID);
      }
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsConfiguration::isSystemSettingTypeSupported ( tCString typeName ) const
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::isSystemSettingTypeSupported(tCString)");

   bool retCode = true;
   dia_UID uid = dia_getHashCodeFromString(typeName);
   std::map<dia_UID,dia_SystemSettingTypeInfo>::const_iterator iter = mTypeInfoRep.find(uid);
   if ( iter == mTypeInfoRep.end() )
   {
      DIA_TR_INF("isSystemSettingTypeSupported(\"%s\",0x%08x) -- FALSE (NOT FOUND)",typeName,uid);
      retCode = false;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_SystemSettingsConfiguration::isSystemSettingTypeSupported ( dia_UID uid ) const
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::isSystemSettingTypeSupported(dia_UID)");

   bool retCode = true;
   std::map<dia_UID,dia_SystemSettingTypeInfo>::const_iterator iter = mTypeInfoRep.find(uid);
   if ( iter == mTypeInfoRep.end() )
   {
      DIA_TR_INF("isSystemSettingTypeSupported(0x%08x) -- FALSE (NOT FOUND)",uid);
      retCode = false;
   }

   return retCode;
}

//-----------------------------------------------------------------------------

tCString
dia_SystemSettingsConfiguration::getSystemSettingTypeName ( dia_UID uid ) const
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::getSystemSettingTypeName(dia_UID)");

   std::map<dia_UID,dia_SystemSettingTypeInfo>::const_iterator iter = mTypeInfoRep.find(uid);
   return ( iter == mTypeInfoRep.end() ) ? strUnknownType : iter->second.mName;
}

//-----------------------------------------------------------------------------

//! load a set of properties
tDiaResult
dia_SystemSettingsConfiguration::loadConfigurationASF ( const dia_SystemSettingsConfigurationASF cfgData[], tU16 numOfItems )
{
   dia_tclFnctTrace oTrace("dia_SystemSettingsConfiguration::loadConfigurationASF");

   tDiaResult retCode = DIA_FAILED;

   if ( cfgData )
   {
      for ( tU16 i=0; i<numOfItems; ++i )
      {
         dia_SystemSettingsConfigurationASF* pConfigItem = OSAL_NEW dia_SystemSettingsConfigurationASF;
         if ( pConfigItem )
         {
            pConfigItem->mPortName              = cfgData[i].mPortName;
            pConfigItem->mSAServicePluginName   = cfgData[i].mSAServicePluginName;
            pConfigItem->mSAFeatureName         = cfgData[i].mSAFeatureName;
            pConfigItem->mSystemSettingID       = cfgData[i].mSystemSettingID;
            pConfigItem->mSystemSettingTypeMask = cfgData[i].mSystemSettingTypeMask;
            pConfigItem->pFunctor               = cfgData[i].pFunctor;
            mConfigASF.push_back(pConfigItem);
         }
      } //lint !e429: custodial pointer is added to container mConfigASF that is freed in destructor
      retCode = DIA_SUCCESS;
   }

   return retCode;
}


