/*!
 * \file       dia_SAFeatureSystemAdapterFactory.cpp
 *
 * \brief      Proxy to create other ASF system adapters
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreSysAdapters
 *
 * \copyright  (c) 2019 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN_ASF__
#include <common/framework/platform/asf/dia_SystemAdapterServicePluginASF.h>
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_SAFEATURE_SYSTEM_ADAPTER_FACTORY__
#include "dia_SAFeatureSystemAdapterFactory.h"
#endif


//-----------------------------------------------------------------------------

dia_SAFeatureSystemAdapterFactory::dia_SAFeatureSystemAdapterFactory ( dia_SystemAdapterServicePluginASF<SystemAdapterFactoryServiceProxy>& pSrvPlugin )
   : dia_SystemAdapterFeatureASF(pSrvPlugin)
{
   dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::dia_SAFeatureSystemAdapterFactory()");

}

//-----------------------------------------------------------------------------

dia_SAFeatureSystemAdapterFactory::~dia_SAFeatureSystemAdapterFactory()
{
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSystemAdapterFactory::registerModule ( const std::string& moduleName, const std::vector<dia::IID>& iidList )
{
   dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::registerModule");

   class FunctorSendRequest : public dia_Functor
   {
   public:
      //! constructor
      FunctorSendRequest ( dia_SAFeatureSystemAdapterFactory* pObj, const std::string& moduleName, const std::vector<dia::IID>& iidList )
         : _pObj(pObj), _moduleName(moduleName), _iidList(iidList) {}
      //! class destructor
      virtual ~FunctorSendRequest () {}
      //! override operator "()" in order to call the member function with stored arguments
      tDiaResult operator() ( void ) final
      {
         dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::registerModule::FunctorSendRequest()");

         tDiaResult retCode = DIA_E_INVALID_POINTER;
         if (_pObj && _pObj->mpSrvPlugin)
         {
            SystemAdapterFactoryServiceProxy* pProxy = _pObj->mpSrvPlugin->getProxy();
            if (pProxy)
            {
               act_t token = pProxy->sendRegisterModuleRequest(*_pObj, _moduleName, _iidList);
               if (0!=token)
               {
                  retCode = DIA_SUCCESS;
               }
            }
         }

         return retCode;
      }

   private:
      //! deprecated default constructor without implementation
      FunctorSendRequest ( void ) = delete;
      //! pointer to object
      dia_SAFeatureSystemAdapterFactory* _pObj;
      //! Dynamic Loader Module name
      std::string _moduleName;
      //! List of Interface Identifiers
      std::vector<dia::IID> _iidList;
   };

   return mpSrvPlugin->executeServiceFunctor( std::unique_ptr<dia_Functor>(new FunctorSendRequest(this, moduleName, iidList)) );
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureSystemAdapterFactory::createSystemAdapter ( dia::IID iid )
{
   dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::createSystemAdapter");

   class FunctorSendRequest : public dia_Functor
   {
   public:
      //! constructor
      FunctorSendRequest ( dia_SAFeatureSystemAdapterFactory* pObj, dia::IID iid ) : _pObj(pObj), _iid(iid) {}
      //! class destructor
      virtual ~FunctorSendRequest () {}
      //! override operator "()" in order to call the member function with stored arguments
      tDiaResult operator() ( void ) final
      {
         dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::createSystemAdapter::FunctorSendRequest()");

         tDiaResult retCode = DIA_E_INVALID_POINTER;
         if (_pObj && _pObj->mpSrvPlugin)
         {
            SystemAdapterFactoryServiceProxy* pProxy = _pObj->mpSrvPlugin->getProxy();
            if (pProxy)
            {
               act_t token = pProxy->sendCreateSystemAdapterRequest(*_pObj, _iid);
               if (0!=token)
               {
                  retCode = DIA_SUCCESS;
               }
            }
         }

         return retCode;
      }

   private:
      //! deprecated default constructor without implementation
      FunctorSendRequest ( void ) = delete;
      //! pointer to object
      dia_SAFeatureSystemAdapterFactory* _pObj;
      //! Interface Identifier
      dia::IID _iid;
   };

   return mpSrvPlugin->executeServiceFunctor( std::unique_ptr<dia_Functor>(new FunctorSendRequest(this, iid)) );
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureSystemAdapterFactory::onRegisterModuleResponse(const ::boost::shared_ptr< SystemAdapterFactoryServiceProxy >&, const ::boost::shared_ptr< RegisterModuleResponse >& response)
{
   dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::onRegisterModuleResponse");

   dia_ISystemAdapterFactoryListener* pListener = 0;
   if ((queryListener<dia_ISystemAdapterFactoryListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->onRegisterModule(response->getModuleName(), response->getResult());
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureSystemAdapterFactory::onCreateSystemAdapterResponse(const ::boost::shared_ptr< SystemAdapterFactoryServiceProxy >&, const ::boost::shared_ptr< CreateSystemAdapterResponse >& response)
{
   dia_tclFnctTrace trc("dia_SAFeatureSystemAdapterFactory::onCreateSystemAdapterResponse");

   dia_ISystemAdapterFactoryListener* pListener = 0;
   if ((queryListener<dia_ISystemAdapterFactoryListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->onCreateSystemAdapter(response->getIid(), response->getResult());
   }
}

//-----------------------------------------------------------------------------
