/*!
 * \file       dia_SAFeatureAudioManager.cpp
 *
 * \brief      System adapter to AudioManager application
 *
 * \details    System adapter to AudioManager application
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSysAdapters
 *
 * \copyright  (c) 2012-2016 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

/* PATHS for ASF communication
 * XML-Definition: x.vws\ai_audio\components\AudioManagerPlugins\AMCommandPlugin\interface\CommandInterfaceIntrospectionData.xml
 * Generated Code: x.vws_GEN\ai_projects\generated\components\asf\asf0100\diag_AudioManagerCommandInterface\dbus\src-gen\org\genivi\audiomanager\CommandInterfaceClientBase.h
 *                 x.vws_GEN\ai_projects\generated\components\asf\asf0100\diag_AudioManagerCommandInterface\dbus\src-gen\org\genivi\audiomanager\CommandInterfaceProxy.h
 */

#if 1
//Redefinition of new and delete when both OSAL and std::stl is used
#define __PLACEMENT_NEW_INLINE
#endif

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_SERVICE_PLUGIN_ASF__
#include <common/framework/platform/asf/dia_SystemAdapterServicePluginASF.h>
#endif

#ifndef __INCLUDED_DIA_AUDIO_MANAGER__
#include <common/framework/platform/cmd/dia_AudioManager.h>
#endif

#ifndef __INCLUDED_DIA_UTILITIES__
#include <common/framework/utils/dia_utilities.h>
#endif

#include "dia_SAFeatureAudioManager.h"

#define DIA_C_U16_AUDIO_SINK_ID_DEFAULT                     ((uint16) 1)
#define DIA_C_S16_AUDIO_CONNECT_RESPONSE_OK                 ((int16)  0) // "E_OK" response
#define DIA_C_S16_AUDIO_CONNECT_RESPONSE_ALREADY_EXISTS     ((int16)  5) // "E_ALREADY_EXISTS" response

using namespace dia;

//------------------------------------------------------------------------------

dia_SAFeatureAudioManager::dia_SAFeatureAudioManager ( dia_SystemAdapterServicePluginASF<CommandInterfaceProxy>& pSrvPlugin )
   : dia_SystemAdapterFeatureASF(pSrvPlugin),
     mNumberOfMainConnectionsChangeReg(0),
     mMainConnectionStateChangeReg(0)
{}

//-----------------------------------------------------------------------------

dia_SAFeatureAudioManager::~dia_SAFeatureAudioManager()
{}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::startMonitoring()
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::startMonitoring");

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      mNumberOfMainConnectionsChangeReg = mpSrvPlugin->getProxy()->sendNumberOfMainConnectionsChangedRegister(*this);
      mMainConnectionStateChangeReg     = mpSrvPlugin->getProxy()->sendMainConnectionStateChangedRegister(*this);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::stopMonitoring()
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::stopMonitoring");

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      mpSrvPlugin->getProxy()->sendNumberOfMainConnectionsChangedDeregister(mNumberOfMainConnectionsChangeReg);
      mpSrvPlugin->getProxy()->sendMainConnectionStateChangedDeregister(mMainConnectionStateChangeReg);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

#ifdef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

tDiaResult
dia_SAFeatureAudioManager::setAudioSource ( dia_eAudioSource sourceID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::setAudioSource(dia_eAudioSource)");

   DIA_TR_INF("::setAudioSource to sourceID=0x%02x , sinkID=DIA_C_U16_AUDIO_SINK_ID_DEFAULT .", sourceID);

   // store information about the requested connection
   mRequestedConnection.reset();
   dia_tAudioSourceType* pSource = querySourceByEnum(sourceID);
   if ( pSource )
   {
      mRequestedConnection.mSource = *pSource;
   }
   else
   {
      DIA_TR_ERR("### dia_SAFeatureAudioManager::setAudioSource pSource is NULL. ERROR.");
      //return DIA_FAILED;
   }
   mRequestedConnection.mSink = dia_tAudioSinkType(DIA_EN_AUDIO_SINK_AMPLIFIER_MAINFIELD,DIA_C_U16_AUDIO_SINK_ID_DEFAULT,0);

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      act_t sendRet = mpSrvPlugin->getProxy()->sendConnectRequest(*this, sourceID, DIA_C_U16_AUDIO_SINK_ID_DEFAULT);
      if (0==sendRet)
      {
         DIA_TR_ERR("### dia_SAFeatureAudioManager::setAudioSource sendRet is NULL. ERROR.");
         return DIA_FAILED;
      }
      DIA_TR_INF("dia_SAFeatureAudioManager::setAudioSource sendConnectRequest sendRet=0x%08X success.", sendRet);
   }

   return DIA_SUCCESS;
}

#endif

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::setAudioSource ( tU8 sourceID, tU8 sinkID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::setAudioSource(tU8,tU8)");

   DIA_TR_INF("dia_SAFeatureAudioManager::setAudioSource to sourceID=0x%02x , sinkID=0x%02x .", sourceID, sinkID);

   mRequestedConnection.reset();
   dia_tAudioSourceType* pSource = querySourceByID(sourceID);
   if ( !pSource ) return DIA_FAILED;
   mRequestedConnection.mSource = *pSource;
   dia_tAudioSinkType* pSink = querySinkByID(sinkID);
   if ( !pSink ) return DIA_FAILED;
   mRequestedConnection.mSink = *pSink;

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      act_t sendRet = mpSrvPlugin->getProxy()->sendConnectRequest(*this, sourceID, sinkID);
      DIA_TR_INF("dia_SAFeatureAudioManager::setAudioSource sendConnectRequest with return=%lu .", sendRet);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::getAudioSource ( void )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::getAudioSource");

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      act_t sendRet = mpSrvPlugin->getProxy()->sendGetListMainConnectionsRequest(*this);
      DIA_TR_INF("dia_SAFeatureAudioManager::getAudioSource sendGetListMainConnectionsRequest return=%lu .", sendRet);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::getAudioConnections ( void )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::getAudioConnections");

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      act_t sendRet = mpSrvPlugin->getProxy()->sendGetListMainConnectionsRequest(*this);
      DIA_TR_INF("dia_SAFeatureAudioManager::getAudioConnections sendGetListMainConnectionsRequest return=%lx .", sendRet);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::getAudioSourceList ( void )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::getAudioSourceList");

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      act_t sendRet = mpSrvPlugin->getProxy()->sendGetListMainSourcesRequest(*this);
      DIA_TR_INF("dia_SAFeatureAudioManager::getAudioSourceList sendGetListMainSourcesRequest return=%lu .", sendRet);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SAFeatureAudioManager::getAudioSinkList ( void )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::getAudioSinkList");

   if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
   {
      act_t sendRet = mpSrvPlugin->getProxy()->sendGetListMainSinksRequest(*this);
      DIA_TR_INF("dia_SAFeatureAudioManager::getAudioSinkList sendGetListMainSinksRequest return=%lu .", sendRet);
   }
   return DIA_SUCCESS;
}

//=============================================================================
//
// ASF callback interface functions
//
//=============================================================================

void
dia_SAFeatureAudioManager::onNumberOfMainConnectionsChangedSignal (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< NumberOfMainConnectionsChangedSignal >& /*signal*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onNumberOfMainConnectionsChangedSignal");
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onNumberOfMainConnectionsChangedError (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< NumberOfMainConnectionsChangedError >& /*error*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onNumberOfMainConnectionsChangedError");
}

//-----------------------------------------------------------------------------

#ifdef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

void
dia_SAFeatureAudioManager::onGetListMainConnectionsResponse (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainConnectionsResponse >& response
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse");

   static dia_eAudioConnectionState connectionStateMapping[DIA_EN_AUDIO_CONNECTION_STATE_COUNT] = {
         DIA_EN_AUDIO_CONNECTION_STATE_UNKNOWN,
         DIA_EN_AUDIO_CONNECTION_STATE_CONNECTING,
         DIA_EN_AUDIO_CONNECTION_STATE_CONNECTED,
         DIA_EN_AUDIO_CONNECTION_STATE_DISCONNECTING,
         DIA_EN_AUDIO_CONNECTION_STATE_DISCONNECTED,
         DIA_EN_AUDIO_CONNECTION_STATE_SUSPENDED,
   };

   if ( !response ) return;

   bool connectionFound = false;

   std::vector< GetListMainConnectionsResponseListConnectionsStruct > ConnList = response->getListConnections();
   std::vector< GetListMainConnectionsResponseListConnectionsStruct >::iterator ConnIter;

   std::vector<dia_tAudioConnectionType> audioConnectionList;

   DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse => result=%d, ConnList.Count=%d", ETG_CENUM(dia_AudioManagerResult, response->getResult()), ConnList.size());

   // First check if a CONNECTED connection is available
   for ( ConnIter = ConnList.begin(); (ConnIter != ConnList.end()) && (connectionFound == false); ++ConnIter )
   {
      int connectionID    = ConnIter->getElem1();
      int sourceID        = ConnIter->getElem2();
      int sinkID          = ConnIter->getElem3();
//    int delay           = ConnIter->getElem4();
      int connectionState = ConnIter->getElem5();

      // create audio source element
      dia_tAudioSourceType source;
      source.mSourceID = (tU8) sourceID;
      dia_tAudioSourceType* pSource = querySourceByID(sourceID);
      if ( pSource ) { source = *pSource; } // copy info like the ID enum

      // create audio sink element
      dia_tAudioSinkType sink;
      sink.mSinkID = (tU8) sinkID;
      dia_tAudioSinkType* pSink = querySinkByID(sinkID);
      if ( pSink ) { sink = *pSink; } // copy info like the ID enum

      // determine state of connection
      dia_eAudioConnectionState state = ((connectionState >= 0) && (connectionState < DIA_EN_AUDIO_CONNECTION_STATE_COUNT))
            ? connectionStateMapping[connectionState]
            : DIA_EN_AUDIO_CONNECTION_STATE_UNKNOWN;

      // create new connection element and add it to the repository of connections
      dia_tAudioConnectionType connection((tU32) connectionID,source,sink,state);
      audioConnectionList.push_back(connection);
   }

   // force thread context switch to context of diagnosis worker thread
   getInstanceOfApplication()->postMessage (
         DIA_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
               DIA_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureAudioManager,dia_tAudioConnectionType,std::vector > (this,&dia_SAFeatureAudioManager::onGetListMainConnectionsResponse,audioConnectionList)
         )
   );
}

#endif

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onGetListMainConnectionsResponse ( const std::vector<dia_tAudioConnectionType>& connectionInfoRep )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse");

   mConnectionInfoRep.clear();
   mConnectionInfoRep = connectionInfoRep;

   dia_IAudioSourceListener* pListener = 0;
   if ( (querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener )
   {
      pListener->vOnAudioConnectionList(mConnectionInfoRep);
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onGetListMainConnectionsError (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainConnectionsError >& /*error*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainConnectionsError");
}

//-----------------------------------------------------------------------------

#ifdef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

void
dia_SAFeatureAudioManager::onMainConnectionStateChangedSignal (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< MainConnectionStateChangedSignal >& signal
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onMainConnectionStateChangedSignal");

   if ( !signal ) return;
   if ( signal->getConnectionState() > DIA_EN_AUDIO_CONNECTION_STATE_SUSPENDED ) return;

   static dia_eAudioConnectionState connectionStateMapping[DIA_EN_AUDIO_CONNECTION_STATE_COUNT] = {
         DIA_EN_AUDIO_CONNECTION_STATE_UNKNOWN,
         DIA_EN_AUDIO_CONNECTION_STATE_CONNECTING,
         DIA_EN_AUDIO_CONNECTION_STATE_CONNECTED,
         DIA_EN_AUDIO_CONNECTION_STATE_DISCONNECTING,
         DIA_EN_AUDIO_CONNECTION_STATE_DISCONNECTED,
         DIA_EN_AUDIO_CONNECTION_STATE_SUSPENDED,
   };

   DIA_TR_INF("dia_SAFeatureAudioManager::onMainConnectionStateChangedSignal => ID = %d, State = %d", signal->getConnectionID(), ETG_CENUM(dia_AudioManagerConnectionState, signal->getConnectionState()));

   tU32 connectionID = signal->getConnectionID();

   dia_tAudioConnectionType  updatedConnection;
   dia_tAudioConnectionType* pConnection = queryConnectionByID(connectionID);
   if ( pConnection )
   {
      pConnection->mState = connectionStateMapping[signal->getConnectionState()];
      updatedConnection = *pConnection;

      if ( mpSrvPlugin && mpSrvPlugin->getProxy() )
      {
         getInstanceOfApplication()->postMessage (
               DIA_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                     DIA_NEW dia_FunctorOneRefArgNoReturnValue<dia_SAFeatureAudioManager,dia_tAudioConnectionType>(this,&dia_SAFeatureAudioManager::onMainConnectionStateChanged,updatedConnection)
               )
         );
      }
   }
   else
   {
      DIA_TR_INF("RECEIVED CONNECTION STATE CHANGE FOR UNKNOWN CONNECTION (ID = %u)", connectionID);
   }
}

#endif

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onMainConnectionStateChanged ( const dia_tAudioConnectionType& connectionInfo )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onMainConnectionStateChanged");

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->vOnAudioConnectionStateChanged(connectionInfo);
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onMainConnectionStateChangedError (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< MainConnectionStateChangedError >& /*error*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onMainConnectionStateChangedError");
}

//---------------------------------------------------------------------------------------------------------------------

#ifdef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

void dia_SAFeatureAudioManager::onGetListMainSourcesResponse (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainSourcesResponse >& response
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSourcesResponse");

   if ( !response ) return;

   std::vector< GetListMainSourcesResponseListMainSourcesStruct > srcList = response->getListMainSources();
   std::vector< GetListMainSourcesResponseListMainSourcesStruct >::iterator srcIter;

   DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainSourcesResponse => result=%d, srcList.Count=%d .", ETG_CENUM(dia_AudioManagerResult, response->getResult()), srcList.size());

   mSourceInfoRep.clear();

   for ( srcIter=srcList.begin(); srcIter != srcList.end(); ++srcIter )
   {
      int sourceID           = srcIter->getElem1();           // am_sourceID_t sourceID;
      std::string sourceName = srcIter->getElem2();           // std::string name;
      int sourceAvailability = srcIter->getElem3().getElem1();// am_Availability_s availability;
      int sourceClassID      = srcIter->getElem4();           // am_sourceClass_t sourceClassID;

      DIA_TR_INF("dia_SAFeatureAudioManager:: ID=%d, Avail=%d, srcClassID=%d, Name='%s'", sourceID, sourceAvailability, sourceClassID, sourceName.c_str());

      dia_tAudioSourceType sourceInfo;
      sourceInfo.mSourceID = (tU8) sourceID;
      evaluateSourceName(sourceName,sourceInfo);

      mSourceInfoRep.push_back(sourceInfo);
   }

   getInstanceOfApplication()->postMessage (
         DIA_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
               DIA_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureAudioManager,dia_tAudioSourceType,std::vector > (this,&dia_SAFeatureAudioManager::onGetListMainSourcesResponse,mSourceInfoRep)
         )
   );
}

#endif

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onGetListMainSourcesResponse ( const std::vector<dia_tAudioSourceType>& srcInfoRep )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSourcesResponse");

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainSourcesResponse send list with size=%zu", srcInfoRep.size());
      pListener->vOnAudioSourceList(srcInfoRep);
   }

}

//-----------------------------------------------------------------------------

void dia_SAFeatureAudioManager::onGetListMainSourcesError (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainSourcesError >& /*error*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSourcesError");
}

//-----------------------------------------------------------------------------

#ifdef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

void dia_SAFeatureAudioManager::onGetListMainSinksResponse (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainSinksResponse >& response
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSinksResponse");

   if ( !response ) return;

   std::vector< GetListMainSinksResponseListMainSinksStruct > sinkList = response->getListMainSinks();
   std::vector< GetListMainSinksResponseListMainSinksStruct >::iterator sinkIter;

   DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainSinksResponse => result=%d, SinkList.Count=%d", ETG_CENUM(dia_AudioManagerResult, response->getResult()), sinkList.size());

   mSinkInfoRep.clear();

   for ( sinkIter=sinkList.begin(); sinkIter != sinkList.end(); ++sinkIter )
   {
      int sinkID           = sinkIter->getElem1();
      std::string sinkName = sinkIter->getElem2();
      int sinkAvailability = sinkIter->getElem3().getElem1();
//    int nVolume          = sinkIter->getElem4();
//    int nMuteState       = sinkIter->getElem5();
      int sinkClassID      = sinkIter->getElem6();

      DIA_TR_INF("dia_SAFeatureAudioManager:: ID=%d, Avail=%d, ClassID=%d, Name='%s'", sinkID, sinkAvailability, sinkClassID, sinkName.c_str());

      dia_tAudioSinkType sinkInfo;
      sinkInfo.mSinkID = (tU8) sinkID;
      sinkInfo.mDevID  = 0;
      evaluateSinkName(sinkName,sinkInfo);

      mSinkInfoRep.push_back(sinkInfo);
   }

   getInstanceOfApplication()->postMessage (
         DIA_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
               DIA_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureAudioManager,dia_tAudioSinkType,std::vector>(this,&dia_SAFeatureAudioManager::onGetListMainSinksResponse,mSinkInfoRep)
         )
   );
}

#endif

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onGetListMainSinksResponse( const std::vector<dia_tAudioSinkType>& audioSinkList )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSinksResponse");

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener) {
      pListener->vOnAudioSinkList(audioSinkList);
   }

}

//-----------------------------------------------------------------------------

void dia_SAFeatureAudioManager::onGetListMainSinksError (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainSinksError >& /*error*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSinksError");
}

//-----------------------------------------------------------------------------

#ifdef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

void
dia_SAFeatureAudioManager::onConnectResponse (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< ConnectResponse >& response
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onConnectResponse");

   if ( !response ) return;

   int16 result = response->getResult();
   tU32 connectionID = response->getMainConnectionID();

   DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => result=%d, ConnID=%u", ETG_CENUM(dia_AudioManagerResult, result), connectionID);

   if ( result == DIA_C_S16_AUDIO_CONNECT_RESPONSE_OK )
   {
      DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => E_OK");
      mRequestedConnection.mID = connectionID;
      dia_tAudioConnectionType* pConnection = queryConnectionByEnum(mRequestedConnection.mSource.mSourceEnum,mRequestedConnection.mSink.mSinkEnum);
      if ( pConnection )
      {
         // update the connection ID of stored element
         pConnection->mID = connectionID;
      }
      else
      {
         // store new connection
         mConnectionInfoRep.push_back(mRequestedConnection);
      }
      DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => mRequestedConnection.mID==%d", mRequestedConnection.mID );
      // Wait for MainConnectionStateChanged signal and check for the currently activated connection ID then inform the listener
   }
   else
   {
      DIA_TR_ERR("dia_SAFeatureAudioManager::onConnectResponse => NOK %d", result);

      getInstanceOfApplication()->postMessage (
            OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                  OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tU32>(this,&dia_SAFeatureAudioManager::onAudioSourceConnectionFailed,DIA_E_AUDIO_CONNECTION_FAILED)
            )
      );
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onAudioSourceConnectionFailed ( tDiaResult errCode )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onAudioSourceConnectionFailed(tDiaResult)");

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->vOnAudioConnectionFailure(errCode);
   }
}

#endif

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onConnectError (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< ConnectError >& /*error*/
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onConnectError");

   mRequestedConnection.reset();

   getInstanceOfApplication()->postMessage (
         OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
               OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tDiaResult>(this,&dia_SAFeatureAudioManager::onAudioSourceConnectionFailed,DIA_E_AUDIO_CONNECTION_FAILED)
         )
   );
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::evaluateSourceName ( const std::string& sourceName, dia_tAudioSourceType& srcInfo ) const
{
   srcInfo.mSourceEnum = DIA_EN_AUDIO_SOURCE_UNKNOWN;

   const std::list<dia::AudioSourceMatcher>& audioSourceMatchers = dia::AudioManager::getAudioSourceMatchers();
   std::list<dia::AudioSourceMatcher>::const_iterator iter = audioSourceMatchers.begin();
   for ( ; iter != audioSourceMatchers.end(); ++iter )
   {
      const dia::AudioSourceMatcher& matcher = *iter;
      if ( (matcher.mStartPos >= 0) && (matcher.mEndPos >= 0) )
      {
         if ( sourceName.compare(matcher.mStartPos,matcher.mEndPos,matcher.mNamePattern.c_str()) == 0 )
         {
            srcInfo.mSourceEnum = matcher.mSrcID;
            if ( matcher.mIsDynamicSource )
            {
               srcInfo.mDevID = readDevIDFromSourceID(matcher.mStartPosDeviceID, sourceName);
            }
            break;
         }
      }
      else
      {
         if ( sourceName.compare(matcher.mNamePattern.c_str()) == 0 )
         {
            srcInfo.mSourceEnum = matcher.mSrcID;
            break;
         }
      }
   }
}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::evaluateSinkName ( const std::string& sinkName, dia_tAudioSinkType& sinkInfo ) const
{
   dia_eAudioSink sinkID = DIA_EN_AUDIO_SINK_UNKNOWN;

   if ( sinkName.compare("AMPLIFIER_MAINFIELD") == 0 )
   {
      sinkID = DIA_EN_AUDIO_SINK_AMPLIFIER_MAINFIELD;
   }
   else if( sinkName.compare("AMP_A") == 0)
   {
	   sinkID = DIA_EN_AUDIO_SINK_AMP_A;
   }
   else if( sinkName.compare("AMP_B") == 0)
      {
   	   sinkID = DIA_EN_AUDIO_SINK_AMP_B;
      }
   else
   {
      DIA_TR_ERR("dia_SAFeatureAudioManager::evaluateSinkName sink Name='%s' UNKNOWN!", sinkName.c_str());
      sinkInfo.mSinkEnum = DIA_EN_AUDIO_SINK_UNKNOWN;
   }

   sinkInfo.mSinkEnum = sinkID;
}

//---------------------------------------------------------------------------------------------------------------------

dia_tAudioSourceType*
dia_SAFeatureAudioManager::querySourceByID ( int sourceID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::querySourceByID");

   dia_tAudioSourceType* pSource = 0;
   std::vector<dia_tAudioSourceType>::iterator srcIter = mSourceInfoRep.begin();
   for ( ; srcIter != mSourceInfoRep.end(); ++srcIter )
   {
      if ( srcIter->mSourceID == sourceID )
      {
         pSource = &(*srcIter);
         break;
      }
   }

   return pSource;
}

//---------------------------------------------------------------------------------------------------------------------

dia_tAudioSourceType*
dia_SAFeatureAudioManager::querySourceByEnum ( dia_eAudioSource sourceID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::querySourceByEnum");

   DIA_TR_INF("dia_SAFeatureAudioManager::querySourceByEnum mSourceInfoRep.size=%zu,sourceID=%u", mSourceInfoRep.size(), sourceID);

   dia_tAudioSourceType* pSource = 0;
   std::vector<dia_tAudioSourceType>::iterator srcIter = mSourceInfoRep.begin();
   for ( ; srcIter != mSourceInfoRep.end(); ++srcIter )
   {
      if ( srcIter->mSourceEnum == sourceID )
      {
         pSource = &(*srcIter);
         break;
      }
      else
      {
         DIA_TR_INF("dia_SAFeatureAudioManager::querySourceByEnum srcIter->mSourceEnum=%d", srcIter->mSourceEnum);
      }
   }

   return pSource;
}

//---------------------------------------------------------------------------------------------------------------------

dia_tAudioSinkType*
dia_SAFeatureAudioManager::querySinkByID ( int sinkID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::querySinkByID");

   dia_tAudioSinkType* pSink = 0;
   std::vector<dia_tAudioSinkType>::iterator sinkIter = mSinkInfoRep.begin();
   for ( ; sinkIter != mSinkInfoRep.end(); ++sinkIter )
   {
      if ( sinkIter->mSinkID == sinkID )
      {
         pSink = &(*sinkIter);
         break;
      }
   }

   return pSink;
}

//---------------------------------------------------------------------------------------------------------------------

dia_tAudioSinkType*
dia_SAFeatureAudioManager::querySinkByEnum ( dia_eAudioSink sinkID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::querySinkByEnum");

   dia_tAudioSinkType* pSink = 0;
   std::vector<dia_tAudioSinkType>::iterator sinkIter = mSinkInfoRep.begin();
   for ( ; sinkIter != mSinkInfoRep.end(); ++sinkIter )
   {
      if ( sinkIter->mSinkEnum == sinkID )
      {
         pSink = &(*sinkIter);
         break;
      }
   }

   return pSink;
}

//---------------------------------------------------------------------------------------------------------------------

dia_tAudioConnectionType*
dia_SAFeatureAudioManager::queryConnectionByID ( tU32 connectionID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::queryConnectionByID");

   dia_tAudioConnectionType* pConnection = 0;
   std::vector<dia_tAudioConnectionType>::iterator cIter = mConnectionInfoRep.begin();
   for ( ; cIter != mConnectionInfoRep.end(); ++cIter )
   {
      if ( cIter->mID == connectionID )
      {
         pConnection = &(*cIter);
         break;
      }
   }

   return pConnection;
}

//---------------------------------------------------------------------------------------------------------------------

dia_tAudioConnectionType*
dia_SAFeatureAudioManager::queryConnectionByEnum ( dia_eAudioSource sourceID, dia_eAudioSink sinkID )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::queryConnectionByID(dia_eAudioSource,dia_eAudioSink");

   dia_tAudioConnectionType* pConnection = 0;
   std::vector<dia_tAudioConnectionType>::iterator cIter = mConnectionInfoRep.begin();
   for ( ; cIter != mConnectionInfoRep.end(); ++cIter )
   {
      if ( (cIter->mSource.mSourceEnum == sourceID) && (cIter->mSink.mSinkEnum == sinkID) )
      {
         pConnection = &(*cIter);
         break;
      }
   }

   return pConnection;
}

//-----------------------------------------------------------------------------

tU8
dia_SAFeatureAudioManager::readDevIDFromSourceID ( int startPos, const std::string& sourceName ) const
{
   std::string strDeviceID = sourceName.substr((tU8)startPos/*till the end*/);
   tU8 devID = (tU8) dia::utils::fromString<int>(strDeviceID);
   DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainSourcesResponse devID %d (int).", devID);

   return devID;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef __DIA_NEW_AUDIO_SOURCE_SWITCHING_FSM__

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onGetListMainConnectionsResponse (
      const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetListMainConnectionsResponse >& response
      )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse");

   tBool mainConnectionFound = FALSE;

   if(response)
   {
      std::vector< GetListMainConnectionsResponseListConnectionsStruct > ConnList = response->getListConnections();
      std::vector< GetListMainConnectionsResponseListConnectionsStruct >::iterator it;

      DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse => result=%u, ConnList.Count=%zu", response->getResult(), ConnList.size());

      int nState = CS_UNKNOWN;
      // First check if a CONNECTED connection is available
      for (it = ConnList.begin(); it != ConnList.end() && mainConnectionFound==FALSE; it++)
      {
         int nConnID = it->getElem1();
         int nSrcID = it->getElem2();
         int nSinkID = it->getElem3();
   //      int nDelay = it->getElem4();
         nState = it->getElem5();

         DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResp.=> ID=%d, SourceID=%d, SinkID=%d, State=%d", nConnID, nSrcID, nSinkID, nState );
         DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse current connectionId == 0x%02x .", mRequestedConnection.mID );

         if (nState == CS_CONNECTED )
         {
            DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse cmp nState(%d) == CS_CONNECTED(%d).", nState, CS_CONNECTED );
            DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResp.use ID=%d, SourceID=%d, SinkID=%d, State=%d", nConnID, nSrcID, nSinkID, nState);

            getInstanceOfApplication()->postMessage (
                  OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                        OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tU8>(this,&dia_SAFeatureAudioManager::onGetListMainConnectionsResponse,(tU8)nSrcID)
                  )
            );
            mainConnectionFound = TRUE;
         } 
         else 
         {
            DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse cmp nState(%d) != CS_CONNECTED(%d).", nState, CS_CONNECTED );
         }
      }

      // Second check if a CONNECT_ING connection is available
      for (it = ConnList.begin(); it != ConnList.end() && mainConnectionFound==FALSE; it++)
      {
         int nConnID = it->getElem1();
         int nSrcID = it->getElem2();
         int nSinkID = it->getElem3();
   //      int nDelay = it->getElem4();
         nState = it->getElem5();

         DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResp.=> ID=%d, SourceID=%d, SinkID=%d, State=%d", nConnID, nSrcID, nSinkID, nState );

         if (nState == CS_CONNECTING)
         {
            DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse cmp nState(%d) == CS_CONNECTING(%d) .", nState, CS_CONNECTING );
            DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResp.use ID=%d, SourceID=%d, SinkID=%d, State=%d", nConnID, nSrcID, nSinkID, nState );

            getInstanceOfApplication()->postMessage (
                  OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                        OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tU8>(this,&dia_SAFeatureAudioManager::onGetListMainConnectionsResponse,(tU8)nSrcID)
                  )
            );
            mainConnectionFound = TRUE;
         } 
         else 
         {
            DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse cmp nState(%d) != CS_CONNECTING(%d) .", nState, CS_CONNECTING );
         }
      }

      if(mainConnectionFound==FALSE)
      {
         DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse NOT HANDLED nState(%d) (no connected or connecting) .", nState);
      }

      // request sources list to match dynamic main source to enum
      if((mpSrvPlugin) && (mpSrvPlugin->getProxy()))
      {
         mpSrvPlugin->getProxy()->sendGetListMainSourcesRequest(*this);
      }
   }

}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onGetListMainConnectionsResponse ( tU8 audioSrcId )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainConnectionsResponse");

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->vOnAudioMainSourceConnection((tU8)audioSrcId);
   }

}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onMainConnectionStateChangedSignal(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< MainConnectionStateChangedSignal >& signal)
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onMainConnectionStateChangedSignal");

   DIA_TR_INF("dia_SAFeatureAudioManager::onMainConnectionStateChangedSignal => ID = %d, State = %d", signal->getConnectionID(), signal->getConnectionState());

   if ( (signal) && ( mRequestedConnection.mID == signal->getConnectionID() ) && (mpSrvPlugin) && (mpSrvPlugin->getProxy()) )
   {
      if ((signal->getConnectionState() == CS_CONNECTED) )
      {
         DIA_TR_INF("REQUESTED SORCE IS NOW CONNECTED");
         getInstanceOfApplication()->postMessage (
               OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                     OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tU32>(this,&dia_SAFeatureAudioManager::onAudioSourceConnectionDone,DIA_SUCCESS)
               )
            );
      }
#if 1
      // Workaround until a connection really goes to "connected"
      else if (signal->getConnectionState() == CS_CONNECTING)
      {
         DIA_TR_INF("STILL CONNECTING, WAIT FOR THE NEXT SIGNAL NOTIFICATION");
      }
#endif
   }
}

void dia_SAFeatureAudioManager::onGetListMainSourcesResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainSourcesResponse >& response)
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSourcesResponse");

   if(response)
   {
   std::vector< GetListMainSourcesResponseListMainSourcesStruct > srcList = response->getListMainSources();
   std::vector< GetListMainSourcesResponseListMainSourcesStruct >::iterator it;

   DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainSourcesResponse => result=%u, srcList.Count=%zu .", response->getResult(), srcList.size());


   std::vector<dia_tAudioSourceType> sourceIDMatchingList;

   for (it=srcList.begin(); it != srcList.end(); it++)
   {
      int nSrcID   = it->getElem1();   // am_sourceID_t sourceID;
//    const char* strSourceName = (it->getElem2()).c_str(); // std::string name;
      std::string strSourceName = it->getElem2(); // std::string name;
      int nSrcAvailability = it->getElem3().getElem1();  // am_Availability_s availability;
      int nClassID = it->getElem4();   // am_sourceClass_t sourceClassID;

      DIA_TR_INF("dia_SAFeatureAudioManager:: ID=%d, Avail=%d, srcClassID=%d, Name='%s'", nSrcID, nSrcAvailability, nClassID, strSourceName.c_str());

      dia_tAudioSourceType sourceListeItem;
      sourceListeItem.mSourceID   = (tU8)nSrcID;
      sourceListeItem.mDevID      = 0;       // max 255 from component

      if(strSourceName.compare("AUX_1")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_AUX_1;
      } else if(strSourceName.compare("AUX_2")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_AUX_2;
      } else if(strSourceName.compare("TUNER_AM")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_AM;
      } else if(strSourceName.compare("TUNER_DAB")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_DAB;
      } else if(strSourceName.compare("TUNER_FM")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_FM;
      } else if(strSourceName.compare("TUNER_TA_DAB")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_TA_DAB;
      } else if(strSourceName.compare("TUNER_TA_FM")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_TA_FM;
      } else if(strSourceName.compare("TUNER_XM")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_XM;
      } else if(strSourceName.compare("MEDIA_CDDA")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_MEDIA_CDDA;
      } else if(strSourceName.compare(0,13,"MEDIA_PLAYER#")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_MEDIA_PLAYER;
         sourceListeItem.mDevID = readDevIDFromSourceID(13, strSourceName);
      } else if(strSourceName.compare("NAVI_SPEECH")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_NAVI_SPEECH;
      } else if(strSourceName.compare(0, 6,"PHONE#")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE;
         sourceListeItem.mDevID = readDevIDFromSourceID( 6, strSourceName);
      } else if(strSourceName.compare(0,14,"PHONE_BTAUDIO#")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE_BTAUDIO;
         sourceListeItem.mDevID = readDevIDFromSourceID(14, strSourceName);
      } else if(strSourceName.compare("PHONE_ECALL")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE_ECALL;
      } else if(strSourceName.compare("PHONE_EXTERNAL_1")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE_EXTERNAL_1;
      } else if(strSourceName.compare("PHONE_OUTBAND_RING")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE_OUTBAND_RING;
      } else if(strSourceName.compare(0,18,"PHONE_INBAND_RING#")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE_INBAND_RING;
         sourceListeItem.mDevID = readDevIDFromSourceID(18, strSourceName);
      } else if(strSourceName.compare("PHONE_SPEECHRECOGNITION")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_PHONE_SPEECHRECOGNITION;
      } else if(strSourceName.compare("SDS_SPEECHRECOGNITION")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_SDS_SPEECHRECOGNITION;
      } else if(strSourceName.compare("SMS_SPEECH")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_SMS_SPEECH;
      } else if(strSourceName.compare("SPI_ENTERTAIN")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_SPI_ENTERTAIN;
      } else if(strSourceName.compare("SPI_INFO")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_SPI_INFO;
      } else if(strSourceName.compare("SPI_PHONE")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_SPI_PHONE;
      } else if(strSourceName.compare("SPI_SPEECHRECOGNITION")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_SPI_SPEECHRECOGNITION;
      } else if(strSourceName.compare("MICROPHONE")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_MICROPHONE;
      } else if(strSourceName.compare("INTERNET_APPS")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_INTERNET_APPS;
      } else if(strSourceName.compare("TUNER_DRM")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_TUNER_DRM;
      } else if(strSourceName.compare("TCU_SERVICE_CALL")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_DCM;
      } else if(strSourceName.compare("MICROPHONE2")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_MICROPHONE2;
      } else if(strSourceName.compare("EXTERNAL_CD")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_ENTERTAIN_EXT_CD;
      } else if(strSourceName.compare("SPARE_CD")==0){
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_ENTERTAIN_SPARE_CD;
      } else {
         DIA_TR_ERR("dia_SAFeatureAudioManager::onGetListMainSourcesResponse source Name='%s' UNKNOWN!", strSourceName.c_str());
         sourceListeItem.mSourceEnum = DIA_EN_AUDIO_SOURCE_UNKNOWN;
      }

      sourceIDMatchingList.push_back(sourceListeItem);
   }

   getInstanceOfApplication()->postMessage (
         OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
               OSAL_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureAudioManager,dia_tAudioSourceType,std::vector>
                 (this,&dia_SAFeatureAudioManager::onGetListMainSourcesResponse,sourceIDMatchingList)
         )
   );

}

}

void dia_SAFeatureAudioManager::onGetListMainSinksResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< GetListMainSinksResponse >& response)
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onGetListMainSinksResponse");

   if(response)
   {
   std::vector< GetListMainSinksResponseListMainSinksStruct > SinkList = response->getListMainSinks();
   std::vector< GetListMainSinksResponseListMainSinksStruct >::iterator it;

   DIA_TR_INF("dia_SAFeatureAudioManager::onGetListMainSinksResponse => result=%u, SinkList.Count=%zu", response->getResult(), SinkList.size());

   std::vector<dia_tAudioSinkType> sinkIDMatchingList;

   for (it=SinkList.begin(); it != SinkList.end(); it++)
   {
      int nSinkID = it->getElem1();
//      const char* strSinkName = (it->getElem2()).c_str();
      std::string strSinkName = it->getElem2();
      int nSinkAvailability = it->getElem3().getElem1();
//      int nVolume = it->getElem4();
//      int nMuteState = it->getElem5();
      int nClassID = it->getElem6();

      DIA_TR_INF("dia_SAFeatureAudioManager:: ID=%d, Avail=%d, ClassID=%d, Name='%s'", nSinkID, nSinkAvailability, nClassID, strSinkName.c_str());
      dia_tAudioSinkType sinkListeItem;
      sinkListeItem.mSinkID = (tU8)nSinkID;
      sinkListeItem.mDevID  = 0;

      if (strSinkName.compare("AMPLIFIER_MAINFIELD") == 0) {
         sinkListeItem.mSinkEnum = DIA_EN_AUDIO_SINK_AMPLIFIER_MAINFIELD;
      } else {
         DIA_TR_ERR("dia_SAFeatureAudioManager::onGetListMainSinksResponse sink Name='%s' UNKNOWN!", strSinkName.c_str());
         sinkListeItem.mSinkEnum = DIA_EN_AUDIO_SINK_UNKNOWN;
      }

      sinkIDMatchingList.push_back(sinkListeItem);
   }

   getInstanceOfApplication()->postMessage (
         OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
               OSAL_NEW dia_FunctorOneTemplateArgNoReturnValue<dia_SAFeatureAudioManager,dia_tAudioSinkType,std::vector>
              (this,&dia_SAFeatureAudioManager::onGetListMainSinksResponse,sinkIDMatchingList)
         )
   );
}

}

//---------------------------------------------------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onAudioSourceConnectionFailed ( tDiaResult errCode )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onAudioSourceConnectionFailed(tDiaResult)");

   DIA_TR_INF("dia_SAFeatureAudioManager::onAudioSourceConnectionFailed errCode=0x%08X", errCode );

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->vOnAudioSourceConnectionDone(errCode);
   }
}

//-----------------------------------------------------------------------------

void
dia_SAFeatureAudioManager::onConnectResponse(const ::boost::shared_ptr< CommandInterfaceProxy >& /*proxy*/, const ::boost::shared_ptr< ConnectResponse >& response)
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onConnectResponse");

   if(response)
   {
      DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => result=%d, ConnID=%d", response->getResult(), response->getMainConnectionID());
      int16 result = response->getResult();
      tU32 connectionID = response->getMainConnectionID();

      if( result == 0 ) // check for "E_OK" response
      {
         DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => E_OK");
         mRequestedConnection.mID = connectionID;
         dia_tAudioConnectionType* pConnection = queryConnectionByID(connectionID );
         if ( pConnection ) { pConnection->mID = connectionID; }
         DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => mRequestedConnection.mID==%d", mRequestedConnection.mID );
//         mControlledConnectionId = response->getMainConnectionID();
//         DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => mControlledConnectionId==%d", mControlledConnectionId );
         // Wait for MainConnectionStateChanged signal and check for the currently activated connection ID then inform the listener
      }
      else if( result == 5 ) // check for "E_ALREADY_EXISTS" response
      {
         DIA_TR_INF("dia_SAFeatureAudioManager::onConnectResponse => E_ALREADY_EXISTS");

         // The requested source is already active, send back the positive acknowledgement to tester.
            getInstanceOfApplication()->postMessage (
                  OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                        OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tU32>(this,&dia_SAFeatureAudioManager::onAudioSourceConnectionDone,DIA_SUCCESS)
                  )
               );
         // }
      }
      else
      {
         DIA_TR_ERR("dia_SAFeatureAudioManager::onConnectResponse => NOK %d", result);

         getInstanceOfApplication()->postMessage (
               OSAL_NEW dia_tclDiagSession::tclEventIntMsgRxGeneric (
                     OSAL_NEW dia_FunctorOneArgNoReturnValue<dia_SAFeatureAudioManager,tU32>(this,&dia_SAFeatureAudioManager::onAudioSourceConnectionDone,DIA_FAILED)
               )
         );
      }
   }
}

void
dia_SAFeatureAudioManager::onAudioSourceConnectionDone ( tU32 status )
{
   dia::ScopeTrace trc("dia_SAFeatureAudioManager::onAudioSourceConnectionDone");

   dia_IAudioSourceListener* pListener = 0;
   if ((querySysAdapterListener<dia_IAudioSourceListener>(&pListener) == DIA_SUCCESS) && pListener)
   {
      pListener->vOnAudioSourceConnectionDone(status);
   }
}

#endif
