/*!
 * \file       dia_SecurityManager.cpp
 *
 * \brief      tbd
 *
 * \details    tbd
 *
 * \component  Diagnosis
 *
 * \ingroup    diaCoreSecurity
 *
 * \copyright  (c) 2015 Robert Bosch Car Multimedia
 *
 */

//----------------------------------------------------------------------------

#ifndef __INCLUDED_DIA_COMMON__
#include "common/framework/application/dia_common.h"
#endif

#ifndef __INCLUDED_DIA_SYSTEM_ADAPTER_FACADE__
#include "common/framework/sysadapters/dia_SystemAdapterFacade.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY__
#include "common/framework/security/dia_security.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_MANAGER__
#include "common/framework/security/dia_SecurityManager.h"
#endif

#ifndef __INCLUDED_DIA_SECURITY_LEVEL__
#include "common/framework/security/dia_SecurityLevel.h"
#endif

#ifndef __DIA_UNIT_TESTING__

dia_SecurityManager*
getInstanceOfSecurityManager ( void )
{
   return dia_SecurityManager::getInstance();
}

void
releaseInstanceOfSecurityManager ( void )
{
   dia_SecurityManager::deleteInstance();
}

#endif

DIA_IMPL_SINGLETON(dia_SecurityManager)

//------------------------------------------------------------------------------

dia_SecurityManager::dia_SecurityManager ( void )
{
    dia_tclFnctTrace trc("dia_SecurityManager::dia_SecurityManager");
}

//------------------------------------------------------------------------------

dia_SecurityManager::~dia_SecurityManager ( void )
{
   _BP_TRY_BEGIN
   {
      mLevelRep.clear();
      mAccessTypeRep.clear();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_SecurityManager::~dia_SecurityManager !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//------------------------------------------------------------------------------

tBool
dia_SecurityManager::addSecurityLevel ( dia_SecurityLevel* pLevel )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::addSecurityLevel()");

   tBool bRetCode = FALSE;

   if ( pLevel )
   {
      tCString levelName = pLevel->getName();

      std::map<dia_UID,dia_SecurityLevel*>::iterator levelIter = mLevelRep.find(pLevel->getUID());
      if ( levelIter == mLevelRep.end() )
      {
         DIA_TR_INF("ADDING SECURITY LEVEL \"%s\"", levelName);

         mLevelRep[pLevel->getUID()] = pLevel;

         std::vector<tU16> accessTypes;
         pLevel->getConfiguration().getAccessTypes(accessTypes);
         for ( std::vector<tU16>::iterator vecIter = accessTypes.begin(); vecIter != accessTypes.end(); vecIter++ )
         {
            tU16 accessType = (*vecIter);

            // add the access type if it was not already added before
            std::map<tU16,dia_SecurityLevel*>::iterator accessTypeIter = mAccessTypeRep.find(accessType);
            if ( accessTypeIter == mAccessTypeRep.end() )
            {
               mAccessTypeRep[accessType] = pLevel;
               DIA_TR_INF("ADDED ACCESS TYPE 0x%04x FOR SECURITY LEVEL \"%s\"", accessType, levelName);
            }
            else
            {
               DIA_TR_INF("ACCESS TYPE 0x%04x FOR SECURITY LEVEL \"%s\" NOT ADDED (ALREADY EXISTS) !!", accessType, levelName);
            }
         }

         bRetCode = TRUE;
      }
   }

   return bRetCode;
}

//------------------------------------------------------------------------------

tBool
dia_SecurityManager::removeSecurityLevel ( dia_UID uid )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::removeSecurityLevel()");

   tBool retCode = FALSE;

   std::map<dia_UID,dia_SecurityLevel*>::iterator iter = mLevelRep.find(uid);

   if ( iter != mLevelRep.end() )
   {
      dia_SecurityLevel* pLevel = iter->second;

      // erase all access types for the given level from the access type repository
      std::vector<tU16> accessTypes;
      pLevel->getConfiguration().getAccessTypes(accessTypes);
      for ( std::vector<tU16>::iterator vecIter = accessTypes.begin(); vecIter != accessTypes.end(); vecIter++ )
      {
         tU16 accessType = (*vecIter);

         std::map<tU16,dia_SecurityLevel*>::iterator accessIter = mAccessTypeRep.find(accessType);
         if ( (accessIter != mAccessTypeRep.end()) && (iter->second == pLevel) )
         {
            mAccessTypeRep.erase(accessType);
         }
      }

      // erase the object pointer from the level repository
      mLevelRep.erase(uid);

      retCode = TRUE;
   }

   return retCode;
}

//------------------------------------------------------------------------------

void
dia_SecurityManager::onSecurityLevelUpdate ( dia_SecurityLevel& secLevel )
{
   bool needToNotify = false;
   dia_eSecurityLevelStatus oldStatus = DIA_EN_SECURITY_LEVEL_STATUS_UNKNOWN;

   std::list<dia_UID>::iterator activeIter = std::find(mActiveLevelRep.begin(),mActiveLevelRep.end(),secLevel.getUID());

   if ( secLevel.isActive() && (activeIter == mActiveLevelRep.end()) )
   {
      // security level state changed from 'not active' to 'active'
      mActiveLevelRep.push_back(secLevel.getUID());
      oldStatus = DIA_EN_SECURITY_LEVEL_STATUS_NOT_ACTIVE;
      needToNotify = true;
   }
   else if ( !(secLevel.isActive()) && (activeIter != mActiveLevelRep.end()) )
   {
      // security level state changed from 'active' to 'not active'
      mActiveLevelRep.erase(activeIter);
      oldStatus = DIA_EN_SECURITY_LEVEL_STATUS_ACTIVE;
      needToNotify = true;
   }
   else
   {
      // nothing to do
   }

   if ( needToNotify )
   {
      std::list<dia_ISecurityListener*>::iterator listenerIter = mListenerRep.begin();
      for ( ; listenerIter != mListenerRep.end(); ++listenerIter )
      {
         (*listenerIter)->vOnSecurityLevelChange(secLevel,secLevel.getStatus(),oldStatus);
      }
   }
}

//------------------------------------------------------------------------------

tBool
dia_SecurityManager::addListener ( dia_ISecurityListener* pListener )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::addListener()");

   tBool bRetCode = FALSE;

   if ( pListener )
   {
      // check if the corresponding object is already registered
      std::list<dia_ISecurityListener*>::iterator iter;
      iter = find(mListenerRep.begin(), mListenerRep.end(), pListener);
      if ( iter == mListenerRep.end() )
      {
         mListenerRep.push_back(pListener);
         bRetCode = TRUE;
      }
   }

   return bRetCode;
}

//------------------------------------------------------------------------------

tBool
dia_SecurityManager::removeListener ( dia_ISecurityListener* pListener )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::removeListener()");

   tBool bRetCode = FALSE;

   if ( pListener )
   {
      // check if the corresponding object is already registered
      std::list<dia_ISecurityListener*>::iterator iter;
      iter = find(mListenerRep.begin(), mListenerRep.end(), pListener);
      if ( iter != mListenerRep.end() )
      {
         mListenerRep.erase(iter);
         bRetCode = TRUE;
      }
   }

   return bRetCode;
}

//------------------------------------------------------------------------------

void
dia_SecurityManager::getActiveLevels ( std::list<dia_SecurityLevel*>& activeLevels )
{
   std::list<dia_UID>::iterator activeIter = mActiveLevelRep.begin();
   for ( ; activeIter != mActiveLevelRep.end(); ++activeIter )
   {
      dia_SecurityLevel* pSecLevel = pGetLevel(*activeIter);
      if ( pSecLevel )
      {
         activeLevels.push_back(pSecLevel);
      }
   }
}

//------------------------------------------------------------------------------

dia_SecurityLevel*
dia_SecurityManager::pGetLevel ( dia_UID uid )
{
   dia_SecurityLevel* pLevel = 0;

   std::map<dia_UID,dia_SecurityLevel*>::iterator iter = mLevelRep.find(uid);
   if ( iter != mLevelRep.end() )
   {
      pLevel = mLevelRep[uid];
   }

   return pLevel;
}

//-----------------------------------------------------------------------------

dia_SecurityLevel*
dia_SecurityManager::pGetLevelUDS ( tU8 udsID )
{
   dia_tclFnctTrace trc("dia_SecurityManager::pGetLevelUDS");

   dia_SecurityLevel* pLevel = 0;

   std::map<tU16,dia_SecurityLevel*>::iterator iter = mAccessTypeRep.find(udsID);

   if ( iter != mAccessTypeRep.end() )
   {
      pLevel = mAccessTypeRep[udsID];
   }

   return pLevel;
}

//------------------------------------------------------------------------------

void
dia_SecurityManager::vOnSessionChanged ( tU8 newSession, tU8 oldSession )
{
    dia_tclFnctTrace trc("dia_SecurityManager::vOnSessionChanged");

    std::map<dia_UID,dia_SecurityLevel*>::iterator iter = mLevelRep.begin();
    for ( ; iter != mLevelRep.end(); iter++ )
    {
       dia_SecurityLevel* pActLevel = iter->second;
       if ( pActLevel != 0 ) pActLevel->onSessionChange(newSession,oldSession);
    }
}

//-----------------------------------------------------------------------------

dia_eSecurityLevelStatus
dia_SecurityManager::getStatus ( dia_UID uid )
{
   dia_SecurityLevel* pLevel = pGetLevel(uid);
   return ( pLevel ) ? pLevel->getStatus() : DIA_EN_SECURITY_LEVEL_STATUS_UNKNOWN;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_SecurityManager::initialize ( void )
{
   dia_tclFnctTrace trc("dia_SecurityManager::initialize");

   std::map<dia_UID,dia_SecurityLevel*>::iterator iter = mLevelRep.begin();
   for ( ; iter != mLevelRep.end(); iter++ )
   {
      dia_SecurityLevel* pActLevel = iter->second;
      if ( pActLevel != 0 ) pActLevel->initialize();
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityManager::lock ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::lock()");

   std::map<dia_UID,dia_SecurityLevel*>::iterator iter = mLevelRep.begin();
   for ( ; iter != mLevelRep.end(); iter++ )
   {
      dia_SecurityLevel* pActLevel = iter->second;
      if ( pActLevel != 0 ) pActLevel->lock();
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityManager::unlock ( void )
{
   dia_tclFnctTrace oTrace("dia_SecurityManager::unlock()");

   std::map<dia_UID,dia_SecurityLevel*>::iterator iter = mLevelRep.begin();
   for ( ; iter != mLevelRep.end(); iter++ )
   {
      dia_SecurityLevel* pActLevel = iter->second;
      if ( pActLevel != 0 ) pActLevel->unlock();
   }

   return DIA_SUCCESS;
}

//------------------------------------------------------------------------------

tDiaResult
dia_SecurityManager::getAccessTypes ( std::vector<tU16>& accessTypes )
{
   std::map<tU16,dia_SecurityLevel*>::iterator iter = mAccessTypeRep.begin();
   for ( ; iter != mAccessTypeRep.end(); iter++ )
   {
      dia_SecurityLevel* pLevel = iter->second;
      if ( pLevel != 0 ) pLevel->getConfiguration().getAccessTypes(accessTypes);
   }

   return DIA_SUCCESS;
}
